"""
Debug dialogs pro zobrazení raw API requestů a odpovědí
"""

import xbmcgui
import xbmcaddon
import json
import xbmc

def is_debug_mode_enabled():
    """Kontroluje, zda je zapnutý debug mód."""
    addon = xbmcaddon.Addon()
    return addon.getSetting("ai_debug_mode") == "true"

# NOVÁ funkce pro debug request s potvrzením
# Vrací True pokud uživatel potvrdil Odeslat, jinak False

def show_api_request_debug(provider, model, request_data):
    if not is_debug_mode_enabled():
        return True  # Debug není aktivní, pokračuj
    try:
        json_formatted = json.dumps(request_data, indent=2, ensure_ascii=False)
    except:
        json_formatted = str(request_data)
    # Přidání API endpointu podle providera
    if provider.lower().startswith("openai"):
        api_endpoint = "https://api.openai.com/v1/responses"
    elif provider.lower().startswith("anthropic"):
        api_endpoint = "https://api.anthropic.com/v1/messages"
    elif provider.lower().startswith("google"):
        # Google endpoint je dynamický podle modelu, ale základ:
        api_endpoint = "https://generativelanguage.googleapis.com/v1beta/models/[model]:generateContent"
    elif provider.lower().startswith("mistral"):
        api_endpoint = "https://api.mistral.ai/v1/chat/completions"
    else:
        api_endpoint = "neznámý"
    # Titulek dialogu s názvem poskytovatele
    title = f"Odeslat API dotaz na {provider}"
    message = f"API ENDPOINT: {api_endpoint}\nRAW API REQUEST:\n{json_formatted}\n\nTento request bude odeslán na {provider} API.\nPotvrďte Odeslat nebo Zrušit."
    result = {'send': False}
    def on_send():
        result['send'] = True
    def on_cancel():
        result['send'] = False
    show_api_request_debug_dialog(message, on_send, on_cancel, title=title, dialogmode='odesilani')
    return result['send']

def show_api_response_debug(provider, model, response_data, response_text=None):
    if not is_debug_mode_enabled():
        return True
    try:
        json_formatted = json.dumps(response_data, indent=2, ensure_ascii=False)
    except:
        json_formatted = str(response_data)
    title = f"Přijata odpověď od {provider}"
    message = f"RAW API RESPONSE:\n{json_formatted}"
    if response_text:
        message += f"\n\nEXTRACTED TEXT:\n{'-'*30}\n{response_text}\n{'-'*30}"
    message += "\n\nPokračujte stisknutím OK."
    result = {'cont': False}
    def on_continue():
        result['cont'] = True
    def on_cancel():
        result['cont'] = False
    show_api_request_debug_dialog(message, on_continue, on_cancel, title=title, dialogmode='response')
    return result['cont']

def show_api_error_debug(provider, model, error_message, error_data=None):
    """
    Zobrazí debug dialog s chybou API.
    
    Args:
        provider (str): Název poskytovatele
        model (str): Název modelu
        error_message (str): Chybová zpráva
        error_data (dict, optional): Raw error data
    """
    if not is_debug_mode_enabled():
        return
    
    title = f"DEBUG: {provider} API Error"
    
    message = f"""Model: {model}

ERROR MESSAGE:
{error_message}

"""
    
    if error_data:
        try:
            json_formatted = json.dumps(error_data, indent=2, ensure_ascii=False)
            message += f"""RAW ERROR DATA:
{'-' * 50}
{json_formatted}
{'-' * 50}

"""
        except:
            message += f"RAW ERROR DATA: {str(error_data)}\n\n"
    
    message += "Stiskněte OK pro pokračování."
    
    # Zobrazí dialog a čeká na potvrzení
    dialog = xbmcgui.Dialog()
    dialog.textviewer(title, message) 

class AIDebugRequestDialog(xbmcgui.WindowXMLDialog):
    def __init__(self, *args, **kwargs):
        super().__init__(*args)
        self.request_text = None
        self.on_send = None
        self.on_cancel = None
        self.addon = xbmcaddon.Addon()
        self.closed = False

    def onInit(self):
        try:
            textbox = self.getControl(7100)
            if self.request_text:
                textbox.setText(self.request_text)
            # Dynamicky nastav header pokud je property
            header = self.getProperty('header')
            if header:
                label = self.getControl(2000)
                label.setLabel(header)
            # Nastavím fokus na správné tlačítko podle property
            defaultcontrol = self.getProperty('defaultcontrol')
            if defaultcontrol:
                try:
                    self.setFocusId(int(defaultcontrol))
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] AIDebugRequestDialog: Nelze nastavit fokus na defaultcontrol {defaultcontrol}: {e}", xbmc.LOGWARNING)
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] AIDebugRequestDialog: Nelze nastavit text nebo header: {e}", xbmc.LOGWARNING)

    def onClick(self, controlId):
        if controlId == 7002:  # Odeslat
            self.closed = True
            self.close()
            if self.on_send:
                self.on_send()
        elif controlId == 7001:  # Zrušit
            self.closed = True
            self.close()
            if self.on_cancel:
                self.on_cancel()

    def onAction(self, action):
        # ESC nebo BACK = Zrušit
        if action in (9, 10):
            if not self.closed:
                self.closed = True
                self.close()
                if self.on_cancel:
                    self.on_cancel()

# Funkce pro zobrazení debug dialogu s raw requestem
# on_send a on_cancel jsou callbacky bez argumentů

def show_api_request_debug_dialog(request_text, on_send, on_cancel, title=None, dialogmode=None):
    import os
    import xbmc
    addon = xbmcaddon.Addon()
    xml_path = os.path.join(addon.getAddonInfo('path'), 'resources', 'skins', 'Default', '1080i', 'ai/DialogAIDebugRequest.xml')
    try:
        dialog = AIDebugRequestDialog('ai/DialogAIDebugRequest.xml', addon.getAddonInfo('path'), 'Default', '1080i')
        dialog.request_text = request_text
        dialog.on_send = on_send
        dialog.on_cancel = on_cancel
        if title:
            dialog.setProperty('header', title)
        if dialogmode:
            dialog.setProperty('dialogmode', dialogmode)
            # Nastavím defaultcontrol property podle režimu
            if dialogmode == 'response':
                dialog.setProperty('defaultcontrol', '7001')
            else:
                dialog.setProperty('defaultcontrol', '7002')
        dialog.doModal()
        del dialog
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Chyba při otevírání ai/DialogAIDebugRequest.xml: {e}", xbmc.LOGERROR)
        xbmcgui.Dialog().ok("Chyba", f"Nelze otevřít debug dialog: {e}") 