# wcs/search_movies.py

import sys
import re
import difflib
import xml.etree.ElementTree as ET
from urllib.parse import quote_plus

import xbmcaddon
import xbmcplugin
import xbmcgui
import xbmc

# Importy našich modulů
from wcs.webshare import WebshareClient as webshare_api
from wcs.ui import UI as ui
from wcs import utils, user_data
from wcs.utils import calculate_advanced_score

# Konstanty přesunuté z router.py
CATEGORIES = ['', 'video', 'images', 'audio', 'archives', 'docs', 'adult']
SORTS = ['', 'recent', 'rating', 'largest', 'smallest']

# === FILMOVĚ SPECIFICKÉ FUNKCE ===

def calculate_quality_score(filename, size_bytes, votes):
    """Vypočítá celkové skóre kvality pro seřazení."""
    score = 0
    try:
        size_bytes = int(size_bytes) if size_bytes else 0
    except (ValueError, TypeError):
        size_bytes = 0
    
    # OPRAVA: Používáme centrální extract_quality() místo duplicitní logiky
    from wcs.utils import extract_quality
    quality = extract_quality(filename)
    filename_lower = filename.lower()  # Stále potřebné pro bonusy
    
    # Kvalita (nejvyšší priorita) - mapování podle centrální detekce
    if quality == 'REMUX':
        score += 1000
    elif quality == '4K':
        score += 900
    elif quality == '1080p':
        score += 800
    elif quality == '720p':
        score += 700
    elif quality == '480p':
        score += 600
    elif quality == '360p':
        score += 500
    # SD soubory bez bonusu
    
    # Bonusy
    if 'hdr' in filename_lower:
        score += 100
    if re.search(r'x265|hevc', filename_lower):
        score += 50
    if 'bluray' in filename_lower:
        score += 30
    
    # Velikost korelace (pro stejnou kvalitu preferuj větší)
    if size_bytes > 40 * 1024**3:  # 40GB+
        score += 50
    elif size_bytes > 15 * 1024**3:  # 15GB+
        score += 30
    elif size_bytes > 5 * 1024**3:   # 5GB+
        score += 20
    
    # User rating
    try:
        positive = int(votes.get('positive_votes', 0))
        negative = int(votes.get('negative_votes', 0))
        if positive > negative:
            score += min(25, positive * 5)
    except (ValueError, TypeError):
        pass
    
    return score

# calculate_advanced_score je nyní v utils.py jako sdílená funkce

def sort_files_by_priority(files, query, filter_series=False):
    """
    Seřadí soubory podle pokročilého skórování s prioritami.
    
    Args:
        files (list): Seznam Webshare souborů
        query (str): Vyhledávací dotaz
        filter_series (bool): Pokud True, filtruje seriály (pro filmy)
        
    Returns:
        list: Seřazené soubory (nejlepší první)
    """
    from wcs import utils
    
    def score_with_filter(item):
        # Pro filmy - filtrujeme seriály
        if filter_series:
            name = item.get('name', '').lower()
            if utils.extract_season_episode(name) != (None, None):
                return 100  # Nízké skóre pro seriály
        
        return calculate_advanced_score(item, query)
    
    # Seřazení podle skóre (sestupně)
    files.sort(key=score_with_filter, reverse=True)
    return files

# === CENTRÁLNÍ FUNKCE PRO VYHLEDÁVÁNÍ A VÝBĚR FILMŮ ===

def search_and_select_movie_file(search_query, title='', year='', plot='', poster='', fanart='', rating='', genre='', director='', runtime='', addon=None, tmdb_id=None, from_library=False):
    """
    CENTRÁLNÍ funkce pro vyhledávání a výběr souboru filmu.
    Používá se ze všech míst - menu doplňku, knihovny, TMDb.
    
    Args:
        search_query (str): Dotaz pro vyhledávání na Webshare
        title (str): Název filmu
        year (str): Rok filmu
        plot (str): Popis
        poster (str): URL obrázku
        fanart (str): URL fanart
        rating (str): Hodnocení
        genre (str): Žánr
        director (str): Režisér
        runtime (str): Délka
        addon: Instance addonu
        tmdb_id (int): TMDb ID
        from_library (bool): Volá se z knihovny Kodi
        
    Returns:
        dict or None: Vybraný soubor nebo None
    """
    if not addon:
        addon = xbmcaddon.Addon()
    
    # Kontrola přihlášení
    if not user_data._check_credentials_and_prompt(addon, lambda: None):
        return None
    
    # Ověření tokenu
    token = webshare_api.revalidate(addon, ui.popinfo)
    if not token:
        return None
    
    # Vyhledání souborů na Webshare
    try:
        response = webshare_api.api_call('search', {
            'what': search_query,
            'wst': token,
            'category': 'video',
            'limit': 50,
            'offset': 0
        })
        xml = ET.fromstring(response.content)
        if not webshare_api.is_ok(xml):
            ui.popinfo(f"Chyba při vyhledávání na Webshare", addon.getAddonInfo('name'))
            return None
        
        files = []
        for file in xml.iter('file'):
            files.append({e.tag: e.text for e in file})
        
        if not files:
            ui.popinfo(f"Nebyly nalezeny žádné soubory pro '{search_query}'", addon.getAddonInfo('name'))
            return None
        
        # Seřazení souborů podle naší logiky
        files = sort_files_by_priority(files, search_query)
        
        # OPRAVA: Vždy používat search_dialog.py pro jednotnou logiku
        from wcs.ui.dialogs.DialogWcsSearch import show_search_dialog
        
        selected_file = show_search_dialog(
            search_query=search_query,
            movie_title=title or search_query,
            movie_year=year,
            poster_url=poster,
            rating=rating,
            genre=genre,
            director=director,
            runtime=runtime,
            plot=plot,
            search_results=files,
            is_episode=False,
            tmdb_id=tmdb_id
        )
        
        return selected_file
        
    except Exception as e:
        ui.popinfo(f"Chyba při vyhledávání: {e}", addon.getAddonInfo('name'))
        return None

# === FUNKCE PRO ŘAZENÍ FILMŮ ===

# Hlavní funkce pro vyhledávání filmů (WRAPPER)
def perform_search_and_display(addon, params, handle):
    """
    WRAPPER: Tenký wrapper pro centrální funkci search_and_select_movie_file.
    Používá se z menu doplňku přes router.
    """
    # OPRAVA: Kontrola zda se nejedná o opětovné zobrazení po ukončení přehrávání
    movie_playing_from_search = addon.getSetting('movie_playing_from_search')
    if movie_playing_from_search == 'true':
        addon.setSetting('movie_playing_from_search', 'false')
        xbmc.log("[plugin.video.milionar] perform_search_and_display: Movie playback ended, preventing re-search", level=xbmc.LOGINFO)
        xbmcplugin.endOfDirectory(handle)
        return

    # Získání hledaného výrazu
    what = params.get('what')
    if not what:
        what = ui.ask('', addon)
        if not what:
            xbmcplugin.endOfDirectory(handle)
            return

    # Extrakce TMDb metadat z parametrů
    title = params.get('tmdb_title', what)
    year = params.get('tmdb_year', '')
    plot = params.get('tmdb_plot', '')
    poster = params.get('tmdb_thumb', '')
    fanart = params.get('tmdb_fanart', '')
    rating = params.get('tmdb_rating', '')
    genre = params.get('tmdb_genre', '')
    director = params.get('tmdb_director', '')
    runtime = params.get('tmdb_runtime', '')

    # Volání centrální funkce
    selected_file = search_and_select_movie_file(
        search_query=f"{title} {year}".strip(),
        title=title,
        year=year,
        plot=plot,
        poster=poster,
        fanart=fanart,
        rating=rating,
        genre=genre,
        director=director,
        runtime=runtime,
        addon=addon,
        tmdb_id=params.get('tmdb_id'),
        from_library=False
    )
    
    # Pokud byl vybrán soubor, přehraj ho
    if selected_file and 'ident' in selected_file and 'name' in selected_file:
        ident = selected_file['ident']
        name = selected_file['name']
        
        params_dict = {
            'tmdb_title': title,
            'tmdb_year': year,
            'tmdb_plot': plot,
            'tmdb_thumb': poster,
            'tmdb_fanart': fanart,
            'tmdb_rating': rating,
            'tmdb_genre': genre,
            'tmdb_director': director,
            'tmdb_runtime': runtime
        }
        
        webshare_api.resolve_playable_item(ident, name, addon, params_dict, handle)
        return
    
    # Ukončení seznamu v Kodi
    xbmcplugin.endOfDirectory(handle) 