import json
import xbmcgui
import xbmcplugin
from wcs import user_data, ui, utils
from wcs.ai.AIProvider import send_prompt, check_ai_api_key_and_handle_error
import xbmc, re, requests, datetime
from wcs.metadata.TMDbClient import get_tmdb_api_key, handle_tmdb_api_error
from wcs.ui.ContextMenu import create_movie_context_menu
import xbmcaddon

def _is_web_search_enabled():
    """Vrací True pokud je povoleno použití web search nástroje v nastavení."""
    addon = xbmcaddon.Addon()
    return addon.getSettingBool('ai_use_web_search')

def _check_tmdb_api_key_and_handle_error(addon, handle):
    """
    Kontroluje TMDb API klíč a v případě chyby zobrazí dialog.
    Vrací True pokud je klíč v pořádku, False pokud nastala chyba.
    """
    try:
        get_tmdb_api_key()
        return True
    except ValueError as e:
        handle_tmdb_api_error(e, addon, handle)
        return False

def _check_ai_api_key_and_handle_error(addon, handle):
    """
    Kontroluje AI API klíč a v případě chyby zobrazí dialog.
    Vrací True pokud je klíč v pořádku, False pokud nastala chyba.
    """
    try:
        return check_ai_api_key_and_handle_error()
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error checking AI API key: {e}", level=xbmc.LOGERROR)
        ui.popinfo('Chyba při kontrole AI API klíče.', addon.getAddonInfo('name'))
        return False

def recommend_movies(params, addon, handle):
    """Zobrazí AI doporučení filmů na základě "Moje filmy"."""
    if not _check_tmdb_api_key_and_handle_error(addon, handle):
        return

    if not _check_ai_api_key_and_handle_error(addon, handle):
        return

    # Nastavení kategorie a obsahu
    xbmcplugin.setPluginCategory(handle, 'AI doporučení filmů')
    xbmcplugin.setContent(handle, 'movies')
    # Načtení uložených filmů
    tmdb_movies = user_data.load_tmdb_movies(addon)
    if not tmdb_movies:
        ui.popinfo('Nemáte žádné filmy v seznamu "Moje filmy".', addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return

    # Sestavení seznamu názvů s rokem pro prompt
    titles = [f"{m.get('title')} ({m.get('year','')})" for m in tmdb_movies]
    prompt = (
        'Jsi odborník na filmy a doporučovací systém, který analyzuje uživatelovu knihovnu filmů a poskytuje přizpůsobená doporučení. Pro získání informací vždy povinně použij nástroj web_search_preview, abys zajistil, že doporučení a novinky reflektují nejaktuálnější data dostupná online. Namísto interní databáze vždy vyhledej na internetu.\n'
        '\n'
        'Uživatel zadá seznam filmů, jeden na řádek, ve formátu: Název filmu (Rok).\n'
        '\n'
        'Úkoly:\n'
        '1. Prvních 5 položek musí být nejaktuálnější položky ze světa filmů získané z web_search_preview; vynechej připravované premiéry.\n'
        '2. Doplň dalších 10 doporučených filmů založených na podobných žánrech, hercích/režisérech, období vydání a tématech, ověř dostupnost pomocí web_search_preview.\n'
        '3. Pokud knihovna obsahuje filmy z kolekce, doplň relevantní díly stejné série.\n'
        '4. Pokud knihovna obsahuje filmy ve více jazycích, zachovej poměr dle vstupu; doplň dostupné filmy online.\n'
        '\n'
        'Seznam filmů v uživateliově knihovně:\n'
        + '\n'.join(titles) + '\n'
        'VÝSTUP MUSÍ BÝT VŽDY POUZE VE TVARU: Název filmu (Rok), KAŽDÝ FILM NA SAMOSTATNÉM ŘÁDKU, BEZ DALŠÍCH KOMENTÁŘŮ.'
    )

    try:
        # Získání odpovědi a usage dat - používáme tools pro doporučování obsahu
        response_text, usage = send_prompt(prompt, use_tools=_is_web_search_enabled())
        # Sjednocení odpovědi na čistý text
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            response_raw = ''.join(parts)
        else:
            response_raw = str(response_text)
        # Rozdělit na jednotlivé řádky s filmy
        rec_lines = [line.strip() for line in response_raw.splitlines() if line.strip()]
        # Odstraníme původní vstupní seznam a případné nadpisy
        input_titles = set(titles)
        rec_lines = [line for line in rec_lines if line not in input_titles and not line.lower().startswith('seznam film')]
        if not rec_lines:
            ui.popinfo('AI nevrátil žádné doporučení.', addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
            return
        # Uložení usage do nového tracking systému
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        provider = ai_provider.get_current_provider()
        model = ai_provider.get_current_model()
        track_ai_usage(provider, model, usage, "movie_recommendation")
    except Exception as e:
        ui.popinfo(f'Chyba AI doporučení: {e}', addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return

    # Seřadit doporučení podle roku sestupně (nejnovější nahoře)
    rec_lines.sort(key=lambda l: int(re.search(r"\((\d{4})\)$", l).group(1)) if re.search(r"\((\d{4})\)$", l) else 0, reverse=True)
    # Stránkování výsledků
    page = int(params.get('page', '1'))
    items_per_page = int(params.get('limit', '20'))
    total = len(rec_lines)
    total_pages = (total + items_per_page - 1) // items_per_page
    start = (page - 1) * items_per_page
    end = start + items_per_page

    for line in rec_lines[start:end]:
        m = re.match(r"(.+)\s+\((\d{4})\)$", line)
        title = m.group(1) if m else line
        year = m.group(2) if m else ''
        # Vyhledání TMDb ID
        try:
            params_tmdb = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ', 'query': title}
            resp = requests.get('https://api.themoviedb.org/3/search/movie', params=params_tmdb, timeout=5)
            resp.raise_for_status()
            results = resp.json().get('results', [])
            tmdb_item = None
            for item in results:
                if year and item.get('release_date', '').startswith(year):
                    tmdb_item = item
                    break
            if not tmdb_item and results:
                tmdb_item = results[0]
            tmdb_id = tmdb_item.get('id') if tmdb_item else None
        except ValueError as e:
            # TMDb API klíč není nastaven - ukončí celou funkci s dialogem
            handle_tmdb_api_error(e, addon, handle)
            return
        except Exception:
            tmdb_id = None

        # Získání detailních metadat z TMDb
        movie_data = {}
        if tmdb_id:
            try:
                detail_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
                detail_resp = requests.get(f'https://api.themoviedb.org/3/movie/{tmdb_id}', params=detail_params, timeout=5)
                detail_resp.raise_for_status()
                movie_data = detail_resp.json()
            except Exception:
                movie_data = {}

        # Sestav label s názvem a rokem v závorce
        release_year = movie_data.get('release_date', '')[:4]
        if release_year:
            label = f"{movie_data.get('title', title)} ({release_year})"
        else:
            label = movie_data.get('title', title)
        listitem = xbmcgui.ListItem(label=label)
        poster = movie_data.get('poster_path')
        if poster:
            listitem.setArt({'thumb': f'https://image.tmdb.org/t/p/w500{poster}'})
        info = {
            'title': movie_data.get('title', title),
            'originaltitle': movie_data.get('original_title', ''),
            'plot': movie_data.get('overview', ''),
            'year': movie_data.get('release_date', '')[:4],
            'rating': movie_data.get('vote_average', 0),
            'genre': [g['name'] for g in movie_data.get('genres', [])],
            'duration': movie_data.get('runtime', 0),
            'studio': [c['name'] for c in movie_data.get('production_companies', [])],
            'tagline': movie_data.get('tagline', '')
        }
        utils.set_video_info_tag(listitem, info)

        # Přidání kontextového menu pomocí pomocné funkce
        if tmdb_id:
            commands = create_movie_context_menu(
                addon=addon,
                tmdb_id=tmdb_id,
                title=info.get('title', title),
                year=info.get('year', year),
                plot=info.get('plot', ''),
                poster=f'https://image.tmdb.org/t/p/w500{poster}' if poster else ''
            )
            listitem.addContextMenuItems(commands)

        url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={tmdb_id}"
        xbmcplugin.addDirectoryItem(handle, url, listitem, False)

    # Navigace stránkování
    if page > 1:
        prev = xbmcgui.ListItem(label=addon.getLocalizedString(30206))
        prev_url = f"plugin://{addon.getAddonInfo('id')}?action=ai_recommend_movies&page={page-1}"
        xbmcplugin.addDirectoryItem(handle, prev_url, prev, True)
    if page < total_pages:
        nxt = xbmcgui.ListItem(label=addon.getLocalizedString(30207))
        nxt_url = f"plugin://{addon.getAddonInfo('id')}?action=ai_recommend_movies&page={page+1}"
        xbmcplugin.addDirectoryItem(handle, nxt_url, nxt, True)

    xbmcplugin.endOfDirectory(handle)

def recommend_by_actors(params, addon, handle):
    """Zobrazí AI doporučení filmů na základě oblíbených herců."""
    if not _check_tmdb_api_key_and_handle_error(addon, handle):
        return
        
    if not _check_ai_api_key_and_handle_error(addon, handle):
        return
        
    # Log volání funkce pro AI doporučení podle herců
    log_level = getattr(xbmc, 'LOGNOTICE', xbmc.LOGINFO)
    xbmc.log(f"[wcs] recommend_by_actors called with params: {params}", log_level)
    xbmcplugin.setPluginCategory(handle, 'AI doporučení podle herců')
    xbmcplugin.setContent(handle, 'movies')
    # Načtení oblíbených herců
    favorite_actors = user_data.load_favorite_actors(addon)
    if not favorite_actors:
        ui.popinfo('Nemáte žádné oblíbené herce.', addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return

    # Příprava promptu pro OpenAI
    names = [actor.get('name', '') for actor in favorite_actors]
    prompt = (
        'Jsi doporučovací systém filmů. Uživatel zadá seznam oblíbených herců, jeden na řádek, ve formátu: Jméno herce.\n\n'
        'Úkoly:\n'
        '1. Vytvoř přibližně 10 filmů, ve kterých tito herci vystupují, ve formátu Název (Rok), jeden na řádek, bez formátování, číslování nebo vysvětlení.\n'
        '2. Doporučení vyber podle rolí a žánrů spjatých s těmito herci.\n\n'
        'Seznam herců:\n'
        + '\n'.join(names)
    )
    try:
        # Pro doporučování podle herců používáme tools
        response_text, usage = send_prompt(prompt, use_tools=_is_web_search_enabled())
        # Sjednocení odpovědi na čistý text
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            response_raw = ''.join(parts)
        else:
            response_raw = str(response_text)
        # Rozdělit na řádky a odstranit záhlaví a vstupní herce
        rec_lines = [line.strip() for line in response_raw.splitlines() if line.strip()]
        input_names = set(names)
        rec_lines = [line for line in rec_lines if line not in input_names and not line.lower().startswith('seznam herc')]
        if not rec_lines:
            ui.popinfo('AI nevrátil žádné doporučení podle herců.', addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
            return
        # Uložení usage do nového tracking systému  
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        provider = ai_provider.get_current_provider()
        model = ai_provider.get_current_model()
        track_ai_usage(provider, model, usage, "movie_recommendation")
    except Exception as e:
        ui.popinfo(f'Chyba AI doporučení podle herců: {e}', addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
    # Seřadit doporučení podle roku sestupně
    rec_lines.sort(key=lambda l: int(re.search(r"\((\d{4})\)$", l).group(1)) if re.search(r"\((\d{4})\)$", l) else 0, reverse=True)
    # Stránkování
    page = int(params.get('page', '1'))
    items_per_page = int(params.get('limit', '20'))
    total = len(rec_lines)
    total_pages = (total + items_per_page - 1) // items_per_page
    start = (page - 1) * items_per_page
    end = start + items_per_page
    # Generování Kodi položek
    for line in rec_lines[start:end]:
        m = re.match(r"(.+)\s+\((\d{4})\)$", line)
        title = m.group(1) if m else line
        year = m.group(2) if m else ''
        # Vyhledání TMDb ID
        try:
            params_tmdb = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ', 'query': title}
            resp = requests.get('https://api.themoviedb.org/3/search/movie', params=params_tmdb, timeout=5)
            resp.raise_for_status()
            results = resp.json().get('results', [])
            tmdb_item = next((item for item in results if year and item.get('release_date', '').startswith(year)), results[0] if results else None)
            tmdb_id = tmdb_item.get('id') if tmdb_item else None
        except Exception:
            tmdb_id = None
        # Získání detailů z TMDb
        movie_data = {}
        if tmdb_id:
            try:
                detail_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
                detail_resp = requests.get(f'https://api.themoviedb.org/3/movie/{tmdb_id}', params=detail_params, timeout=5)
                detail_resp.raise_for_status()
                movie_data = detail_resp.json()
            except Exception:
                movie_data = {}
        # Sestavení labelu a metadat
        release_year = movie_data.get('release_date', '')[:4]
        label = f"{movie_data.get('title', title)} ({release_year})" if release_year else movie_data.get('title', title)
        listitem = xbmcgui.ListItem(label=label)
        poster = movie_data.get('poster_path')
        if poster:
            listitem.setArt({'thumb': f'https://image.tmdb.org/t/p/w500{poster}'})
        info = {
            'title': movie_data.get('title', title),
            'originaltitle': movie_data.get('original_title', ''),
            'plot': movie_data.get('overview', ''),
            'year': movie_data.get('release_date', '')[:4],
            'rating': movie_data.get('vote_average', 0),
            'genre': [g['name'] for g in movie_data.get('genres', [])],
            'duration': movie_data.get('runtime', 0),
            'studio': [c['name'] for c in movie_data.get('production_companies', [])],
            'tagline': movie_data.get('tagline', '')
        }
        utils.set_video_info_tag(listitem, info)

        # Přidání kontextového menu pomocí pomocné funkce
        if tmdb_id:
            commands = create_movie_context_menu(
                addon=addon,
                tmdb_id=tmdb_id,
                title=info.get('title', title),
                year=info.get('year', year),
                plot=info.get('plot', ''),
                poster=f'https://image.tmdb.org/t/p/w500{poster}' if poster else ''
            )
            listitem.addContextMenuItems(commands)

        url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={tmdb_id}"
        xbmcplugin.addDirectoryItem(handle, url, listitem, False)

    # Navigace
    if page > 1:
        prev = xbmcgui.ListItem(label=addon.getLocalizedString(30206))
        prev_url = f"plugin://{addon.getAddonInfo('id')}?action=ai_recommend_by_actors&page={page-1}"
        xbmcplugin.addDirectoryItem(handle, prev_url, prev, True)
    if page < total_pages:
        nxt = xbmcgui.ListItem(label=addon.getLocalizedString(30207))
        nxt_url = f"plugin://{addon.getAddonInfo('id')}?action=ai_recommend_by_actors&page={page+1}"
        xbmcplugin.addDirectoryItem(handle, nxt_url, nxt, True)
    xbmcplugin.endOfDirectory(handle)

def recommend_by_directors(params, addon, handle):
    """Zobrazí AI doporučení filmů na základě oblíbených režisérů."""
    if not _check_tmdb_api_key_and_handle_error(addon, handle):
        return
        
    if not _check_ai_api_key_and_handle_error(addon, handle):
        return
        
    # Log volání funkce pro AI doporučení podle režisérů
    log_level = getattr(xbmc, 'LOGNOTICE', xbmc.LOGINFO)
    xbmc.log(f"[wcs] recommend_by_directors called with params: {params}", log_level)
    xbmcplugin.setPluginCategory(handle, 'AI doporučení podle režisérů')
    xbmcplugin.setContent(handle, 'movies')
    # Načtení oblíbených režisérů
    favorite_directors = user_data.load_favorite_directors(addon)
    if not favorite_directors:
        ui.popinfo('Nemáte žádné oblíbené režiséry.', addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return

    # Příprava promptu pro OpenAI
    names = [director.get('name', '') for director in favorite_directors]
    prompt = (
        'Jsi doporučovací systém filmů. Uživatel zadá seznam oblíbených režisérů, jeden na řádek, ve formátu: Jméno režiséra.\n\n'
        'Úkoly:\n'
        '1. Vytvoř přibližně 10 filmů režírovaných těmito režiséry, ve formátu Název (Rok), jeden na řádek, bez formátování, číslování nebo vysvětlení.\n'
        '2. Doporučení vyber podle žánrů a tématech spjatých s těmito režiséry.\n\n'
        'Seznam režisérů:\n'
        + '\n'.join(names)
    )
    try:
        # Pro doporučování podle režisérů používáme tools
        response_text, usage = send_prompt(prompt, use_tools=_is_web_search_enabled())
        # Sjednocení odpovědi na čistý text
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            response_raw = ''.join(parts)
        else:
            response_raw = str(response_text)
        # Rozdělit na řádky a odstranit vstupní režiséry
        rec_lines = [line.strip() for line in response_raw.splitlines() if line.strip()]
        input_names = set(names)
        rec_lines = [line for line in rec_lines if line not in input_names and not line.lower().startswith('seznam režisér')]
        if not rec_lines:
            ui.popinfo('AI nevrátil žádné doporučení podle režisérů.', addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
            return
        # Uložení usage do nového tracking systému  
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        provider = ai_provider.get_current_provider()
        model = ai_provider.get_current_model()
        track_ai_usage(provider, model, usage, "movie_recommendation")
    except Exception as e:
        ui.popinfo(f'Chyba AI doporučení podle režisérů: {e}', addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
    # Seřadit doporučení podle roku sestupně
    rec_lines.sort(key=lambda l: int(re.search(r"\((\d{4})\)$", l).group(1)) if re.search(r"\((\d{4})\)$", l) else 0, reverse=True)
    # Stránkování výsledků
    page = int(params.get('page', '1'))
    items_per_page = int(params.get('limit', '20'))
    total = len(rec_lines)
    total_pages = (total + items_per_page - 1) // items_per_page
    start = (page - 1) * items_per_page
    end = start + items_per_page

    for line in rec_lines[start:end]:
        m = re.match(r"(.+)\s+\((\d{4})\)$", line)
        title = m.group(1) if m else line
        year = m.group(2) if m else ''
        # Vyhledání TMDb ID
        try:
            params_tmdb = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ', 'query': title}
            resp = requests.get('https://api.themoviedb.org/3/search/movie', params=params_tmdb, timeout=5)
            resp.raise_for_status()
            results = resp.json().get('results', [])
            tmdb_item = next((item for item in results if year and item.get('release_date', '').startswith(year)), results[0] if results else None)
            tmdb_id = tmdb_item.get('id') if tmdb_item else None
        except Exception:
            tmdb_id = None

        # Získání detailních metadat z TMDb
        movie_data = {}
        if tmdb_id:
            try:
                detail_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
                detail_resp = requests.get(f'https://api.themoviedb.org/3/movie/{tmdb_id}', params=detail_params, timeout=5)
                detail_resp.raise_for_status()
                movie_data = detail_resp.json()
            except Exception:
                movie_data = {}

        # Sestav label s názvem a rokem v závorce
        release_year = movie_data.get('release_date', '')[:4]
        if release_year:
            label = f"{movie_data.get('title', title)} ({release_year})"
        else:
            label = movie_data.get('title', title)
        listitem = xbmcgui.ListItem(label=label)
        poster = movie_data.get('poster_path')
        if poster:
            listitem.setArt({'thumb': f'https://image.tmdb.org/t/p/w500{poster}'})
        info = {
            'title': movie_data.get('title', title),
            'originaltitle': movie_data.get('original_title', ''),
            'plot': movie_data.get('overview', ''),
            'year': movie_data.get('release_date', '')[:4],
            'rating': movie_data.get('vote_average', 0),
            'genre': [g['name'] for g in movie_data.get('genres', [])],
            'duration': movie_data.get('runtime', 0),
            'studio': [c['name'] for c in movie_data.get('production_companies', [])],
            'tagline': movie_data.get('tagline', '')
        }
        utils.set_video_info_tag(listitem, info)

        # Přidání kontextového menu pomocí pomocné funkce
        if tmdb_id:
            commands = create_movie_context_menu(
                addon=addon,
                tmdb_id=tmdb_id,
                title=info.get('title', title),
                year=info.get('year', year),
                plot=info.get('plot', ''),
                poster=f'https://image.tmdb.org/t/p/w500{poster}' if poster else ''
            )
            listitem.addContextMenuItems(commands)

        url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={tmdb_id}"
        xbmcplugin.addDirectoryItem(handle, url, listitem, False)

    # Navigace stránkování
    if page > 1:
        prev = xbmcgui.ListItem(label=addon.getLocalizedString(30206))
        prev_url = f"plugin://{addon.getAddonInfo('id')}?action=ai_recommend_by_directors&page={page-1}"
        xbmcplugin.addDirectoryItem(handle, prev_url, prev, True)
    if page < total_pages:
        nxt = xbmcgui.ListItem(label=addon.getLocalizedString(30207))
        nxt_url = f"plugin://{addon.getAddonInfo('id')}?action=ai_recommend_by_directors&page={page+1}"
        xbmcplugin.addDirectoryItem(handle, nxt_url, nxt, True)
    xbmcplugin.endOfDirectory(handle)

def recommend_by_writers(params, addon, handle):
    """Zobrazí AI doporučení filmů na základě oblíbených scénáristů."""
    if not _check_tmdb_api_key_and_handle_error(addon, handle):
        return
        
    if not _check_ai_api_key_and_handle_error(addon, handle):
        return
        
    # Log volání funkce pro AI doporučení podle scénáristů
    log_level = getattr(xbmc, 'LOGNOTICE', xbmc.LOGINFO)
    xbmc.log(f"[wcs] recommend_by_writers called with params: {params}", log_level)
    xbmcplugin.setPluginCategory(handle, 'AI doporučení podle scénáristů')
    xbmcplugin.setContent(handle, 'movies')
    # Načtení oblíbených scénáristů
    favorite_writers = user_data.load_favorite_writers(addon)
    if not favorite_writers:
        ui.popinfo('Nemáte žádné oblíbené scénáristy.', addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return

    # Příprava promptu pro OpenAI
    names = [writer.get('name', '') for writer in favorite_writers]
    prompt = (
        'Jsi doporučovací systém filmů. Uživatel zadá seznam oblíbených scénáristů, jeden na řádek, ve formátu: Jméno scénáristy.\n\n'
        'Úkoly:\n'
        '1. Vytvoř přibližně 10 filmů napsaných těmito scénáristy, ve formátu Název (Rok), jeden na řádek, bez formátování, číslování nebo vysvětlení.\n'
        '2. Doporučení vyber podle témat a stylu spjatého s těmito scénáristy.\n\n'
        'Seznam scénáristů:\n'
        + '\n'.join(names)
    )
    try:
        # Pro doporučování podle scénáristů používáme tools
        response_text, usage = send_prompt(prompt, use_tools=_is_web_search_enabled())
        # Sjednocení odpovědi na čistý text
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            response_raw = ''.join(parts)
        else:
            response_raw = str(response_text)
        # Rozdělit na řádky a odstranit vstupní scénáristy
        rec_lines = [line.strip() for line in response_raw.splitlines() if line.strip()]
        input_names = set(names)
        rec_lines = [line for line in rec_lines if line not in input_names and not line.lower().startswith('seznam scénárist')]
        if not rec_lines:
            ui.popinfo('AI nevrátil žádné doporučení podle scénáristů.', addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
            return
        # Uložení usage do nového tracking systému  
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        provider = ai_provider.get_current_provider()
        model = ai_provider.get_current_model()
        track_ai_usage(provider, model, usage, "movie_recommendation")
    except Exception as e:
        ui.popinfo(f'Chyba AI doporučení podle scénáristů: {e}', addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
    # Seřadit doporučení podle roku sestupně
    rec_lines.sort(key=lambda l: int(re.search(r"\((\d{4})\)$", l).group(1)) if re.search(r"\((\d{4})\)$", l) else 0, reverse=True)
    # Stránkování výsledků
    page = int(params.get('page', '1'))
    items_per_page = int(params.get('limit', '20'))
    total = len(rec_lines)
    total_pages = (total + items_per_page - 1) // items_per_page
    start = (page - 1) * items_per_page
    end = start + items_per_page

    for line in rec_lines[start:end]:
        m = re.match(r"(.+)\s+\((\d{4})\)$", line)
        title = m.group(1) if m else line
        year = m.group(2) if m else ''
        # Vyhledání TMDb ID
        try:
            params_tmdb = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ', 'query': title}
            resp = requests.get('https://api.themoviedb.org/3/search/movie', params=params_tmdb, timeout=5)
            resp.raise_for_status()
            results = resp.json().get('results', [])
            tmdb_item = next((item for item in results if year and item.get('release_date', '').startswith(year)), results[0] if results else None)
            tmdb_id = tmdb_item.get('id') if tmdb_item else None
        except Exception:
            tmdb_id = None

        # Získání detailních metadat z TMDb
        movie_data = {}
        if tmdb_id:
            try:
                detail_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
                detail_resp = requests.get(f'https://api.themoviedb.org/3/movie/{tmdb_id}', params=detail_params, timeout=5)
                detail_resp.raise_for_status()
                movie_data = detail_resp.json()
            except Exception:
                movie_data = {}

        # Sestav label s názvem a rokem v závorce
        release_year = movie_data.get('release_date', '')[:4]
        if release_year:
            label = f"{movie_data.get('title', title)} ({release_year})"
        else:
            label = movie_data.get('title', title)
        listitem = xbmcgui.ListItem(label=label)
        poster = movie_data.get('poster_path')
        if poster:
            listitem.setArt({'thumb': f'https://image.tmdb.org/t/p/w500{poster}'})
        info = {
            'title': movie_data.get('title', title),
            'originaltitle': movie_data.get('original_title', ''),
            'plot': movie_data.get('overview', ''),
            'year': movie_data.get('release_date', '')[:4],
            'rating': movie_data.get('vote_average', 0),
            'genre': [g['name'] for g in movie_data.get('genres', [])],
            'duration': movie_data.get('runtime', 0),
            'studio': [c['name'] for c in movie_data.get('production_companies', [])],
            'tagline': movie_data.get('tagline', '')
        }
        utils.set_video_info_tag(listitem, info)

        # Přidání kontextového menu pomocí pomocné funkce
        if tmdb_id:
            commands = create_movie_context_menu(
                addon=addon,
                tmdb_id=tmdb_id,
                title=info.get('title', title),
                year=info.get('year', year),
                plot=info.get('plot', ''),
                poster=f'https://image.tmdb.org/t/p/w500{poster}' if poster else ''
            )
            listitem.addContextMenuItems(commands)

        url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={tmdb_id}"
        xbmcplugin.addDirectoryItem(handle, url, listitem, False)

    # Navigace stránkování
    if page > 1:
        prev = xbmcgui.ListItem(label=addon.getLocalizedString(30206))
        prev_url = f"plugin://{addon.getAddonInfo('id')}?action=ai_recommend_by_writers&page={page-1}"
        xbmcplugin.addDirectoryItem(handle, prev_url, prev, True)
    if page < total_pages:
        nxt = xbmcgui.ListItem(label=addon.getLocalizedString(30207))
        nxt_url = f"plugin://{addon.getAddonInfo('id')}?action=ai_recommend_by_writers&page={page+1}"
        xbmcplugin.addDirectoryItem(handle, nxt_url, nxt, True)
    xbmcplugin.endOfDirectory(handle)

def start_text_game(movie_title, movie_plot="", movie_genre=""):
    """
    Spustí novou textovou hru na základě filmu.
    
    Args:
        movie_title (str): Název filmu
        movie_plot (str): Děj filmu (volitelné)
        movie_genre (str): Žánr filmu (volitelné)
    
    Returns:
        tuple: (intro_text, [choice_a, choice_b, choice_c], usage)
    """
    prompt = (
        f"Jsi vypravěč interaktivní textové adventury inspirované filmem '{movie_title}'.\n\n"
        f"KONTEXT FILMU:\n"
        f"Název: {movie_title}\n"
        f"Žánr: {movie_genre if movie_genre else 'Neurčen'}\n"
        f"Děj: {movie_plot[:300] if movie_plot else 'Využij své znalosti o tomto filmu'}\n\n"
        "INSTRUKCE:\n"
        "1. Využij VŠECHNY své znalosti o tomto filmu - postavy, děj, lokace, atmosféru, klíčové momenty\n"
        "2. Vytvoř úvodní situaci (2-3 věty) zasazenou do světa filmu\n"
        "3. Nabídni hráčovi výběr ze 3 hlavních postav filmu jako volby a), b), c)\n"
        "4. Každá volba: 'Hrát jako [Jméno postavy]'\n"
        "5. Toto je začátek 5-dílné adventure, kde volba postavy ovlivní celý příběh\n\n"
        "FORMÁT ODPOVĚDI MUSÍ BÝT PŘESNĚ TAKTO:\n"
        "{\n"
        '  "intro": "Úvodní text příběhu",\n'
        '  "choices": {\n'
        '    "a": "Hrát jako [Hlavní postava 1]",\n'
        '    "b": "Hrát jako [Hlavní postava 2]",\n'
        '    "c": "Hrát jako [Hlavní postava 3]"\n'
        '  }\n'
        "}"
    )
    
    try:
        # Pro textové hry nepoužíváme tools
        response_text, usage = send_prompt(prompt)
        
        # Zpracování odpovědi
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            response_raw = ''.join(parts)
        else:
            response_raw = str(response_text)
        
        try:
            response_json = json.loads(response_raw)
            intro_text = response_json.get('intro', '')
            choices_dict = response_json.get('choices', {})
            choices = [choices_dict.get('a', ''), choices_dict.get('b', ''), choices_dict.get('c', '')]
            if not intro_text or not all(choices):
                raise ValueError("Neplatná struktura JSON odpovědi")
            return intro_text, choices[:3], usage
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error parsing JSON response in start_text_game: {e}", level=xbmc.LOGERROR)
            # Fallback odpověď
            fallback_intro = f"Vítej ve světě filmu {movie_title}! Příběh začíná...\n\nKterou postavu si vybereš?"
            fallback_choices = ["Hrát jako hlavní hrdina", "Hrát jako vedlejší postava", "Hrát jako antagonista"]
            return fallback_intro, fallback_choices, None
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error in start_text_game: {e}", level=xbmc.LOGERROR)
        # Fallback odpověď
        fallback_intro = f"Vítej ve světě filmu {movie_title}! Příběh začíná...\n\nKterou postavu si vybereš?"
        fallback_choices = ["Hrát jako hlavní hrdina", "Hrát jako vedlejší postava", "Hrát jako antagonista"]
        return fallback_intro, fallback_choices, None

def continue_text_game(movie_title, story_history, choice_made, turn_number=1):
    """
    Pokračuje v textové hře na základě hráčovy volby.
    
    Args:
        movie_title (str): Název filmu
        story_history (list): Historie příběhu
        choice_made (str): Volba kterou hráč udělal
        turn_number (int): Číslo aktuálního tahu (1-5)
    
    Returns:
        tuple: (new_text, [choice_a, choice_b, choice_c], is_ending, usage)
    """
    # Sestavení kontextu z historie
    context = "\n".join(story_history[-4:]) if story_history else ""
    
    # Určení fáze hry
    if turn_number == 1:
        phase_instruction = "Hráč si vybral postavu. Nyní začni první herní situaci - umísti ho do klíčové scény z filmu."
    elif turn_number <= 3:
        phase_instruction = f"Toto je {turn_number}. kolo z 5. Pokračuj v napínavém příběhu s akcí a dilematem."
    elif turn_number == 4:
        phase_instruction = "Toto je 4. kolo z 5. Blížíme se ke klimaxu - vytvoř dramatickou situaci."
    else:  # turn_number >= 5
        phase_instruction = "Toto je finální 5. kolo. Doved příběh k logickému konci podle hráčových voleb."
    
    prompt = (
        f"Pokračuj v textové adventuře na téma filmu '{movie_title}'.\n\n"
        f"KONTEXT HRÁČOVÝCH VOLEB:\n{context}\n\n"
        f"AKTUÁLNÍ VOLBA HRÁČE: {choice_made}\n\n"
        f"FÁZE HRY: {phase_instruction}\n\n"
        "INSTRUKCE:\n"
        "1. Využij své znalosti o filmu - postavy, lokace, zápletka, atmosféra\n"
        "2. Reaguj na hráčovu volbu logicky v kontextu filmu\n"
        "3. Vytvoř pokračování příběhu (2-3 věty)\n"
        "4. Nabídni 3 smysluplné volby související s filmem\n"
    )
    
    if turn_number >= 5:
        prompt += (
            "5. DŮLEŽITÉ: Toto je poslední kolo - jako volbu c) napiš 'KONEC PŘÍBĚHU'\n\n"
            "FORMÁT ODPOVĚDI MUSÍ BÝT PŘESNĚ TAKTO:\n"
            "{\n"
            '  "story": "Finální pokračování příběhu - zakončení podle hráčových voleb",\n'
            '  "choices": {\n'
            '    "a": "finální volba A",\n'
            '    "b": "finální volba B",\n'
            '    "c": "KONEC PŘÍBĚHU"\n'
            '  }\n'
            "}\n"
            "DŮLEŽITÉ: Nevkládej žádný text před ani za JSON, pouze čistý JSON!"
        )
    else:
        prompt += (
            "5. Každá volba max 8-10 slov\n\n"
            "FORMÁT ODPOVĚDI MUSÍ BÝT PŘESNĚ TAKTO:\n"
            "{\n"
            '  "story": "Pokračování příběhu",\n'
            '  "choices": {\n'
            '    "a": "akční volba",\n'
            '    "b": "diplomatická volba",\n'
            '    "c": "kreativní volba"\n'
            '  }\n'
            "}\n"
            "DŮLEŽITÉ: Nevkládej žádný text před ani za JSON, pouze čistý JSON!"
        )
    
    try:
        # Pro textové hry nepoužíváme tools
        response_text, usage = send_prompt(prompt)
        
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            response_raw = ''.join(parts)
        else:
            response_raw = str(response_text)
        
        try:
            # Robustní extrakce JSON: najdi první { a poslední }
            start = response_raw.find('{')
            end = response_raw.rfind('}')
            if start != -1 and end != -1 and end > start:
                json_str = response_raw[start:end+1]
            else:
                json_str = response_raw
            
            # Parsování JSON odpovědi
            response_json = json.loads(json_str)
            story_text = response_json.get('story', '')
            choices_dict = response_json.get('choices', {})
            choices = [choices_dict.get('a', ''), choices_dict.get('b', ''), choices_dict.get('c', '')]
            is_ending = (turn_number >= 5) or (len(choices) > 2 and 'KONEC' in choices[2].upper())
            if not story_text or not all(choices):
                raise ValueError("Neplatná struktura JSON odpovědi")
            return story_text, choices[:3], is_ending, usage
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error parsing JSON response in continue_text_game: {e}", level=xbmc.LOGERROR)
            xbmc.log(f"[plugin.video.milionar] AI response_raw: {response_raw}", level=xbmc.LOGERROR)
            # Fallback odpověď
            if turn_number >= 5:
                fallback_text = "Příběh se chýlí ke konci..."
                fallback_choices = ["Dokončit cestu", "Užít si okamžik", "KONEC PŘÍBĚHU"]
                return fallback_text, fallback_choices, True, None
            else:
                fallback_text = "Dobrodružství pokračuje..."
                fallback_choices = ["Jdi vpřed", "Buď opatrný", "Zkus něco jiného"]
                return fallback_text, fallback_choices, False, None
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error in continue_text_game: {e}", level=xbmc.LOGERROR)
        # Fallback odpověď
        if turn_number >= 5:
            fallback_text = "Příběh se chýlí ke konci..."
            fallback_choices = ["Dokončit cestu", "Užít si okamžik", "KONEC PŘÍBĚHU"]
            return fallback_text, fallback_choices, True, None
        else:
            fallback_text = "Dobrodružství pokračuje..."
            fallback_choices = ["Jdi vpřed", "Buď opatrný", "Zkus něco jiného"]
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error in continue_text_game: {e}", level=xbmc.LOGERROR)
        # Fallback odpověď
        if turn_number >= 5:
            fallback_text = "Příběh se chýlí ke konci..."
            fallback_choices = ["Dokončit cestu", "Užít si okamžik", "KONEC PŘÍBĚHU"]
            return fallback_text, fallback_choices, True, None
        else:
            fallback_text = "Dobrodružství pokračuje..."
            fallback_choices = ["Jdi vpřed", "Buď opatrný", "Zkus něco jiného"]
            return fallback_text, fallback_choices, False, None

def chat_recommendation_step(chat_history, user_input, media_type="movie", context=None):
    """
    Zpracuje jeden krok chatu pro doporučování filmů/seriálů.
    
    Args:
        chat_history (list): Historie konverzace [{"role": "user/assistant", "content": "..."}]
        user_input (str): Vstup uživatele
        media_type (str): "movie" nebo "series"
        context (dict): Kontext aplikace (např. 'my_movies', 'data': [...])
    
    Returns:
        tuple: (response_data, usage)
        response_data je dict s klíči: message, buttons, recommendations
    """
    
    # Sestavení historie do textové podoby
    history_text = ""
    for msg in chat_history:
        role = "Uživatel" if msg['role'] == "user" else "AI"
        history_text += f"{role}: {msg['content']}\n"
    
    # Specifikace typu média
    # Dynamická volba labelu pro média a detekce mixu
    is_mixed_content = False
    if context:
        c_type = context.get('type')
        if c_type in ['my_mixed_collection', 'recently_played']:
            media_label = "filmů a seriálů"
            is_mixed_content = True
        elif c_type in ['favorite_actors', 'favorite_directors', 'favorite_writers']:
            media_label = "filmů a seriálů"
            is_mixed_content = True
        else:
            media_label = "filmů" if media_type == "movie" else "seriálů"
    else:
        media_label = "filmů" if media_type == "movie" else "seriálů"

    # Příprava instrukcí pro formát (podmíněný type)
    if is_mixed_content:
        json_type_field = ', "type": "movie" nebo "tv"'
        json_type_rule = "- Pole 'type' MUSÍ být buď 'movie' (pro filmy) nebo 'tv' (pro seriály). SPRÁVNĚ URČI TYP.\n"
    else:
        json_type_field = ""
        json_type_rule = ""

    # Zpracování kontextu pro prompt
    context_instruction = ""
    if context:
        ctx_type = context.get('type')
        ctx_val = context.get('value')
        ctx_data = context.get('data', [])
        
        if ctx_type in ['my_movies', 'my_series'] and ctx_data:
            titles = [f"{item.get('title', item.get('name'))} ({item.get('year')})" for item in ctx_data[:30]] # Limit 30 items
            titles_str = ", ".join(titles)
            context_instruction = (
                f"KONTEXT UŽIVATELE ({ctx_type}):\n"
                f"Uživatel má v knihovně tyto tituly: {titles_str}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. Přeskoč úvodní vyzvídání na žánry/nálady. Rovnou analyzuj vkus podle těchto titulů.\n"
                "2. Navrhni 10 PODOBNÝCH děl, která by se mohla líbit (ale nejsou v seznamu).\n"
                "3. První zpráva má být: 'Na základě vaší knihovny {media_label} doporučuji...'\n\n"
            )

        elif ctx_type == 'my_mixed_collection' and ctx_data:
            movies = ctx_data.get('movies', [])
            series = ctx_data.get('series', [])
            m_titles = [f"{m.get('title')} ({m.get('year')})" for m in movies[:20]]
            s_titles = [f"{s.get('name')} ({s.get('year')})" for s in series[:20]]
            all_str = ", ".join(m_titles + s_titles)
            
            context_instruction = (
                f"KONTEXT UŽIVATELE (Moje sbírka - Mix):\n"
                f"Uživatel má ve sbírce toto: {all_str}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. Přeskoč úvodní dotazy.\n"
                "2. Analyzuj vkus napříč filmy i seriály.\n"
                "3. Nabízej MIX doporučení (filmy i seriály), pokud si uživatel neřekne jinak.\n"
                "4. Zkus najít skryté souvislosti (např. má rád sci-fi filmy -> doporuč sci-fi seriál).\n"
                "5. Prví zpráva: 'Na základě vaší celé sbírky doporučuji...'\n\n"
            )
        elif ctx_type == 'recently_played' and ctx_data:
            # Načtení limitu z nastavení
            try:
                limit = int(xbmcaddon.Addon().getSetting('ai_context_history_limit') or 20)
            except:
                limit = 20

            titles = []
            for item in ctx_data[:limit]:
                title = item.get('title', '')
                year = item.get('year', '')
                ts = item.get('timestamp')

                if item.get('media_type') == 'series':
                     # Pro seriály přidáme i info o epizodě ve formátu (SxxExx Název)
                     ep_title = item.get('episode_title', '')
                     season = item.get('season')
                     episode = item.get('episode')
                     
                     ep_string = ""
                     if season is not None and episode is not None:
                         ep_string = f"S{int(season):02d}E{int(episode):02d}"
                     
                     if ep_title:
                         ep_string += f" {ep_title}"
                     else:
                         ep_string += f" Epizoda {episode}"
                    
                     if ep_string:
                        title += f" ({ep_string.strip()})"
                if year:
                    title += f" ({year})"
                
                # Přidání data přehrání
                date_str = ""
                if ts:
                    try:
                        dt = datetime.datetime.strptime(ts, "%Y-%m-%d %H:%M:%S")
                        date_str = dt.strftime("%d.%m. %H:%M")
                    except:
                        # Fallback pokud parsování selže (např. starý formát nebo chyba)
                        date_str = str(ts)
                
                if date_str:
                    title += f" [Viděno: {date_str}]"

                titles.append(title)
            
            titles_str = ", ".join(titles)
            
            context_instruction = (
                f"KONTEXT UŽIVATELE (Naposledy přehrané – MIX Filmy/Seriály):\n"
                f"Uživatel si naposledy pustil tyto tituly (s časem přehrání): {titles_str}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. Toto je smíšená historie. Všímej si ČASU ('Viděno: ...'), kdy to viděl.\n"
                "2. Zohledni čas v doporučení (např. večer doporuč filmy, přes den seriály, o víkendu maratony).\n"
                "3. Pokud si uživatel neřekne jinak, nabízej MIX (filmy i seriály).\n"
                "4. První zpráva: 'Vidím, že jste naposledy sledovali...'\n\n"
            )

        elif ctx_type in ['favorite_actors', 'favorite_directors', 'favorite_writers'] and ctx_data:
            names = [item.get('name') for item in ctx_data]
            names_str = ", ".join(names)
            what_map = {'favorite_actors': 'oblíbených herců', 'favorite_directors': 'oblíbených režisérů', 'favorite_writers': 'oblíbených scenáristů'}
            what = what_map.get(ctx_type, 'oblíbených osobností')
            
            context_instruction = (
                f"KONTEXT UŽIVATELE ({ctx_type}):\n"
                f"Uživatel má seznam {what}: {names_str}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. Přeskoč úvodní vyzvídání.\n"
                "2. Analyzuj styl těchto osobností.\n"
                "3. Doporuč filmy/seriály, které jsou typické pro jejich tvorbu nebo kde hrají/režírují (i vzájemné spolupráce).\n"
                "4. První zpráva: 'Na základě vašich {what} doporučuji...'\n\n"
            )
        elif ctx_type == 'genre':
            context_instruction = (
                f"KONTEXT UŽIVATELE ({ctx_type}):\n"
                f"Uživatel si právě prohlíží kategorii ŽÁNR: {ctx_val}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. PŘESKOČ úvodní dotaz na žánr.\n"
                "2. Hned doporuč ty nejlepší tituly z žánru {ctx_val}.\n"
                "3. Dále se ptej na podžánry nebo specifickou náladu v rámci tohoto žánru.\n\n"
            )
        elif ctx_type == 'person':
            p_type = context.get('person_type', 'osoba')
            role_map = {'actor': 'tohoto herce', 'director': 'tohoto režiséra', 'writer': 'tohoto scenáristy'}
            role_str = role_map.get(p_type, 'této osobnosti')
            context_instruction = (
                f"KONTEXT UŽIVATELE ({ctx_type}):\n"
                f"Uživatel si prohlíží profil: {ctx_val} ({p_type}).\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. PŘESKOČ standardní zjišťování vkusu.\n"
                "2. Doporuč filmy/seriály, které jsou STYLOVĚ PODOBNÉ tvorbě {ctx_val}.\n"
                "3. Cíl: 'Pokud máš rád {ctx_val}, bude se ti líbit i toto...'\n"
                "4. Můžeš doporučit i méně známá díla přimo od {role_str}.\n\n"
            )
        elif ctx_type == 'country':
            context_instruction = (
                f"KONTEXT UŽIVATELE ({ctx_type}):\n"
                f"Uživatel hledá díla ze země: {ctx_val}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. PŘESKOČ dotaz na zemi původu.\n"
                "2. Doporuč to nejlepší z produkce země {ctx_val}.\n"
                "3. Ptej se na žánr nebo období.\n\n"
            )
        elif ctx_type == 'year':
            context_instruction = (
                f"KONTEXT UŽIVATELE ({ctx_type}):\n"
                f"Uživatel se zajímá o rok (nebo období): {ctx_val}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. PŘESKOČ dotaz na rok/období.\n"
                "2. Doporuč hity z roku/období {ctx_val}.\n"
                "3. Ptej se na žánr.\n\n"
            )
        elif ctx_type == 'collection':
             context_instruction = (
                f"KONTEXT UŽIVATELE ({ctx_type}):\n"
                f"Uživatel si prohlíží kolekci: {ctx_val}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. Doporuč filmy, které jsou podobné této sérii/kolekci.\n"
                "2. Nabídni podobné franšízy nebo 'co vidět dál'.\n\n"
            )
        elif ctx_type == 'category':
             context_instruction = (
                f"KONTEXT UŽIVATELE ({ctx_type}):\n"
                f"Uživatel je v sekci: {ctx_val}.\n"
                "INSTRUKCE PRO KONTEXT:\n"
                "1. Zohledni tuto sekci při doporučování (např. pokud je to 'Populární', nabízej mainstreamové hity).\n\n"
            )

    # Získání systémového kontextu (čas, lokace)
    from datetime import datetime
    now = datetime.now()
    current_time_str = now.strftime("%Y-%m-%d %H:%M")
    
    # Bezpečné získání jazyka a regionu
    try:
        user_lang = xbmc.getLanguage(xbmc.ISO_639_1) # 'cs', 'en'
    except:
        user_lang = 'cs' # Fallback
        
    try:
        # Některé verze Kodi nemají ISO_3166_1, zkusíme získat region jinak nebo použijeme jazyk
        if hasattr(xbmc, 'ISO_3166_1'):
            user_region = xbmc.getRegion(xbmc.ISO_3166_1)
        else:
            # Fallback na jazyk v uppercase jako proxy pro region
            user_region = user_lang.upper()
    except:
        user_region = 'CZ' # Fallback

    # Načtení promptu ze souboru
    def _load_prompt_template():
        try:
            import os
            addon_dir = xbmcaddon.Addon().getAddonInfo('path')
            prompt_file = os.path.join(addon_dir, 'resources', 'prompts', 'ai_recommendation.txt')
            if os.path.exists(prompt_file):
                with open(prompt_file, 'r', encoding='utf-8') as f:
                    return f.read()
            xbmc.log(f"[plugin.video.milionar] Prompt file not found: {prompt_file}", level=xbmc.LOGWARNING)
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error loading prompt file: {e}", level=xbmc.LOGERROR)
        
        # Fallback hardcoded prompt (simplified safe fallback)
        return (
            "Systémová chyba načtení promptu.\n"
            "Jsi asistent pro filmy/seriály {media_label}.\n"
            "{context_instruction}\n"
            "Historie:\n{history_text}\n"
            "Uživatel: {user_input}\n"
            "Odpověz JSONem s 10 doporučeními."
        )

    prompt_template = _load_prompt_template()
    
    try:
        prompt = prompt_template.format(
            current_time_str=current_time_str,
            user_region=user_region,
            user_lang=user_lang,
            media_label=media_label,
            context_instruction=context_instruction,
            history_text=history_text,
            user_input=user_input,
            json_type_field=json_type_field,
            json_type_rule=json_type_rule
        )
    except Exception as e:
         xbmc.log(f"[plugin.video.milionar] Error formatting prompt: {e}", level=xbmc.LOGERROR)
         # Emergency Fallback if format fails (e.g. user broke braces)
         prompt = (
            f"Jsi asistent pro {media_label}. Uživatelský vstup: {user_input}. "
            f"Doporuč 10 titulů. Odpověz v validním JSON."
         )
    
    try:
        # Volání AI s tools (pro případné vyhledávání informací)
        response_text, usage = send_prompt(prompt, use_tools=_is_web_search_enabled())
        
        # Sjednocení odpovědi
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            response_raw = ''.join(parts)
        else:
            response_raw = str(response_text)
            
        # Extrakce JSON
        try:
            start = response_raw.find('{')
            end = response_raw.rfind('}')
            if start != -1 and end != -1 and end > start:
                json_str = response_raw[start:end+1]
            else:
                json_str = response_raw
                
            data = json.loads(json_str)
            
            # Validace struktury
            if 'message' not in data:
                data['message'] = "Omlouvám se, došlo k chybě ve zpracování."
            if 'buttons' not in data:
                data['buttons'] = []
            if 'recommendations' not in data:
                data['recommendations'] = []
                
            # Logování usage
            from wcs.ai.UsageTracker import track_ai_usage
            from wcs.ai.AIProvider import ai_provider
            provider = ai_provider.get_current_provider()
            model = ai_provider.get_current_model()
            track_ai_usage(provider, model, usage, "chat_recommendation")
            
            return data, usage
            
        except json.JSONDecodeError:
            xbmc.log(f"[plugin.video.milionar] JSON Decode Error: {response_raw}", level=xbmc.LOGERROR)
            return {
                "message": "Omlouvám se, nerozuměl jsem odpovědi AI. Zkus to prosím znovu.",
                "buttons": [{"label": "Zkusit znovu", "value": user_input}],
                "recommendations": []
            }, None
            
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error in chat_recommendation_step: {e}", level=xbmc.LOGERROR)
        return {
            "message": f"Došlo k chybě komunikace: {str(e)}",
            "buttons": [{"label": "Zpět", "value": "back"}],
            "recommendations": []
        }, None


# =========================
# MILLIONAIRE GAME FUNCTIONS
# =========================

def generate_millionaire_easy_questions(media_type="movie", title="", plot="", genre="", episode_title="", season_number=None, episode_number=None):
    """
    Generuje 3 snadné otázky pro Filmový Milionář - základní porozumění příběhu.
    Témata: Hlavní zápletka, klíčové události, základní postavy.
    """
    # Sestavení kontextu podle typu média
    if media_type == "series":
        if episode_title:
            media_context = f'seriál "{title}", epizoda "{episode_title}"'
            if season_number and episode_number:
                media_context += f" (série {season_number}, epizoda {episode_number})"
        else:
            media_context = f'seriál "{title}"'
    else:
        media_context = f'film "{title}"'
    
    plot_info = f"\n\nDěj: {plot}" if plot else ""
    genre_info = f"\nŽánr: {genre}" if genre else ""
    
    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 SNADNÉ otázky pro kvíz typu "Chcete být milionářem" o {media_context}.{plot_info}{genre_info}

OBTÍŽNOST: SNADNÁ - "Základní porozumění příběhu"
TÉMATA: Hlavní zápletka, klíčové události, základní postavy
CÍLOVÁ SKUPINA: Každý divák, kdo film/seriál sledoval

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Kdo je hlavní postava?
- Kde se odehrává příběh?
- Co je hlavní cíl protagonisty?
- Jak film/epizoda končí? (základní koncovka)
- Kdo je hlavní antagonista?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být JEDNODUCHÉ pro běžného diváka
- Zaměř se POUZE na základní děj a hlavní postavy
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""

    try:
        response_text, usage = send_prompt(prompt)
        
        # Vyčištění a parsování JSON odpovědi
        clean_response = _clean_ai_response(response_text)
        questions_data = json.loads(clean_response)
        
        # Převedení do standardního formátu pro millionaire controller
        questions = []
        for q_data in questions_data.get("questions", []):
            question_text = q_data.get('question', '')
            options_dict = q_data.get('options', {})
            correct_answer = q_data.get('correct', 'A')
            explanation = q_data.get('explanation', '')
            
            # Sestavení seznamu možností v pořadí A, B, C, D s prefixem
            options = [
                f"A) {options_dict.get('A', '')}",
                f"B) {options_dict.get('B', '')}",
                f"C) {options_dict.get('C', '')}",
                f"D) {options_dict.get('D', '')}"
            ]
            
            # Převedení písmena na index
            correct_index = {'A': 0, 'B': 1, 'C': 2, 'D': 3}.get(correct_answer.upper(), 0)
            
            questions.append({
                'question': question_text,
                'options': options,
                'correct': correct_index,
                'explanation': explanation
            })
        
        # Tracking AI usage
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        provider = ai_provider.get_current_provider()
        model = ai_provider.get_current_model()
        track_ai_usage(provider, model, usage, "millionaire_easy_questions")
        
        return questions
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating millionaire easy questions: {e}", level=xbmc.LOGERROR)
        return []

def generate_millionaire_medium_questions(media_type="movie", title="", plot="", genre="", episode_title="", season_number=None, episode_number=None):
    """
    Generuje 3 střední otázky pro Filmový Milionář - pozornost k detailům.
    Témata: Vztahy postav, motivace, charakterový vývoj, vedlejší postavy.
    """
    # Sestavení kontextu podle typu média
    if media_type == "series":
        if episode_title:
            media_context = f'seriál "{title}", epizoda "{episode_title}"'
            if season_number and episode_number:
                media_context += f" (série {season_number}, epizoda {episode_number})"
        else:
            media_context = f'seriál "{title}"'
    else:
        media_context = f'film "{title}"'
    
    plot_info = f"\n\nDěj: {plot}" if plot else ""
    genre_info = f"\nŽánr: {genre}" if genre else ""
    
    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 STŘEDNÍ otázky pro kvíz typu "Chcete být milionářem" o {media_context}.{plot_info}{genre_info}

OBTÍŽNOST: STŘEDNÍ - "Pozornost k detailům"
TÉMATA: Vztahy postav, motivace, charakterový vývoj, vedlejší postavy
CÍLOVÁ SKUPINA: Pozorný divák si všimne

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Jaký je vztah mezi [postava A] a [postava B]?
- Co motivuje [vedlejší postavu] k jednání?
- Jak se změní [hlavní postava] během příběhu?
- Kdo je [specifická vedlejší postava]?
- Jaké tajemství skrývá [postava]?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být STŘEDNÍ obtížnosti
- Zaměř se na DETAILY postav a jejich vztahy
- Vyžadují pozorné sledování, ne jen základní děj
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""

    try:
        response_text, usage = send_prompt(prompt)
        
        # Vyčištění a parsování JSON odpovědi
        clean_response = _clean_ai_response(response_text)
        questions_data = json.loads(clean_response)
        
        # Převedení do standardního formátu pro millionaire controller
        questions = []
        for q_data in questions_data.get("questions", []):
            question_text = q_data.get('question', '')
            options_dict = q_data.get('options', {})
            correct_answer = q_data.get('correct', 'A')
            explanation = q_data.get('explanation', '')
            
            # Sestavení seznamu možností v pořadí A, B, C, D s prefixem
            options = [
                f"A) {options_dict.get('A', '')}",
                f"B) {options_dict.get('B', '')}",
                f"C) {options_dict.get('C', '')}",
                f"D) {options_dict.get('D', '')}"
            ]
            
            # Převedení písmena na index
            correct_index = {'A': 0, 'B': 1, 'C': 2, 'D': 3}.get(correct_answer.upper(), 0)
            
            questions.append({
                'question': question_text,
                'options': options,
                'correct': correct_index,
                'explanation': explanation
            })
        
        # Tracking AI usage
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        provider = ai_provider.get_current_provider()
        model = ai_provider.get_current_model()
        track_ai_usage(provider, model, usage, "millionaire_medium_questions")
        
        return questions
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating millionaire medium questions: {e}", level=xbmc.LOGERROR)
        return []

def generate_millionaire_hard_questions(media_type="movie", title="", plot="", genre="", episode_title="", season_number=None, episode_number=None):
    """
    Generuje 3 těžké otázky pro Filmový Milionář - filmová erudice.
    Témata: Produkce, herci, režie, skryté detaily, trivia, easter eggs.
    """
    # Sestavení kontextu podle typu média
    if media_type == "series":
        if episode_title:
            media_context = f'seriál "{title}", epizoda "{episode_title}"'
            if season_number and episode_number:
                media_context += f" (série {season_number}, epizoda {episode_number})"
        else:
            media_context = f'seriál "{title}"'
    else:
        media_context = f'film "{title}"'
    
    plot_info = f"\n\nDěj: {plot}" if plot else ""
    genre_info = f"\nŽánr: {genre}" if genre else ""
    
    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 TĚŽKÉ otázky pro kvíz typu "Chcete být milionářem" o {media_context}.{plot_info}{genre_info}

OBTÍŽNOST: TĚŽKÁ - "Filmová erudice"
TÉMATA: Produkce, herci, režie, skryté detaily, trivia, easter eggs
CÍLOVÁ SKUPINA: Skutečný fanoušek pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Kdo režíroval tento film/epizodu?
- Který herec hrál [specifickou roli]?
- V jakém roce byl film natočen?
- Jaký easter egg se objevuje v [konkrétní scéně]?
- Kdo složil hudbu k filmu?
- Jaké jsou zajímavosti ze zákulisí?
- Originální název nebo lokace natáčení

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být TĚŽKÉ - pro skutečné znalce
- Zaměř se na PRODUKCI, HERCE, REŽII a TRIVIA
- Kombinace znalosti díla + filmové kultury
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""

    try:
        response_text, usage = send_prompt(prompt)
        
        # Vyčištění a parsování JSON odpovědi
        clean_response = _clean_ai_response(response_text)
        questions_data = json.loads(clean_response)
        
        # Převedení do standardního formátu pro millionaire controller
        questions = []
        for q_data in questions_data.get("questions", []):
            question_text = q_data.get('question', '')
            options_dict = q_data.get('options', {})
            correct_answer = q_data.get('correct', 'A')
            explanation = q_data.get('explanation', '')
            
            # Sestavení seznamu možností v pořadí A, B, C, D s prefixem
            options = [
                f"A) {options_dict.get('A', '')}",
                f"B) {options_dict.get('B', '')}",
                f"C) {options_dict.get('C', '')}",
                f"D) {options_dict.get('D', '')}"
            ]
            
            # Převedení písmena na index
            correct_index = {'A': 0, 'B': 1, 'C': 2, 'D': 3}.get(correct_answer.upper(), 0)
            
            questions.append({
                'question': question_text,
                'options': options,
                'correct': correct_index,
                'explanation': explanation
            })
        
        # Tracking AI usage
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        provider = ai_provider.get_current_provider()
        model = ai_provider.get_current_model()
        track_ai_usage(provider, model, usage, "millionaire_hard_questions")
        
        return questions
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating millionaire hard questions: {e}", level=xbmc.LOGERROR)
        return []

def generate_director_tip(media_type="movie", title="", question="", options=[], plot="", genre="", episode_title="", season_number=None, episode_number=None):
    """
    Generuje osobní radu od režiséra/tvůrce k aktuální otázce v millionaire hře.
    
    Args:
        media_type (str): "movie" nebo "series"
        title (str): Název filmu/seriálu
        question (str): Text aktuální otázky
        options (list): Seznam 4 možností odpovědí [A, B, C, D]
        plot (str): Děj filmu/epizody (volitelné)
        genre (str): Žánr (volitelné)
        episode_title (str): Název epizody (jen pro seriály)
        season_number (int): Číslo sezóny (jen pro seriály)
        episode_number (int): Číslo epizody (jen pro seriály)
    
    Returns:
        str: Osobní rada od režiséra/tvůrce nebo None při chybě
    """
    if media_type == "series":
        director_role = "tvůrce seriálu"
        context = f"seriálu {title}"
        if episode_title:
            context += f", konkrétně epizody '{episode_title}'"
            if season_number and episode_number:
                context += f" (S{season_number:02d}E{episode_number:02d})"
            character_note = "Odpověz jako tvůrce, který zná každou scénu a každý detail této epizody."
        else:
            character_note = "Odpověz jako tvůrce, který zná celý seriál, všechny postavy a dějové linky."
    elif media_type == "actor":
        director_role = "herec"
        context = f"herce {title}"
        character_note = "Odpověz jako herec, který zná svůj životopis, kariéru a všechny své role."
    elif media_type == "director":
        director_role = "režisér"
        context = f"režiséra {title}"
        character_note = "Odpověz jako režisér, který zná svůj životopis, kariéru a všechny své filmy."
    elif media_type == "creator":
        director_role = "tvůrce"
        context = f"tvůrce {title}"
        character_note = "Odpověz jako tvůrce, který zná svůj životopis, kariéru a všechny své seriály."
    else:
        director_role = "režiséra filmu"
        context = f"filmu {title}"
        character_note = "Odpověz jako režisér, který zná každou scénu a každý detail tohoto filmu."
    
    # Sestavení kontextu
    context_info = f"Kontext {context}"
    if plot:
        context_info += f"\nDěj: {plot[:300]}"
    if genre:
        context_info += f"\nŽánr: {genre}"
    
    # Sestavení možností pro lepší orientaci
    options_text = ""
    if len(options) >= 4:
        options_text = f"""
MOŽNOSTI ODPOVĚDÍ:
A) {options[0]}
B) {options[1]}
C) {options[2]}
D) {options[3]}"""
    
    prompt = f"""Jsi {director_role} {context}. Hráč se tě ptá na tip k této otázce:

{context_info}

OTÁZKA: {question}{options_text}

{character_note}

Poskytni krátký, užitečný tip (maximálně 2-3 věty), který:
- Neodhalí přímo správnou odpověď
- Nasměruje k správnému uvažování nebo poskytne relevantní kontext
- Použije tvoje znalosti jako {director_role}
- Bude v charakteru a stylu {director_role}
- Pomůže hráči lépe se rozhodnout

Odpověz POUZE tipem, bez úvodu či závěru."""

    try:
        # Pro tipy nepoužíváme tools
        response_text, usage = send_prompt(prompt)
        
        # Zpracování odpovědi
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            tip_text = ''.join(parts).strip()
        else:
            tip_text = str(response_text).strip()
        
        # Tracking AI usage
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        provider = ai_provider.get_current_provider()
        model = ai_provider.get_current_model()
        track_ai_usage(provider, model, usage, "director_tip")
        
        return tip_text if tip_text else None
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating director tip: {e}", level=xbmc.LOGERROR)
        return None

def _clean_ai_response(response_text):
    """Vyčistí AI odpověď pro JSON parsing."""
    if isinstance(response_text, list):
        # Spojit všechny části odpovědi
        parts = []
        for part in response_text:
            if isinstance(part, dict):
                parts.append(part.get('text', str(part)))
            else:
                parts.append(str(part))
        response_text = ''.join(parts)
    
    # Najít JSON blok
    json_start = response_text.find('{')
    json_end = response_text.rfind('}') + 1
    
    if json_start != -1 and json_end > json_start:
        return response_text[json_start:json_end]
    else:
        return response_text

def generate_quiz_batch(media_type="movie", title="", plot="", genre="", episode_title="", season_number=None, episode_number=None, total_questions=5):
    """
    Generuje všech 5 kvízových otázek najednou pro prevenci opakování.
    
    Args:
        media_type (str): "movie" nebo "series"
        title (str): Název filmu/seriálu
        plot (str): Děj filmu/epizody (volitelné)
        genre (str): Žánr (volitelné)
        episode_title (str): Název epizody (jen pro seriály)
        season_number (int): Číslo sezóny (jen pro seriály)
        episode_number (int): Číslo epizody (jen pro seriály)
        total_questions (int): Celkový počet otázek (default 5)
    
    Returns:
        tuple: (list_of_questions, usage) kde list_of_questions obsahuje dictionary pro každou otázku
    """
    categories = ["trivia", "děj", "postavy", "herci", "režie", "produkce"]
    
    if media_type == "series":
        if episode_title:
            # Pro seriály se zaměřit na konkrétní epizodu
            prompt = (
                f"Jsi tvůrce televizního kvízu. Vytvoř {total_questions} různých kvízových otázek o seriálu '{title}' specificky o epizodě '{episode_title}' (S{season_number}E{episode_number}).\n\n"
                f"KONTEXT EPIZODY:\n"
                f"Seriál: {title}\n"
                f"Epizoda: {episode_title}\n"
                f"Sezóna: {season_number}, Epizoda: {episode_number}\n"
                f"Děj epizody: {plot[:200] if plot else 'Využij své znalosti o této konkrétní epizodě'}\n\n"
                
                "KRITICKÉ INSTRUKCE:\n"
                "1. Využij VŠECHNY své znalosti o TÉTO KONKRÉTNÍ EPIZODĚ (ne celém seriálu)\n"
                "2. Každá otázka musí být z JINÉ kategorie nebo aspektu epizody\n"
                f"3. Pokryj různé kategorie: {', '.join(categories)}\n"
                "4. Žádné dvě otázky se nesmí opakovat ani podobat\n"
                "5. Otázky ať testují skutečné znalosti této epizody\n"
                "6. Každá otázka 3 možnosti odpovědi (A, B, C)\n"
                "7. Pouze JEDNA odpověď je správná u každé otázky\n"
                "8. Ostatní odpovědi ať jsou věrohodné ale špatné\n"
                "9. Stupňuj obtížnost: 1-2 lehké, 3-4 střední, 5 těžká\n\n"
                
                "FORMÁT ODPOVĚDI MUSÍ BÝT PŘESNĚ TAKTO:\n"
                "{\n"
                '  "questions": [\n'
                '    {\n'
                '      "question": "Text otázky 1",\n'
                '      "options": {"A": "Možnost A", "B": "Možnost B", "C": "Možnost C"},\n'
                '      "correct": "A|B|C"\n'
                '    },\n'
                '    {\n'
                '      "question": "Text otázky 2",\n'
                '      "options": {"A": "Možnost A", "B": "Možnost B", "C": "Možnost C"},\n'
                '      "correct": "A|B|C"\n'
                '    }\n'
                '    // ... další otázky až do 5\n'
                '  ]\n'
                "}"
            )
        else:
            # Pro seriály bez konkrétní epizody
            prompt = (
                f"Jsi tvůrce televizního kvízu. Vytvoř {total_questions} různých kvízových otázek o seriálu '{title}'.\n\n"
                f"KONTEXT SERIÁLU:\n"
                f"Seriál: {title}\n"
                f"Žánr: {genre if genre else 'Využij své znalosti'}\n"
                f"Děj: {plot[:200] if plot else 'Využij své znalosti o tomto seriálu'}\n\n"
                
                "KRITICKÉ INSTRUKCE:\n"
                "1. Využij VŠECHNY své znalosti o tomto seriálu\n"
                "2. Každá otázka musí být z JINÉ kategorie nebo aspektu seriálu\n"
                f"3. Pokryj různé kategorie: {', '.join(categories)}\n"
                "4. Žádné dvě otázky se nesmí opakovat ani podobat\n"
                "5. Otázky ať testují skutečné znalosti o seriálu\n"
                "6. Každá otázka 3 možnosti odpovědi (A, B, C)\n"
                "7. Pouze JEDNA odpověď je správná u každé otázky\n"
                "8. Ostatní odpovědi ať jsou věrohodné ale špatné\n"
                "9. Stupňuj obtížnost: 1-2 lehké, 3-4 střední, 5 těžká\n\n"
                
                "FORMÁT ODPOVĚDI MUSÍ BÝT PŘESNĚ TAKTO:\n"
                "{\n"
                '  "questions": [\n'
                '    {\n'
                '      "question": "Text otázky 1",\n'
                '      "options": {"A": "Možnost A", "B": "Možnost B", "C": "Možnost C"},\n'
                '      "correct": "A|B|C"\n'
                '    },\n'
                '    {\n'
                '      "question": "Text otázky 2",\n'
                '      "options": {"A": "Možnost A", "B": "Možnost B", "C": "Možnost C"},\n'
                '      "correct": "A|B|C"\n'
                '    }\n'
                '    // ... další otázky až do 5\n'
                '  ]\n'
                "}"
            )
    else:
        # Pro filmy
        prompt = (
            f"Jsi tvůrce filmového kvízu. Vytvoř {total_questions} různých kvízových otázek o filmu '{title}'.\n\n"
            f"KONTEXT FILMU:\n"
            f"Název: {title}\n"
            f"Žánr: {genre if genre else 'Využij své znalosti'}\n"
            f"Děj: {plot[:200] if plot else 'Využij své znalosti o tomto filmu'}\n\n"
            
            "KRITICKÉ INSTRUKCE:\n"
            "1. Využij VŠECHNY své znalosti o tomto filmu\n"
            "2. Každá otázka musí být z JINÉ kategorie nebo aspektu filmu\n"
            f"3. Pokryj různé kategorie: {', '.join(categories)}\n"
            "4. Žádné dvě otázky se nesmí opakovat ani podobat\n"
            "5. Otázky ať testují skutečné znalosti o filmu\n"
            "6. Každá otázka 3 možnosti odpovědi (A, B, C)\n"
            "7. Pouze JEDNA odpověď je správná u každé otázky\n"
            "8. Ostatní odpovědi ať jsou věrohodné ale špatné\n"
            "9. Stupňuj obtížnost: 1-2 lehké, 3-4 střední, 5 těžká\n\n"
            
            "FORMÁT ODPOVĚDI MUSÍ BÝT PŘESNĚ TAKTO:\n"
            "{\n"
            '  "questions": [\n'
            '    {\n'
            '      "question": "Text otázky 1",\n'
            '      "options": {"A": "Možnost A", "B": "Možnost B", "C": "Možnost C"},\n'
            '      "correct": "A|B|C"\n'
            '    },\n'
            '    {\n'
            '      "question": "Text otázky 2",\n'
            '      "options": {"A": "Možnost A", "B": "Možnost B", "C": "Možnost C"},\n'
            '      "correct": "A|B|C"\n'
            '    }\n'
            '    // ... další otázky až do 5\n'
            '  ]\n'
            "}"
        )
    
    try:
        # Pro kvízy nepoužíváme tools
        response_text, usage = send_prompt(prompt)
        
        # Zpracování odpovědi
        if isinstance(response_text, list):
            parts = [evt.get('text', str(evt)) if isinstance(evt, dict) else str(evt) for evt in response_text]
            response_raw = ''.join(parts)
        else:
            response_raw = str(response_text)
        
        try:
            # Robustní extrakce JSON: najdi první { a poslední }
            start = response_raw.find('{')
            end = response_raw.rfind('}')
            if start != -1 and end != -1 and end > start:
                json_str = response_raw[start:end+1]
            else:
                json_str = response_raw
            
            # Parsování JSON odpovědi
            response_json = json.loads(json_str)
            questions_data = response_json.get('questions', [])
            
            if len(questions_data) != total_questions:
                raise ValueError(f"Očekáváno {total_questions} otázek, získáno {len(questions_data)}")
            
            # Převedení do standardního formátu
            questions = []
            for i, q_data in enumerate(questions_data):
                question_text = q_data.get('question', '')
                options_dict = q_data.get('options', {})
                correct_answer = q_data.get('correct', 'A')
                
                # Sestavení seznamu možností v pořadí A, B, C
                options = [
                    options_dict.get('A', ''),
                    options_dict.get('B', ''),
                    options_dict.get('C', '')
                ]
                
                # Převedení písmena na index
                correct_index = {'A': 0, 'B': 1, 'C': 2}.get(correct_answer.upper(), 0)
                
                # Validace dat
                if not question_text or not all(options):
                    raise ValueError(f"Neplatná struktura otázky {i+1}")
                
                questions.append({
                    'question': question_text,
                    'options': options,
                    'correct_index': correct_index
                })
            
            return questions, usage
            
        except json.JSONDecodeError as e:
            xbmc.log(f"[plugin.video.milionar] Error parsing JSON response in generate_quiz_batch: {e}", level=xbmc.LOGERROR)
            raise ValueError("Neplatný JSON formát odpovědi")
            
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error in generate_quiz_batch: {e}", level=xbmc.LOGERROR)
        # Fallback - vytvořit základní otázky
        questions = []
        for i in range(total_questions):
            if media_type == "series":
                fallback_question = f"Obecná otázka {i+1} o epizodě '{episode_title}' ze seriálu {title}"
                fallback_options = [f"Možnost A", f"Možnost B", f"Možnost C"]
            else:
                fallback_question = f"Obecná otázka {i+1} o filmu {title}"
                fallback_options = [f"Možnost A", f"Možnost B", f"Možnost C"]
            
            questions.append({
                'question': fallback_question,
                'options': fallback_options,
                'correct_index': 0
            })
        return questions, None

# === NOVÉ FUNKCE PRO MILIONÁŘE NA TÉMA HERCE ===

def generate_actor_millionaire_easy_questions(actor_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 snadné otázky o herci pro hru Milionář - základní znalosti kariéry.
    Témata: Narození, nejslavnější role, základní ocenění, kariérní fakta.
    """
    known_for = known_for or []
    roles_info = "\nZnámé role: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 SNADNÉ otázky pro kvíz typu "Chcete být milionářem" o herci {actor_name}.{bio_info}{roles_info}

OBTÍŽNOST: SNADNÁ - "Základní znalosti kariéry"
TÉMATA: Narození, nejslavnější role, základní ocenění, kariérní fakta
CÍLOVÁ SKUPINA: Každý fanoušek filmů pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Kdy se herec narodil?
- Ve kterém filmu hrál hlavní roli?
- Za jaký film získal Oscar?
- Jaké je jeho nejslavnější dílo?
- V jakém roce debutoval v kině?
- Jaké je jeho rodné město?
- Který film ho proslavil?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být JEDNODUCHÉ pro běžného fanouška
- Zaměř se POUZE na základní fakta a nejslavnější díla
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "actor_millionaire_easy")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating actor easy questions: {e}", xbmc.LOGERROR)
        return []

def generate_actor_millionaire_medium_questions(actor_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 střední otázky o herci - pozornost k detailům kariéry.
    Témata: Konkrétní role, spolupráce, méně známá díla, citované výroky.
    """
    known_for = known_for or []
    roles_info = "\nZnámé role: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 STŘEDNÍ otázky pro kvíz typu "Chcete být milionářem" o herci {actor_name}.{bio_info}{roles_info}

OBTÍŽNOST: STŘEDNÍ - "Pozornost k detailům kariéry"
TÉMATA: Konkrétní role, spolupráce s dalšími osobnostmi, méně známá díla, citované výroky
CÍLOVÁ SKUPINA: Pozorný fanoušek si všimne

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Jakou roli hrál v [konkrétním filmu]?
- S kým spolupracoval v [rok]?
- Jaký slavný výrok pronesl?
- V kterém méně známém filmu hrál?
- Jakou postavu ztvárnil v [seriálu]?
- S jakým režisérem spolupracoval?
- Jaké má herecké vzdělání?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být STŘEDNÍ obtížnosti
- Zaměř se na DETAILY kariéry a spolupráce
- Vyžadují lepší znalost herce, ne jen základní fakta
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "actor_millionaire_medium")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating actor medium questions: {e}", xbmc.LOGERROR)
        return []

def generate_actor_millionaire_hard_questions(actor_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 těžké otázky o herci - herecká erudice.
    Témata: Detailní trivia, specifická ocenění, méně známé projekty, historické souvislosti.
    """
    known_for = known_for or []
    roles_info = "\nZnámé role: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi filmový historik. Vytvoř přesně 3 TĚŽKÉ otázky pro kvíz typu "Chcete být milionářem" o herci {actor_name}.{bio_info}{roles_info}

OBTÍŽNOST: TĚŽKÁ - "Herecká erudice"
TÉMATA: Detailní trivia, specifická ocenění, méně známé projekty, historické souvislosti
CÍLOVÁ SKUPINA: Skutečný znalec herce pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Za jakou roli byl nominován na [konkrétní rok]?
- V kterém divadelním představení hrál?
- Jaké má konkrétní ocenění z [festival]?
- V jakém roce debutoval v televizi?
- Jaké má vzdělání v herectví?
- Který méně známý film režíroval?
- Jaké má konkrétní hobby nebo zájmy?
- V kterém roce získal [konkrétní cenu]?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být TĚŽKÉ - pro skutečné znalce
- Zaměř se na DETAILNÍ TRIVIA a méně známé fakta
- Kombinace znalosti herce + filmové kultury
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "actor_millionaire_hard")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating actor hard questions: {e}", xbmc.LOGERROR)
        return []

# === NOVÉ FUNKCE PRO QUIZ NA TÉMA HERCE ===

def generate_actor_quiz_batch(actor_name: str, actor_bio: str = "", known_for=None, total_questions: int = 5):
    """Generuje dávku kvízových otázek o herci."""
    known_for = known_for or []
    roles_info = "\nZnámé role: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {actor_bio[:400]}" if actor_bio else ""

    prompt = f"""Jsi tvůrce filmových kvízů. Vytvoř {total_questions} otázek (A, B, C) o herci {actor_name}.{bio_info}{roles_info}

KRITÉRIA:
1. Pokryj různé oblasti: životopis, filmy, ocenění, zajímavosti, spolupráce.
2. Obtížnost stupňuj: 1-2 lehké, 3-4 střední, 5 těžká.
3. Každá otázka 3 možnosti (A, B, C) a jedna správná.
4. Žádné opakování témat.

FORMÁT:
{{
  "questions": [{{"question": "", "options": {{"A":"","B":"","C":""}}, "correct":"A"}}]
}}
"""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([str(r.get('text', r)) if isinstance(r, dict) else str(r) for r in resp])
        start = resp.find('{'); end = resp.rfind('}')
        json_str = resp[start:end+1] if start!=-1 and end!=-1 else resp
        data = json.loads(json_str)
        questions = []
        for q in data.get('questions', []):
            opts = q.get('options', {})
            correct_letter = q.get('correct', 'A').upper()
            questions.append({
                'question': q.get('question',''),
                'options': [opts.get('A',''), opts.get('B',''), opts.get('C','')],
                'correct_index': {'A':0,'B':1,'C':2}.get(correct_letter,0)
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "actor_quiz")
        return questions, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] actor quiz error: {e}", xbmc.LOGERROR)
        # fallback simple questions
        q=[]
        for i in range(total_questions):
            q.append({'question':f'Obecná otázka {i+1} o herci {actor_name}','options':['A','B','C'],'correct_index':0})
        return q, None

# === TEXT GAME PRO HERCE ===

def start_actor_text_game(actor_name: str, actor_bio: str = "", known_for=None):
    known_for = known_for or []
    roles = ", ".join(known_for[:5]) if known_for else "jeho kariéry"
    prompt = f"""Jsi vypravěč. Vytvoř úvod interaktivního příběhu ze života herce {actor_name}. Hráč si vybere, zda chce hrát období A) začátky kariéry, B) vrchol kariéry, C) osobní život. FORMÁT: {{\n  \"intro\": \"text\",\n  \"choices\":{{\"a\":\"Začátky kariéry\",\"b\":\"Vrchol kariéry\",\"c\":\"Osobní život\"}}\n}}"""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([r.get('text', str(r)) if isinstance(r, dict) else str(r) for r in resp])
        data = json.loads(resp[resp.find('{'):resp.rfind('}')+1])
        intro = data.get('intro','')
        ch = data.get('choices',{})
        choices=[ch.get('a',''), ch.get('b',''), ch.get('c','')]
        return intro, choices, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] start_actor_text_game error: {e}", xbmc.LOGERROR)
        return f"Příběh herce {actor_name} začíná...", ["Začátky kariéry","Vrchol kariéry","Osobní život"], None

def continue_actor_text_game(actor_name: str, story_history, choice_made, turn_number=1):
    context = "\n".join(story_history[-4:])
    phase = "pokračuj" if turn_number<5 else "dokonči příběh"
    prompt = f"""Pokračuj v příběhu herce {actor_name}. Kontekst:\n{context}\nVolba hráče: {choice_made}\nInstrukce: {phase}. Dej 3 možnosti a),b),c). Pokud konec, volba c) = KONEC PŘÍBĚHU. FORMÁT stejný jako dříve."""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([r.get('text', str(r)) if isinstance(r, dict) else str(r) for r in resp])
        data = json.loads(resp[resp.find('{'):resp.rfind('}')+1])
        story=data.get('story','')
        ch=data.get('choices',{})
        choices=[ch.get('a',''), ch.get('b',''), ch.get('c','')]
        is_end = (turn_number>=5) or ('KONEC' in choices[2].upper())
        return story, choices, is_end, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] continue_actor_text_game error: {e}", xbmc.LOGERROR)
        return "Příběh pokračuje...", ["Volba A","Volba B","KONEC PŘÍBĚHU"], turn_number>=5, None

# === FUNKCE PRO MILIONÁŘE S REŽISÉREM ===

def generate_director_millionaire_easy_questions(director_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 snadné otázky o režisérovi pro hru Milionář - základní znalosti kariéry.
    Témata: Narození, nejslavnější filmy, základní ocenění, kariérní fakta.
    """
    known_for = known_for or []
    films_info = "\nZnámé filmy: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 SNADNÉ otázky pro kvíz typu "Chcete být milionářem" o režisérovi {director_name}.{bio_info}{films_info}

OBTÍŽNOST: SNADNÁ - "Základní znalosti kariéry"
TÉMATA: Narození, nejslavnější filmy, základní ocenění, kariérní fakta
CÍLOVÁ SKUPINA: Každý fanoušek filmů pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Kdy se režisér narodil?
- Jaký je jeho nejslavnější film?
- Za jaký film získal Oscar?
- V jakém roce debutoval?
- Jaké je jeho rodné město?
- Který film ho proslavil?
- Jaké má základní vzdělání?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být JEDNODUCHÉ pro běžného fanouška
- Zaměř se POUZE na základní fakta a nejslavnější díla
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "director_millionaire_easy")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating director easy questions: {e}", xbmc.LOGERROR)
        return []

def generate_director_millionaire_medium_questions(director_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 střední otázky o režisérovi - pozornost k detailům kariéry.
    Témata: Konkrétní filmy, spolupráce, technika, méně známá díla.
    """
    known_for = known_for or []
    films_info = "\nZnámé filmy: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 STŘEDNÍ otázky pro kvíz typu "Chcete být milionářem" o režisérovi {director_name}.{bio_info}{films_info}

OBTÍŽNOST: STŘEDNÍ - "Pozornost k detailům kariéry"
TÉMATA: Konkrétní filmy, spolupráce s herci, technika natáčení, méně známá díla
CÍLOVÁ SKUPINA: Pozorný fanoušek si všimne

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Jakou techniku použil v [konkrétním filmu]?
- S jakým hercem spolupracoval nejčastěji?
- Jaký je jeho charakteristický styl?
- V kterém roce natočil [film]?
- Jaké má vzdělání v režii?
- S jakým kameramanem spolupracoval?
- Jaký žánr preferuje?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být STŘEDNÍ obtížnosti
- Zaměř se na DETAILY kariéry a spolupráce
- Vyžadují lepší znalost režiséra, ne jen základní fakta
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "director_millionaire_medium")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating director medium questions: {e}", xbmc.LOGERROR)
        return []

def generate_director_millionaire_hard_questions(director_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 těžké otázky o režisérovi - režisérská erudice.
    Témata: Detailní trivia, specifická ocenění, méně známé projekty, historické souvislosti.
    """
    known_for = known_for or []
    films_info = "\nZnámé filmy: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi filmový historik. Vytvoř přesně 3 TĚŽKÉ otázky pro kvíz typu "Chcete být milionářem" o režisérovi {director_name}.{bio_info}{films_info}

OBTÍŽNOST: TĚŽKÁ - "Režisérská erudice"
TÉMATA: Detailní trivia, specifická ocenění, méně známé projekty, historické souvislosti
CÍLOVÁ SKUPINA: Skutečný znalec režiséra pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Za jaký film byl nominován na [konkrétní rok]?
- V kterém divadle režíroval?
- Jaké má konkrétní ocenění z [festival]?
- V jakém roce debutoval v televizi?
- Jaké má vzdělání v režii?
- Který méně známý film produkoval?
- Jaké má konkrétní hobby nebo zájmy?
- V kterém roce získal [konkrétní cenu]?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být TĚŽKÉ - pro skutečné znalce
- Zaměř se na DETAILNÍ TRIVIA a méně známé fakta
- Kombinace znalosti režiséra + filmové kultury
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "director_millionaire_hard")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating director hard questions: {e}", xbmc.LOGERROR)
        return []

def generate_director_quiz_batch(director_name: str, director_bio: str = "", known_for=None, total_questions: int = 5):
    """Generuje dávku kvízových otázek o režisérovi."""
    known_for = known_for or []
    films_info = "\nZnámé filmy: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {director_bio[:400]}" if director_bio else ""

    prompt = f"""Jsi tvůrce filmových kvízů. Vytvoř {total_questions} otázek (A, B, C) o režisérovi {director_name}.{bio_info}{films_info}

KRITÉRIA:
1. Pokryj různé oblasti: životopis, filmy, ocenění, zajímavosti, spolupráce.
2. Obtížnost stupňuj: 1-2 lehké, 3-4 střední, 5 těžká.
3. Každá otázka 3 možnosti (A, B, C) a jedna správná.
4. Žádné opakování témat.

FORMÁT:
{{
  "questions": [{{"question": "", "options": {{"A":"","B":"","C":""}}, "correct":"A"}}]
}}
"""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([str(r.get('text', r)) if isinstance(r, dict) else str(r) for r in resp])
        start = resp.find('{'); end = resp.rfind('}')
        json_str = resp[start:end+1] if start!=-1 and end!=-1 else resp
        data = json.loads(json_str)
        questions = []
        for q in data.get('questions', []):
            opts = q.get('options', {})
            correct_letter = q.get('correct', 'A').upper()
            questions.append({
                'question': q.get('question',''),
                'options': [opts.get('A',''), opts.get('B',''), opts.get('C','')],
                'correct_index': {'A':0,'B':1,'C':2}.get(correct_letter,0)
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "director_quiz")
        return questions, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] director quiz error: {e}", xbmc.LOGERROR)
        # fallback simple questions
        q=[]
        for i in range(total_questions):
            q.append({'question':f'Obecná otázka {i+1} o režisérovi {director_name}','options':['A','B','C'],'correct_index':0})
        return q, None

# === TEXT GAME PRO REŽISÉRA ===

def start_director_text_game(director_name: str, director_bio: str = "", known_for=None):
    known_for = known_for or []
    films = ", ".join(known_for[:5]) if known_for else "jeho kariéry"
    prompt = f"""Jsi vypravěč. Vytvoř úvod interaktivního příběhu ze života režiséra {director_name}. Hráč si vybere, zda chce hrát období A) začátky kariéry, B) vrchol kariéry, C) osobní život. FORMÁT: {{\n  \"intro\": \"text\",\n  \"choices\":{{\"a\":\"Začátky kariéry\",\"b\":\"Vrchol kariéry\",\"c\":\"Osobní život\"}}\n}}"""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([r.get('text', str(r)) if isinstance(r, dict) else str(r) for r in resp])
        data = json.loads(resp[resp.find('{'):resp.rfind('}')+1])
        intro = data.get('intro','')
        ch = data.get('choices',{})
        choices=[ch.get('a',''), ch.get('b',''), ch.get('c','')]
        return intro, choices, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] start_director_text_game error: {e}", xbmc.LOGERROR)
        return f"Příběh režiséra {director_name} začíná...", ["Začátky kariéry","Vrchol kariéry","Osobní život"], None

def continue_director_text_game(director_name: str, story_history, choice_made, turn_number=1):
    context = "\n".join(story_history[-4:])
    phase = "pokračuj" if turn_number<5 else "dokonči příběh"
    prompt = f"""Pokračuj v příběhu režiséra {director_name}. Kontekst:\n{context}\nVolba hráče: {choice_made}\nInstrukce: {phase}. Dej 3 možnosti a),b),c). Pokud konec, volba c) = KONEC PŘÍBĚHU. FORMÁT stejný jako dříve."""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([r.get('text', str(r)) if isinstance(r, dict) else str(r) for r in resp])
        data = json.loads(resp[resp.find('{'):resp.rfind('}')+1])
        story=data.get('story','')
        ch=data.get('choices',{})
        choices=[ch.get('a',''), ch.get('b',''), ch.get('c','')]
        is_end = (turn_number>=5) or ('KONEC' in choices[2].upper())
        return story, choices, is_end, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] continue_director_text_game error: {e}", xbmc.LOGERROR)
        return "Příběh pokračuje...", ["Volba A","Volba B","KONEC PŘÍBĚHU"], turn_number>=5, None

# === FUNKCE PRO MILIONÁŘE S TVŮRCEM ===

def generate_creator_millionaire_easy_questions(creator_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 snadné otázky o tvůrci seriálů pro hru Milionář - základní znalosti kariéry.
    Témata: Narození, nejslavnější seriály, základní ocenění, kariérní fakta.
    """
    known_for = known_for or []
    shows_info = "\nZnámé seriály: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na seriálové kvízy. Vytvoř přesně 3 SNADNÉ otázky pro kvíz typu "Chcete být milionářem" o tvůrci seriálů {creator_name}.{bio_info}{shows_info}

OBTÍŽNOST: SNADNÁ - "Základní znalosti kariéry"
TÉMATA: Narození, nejslavnější seriály, základní ocenění, kariérní fakta
CÍLOVÁ SKUPINA: Každý fanoušek seriálů pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Kdy se tvůrce narodil?
- Jaký je jeho nejslavnější seriál?
- Za jaký seriál získal ocenění?
- V jakém roce debutoval?
- Jaké je jeho rodné město?
- Který seriál ho proslavil?
- Jaké má základní vzdělání?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být JEDNODUCHÉ pro běžného fanouška
- Zaměř se POUZE na základní fakta a nejslavnější díla
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "creator_millionaire_easy")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating creator easy questions: {e}", xbmc.LOGERROR)
        return []

def generate_creator_millionaire_medium_questions(creator_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 střední otázky o tvůrci seriálů - pozornost k detailům kariéry.
    Témata: Konkrétní seriály, spolupráce, technika, méně známá díla.
    """
    known_for = known_for or []
    shows_info = "\nZnámé seriály: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na seriálové kvízy. Vytvoř přesně 3 STŘEDNÍ otázky pro kvíz typu "Chcete být milionářem" o tvůrci seriálů {creator_name}.{bio_info}{shows_info}

OBTÍŽNOST: STŘEDNÍ - "Pozornost k detailům kariéry"
TÉMATA: Konkrétní seriály, spolupráce s herci a režiséry, technika tvorby, méně známá díla
CÍLOVÁ SKUPINA: Pozorný fanoušek si všimne

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Jakou techniku použil v [konkrétním seriálu]?
- S jakým hercem spolupracoval nejčastěji?
- Jaký je jeho charakteristický styl?
- V kterém roce vytvořil [seriál]?
- Jaké má vzdělání v tvorbě?
- S jakým režisérem spolupracoval?
- Jaký žánr preferuje?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být STŘEDNÍ obtížnosti
- Zaměř se na DETAILY kariéry a spolupráce
- Vyžadují lepší znalost tvůrce, ne jen základní fakta
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "creator_millionaire_medium")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating creator medium questions: {e}", xbmc.LOGERROR)
        return []

def generate_creator_millionaire_hard_questions(creator_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 těžké otázky o tvůrci seriálů - tvůrčí erudice.
    Témata: Detailní trivia, specifická ocenění, méně známé projekty, historické souvislosti.
    """
    known_for = known_for or []
    shows_info = "\nZnámé seriály: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi televizní historik. Vytvoř přesně 3 TĚŽKÉ otázky pro kvíz typu "Chcete být milionářem" o tvůrci seriálů {creator_name}.{bio_info}{shows_info}

OBTÍŽNOST: TĚŽKÁ - "Tvůrčí erudice"
TÉMATA: Detailní trivia, specifická ocenění, méně známé projekty, historické souvislosti
CÍLOVÁ SKUPINA: Skutečný znalec tvůrce pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Za jaký seriál byl nominován na [konkrétní rok]?
- V kterém divadle působil?
- Jaké má konkrétní ocenění z [festival]?
- V jakém roce debutoval v televizi?
- Jaké má vzdělání v tvorbě?
- Který méně známý projekt vytvořil?
- Jaké má konkrétní hobby nebo zájmy?
- V kterém roce získal [konkrétní cenu]?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být TĚŽKÉ - pro skutečné znalce
- Zaměř se na DETAILNÍ TRIVIA a méně známé fakta
- Kombinace znalosti tvůrce + televizní kultury
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "creator_millionaire_hard")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating creator hard questions: {e}", xbmc.LOGERROR)
        return []

def generate_creator_quiz_batch(creator_name: str, creator_bio: str = "", known_for=None, total_questions: int = 5):
    """Generuje dávku kvízových otázek o tvůrci."""
    known_for = known_for or []
    shows_info = "\nZnámé seriály: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {creator_bio[:400]}" if creator_bio else ""

    prompt = f"""Jsi tvůrce televizních kvízů. Vytvoř {total_questions} otázek (A, B, C) o tvůrci seriálů {creator_name}.{bio_info}{shows_info}

KRITÉRIA:
1. Pokryj různé oblasti: životopis, seriály, ocenění, zajímavosti, spolupráce s herci a režiséry.
2. Obtížnost stupňuj: 1-2 lehké, 3-4 střední, 5 těžká.
3. Každá otázka 3 možnosti (A, B, C) a jedna správná.
4. Žádné opakování témat.

FORMÁT:
{{
  "questions": [{{"question": "", "options": {{"A":"","B":"","C":""}}, "correct":"A"}}]
}}
"""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([str(r.get('text', r)) if isinstance(r, dict) else str(r) for r in resp])
        start = resp.find('{'); end = resp.rfind('}')
        json_str = resp[start:end+1] if start!=-1 and end!=-1 else resp
        data = json.loads(json_str)
        questions = []
        for q in data.get('questions', []):
            opts = q.get('options', {})
            correct_letter = q.get('correct', 'A').upper()
            questions.append({
                'question': q.get('question',''),
                'options': [opts.get('A',''), opts.get('B',''), opts.get('C','')],
                'correct_index': {'A':0,'B':1,'C':2}.get(correct_letter,0)
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "creator_quiz")
        return questions, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] creator quiz error: {e}", xbmc.LOGERROR)
        # fallback simple questions
        q=[]
        for i in range(total_questions):
            q.append({'question':f'Obecná otázka {i+1} o tvůrci {creator_name}','options':['A','B','C'],'correct_index':0})
        return q, None

# === TEXT GAME PRO TVŮRCE ===

def start_creator_text_game(creator_name: str, creator_bio: str = "", known_for=None):
    known_for = known_for or []
    shows = ", ".join(known_for[:5]) if known_for else "jeho kariéry"
    prompt = f"""Jsi vypravěč. Vytvoř úvod interaktivního příběhu ze života tvůrce seriálů {creator_name}. Hráč si vybere, zda chce hrát období A) začátky kariéry, B) vrchol kariéry, C) osobní život. FORMÁT: {{\n  \"intro\": \"text\",\n  \"choices\":{{\"a\":\"Začátky kariéry\",\"b\":\"Vrchol kariéry\",\"c\":\"Osobní život\"}}\n}}"""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([r.get('text', str(r)) if isinstance(r, dict) else str(r) for r in resp])
        data = json.loads(resp[resp.find('{'):resp.rfind('}')+1])
        intro = data.get('intro','')
        ch = data.get('choices',{})
        choices=[ch.get('a',''), ch.get('b',''), ch.get('c','')]
        return intro, choices, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] start_creator_text_game error: {e}", xbmc.LOGERROR)
        return f"Příběh tvůrce {creator_name} začíná...", ["Začátky kariéry","Vrchol kariéry","Osobní život"], None

def continue_creator_text_game(creator_name: str, story_history, choice_made, turn_number=1):
    context = "\n".join(story_history[-4:])
    phase = "pokračuj" if turn_number<5 else "dokonči příběh"
    prompt = f"""Pokračuj v příběhu tvůrce seriálů {creator_name}. Kontekst:\n{context}\nVolba hráče: {choice_made}\nInstrukce: {phase}. Dej 3 možnosti a),b),c). Pokud konec, volba c) = KONEC PŘÍBĚHU. FORMÁT stejný jako dříve."""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([r.get('text', str(r)) if isinstance(r, dict) else str(r) for r in resp])
        data = json.loads(resp[resp.find('{'):resp.rfind('}')+1])
        story=data.get('story','')
        ch=data.get('choices',{})
        choices=[ch.get('a',''), ch.get('b',''), ch.get('c','')]
        is_end = (turn_number>=5) or ('KONEC' in choices[2].upper())
        return story, choices, is_end, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] continue_creator_text_game error: {e}", xbmc.LOGERROR)
        return "Příběh pokračuje...", ["Volba A","Volba B","KONEC PŘÍBĚHU"], turn_number>=5, None

def get_millionaire_difficulty_header():
    """
    Vrací text hlavní obtížnosti podle nastavení addonu.
    Tento text se přidá na začátek promptů pro milionář hru.
    """
    addon = xbmcaddon.Addon()
    difficulty_setting = addon.getSetting("millionaire_difficulty")
    
    # Mapování hodnot na obtížnost (0=snadná, 1=střední, 2=těžká)
    if difficulty_setting == "0":  # Snadná
        return "HLAVNÍ OBTÍŽNOST: SNADNÁ - Formuluj otázky jednoduše z obecně známých informací. Vyhýbej se detailům a insider faktům. Možnosti odpovědí musí být zřetelně odlišitelné.\n\n"
    elif difficulty_setting == "2":  # Těžká
        return "HLAVNÍ OBTÍŽNOST: TĚŽKÁ - Opírej otázky o hlubší znalosti a přesné kontexty. Využívej odbornější pojmy, pokud jsou relevantní. Možnosti odpovědí musí být velmi věrohodné a pečlivě odlišitelné až po důkladném uvážení.\n\n"
    else:  # Střední (default) nebo "1"
        return "HLAVNÍ OBTÍŽNOST: STŘEDNÍ - Pracuj s mírnou mírou detailu nad mainstream. Můžeš použít méně známé souvislosti, ale zachovej srozumitelnost. Možnosti odpovědí musí být věrohodné, nikoli zavádějící.\n\n"

def get_millionaire_difficulty_instruction():
    """
    Vrací doplňující instrukci podle nastavení obtížnosti.
    Tato instrukce se vkládá doprostřed promptů.
    """
    addon = xbmcaddon.Addon()
    difficulty_setting = addon.getSetting("millionaire_difficulty")
    
    # Mapování hodnot na obtížnost (0=snadná, 1=střední, 2=těžká)
    if difficulty_setting == "0":  # Snadná
        return "DOPLŇUJÍCÍ INSTRUKCE: Uplatni hlavní obtížnost v duchu základů bez detailů."
    elif difficulty_setting == "2":  # Těžká
        return "DOPLŇUJÍCÍ INSTRUKCE: Uplatni hlavní obtížnost s důrazem na hlubší znalosti a věrohodné distraktory."
    else:  # Střední (default) nebo "1"
        return "DOPLŇUJÍCÍ INSTRUKCE: Uplatni hlavní obtížnost s mírným navýšením detailu, zároveň zachovej srozumitelnost."

# === NOVÉ FUNKCE PRO SCÉNÁRISTY (WRITER) ===

def generate_writer_millionaire_easy_questions(writer_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 snadné otázky o scénáristovi pro hru Milionář - základní znalosti kariéry.
    Témata: Narození, nejslavnější filmy, základní ocenění, kariérní fakta.
    """
    known_for = known_for or []
    movies_info = "\nZnámé filmy: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 SNADNÉ otázky pro kvíz typu "Chcete být milionářem" o scénáristovi {writer_name}.{bio_info}{movies_info}

OBTÍŽNOST: SNADNÁ - "Základní znalosti kariéry"
TÉMATA: Narození, nejslavnější filmy, základní ocenění, kariérní fakta
CÍLOVÁ SKUPINA: Každý fanoušek filmů pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Kdy se scénárista narodil?
- Jaký je jeho nejslavnější film?
- Za jaký film získal ocenění?
- V jakém roce debutoval?
- Jaké je jeho rodné město?
- Který film ho proslavil?
- Jaké má základní vzdělání?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být JEDNODUCHÉ pro běžného fanouška
- Zaměř se POUZE na základní fakta a nejslavnější díla
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "writer_millionaire_easy")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating writer easy questions: {e}", xbmc.LOGERROR)
        return []

def generate_writer_millionaire_medium_questions(writer_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 střední otázky o scénáristovi - pozornost k detailům kariéry.
    Témata: Konkrétní filmy, spolupráce, technika psaní, méně známá díla.
    """
    known_for = known_for or []
    movies_info = "\nZnámé filmy: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi expert na filmové kvízy. Vytvoř přesně 3 STŘEDNÍ otázky pro kvíz typu "Chcete být milionářem" o scénáristovi {writer_name}.{bio_info}{movies_info}

OBTÍŽNOST: STŘEDNÍ - "Pozornost k detailům kariéry"
TÉMATA: Konkrétní filmy, spolupráce s režiséry a herci, technika psaní, méně známá díla
CÍLOVÁ SKUPINA: Pozorný fanoušek si všimne

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Jakou techniku psaní použil v [konkrétním filmu]?
- S jakým režisérem spolupracoval nejčastěji?
- Jaký je jeho charakteristický styl?
- V kterém roce napsal [film]?
- Jaké má vzdělání v psaní?
- S jakým hercem spolupracoval?
- Jaký žánr preferuje?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být STŘEDNÍ obtížnosti
- Zaměř se na DETAILY kariéry a spolupráce
- Vyžadují lepší znalost scénáristy, ne jen základní fakta
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "writer_millionaire_medium")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating writer medium questions: {e}", xbmc.LOGERROR)
        return []

def generate_writer_millionaire_hard_questions(writer_name: str, bio: str = "", known_for=None):
    """
    Generuje 3 těžké otázky o scénáristovi - scénáristická erudice.
    Témata: Detailní trivia, specifická ocenění, méně známé projekty, historické souvislosti.
    """
    known_for = known_for or []
    movies_info = "\nZnámé filmy: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {bio[:500]}" if bio else ""

    difficulty_header = get_millionaire_difficulty_header()
    difficulty_instruction = get_millionaire_difficulty_instruction()
    prompt = f"""{difficulty_header}Jsi filmový historik. Vytvoř přesně 3 TĚŽKÉ otázky pro kvíz typu "Chcete být milionářem" o scénáristovi {writer_name}.{bio_info}{movies_info}

OBTÍŽNOST: TĚŽKÁ - "Scénáristická erudice"
TÉMATA: Detailní trivia, specifická ocenění, méně známé projekty, historické souvislosti
CÍLOVÁ SKUPINA: Skutečný znalec scénáristy pozná

{difficulty_instruction}

PŘÍKLADY TÉMAT:
- Za jaký film byl nominován na [konkrétní rok]?
- V kterém divadle působil?
- Jaké má konkrétní ocenění z [festival]?
- V jakém roce debutoval v televizi?
- Jaké má vzdělání v psaní?
- Který méně známý projekt napsal?
- Jaké má konkrétní hobby nebo zájmy?
- V kterém roce získal [konkrétní cenu]?

FORMÁT ODPOVĚDI (PŘESNĚ):
{{
  "questions": [
    {{
      "question": "Text otázky",
      "options": {{"A": "Možnost A", "B": "Možnost B", "C": "Možnost C", "D": "Možnost D"}},
      "correct": "A|B|C|D",
      "explanation": "Krátké vysvětlení správné odpovědi"
    }}
  ]
}}

DŮLEŽITÉ:
- Přesně 3 otázky
- Každá otázka má 4 možnosti (A, B, C, D)
- Správná odpověď jako písmeno (A, B, C nebo D)
- Otázky musí být TĚŽKÉ - pro skutečné znalce
- Zaměř se na DETAILNÍ TRIVIA a méně známé fakta
- Kombinace znalosti scénáristy + filmové kultury
- KAŽDÁ MOŽNOST ODPOVĚDI MÁ MAXIMÁLNĚ 25 ZNAKŮ (včetně mezer) - buď stručná a jasná"""
    try:
        response_text, usage = send_prompt(prompt)
        clean_resp = _clean_ai_response(response_text)
        data = json.loads(clean_resp)
        questions = []
        for q in data.get("questions", []):
            opts = q.get("options", {})
            correct_letter = q.get("correct", "A").upper()
            questions.append({
                "question": q.get("question", ""),
                "options": [f"A) {opts.get('A', '')}", f"B) {opts.get('B', '')}", f"C) {opts.get('C', '')}", f"D) {opts.get('D', '')}"],
                "correct": {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter, 0),
                "explanation": q.get("explanation", "")
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "writer_millionaire_hard")
        return questions
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error generating writer hard questions: {e}", xbmc.LOGERROR)
        return []

def generate_writer_quiz_batch(writer_name: str, writer_bio: str = "", known_for=None, total_questions: int = 5):
    """Generuje dávku kvízových otázek o scénáristovi."""
    known_for = known_for or []
    movies_info = "\nZnámé filmy: " + ", ".join(known_for[:10]) if known_for else ""
    bio_info = f"\n\nŽivotopis: {writer_bio[:400]}" if writer_bio else ""

    prompt = f"""Jsi tvůrce filmových kvízů. Vytvoř {total_questions} otázek (A, B, C) o scénáristovi {writer_name}.{bio_info}{movies_info}

KRITÉRIA:
1. Pokryj různé oblasti: životopis, filmy, ocenění, zajímavosti, spolupráce s režiséry a herci.
2. Obtížnost stupňuj: 1-2 lehké, 3-4 střední, 5 těžká.
3. Každá otázka 3 možnosti (A, B, C) a jedna správná.
4. Žádné opakování témat.

FORMÁT:
{{
  "questions": [{{"question": "", "options": {{"A":"","B":"","C":""}}, "correct":"A"}}]
}}
"""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([str(r.get('text', r)) if isinstance(r, dict) else str(r) for r in resp])
        start = resp.find('{'); end = resp.rfind('}')
        json_str = resp[start:end+1] if start!=-1 and end!=-1 else resp
        data = json.loads(json_str)
        questions = []
        for q in data.get('questions', []):
            opts = q.get('options', {})
            correct_letter = q.get('correct', 'A').upper()
            questions.append({
                'question': q.get('question',''),
                'options': [opts.get('A',''), opts.get('B',''), opts.get('C','')],
                'correct_index': {'A':0,'B':1,'C':2}.get(correct_letter,0)
            })
        from wcs.ai.UsageTracker import track_ai_usage
        from wcs.ai.AIProvider import ai_provider
        track_ai_usage(ai_provider.get_current_provider(), ai_provider.get_current_model(), usage, "writer_quiz")
        return questions, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] writer quiz error: {e}", xbmc.LOGERROR)
        # fallback simple questions
        q=[]
        for i in range(total_questions):
            q.append({'question':f'Obecná otázka {i+1} o scénáristovi {writer_name}','options':['A','B','C'],'correct_index':0})
        return q, None

# === TEXT GAME PRO SCÉNÁRISTY ===

def start_writer_text_game(writer_name: str, writer_bio: str = "", known_for=None):
    known_for = known_for or []
    movies = ", ".join(known_for[:5]) if known_for else "jeho kariéry"
    prompt = f"""Jsi vypravěč. Vytvoř úvod interaktivního příběhu ze života scénáristy {writer_name}. Hráč si vybere, zda chce hrát období A) začátky kariéry, B) vrchol kariéry, C) osobní život. FORMÁT: {{\n  \"intro\": \"text\",\n  \"choices\":{{\"a\":\"Začátky kariéry\",\"b\":\"Vrchol kariéry\",\"c\":\"Osobní život\"}}\n}}"""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([r.get('text', str(r)) if isinstance(r, dict) else str(r) for r in resp])
        data = json.loads(resp[resp.find('{'):resp.rfind('}')+1])
        intro = data.get('intro','')
        ch = data.get('choices',{})
        choices=[ch.get('a',''), ch.get('b',''), ch.get('c','')]
        return intro, choices, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] start_writer_text_game error: {e}", xbmc.LOGERROR)
        return f"Příběh scénáristy {writer_name} začíná...", ["Začátky kariéry","Vrchol kariéry","Osobní život"], None

def continue_writer_text_game(writer_name: str, story_history, choice_made, turn_number=1):
    context = "\n".join(story_history[-4:])
    phase = "pokračuj" if turn_number<5 else "dokonči příběh"
    prompt = f"""Pokračuj v příběhu scénáristy {writer_name}. Kontekst:\n{context}\nVolba hráče: {choice_made}\nInstrukce: {phase}. Dej 3 možnosti a),b),c). Pokud konec, volba c) = KONEC PŘÍBĚHU. FORMÁT stejný jako dříve."""
    try:
        resp, usage = send_prompt(prompt)
        if isinstance(resp, list):
            resp = ''.join([r.get('text', str(r)) if isinstance(r, dict) else str(r) for r in resp])
        data = json.loads(resp[resp.find('{'):resp.rfind('}')+1])
        story=data.get('story','')
        ch=data.get('choices',{})
        choices=[ch.get('a',''), ch.get('b',''), ch.get('c','')]
        is_end = (turn_number>=5) or ('KONEC' in choices[2].upper())
        return story, choices, is_end, usage
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] continue_writer_text_game error: {e}", xbmc.LOGERROR)
        return "Příběh pokračuje...", ["Volba A","Volba B","KONEC PŘÍBĚHU"], turn_number>=5, None