import xbmc
import xbmcaddon
from wcs.ai import AIHandler as ai_handler
from wcs import user_data
from wcs.ai.AIProvider import ai_provider


class MillionaireGameController:
    """
    Kontrolér specificky pro hru Filmový Milionář.
    Obsahuje logiku pro otázky ve stylu "Chcete být milionářem" na téma filmů.
    """
    
    def __init__(self, title="", plot="", genre="", media_type="movie", episode_title="", season_number=None, episode_number=None, actor_data=None):
        self.addon = xbmcaddon.Addon()
        
        # Informace o médiu
        self.title = title
        self.plot = plot
        self.genre = genre
        self.media_type = media_type  # "movie", "series", "actor" nebo "director"
        self.episode_title = episode_title  # Pro seriály
        self.season_number = season_number  # Pro seriály
        self.episode_number = episode_number  # Pro seriály
        self.actor_data = actor_data or {}
        
        # Základní herní stav
        self.game_state = "intro"
        self.game_type = "millionaire"
        
        # UI callbacks
        self.update_display_callback = None
        self.update_buttons_callback = None
        self.update_sidebar_callback = None
        self.show_error_callback = None
        self.set_focus_callback = None
        self.start_countdown_callback = None
        self.close_dialog_callback = None
        
        # Millionaire-specifické vlastnosti
        self.current_level = "easy"  # "easy", "medium", "hard"
        self.current_question_index = 0
        self.score = 0
        self.total_winnings = 0
        self.lifelines_used = []
        self.current_choices = []
        self.lifeline_5050_used = False
        self.lifeline_5050_indices = None  # (správná, náhodná špatná)
        self.lifeline_audience_used = False
        self.lifeline_audience_results = None  # [procento_A, procento_B, procento_C, procento_D]
        self.lifeline_tip_used = False
        self.ladder_anim_toggle = False
        self.ladder_anim_running = False
        
    def set_ui_callbacks(self, update_display, update_buttons, update_sidebar, show_error, set_focus=None, start_countdown=None, close_dialog=None):
        """Nastaví callback funkce pro komunikaci s UI."""
        self.update_display_callback = update_display
        self.update_buttons_callback = update_buttons
        self.update_sidebar_callback = update_sidebar
        self.show_error_callback = show_error
        self.set_focus_callback = set_focus
        self.start_countdown_callback = start_countdown
        self.close_dialog_callback = close_dialog
        
        # Nový callback pro aktualizaci grafického žebříčku
        self.update_millionaire_ladder_callback = None
    
    def set_return_to_menu_callback(self, callback):
        """Nastaví callback pro návrat na hlavní menu."""
        self.return_to_menu_callback = callback
    
    def _add_ai_info_header(self, text: str) -> str:
        """Přidá informaci o AI modelu na začátek textu velmi malým písmem."""
        # AI info odstraněno na požádání uživatele
        return text
        
    def _get_tip_text(self, short=False):
        """Vrací dynamický text pro tip podle typu média."""
        base_text = ""
        if self.media_type == "series":
            base_text = "Tip od tvůrce"
        elif self.media_type == "actor":
            base_text = "Tip od herce"
        elif self.media_type == "director":
            base_text = "Tip od režiséra"
        elif self.media_type == "writer":
            base_text = "Tip od scénáristy"
        elif self.media_type == "creator":
            base_text = "Tip od tvůrce"
        else:
            base_text = "Tip od režiséra"
        
        if short:
            return base_text
        else:
            # Přidat pomlčku a popis
            if self.media_type == "series":
                return f"{base_text} - osobní rada od tvůrce seriálu"
            elif self.media_type == "actor":
                return f"{base_text} - osobní rada od herce"
            elif self.media_type == "director":
                return f"{base_text} - osobní rada od režiséra"
            elif self.media_type == "writer":
                return f"{base_text} - osobní rada od scénáristy"
            elif self.media_type == "creator":
                return f"{base_text} - osobní rada od tvůrce"
            else:
                return f"{base_text} - osobní rada od režiséra"
    
    def _track_ai_usage(self, usage, usage_type):
        """Společná metoda pro sledování AI usage."""
        if usage:
            from wcs.ai.UsageTracker import track_ai_usage
            provider = ai_provider.get_current_provider()
            model = ai_provider.get_current_model()
            track_ai_usage(provider, model, usage, usage_type)
        
    def start_specific_game(self):
        """Spustí hru Filmový Milionář s welcome screenem."""
        xbmc.log("[plugin.video.milionar] MillionaireGameController.start_specific_game called", level=xbmc.LOGINFO)
        self.reset_lifelines()
        self.game_state = "millionaire_welcome"
        self.current_level = 0
        self.total_winnings = 0
        self.lifelines_used = []
        
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
            
        # Zobrazit welcome screen
        self.show_millionaire_welcome()

    def show_millionaire_welcome(self):
        """Zobrazí uvítací obrazovku hry Filmový Milionář."""
        # Sestavení textu podle typu média
        if self.media_type == "series" and self.episode_title:
            media_info = f'"{self.title}" - {self.episode_title}'
            if self.season_number and self.episode_number:
                media_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
            millionaire_headline = f"Milionář na základě epizody seriálu {media_info}"
            millionaire_type_line = f"• 9 otázek různé obtížnosti o epizodě seriálu"
            challenge_line = "Jste připraveni vsadit své znalosti seriálu?"
        elif self.media_type == "series":
            media_info = f'"{self.title}"'
            millionaire_headline = f"Milionář na základě seriálu {media_info}"
            millionaire_type_line = f"• 9 otázek různé obtížnosti o seriálu"
            challenge_line = "Jste připraveni vsadit své znalosti seriálu?"
        elif self.media_type == "actor":
            media_info = f'"{self.title}"'
            millionaire_headline = f"Milionář na základě herce {media_info}"
            millionaire_type_line = f"• 9 otázek různé obtížnosti o herci"
            challenge_line = "Jste připraveni vsadit své znalosti o herci?"
        elif self.media_type == "director":
            media_info = f'"{self.title}"'
            millionaire_headline = f"Milionář na základě režiséra {media_info}"
            millionaire_type_line = f"• 9 otázek různé obtížnosti o režiséru"
            challenge_line = "Jste připraveni vsadit své znalosti o režiséru?"
        elif self.media_type == "writer":
            media_info = f'"{self.title}"'
            millionaire_headline = f"Milionář na základě scénáristy {media_info}"
            millionaire_type_line = f"• 9 otázek různé obtížnosti o scénáristovi"
            challenge_line = "Jste připraveni vsadit své znalosti o scénáristovi?"
        elif self.media_type == "creator":
            media_info = f'"{self.title}"'
            millionaire_headline = f"Milionář na základě tvůrce {media_info}"
            millionaire_type_line = f"• 9 otázek různé obtížnosti o tvůrci"
            challenge_line = "Jste připraveni vsadit své znalosti o tvůrci seriálů?"
        else:
            media_info = f'"{self.title}"'
            millionaire_headline = f"Milionář na základě filmu {media_info}"
            millionaire_type_line = f"• 9 otázek různé obtížnosti o filmu"
            challenge_line = "Jste připraveni vsadit své znalosti filmu?"

        # Přidání žánru pokud je dostupný
        genre_text = f" • Žánr: {self.genre}" if self.genre else ""

        # Krátký popis děje pokud je dostupný
        plot_preview = ""
        if self.plot and len(self.plot) > 50:
            plot_preview = f"\nDĚJ: {self.plot[:150]}{'...' if len(self.plot) > 150 else ''}\n"

        # Dynamický text pro tip podle typu média
        tip_text = self._get_tip_text()

        welcome_text = f"""CO VÁS ČEKÁ:
• {millionaire_headline}
{millionaire_type_line}
• Postupně rostoucí virtuální výhry
{plot_preview}
POMOCNÍCI:
• Nápověda 50:50 - odstraní 2 špatné odpovědi
• {tip_text}
• Hlasování diváků - statistická pomoc

Připravte se na cestu za miliónem!

{challenge_line}"""

        # Nastavit property pro welcome sidebar
        self._set_millionaire_film_info()

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(welcome_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Začít hru", "Pravidla", "Zpět na menu", "Zavřít")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()
        # Nastavit fokus na tlačítko "Začít hru" (index 0)
        if self.set_focus_callback:
            self.set_focus_callback(0)

    def _set_millionaire_film_info(self):
        """Nastaví property s informacemi o filmu pro welcome sidebar."""
        try:
            # Získat reference na hlavní window pro nastavení property
            import xbmcgui
            window = xbmcgui.Window(xbmcgui.getCurrentWindowId())
            
            # Sestavit informace o filmu podle typu média
            if self.media_type == "series" and self.episode_title:
                film_info = f"Seriál: {self.title}\nEpizoda: {self.episode_title}"
                if self.season_number and self.episode_number:
                    film_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
            elif self.media_type == "series":
                film_info = f"Seriál: {self.title}"
            elif self.media_type == "actor":
                film_info = f"Herec: {self.title}"
            elif self.media_type == "director":
                film_info = f"Režisér: {self.title}"
            elif self.media_type == "creator":
                film_info = f"Tvůrce: {self.title}"
            else:
                film_info = f"Film: {self.title}"
            
            # Přidat žánr pokud je dostupný
            if self.genre:
                film_info += f"\nŽánr: {self.genre}"
            
            window.setProperty('WCS.Millionaire.FilmInfo', film_info)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error setting millionaire film info: {e}", level=xbmc.LOGERROR)

    def show_millionaire_rules(self):
        """Zobrazí detailní pravidla hry."""
        # Texty podle typu média
        if self.media_type == "series":
            game_title = "SERIÁLOVÉHO MILIONÁŘE"
            if self.episode_title:
                episode_info = f"{self.title} - {self.episode_title}"
                if self.season_number and self.episode_number:
                    episode_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
            else:
                episode_info = f"{self.title}"
            intro_line = f"Odpovídáte na 9 otázek o seriálu"
            ai_line = "AI vytvoří otázky na základě seriálu"
            final_line = "Jste připraveni vstoupit do světa seriálového milionáře?"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
            intro_line = f"Odpovídáte na 9 otázek o herci"
            ai_line = "AI vytvoří otázky na základě životopisu"
            final_line = "Jste připraveni vstoupit do světa milionáře s hercem?"
        elif self.media_type == "director":
            game_title = f"MILIONÁŘ S REŽISÉREM"
            intro_line = f"Odpovídáte na 9 otázek o režisérovi"
            ai_line = "AI vytvoří otázky na základě životopisu"
            final_line = "Jste připraveni vstoupit do světa milionáře s režisérem?"
        elif self.media_type == "creator":
            game_title = f"MILIONÁŘ S TVŮRCEM"
            intro_line = f"Odpovídáte na 9 otázek o tvůrci"
            ai_line = "AI vytvoří otázky na základě životopisu"
            final_line = "Jste připraveni vstoupit do světa milionáře s tvůrcem?"
        else:
            game_title = "FILMOVÉHO MILIONÁŘE"
            intro_line = f"Odpovídáte na 9 otázek o filmu"
            ai_line = "AI vytvoří otázky na základě filmu"
            final_line = "Jste připraveni vstoupit do světa filmového milionáře?"

        rules_text = f"""PRŮBĚH HRY:
• {intro_line}
• Otázky jsou rozdělené do 3 úrovní
• Špatná odpověď znamená konec hry

HERNÍ MECHANIKA:
• {ai_line}
• Vždy máte 4 možnosti odpovědí (A, B, C, D)
• Postupná eskalace obtížnosti

POMOCNÍCI:
• 50:50 - odstraní 2 špatné odpovědi
• {self._get_tip_text()}
• Hlasování diváků - statistická pomoc

{final_line}"""

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(rules_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Začít hru", "Zpět", "Zpět na menu", "Zavřít")

    def make_choice(self, choice_index):
        """Hlavní metoda pro zpracování voleb."""
        self.handle_game_choice(choice_index)
    
    def handle_game_choice(self, choice_index):
        """Zpracuje volbu specifickou pro millionaire hru."""
        if self.game_state == "millionaire_welcome":
            if choice_index == 0:  # Začít hru
                self.show_placeholder_game()
            elif choice_index == 1:  # Pravidla
                self.game_state = "millionaire_rules"
                self.show_millionaire_rules()
            elif choice_index == 2:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "millionaire_rules":
            if choice_index == 0:  # Začít hru
                self.show_placeholder_game()
            elif choice_index == 1:  # Zpět
                self.game_state = "millionaire_welcome"
                self.show_millionaire_welcome()
            elif choice_index == 2:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state in ["generating_easy", "generating_medium", "generating_hard"]:
            # Loading screen - tlačítka jsou jen informativní, nic nedělají
            pass
        elif self.game_state == "answer_pending":
            # Během countdown nelze nic dělat - tlačítka jsou jen informativní
            pass
        elif self.game_state in ["playing_easy", "playing_medium", "playing_hard"]:
            # Odpověď na otázku (tlačítka 0-3 jsou odpovědi A-D)
            xbmc.log(f"[plugin.video.milionar] Button {choice_index} pressed in game state {self.game_state}", level=xbmc.LOGINFO)
            self.answer_question(choice_index)

        elif self.game_state == "answer_feedback":
            if choice_index == 0:  # Pokračovat
                self.continue_after_feedback()
            elif choice_index == 1:  # Nová hra
                self.show_placeholder_game()
            elif choice_index == 2:  # Zpět na menu
                self.start_game_selection()  
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "error":
            if choice_index == 0:  # Zkusit znovu
                # Zopakovat volání, které selhalo, ale zachovat stav
                if self.current_level == "easy" or not self.questions_easy:
                    # Selhaly snadné otázky - začít od začátku
                    self.start_millionaire_game()
                elif self.current_level == "medium" or (self.questions_easy and not self.questions_medium):
                    # Selhaly střední otázky - zopakovat střední
                    # Zachovat snadné otázky a score
                    # Použít existující loading screen
                    self.show_level_transition("medium")
                elif self.current_level == "hard" or (self.questions_medium and not self.questions_hard):
                    # Selhaly těžké otázky - zopakovat těžké
                    # Zachovat snadné a střední otázky a score
                    # Použít existující loading screen
                    self.show_level_transition("hard")
                else:
                    # Fallback - začít od začátku
                    self.start_millionaire_game()
            elif choice_index == 1:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 2:  # Nová hra
                self.show_placeholder_game()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "game_over":
            if choice_index == 0:  # Nová hra
                self.show_placeholder_game()
            elif choice_index == 1:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 2:  # Jiná hra
                self.start_game_selection()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "game_finished":
            # Rozliším, zda už jsou zobrazené výsledky nebo ne
            if not hasattr(self, '_results_shown') or not self._results_shown:
                if choice_index == 0:  # Výsledky
                    self.show_final_score()
                elif choice_index == 1:  # Nová hra
                    self.show_placeholder_game()
                elif choice_index == 2:  # Zpět na menu
                    self.start_game_selection()
                elif choice_index == 3:  # Zavřít
                    self.end_game()
            else:
                if choice_index == 0:  # Nová hra
                    self.show_placeholder_game()
                elif choice_index == 1:  # Zpět na menu
                    self.start_game_selection()
                elif choice_index == 2:  # Zavřít
                    self.end_game()

    def show_placeholder_game(self):
        """Spustí skutečnou hru Filmový Milionář."""
        self.start_millionaire_game()
    
    def start_millionaire_game(self):
        """Spustí hru Filmový Milionář - generuje první sadu otázek."""
        xbmc.log("[plugin.video.milionar] Starting Millionaire game", level=xbmc.LOGINFO)
        
        self.reset_lifelines()
        self.game_state = "generating_easy"
        self.current_level = "easy"
        self.current_question_index = 0
        self.score = 0
        self.questions_easy = []
        self.questions_medium = []
        self.questions_hard = []
        self.current_question = None
        
        # Zobrazit loading screen
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        elif self.media_type == "director":
            game_title = f"MILIONÁŘ S REŽISÉREM"
        elif self.media_type == "creator":
            game_title = f"MILIONÁŘ S TVŮRCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
        
        # Sestavím info o epizodě pokud je dostupné
        episode_info = ""
        if self.media_type == "series" and self.episode_title:
            episode_info = f"Epizoda: {self.episode_title}"
            if self.season_number and self.episode_number:
                episode_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
            episode_info = f"\n{episode_info}"
        
        # Přidám zkrácený popis filmu/epizody na max 3 řádky (cca 240 znaků)
        plot_info = ""
        if self.plot:
            max_len = 240
            short_plot = self.plot.strip().replace('\n', ' ')
            if len(short_plot) > max_len:
                cut = short_plot[:max_len].rsplit(' ', 1)[0]
                short_plot = cut + "..."
            plot_info = f"\n{short_plot}"
        
        # Sestavím bloky s prázdnými řádky mezi sekcemi
        if self.media_type == "movie":
            main_label = f"Film: {self.title}"
        elif self.media_type == "actor":
            main_label = f"Herec: {self.title}"
        elif self.media_type == "director":
            main_label = f"Režisér: {self.title}"
        elif self.media_type == "creator":
            main_label = f"Tvůrce: {self.title}"
        elif self.media_type == "series":
            main_label = f"Seriál: {self.title}"
        else:
            main_label = f"Film: {self.title}"
        info_lines = [main_label]
        if episode_info:
            info_lines.append(episode_info.strip())
        if plot_info:
            info_lines.append(f"Děj: {plot_info.strip()}")
        info_block = "\n\n".join(info_lines)

        loading_text = f"""AI připravuje {game_title.lower()}

{info_block}

Generuji SNADNÉ otázky (1-3)

Chvíli strpení..."""

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(loading_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Načítání...", "...", "...", "...")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()
        self.stop_ladder_animation()
        self.animate_generating_ladder([111,112,113], duration=3.0, interval=0.2)
        
        # Spustit generování otázek v separátním threadu
        import threading
        thread = threading.Thread(target=self._generate_easy_questions)
        thread.daemon = True
        thread.start()
    
    def _generate_easy_questions(self):
        """Generuje snadné otázky v background."""
        try:
            from wcs.ai import AIHandler as ai_handler
            
            if self.media_type == "actor":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_easy = ai_handler.generate_actor_millionaire_easy_questions(name, bio, known_for)
            elif self.media_type == "director":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_easy = ai_handler.generate_director_millionaire_easy_questions(name, bio, known_for)
            elif self.media_type == "writer":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_easy = ai_handler.generate_writer_millionaire_easy_questions(name, bio, known_for)
            elif self.media_type == "creator":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_easy = ai_handler.generate_creator_millionaire_easy_questions(name, bio, known_for)
            else:
                self.questions_easy = ai_handler.generate_millionaire_easy_questions(
                    media_type=self.media_type,
                    title=self.title,
                    plot=self.plot,
                    genre=self.genre,
                    episode_title=self.episode_title,
                    season_number=self.season_number,
                    episode_number=self.episode_number
                )
            
            xbmc.log(f"[plugin.video.milionar] Generated {len(self.questions_easy)} easy questions", level=xbmc.LOGINFO)
            
            if self.questions_easy:
                self.game_state = "playing_easy"
                self.current_level = "easy"
                self.current_question_index = 0
                self.show_current_question()
            else:
                self.show_generation_error("Nepodařilo se vygenerovat snadné otázky.")
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error generating easy questions: {e}", level=xbmc.LOGERROR)
            self.show_generation_error(f"Chyba při generování otázek: {str(e)}")
    
    def show_generation_error(self, error_message):
        """Zobrazí chybu při generování otázek."""
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        elif self.media_type == "director":
            game_title = f"MILIONÁŘ S REŽISÉREM"
        elif self.media_type == "writer":
            game_title = f"MILIONÁŘ SE SCÉNÁRISTOU"
        elif self.media_type == "creator":
            game_title = f"MILIONÁŘ S TVŮRCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            
        error_text = f"""{game_title} - CHYBA

{error_message}

Možné příčiny:
• Problém s AI službou
• Síťové připojení

Zkuste to znovu."""
        
        self.game_state = "error"
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(error_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Zkusit znovu", "Zpět na menu", "Nová hra", "Zavřít")
    

    
    def show_current_question(self):
        """Zobrazí aktuální otázku a případně ji přečte AI hlasem."""
        # Určit aktuální sadu otázek podle stavu hry
        if self.game_state == "playing_easy":
            questions = self.questions_easy
            level_name = "SNADNÁ"
            level_progress = f"{self.current_question_index + 1}/3"
        elif self.game_state == "playing_medium":
            questions = self.questions_medium
            level_name = "STŘEDNÍ"
            level_progress = f"{self.current_question_index + 1}/3"
        elif self.game_state == "playing_hard":
            questions = self.questions_hard
            level_name = "TĚŽKÁ"
            level_progress = f"{self.current_question_index + 1}/3"
        else:
            return
        if not questions or self.current_question_index >= len(questions):
            return
        question_data = questions[self.current_question_index]
        self.current_question = question_data
        # Sestavit text otázky
        total_question_number = self.get_total_question_number()
        money_amount = self.get_money_amount_for_question(total_question_number)
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        elif self.media_type == "director":
            game_title = f"MILIONÁŘ S REŽISÉREM"
        elif self.media_type == "writer":
            game_title = f"MILIONÁŘ SE SCÉNÁRISTOU"
        elif self.media_type == "creator":
            game_title = f"MILIONÁŘ S TVŮRCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
        question_text = f"""{game_title} - OTÁZKA {total_question_number}\n\nOTÁZKA ZA: {money_amount}\n\n{question_data['question']}"""
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(question_text))
        # --- TTS čtení otázky ---
        try:
            tts_enabled = self.addon.getSetting('milionar_tts_enabled') == 'true'
            tts_voice = self.addon.getSetting('milionar_tts_voice') or 'alloy'
            tts_style_value = self.addon.getSetting('milionar_tts_style')
        except Exception:
            tts_enabled = False
            tts_voice = 'alloy'
            tts_style_value = 0
        if tts_enabled:
            import threading
            def tts_thread():
                try:
                    from wcs.ai import TTSClient as openai_tts
                    audio_path = openai_tts.generate_tts(
                        question_data['question'],
                        voice=tts_voice,
                        style=tts_style_value
                    )
                    xbmc.executebuiltin(f'PlayMedia("{audio_path}", noresume)')
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] TTS chyba: {e}", level=xbmc.LOGWARNING)
            threading.Thread(target=tts_thread, daemon=True).start()
        # ---
        # Nastavit odpovědi na tlačítka
        options = question_data.get('options', [])
        correct_index = question_data.get('correct', 0)
        show_correct_marker = self.addon.getSetting('millionaire_show_correct_answer') == 'true'
        if self.lifeline_5050_used and self.lifeline_5050_indices:
            keep = set(self.lifeline_5050_indices)
            button_texts = []
            for i in range(4):
                if i in keep:
                    text = options[i]
                    button_texts.append(text)
                else:
                    button_texts.append("")
        else:
            button_texts = []
            for i in range(4):
                if i < len(options):
                    text = options[i]
                    button_texts.append(text)
                else:
                    button_texts.append("")
        if self.update_buttons_callback:
            self.update_buttons_callback(*button_texts)
            xbmc.log(f"[plugin.video.milionar] Setting question buttons: {button_texts}", level=xbmc.LOGINFO)
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()
    
    def get_total_question_number(self):
        """Vrátí celkové číslo aktuální otázky (1-9)."""
        if self.current_level == "easy":
            return self.current_question_index + 1
        elif self.current_level == "medium":
            return 3 + self.current_question_index + 1
        elif self.current_level == "hard":
            return 6 + self.current_question_index + 1
        return 1

    def get_money_amount_for_question(self, question_number):
        """Vrátí peněžní částku pro danou otázku (1-9)."""
        money_amounts = {
            1: "1 000 Kč",
            2: "2 000 Kč", 
            3: "5 000 Kč",
            4: "10 000 Kč",
            5: "20 000 Kč",
            6: "50 000 Kč",
            7: "100 000 Kč",
            8: "500 000 Kč",
            9: "1 000 000 Kč"
        }
        return money_amounts.get(question_number, "1 000 Kč")

    def get_continue_text(self, next_question_number, next_money_amount):
        """Vrátí dynamický text pro pokračování podle čísla otázky."""
        import random
        
        continue_texts = {
            2: [
                f"""Výborně! To byla jednoduchá rozcvička!

Čeká vás otázka č. {next_question_number} za {next_money_amount}.

Dokážete pokračovat v této lehké jízdě?""",
                f"""Správně! Pěkný začátek!

Nyní přichází otázka č. {next_question_number} za {next_money_amount}.

Zvládnete i tuhle bez problémů?""",
                f"""Výtečně! Zatím vám to jde skvěle!

Před vámi je otázka č. {next_question_number} za {next_money_amount}.

Půjdete dál touto snadnou cestou?"""
            ],
            
            3: [
                f"""Správně! Zatím to jde snadno, že?

Otázka č. {next_question_number} za {next_money_amount} vás čeká.

Odvážíte se postoupit dál?""",
                f"""Perfektně! Máte dobrou formu!

Čeká vás otázka č. {next_question_number} za {next_money_amount}.

Pokračujete v této jízdě?""",
                f"""Skvěle! Dosud žádný problém!

Před vámi stojí otázka č. {next_question_number} za {next_money_amount}.

Máte chuť na další výzvu?"""
            ],
            
            4: [
                f"""Skvěle! Nyní vstupujete do vážnějších vod...

Před vámi je otázka č. {next_question_number} za {next_money_amount}.

Cítíte se na to pokračovat?""",
                f"""Výborně! Teď se to začíná komplikovat!

Čeká vás otázka č. {next_question_number} za {next_money_amount}.

Jste připraveni na vyšší obtížnost?""",
                f"""Úžasně! Přicházejí těžší časy...

Otázka č. {next_question_number} za {next_money_amount} je před vámi.

Odvážíte se čelit větší výzvě?"""
            ],
            
            5: [
                f"""Výborně! Už to začíná být zajímavé!

Čeká vás otázka č. {next_question_number} za {next_money_amount}.

Máte dostatek odvahy jít dál?""",
                f"""Fantastické! Částky rostou a s nimi i riziko!

Před vámi je otázka č. {next_question_number} za {next_money_amount}.

Pocítíte už první náznaky nervozity?""",
                f"""Skvěle! Vstupujete do nebezpečnější zóny!

Otázka č. {next_question_number} za {next_money_amount} vás čeká.

Máte odvahu pokračovat?"""
            ],
            
            6: [
                f"""Fantastické! Nyní se hra začíná správně rozbíhat!

Před vámi stojí otázka č. {next_question_number} za {next_money_amount}!

Odvážíte se vstoupit do nebezpečných vod?""",
                f"""Neuvěřitelné! Dostáváte se do vysokých částek!

Čeká vás otázka č. {next_question_number} za {next_money_amount}!

Cítíte už to napětí ve vzduchu?""",
                f"""Úžasné! Teď to začíná být vážné!

Otázka č. {next_question_number} za {next_money_amount} je před vámi!

Vydáte se do riskantních výšin?"""
            ],
            
            7: [
                f"""Úžasné! Dostáváte se do velmi vysokých částek!

Otázka č. {next_question_number} za {next_money_amount} je před vámi.

Riskujete vše pro šanci na ještě větší výhru?""",
                f"""Senzační! Jste v nebezpečné zóně velkých peněz!

Před vámi čeká otázka č. {next_question_number} za {next_money_amount}.

Odvážíte se jít až na samý vrchol?""",
                f"""Neuvěřitelné! Částky jsou už opravdu vysoké!

Čeká vás otázka č. {next_question_number} za {next_money_amount}.

Máte nervy na to risknout tak velkou sumu?"""
            ],
            
            8: [
                f"""Neuvěřitelné! Jste jen dva kroky od miliónu!

Čeká vás kritická otázka č. {next_question_number} za {next_money_amount}!

Odvážíte se riskovat vše pro šanci na životní výhru?""",
                f"""Senzační! Miliónová meta je na dosah ruky!

Před vámi je osudová otázka č. {next_question_number} za {next_money_amount}!

Vydáte všechno na jednu kartu?""",
                f"""Fantastické! Dva kroky od životního triumfu!

Kritická otázka č. {next_question_number} za {next_money_amount} vás čeká!

Máte odvahu jít až do finále?"""
            ],
            
            9: [
                f"""Senzační! Pouze jedna otázka vás dělí od MILIÓNU!

Finální otázka č. {next_question_number} za {next_money_amount}!

Vydáte vše na jednu kartu a pokusíte se o životní triumf?""",
                f"""Neuvěřitelné! Finále! Poslední krok k MILIÓNU!

Osudová otázka č. {next_question_number} za {next_money_amount}!

Odvážíte se učinit poslední krok k dokonalému vítězství?""",
                f"""Úžasné! Toto je ten okamžik! MILIÓN je na dosah!

Závěrečná otázka č. {next_question_number} za {next_money_amount}!

Máte odvahu dokončit svou cestu k absolutnímu triumfu?"""
            ]
        }
        
        # Vybrat náhodný text pro dané číslo otázky
        if next_question_number in continue_texts:
            return random.choice(continue_texts[next_question_number])
        
        return f"""Výborně! Postoupili jste dál!

Čeká vás otázka č. {next_question_number} za {next_money_amount}!

Odvážíte se pokračovat na cestě k miliónu?"""
    
    def answer_question(self, answer_index):
        """Zpracuje odpověď na otázku - spustí countdown."""
        if self.lifeline_5050_used and self.lifeline_5050_indices:
            if answer_index not in self.lifeline_5050_indices:
                return  # Ignoruj kliknutí na prázdné tlačítko
        if not self.current_question:
            xbmc.log(f"[plugin.video.milionar] No current question for answer {answer_index}", level=xbmc.LOGWARNING)
            return
            
        options = self.current_question.get('options', [])
        if answer_index >= len(options):
            xbmc.log(f"[plugin.video.milionar] Invalid answer index {answer_index}, only {len(options)} options available", level=xbmc.LOGWARNING)
            return
            
        # Uložit vybranou odpověď
        self.selected_answer = options[answer_index]
        self.selected_answer_index = answer_index
        
        # OKAMŽITĚ aktualizovat text s vybranou odpovědí PŘED změnou stavu
        total_question_number = self.get_total_question_number()
        money_amount = self.get_money_amount_for_question(total_question_number)
        
        # Určit aktuální úroveň pro zobrazení
        if self.game_state == "playing_easy":
            level_name = "SNADNÁ"
            level_progress = f"{self.current_question_index + 1}/3"
        elif self.game_state == "playing_medium":
            level_name = "STŘEDNÍ"
            level_progress = f"{self.current_question_index + 1}/3"
        elif self.game_state == "playing_hard":
            level_name = "TĚŽKÁ"
            level_progress = f"{self.current_question_index + 1}/3"
        
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            
        display_text = f"""{game_title} - OTÁZKA {total_question_number}

OTÁZKA ZA: {money_amount}

{self.current_question['question']}

VAŠE ODPOVĚĎ: {self.selected_answer}

Vyhodnocení odpovědi za 3s"""

        # IHNED aktualizovat UI - to musí být první věc!
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(display_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Vyhodnocuje se...", "Počkejte...", "Čekám...", "Prosím počkejte")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()
        
        # Teprve nyní změnit stav
        self.game_state = "answer_pending"
        
        xbmc.log(f"[plugin.video.milionar] Answer {answer_index} selected: {self.selected_answer}", level=xbmc.LOGINFO)
        
        # Připravit dynamické tlačítka pro countdown
        self.countdown_button_texts = self.generate_countdown_button_texts()
        
        # Spustit countdown (4 sekundy) s dynamickými tlačítky
        if self.start_countdown_callback:
            self.start_countdown_with_dynamic_buttons(4, self.finish_answer_evaluation)
        else:
            # Fallback pokud není callback - ihned vyhodnotit
            self.finish_answer_evaluation()
        
        # Po odpovědi resetuj 50:50 pro další otázku
        self.lifeline_5050_indices = None
    
    def finish_answer_evaluation(self):
        """Dokončí vyhodnocení odpovědi po countdown."""
        correct_index = self.current_question.get('correct', 0)
        is_correct = (self.selected_answer_index == correct_index)
        
        xbmc.log(f"[plugin.video.milionar] Answer evaluation: selected {self.selected_answer_index}, correct {correct_index}, result: {'CORRECT' if is_correct else 'WRONG'}", level=xbmc.LOGINFO)
        
        if is_correct:
            self.score += 1
            self.show_answer_feedback(True)
        else:
            # Špatná odpověď = konec hry
            correct_option = self.current_question.get('options', [])[correct_index]
            self.show_game_over(correct_option, self.current_question.get('explanation', ''))
    
    def generate_countdown_button_texts(self):
        """Generuje náhodné texty pro dynamická tlačítka během odpočtu."""
        import random
        
        # Varianty textů pro každou pozici (zleva doprava s gradací napětí)
        position_texts = {
            1: ["Myslíte si...", "Věříte...", "Opravdu..."],
            2: ["Správně?", "Dobře?", "Že ano?"], 
            3: ["Uvidíme...", "Zjistíme...", "Moment..."],
            4: ["Nervózní?", "Napětí!", "Výsledek!"]
        }
        
        # Vybrat náhodný text pro každou pozici
        selected_texts = {}
        for position in range(1, 5):
            selected_texts[position] = random.choice(position_texts[position])
        
        return selected_texts
    
    def start_countdown_with_dynamic_buttons(self, seconds, callback):
        """Spustí countdown s dynamickými tlačítky."""
        import threading
        import time
        
        def countdown_thread():
            # Celkový čas 4 sekundy: 3 tlačítka po 0.75s + poslední tlačítko 1.25s
            step_duration = 0.75  # 0.75 sekundy na první 3 tlačítka
            
            # Zobrazit tlačítka postupně během prvních 3 sekund
            for step in range(3):  # První 3 tlačítka
                # Aktualizovat tlačítka podle aktuálního kroku
                self.update_countdown_buttons(step + 1, seconds, step_duration)
                
                # Aktualizovat odpočet v textu (3, 2, 1)
                countdown_number = seconds - step - 1
                self.update_countdown_display_number(countdown_number)
                
                time.sleep(step_duration)
            
            # 4. tlačítko zobrazit a nechat vidět 1.25 sekundy
            self.update_countdown_buttons(4, seconds, step_duration)
            
            # Odpočet 1 -> 0 během posledních 1.25 sekundy
            self.update_countdown_display_number(1)
            time.sleep(0.25)  # Krátce ukázat "1"
            
            self.update_countdown_display_number(0)
            time.sleep(1.0)   # 1 sekunda na "0"
            
            # Po skončení countdown zavolat callback
            callback()
        
        # Spustit countdown v background thread
        thread = threading.Thread(target=countdown_thread)
        thread.daemon = True
        thread.start()
    
    def update_countdown_buttons(self, current_step, total_seconds, step_duration):
        """Aktualizuje tlačítka podle aktuálního kroku odpočtu."""
        if not self.update_buttons_callback:
            return
            
        button1 = "..."
        button2 = "..."
        button3 = "..."
        button4 = "..."
        
        # Postupně zobrazovat texty zleva doprava - každé tlačítko v jiném kroku
        if current_step >= 1:
            button1 = self.countdown_button_texts[1]
        if current_step >= 2:
            button2 = self.countdown_button_texts[2]
        if current_step >= 3:
            button3 = self.countdown_button_texts[3]
        if current_step >= 4:
            button4 = self.countdown_button_texts[4]
        
        self.update_buttons_callback(button1, button2, button3, button4)
    
    def update_countdown_display_number(self, countdown_number):
        """Aktualizuje odpočet v textu hry s číslem."""
        if not self.update_display_callback:
            return
            
        total_question_number = self.get_total_question_number()
        money_amount = self.get_money_amount_for_question(total_question_number)
        
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            
        display_text = f"""{game_title} - OTÁZKA {total_question_number}

OTÁZKA ZA: {money_amount}

{self.current_question['question']}

VAŠE ODPOVĚĎ: {self.selected_answer}

Vyhodnocení odpovědi za {countdown_number}s"""

        self.update_display_callback(self._add_ai_info_header(display_text))
    
    def show_answer_feedback(self, is_correct):
        """Zobrazí zpětnou vazbu k odpovědi na stejné obrazovce a případně ji přečte AI hlasem."""
        question_data = self.current_question
        explanation = question_data.get('explanation', '')
        total_question_number = self.get_total_question_number()
        
        # Určit aktuální úroveň pro zobrazení
        if self.current_level == "easy":
            level_name = "SNADNÁ"
            level_progress = f"{self.current_question_index + 1}/3"
        elif self.current_level == "medium":
            level_name = "STŘEDNÍ"
            level_progress = f"{self.current_question_index + 1}/3"
        elif self.current_level == "hard":
            level_name = "TĚŽKÁ"
            level_progress = f"{self.current_question_index + 1}/3"
        
        # Zobrazit otázku + odpověď + výsledek na stejné stránce
        money_amount = self.get_money_amount_for_question(total_question_number)
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            
        # Zjistit jaká bude další otázka a její hodnota
        next_question_number = total_question_number + 1
        if next_question_number <= 9:
            next_money_amount = self.get_money_amount_for_question(next_question_number)
            next_question_text = self.get_continue_text(next_question_number, next_money_amount)
        else:
            # Poslední otázka byla zodpovězena - hra končí vítězstvím
            next_question_text = "Fantastické! Zodpověděli jste všechny otázky a stali se MILIONÁŘEM!"

        feedback_text = f"""{game_title} - OTÁZKA {total_question_number}

OTÁZKA ZA: {money_amount}

{question_data['question']}

VAŠE ODPOVĚĎ: {self.selected_answer}

SPRÁVNĚ!

{explanation}

{next_question_text}"""
        
        # Nastavit správný stav podle toho, zda je to poslední otázka
        if total_question_number == 9:
            self.game_state = "game_finished"
            self._results_shown = False
            if self.update_buttons_callback:
                self.update_buttons_callback("Výsledky", "Nová hra", "Zpět na menu", "Zavřít")
        else:
            # Není poslední otázka - standardní stav
            self.game_state = "answer_feedback"
        
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(feedback_text))
        
        # Nastavit správná tlačítka podle toho, zda je to poslední otázka
        if total_question_number == 9:
            # Poslední otázka - první tlačítko je "Výsledky"
            if self.update_buttons_callback:
                self.update_buttons_callback("Výsledky", "Nová hra", "Zpět na menu", "Zavřít")
        else:
            # Není poslední otázka - standardní tlačítka
            if self.update_buttons_callback:
                self.update_buttons_callback("Další otázka", "Nová hra", "Zpět na menu", "Zavřít")
        
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()
        # Nastavit fokus na tlačítko "Pokračovat" (index 0)
        if self.set_focus_callback:
            self.set_focus_callback(0)
        # --- TTS čtení správné odpovědi a vysvětlení ---
        try:
            tts_expl_enabled = self.addon.getSetting('milionar_tts_explanation_enabled') == 'true'
            tts_voice = self.addon.getSetting('milionar_tts_voice') or 'alloy'
            tts_style_value = self.addon.getSetting('milionar_tts_style')
        except Exception:
            tts_expl_enabled = False
            tts_voice = 'alloy'
            tts_style_value = 0
        if tts_expl_enabled:
            import threading
            def tts_expl_thread():
                try:
                    from wcs.ai import TTSClient as openai_tts
                    tts_text = f"Správně! {explanation}".strip()
                    if tts_text:
                        audio_path = openai_tts.generate_tts(
                            tts_text,
                            voice=tts_voice,
                            style=tts_style_value
                        )
                        xbmc.executebuiltin(f'PlayMedia("{audio_path}", noresume)')
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] TTS vysvětlení chyba: {e}", level=xbmc.LOGWARNING)
            threading.Thread(target=tts_expl_thread, daemon=True).start()
        # ---
    
    def show_game_over(self, correct_option, explanation):
        """Zobrazí screen pro konec hry při špatné odpovědi na stejné obrazovce a případně přečte správnou odpověď a vysvětlení AI hlasem."""
        total_questions = self.get_total_question_number()
        
        # Určit aktuální úroveň pro zobrazení
        if self.current_level == "easy":
            level_name = "SNADNÁ"
            level_progress = f"{self.current_question_index + 1}/3"
        elif self.current_level == "medium":
            level_name = "STŘEDNÍ"
            level_progress = f"{self.current_question_index + 1}/3"
        elif self.current_level == "hard":
            level_name = "TĚŽKÁ"
            level_progress = f"{self.current_question_index + 1}/3"
        
        # Zobrazit otázku + odpověď + výsledek na stejné stránce
        money_amount = self.get_money_amount_for_question(total_questions)
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
            knowledge_reference = "seriálové znalosti"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
            knowledge_reference = "znalosti o herci"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            knowledge_reference = "filmové znalosti"
            
        game_over_text = f"""{game_title} - OTÁZKA {total_questions}

OTÁZKA ZA: {money_amount}

{self.current_question['question']}

VAŠE ODPOVĚĎ: {self.selected_answer}

ŠPATNĚ!

Správná odpověď: {correct_option}

{explanation}

Bohužel jste odpověděl špatně a hra končí.
Zkuste to znovu a otestujte své {knowledge_reference}!"""
        
        self.game_state = "game_over"
        
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(game_over_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Nová hra", "Zpět na menu", "Jiná hra", "Zavřít")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()
        # --- TTS čtení správné odpovědi a vysvětlení při špatné odpovědi ---
        try:
            tts_expl_enabled = self.addon.getSetting('milionar_tts_explanation_enabled') == 'true'
            tts_voice = self.addon.getSetting('milionar_tts_voice') or 'alloy'
            tts_style_value = self.addon.getSetting('milionar_tts_style')
        except Exception:
            tts_expl_enabled = False
            tts_voice = 'alloy'
            tts_style_value = 0
        if tts_expl_enabled:
            import threading
            def tts_expl_thread():
                try:
                    from wcs.ai import TTSClient as openai_tts
                    tts_text = f"Špatně! Správná odpověď: {correct_option}. {explanation}".strip()
                    if tts_text:
                        audio_path = openai_tts.generate_tts(
                            tts_text,
                            voice=tts_voice,
                            style=tts_style_value
                        )
                        xbmc.executebuiltin(f'PlayMedia("{audio_path}", noresume)')
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] TTS vysvětlení chyba: {e}", level=xbmc.LOGWARNING)
            threading.Thread(target=tts_expl_thread, daemon=True).start()
        # ---
    
    def continue_after_feedback(self):
        """Pokračuje po zobrazení zpětné vazby."""
        self.current_question_index += 1
        
        # Kontrola, zda máme další otázky v aktuální úrovni
        if self.game_state == "answer_feedback":
            if self.current_question_index < 3:
                # Vrátit se k aktuální úrovni na základě self.current_level
                if self.current_level == "easy":
                    self.game_state = "playing_easy"
                elif self.current_level == "medium":
                    self.game_state = "playing_medium"
                elif self.current_level == "hard":
                    self.game_state = "playing_hard"
                self.show_current_question()
            else:
                # Přechod na další úroveň
                self.proceed_to_next_level()
    
    def proceed_to_next_level(self):
        """Přechod na další úroveň obtížnosti."""
        self.current_question_index = 0
        
        if self.current_level == "easy":
            # Dokončili jsme Easy, přejdeme na Medium
            self.current_level = "medium"
            self.show_level_transition("medium")
        elif self.current_level == "medium":
            # Dokončili jsme Medium, přejdeme na Hard
            self.current_level = "hard"
            self.show_level_transition("hard")
        else:
            # Dokončili jsme všechny úrovně
            self.show_final_score()
    
    def show_level_transition(self, next_level):
        """Zobrazí přechod mezi úrovněmi."""
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            
        if next_level == "medium":
            self.game_state = "generating_medium"
            # Sestavím info o epizodě pokud je dostupné
            episode_info = ""
            if self.media_type == "series" and self.episode_title:
                episode_info = f"Epizoda: {self.episode_title}"
                if self.season_number and self.episode_number:
                    episode_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
                episode_info = f"\n{episode_info}"
            # Přidám zkrácený popis filmu/epizody na max 3 řádky (cca 240 znaků)
            plot_info = ""
            if self.plot:
                max_len = 240
                short_plot = self.plot.strip().replace('\n', ' ')
                if len(short_plot) > max_len:
                    cut = short_plot[:max_len].rsplit(' ', 1)[0]
                    short_plot = cut + "..."
                plot_info = f"\n{short_plot}"
            # Sestavím bloky s prázdnými řádky mezi sekcemi
            info_lines = [f"Téma: {self.title}"]
            if episode_info:
                info_lines.append(episode_info.strip())
            if plot_info:
                info_lines.append(f"Děj: {plot_info.strip()}")
            info_block = "\n\n".join(info_lines)
            # Zobrazit loading screen pro střední otázky
            loading_text = f"""AI připravuje {game_title.lower()}

{info_block}

Generuji STŘEDNÍ otázky (4-6)

Chvíli strpení..."""
        else:  # hard
            self.game_state = "generating_hard"
            # Sestavím info o epizodě pokud je dostupné
            episode_info = ""
            if self.media_type == "series" and self.episode_title:
                episode_info = f"Epizoda: {self.episode_title}"
                if self.season_number and self.episode_number:
                    episode_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
                episode_info = f"\n{episode_info}"
            # Přidám zkrácený popis filmu/epizody na max 3 řádky (cca 240 znaků)
            plot_info = ""
            if self.plot:
                max_len = 240
                short_plot = self.plot.strip().replace('\n', ' ')
                if len(short_plot) > max_len:
                    cut = short_plot[:max_len].rsplit(' ', 1)[0]
                    short_plot = cut + "..."
                plot_info = f"\n{short_plot}"
            # Sestavím bloky s prázdnými řádky mezi sekcemi
            info_lines = [f"Téma: {self.title}"]
            if episode_info:
                info_lines.append(episode_info.strip())
            if plot_info:
                info_lines.append(f"Děj: {plot_info.strip()}")
            info_block = "\n\n".join(info_lines)
            # Zobrazit loading screen pro těžké otázky
            loading_text = f"""AI připravuje {game_title.lower()}

{info_block}

Generuji TĚŽKÉ otázky (7-9)

Chvíli strpení..."""
        
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(loading_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Načítání...", "...", "...", "...")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()
        self.stop_ladder_animation()
        if next_level == "medium":
            self.animate_generating_ladder([114,115,116], duration=3.0, interval=0.2)
        elif next_level == "hard":
            self.animate_generating_ladder([107,108,109], duration=3.0, interval=0.2)
        
        # Spustit generování další sady otázek
        import threading
        if next_level == "medium":
            thread = threading.Thread(target=self._generate_medium_questions)
        else:
            thread = threading.Thread(target=self._generate_hard_questions)
        thread.daemon = True
        thread.start()
    
    def _generate_medium_questions(self):
        """Generuje střední otázky v background."""
        try:
            from wcs.ai import AIHandler as ai_handler
            
            if self.media_type == "actor":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_medium = ai_handler.generate_actor_millionaire_medium_questions(name, bio, known_for)
            elif self.media_type == "director":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_medium = ai_handler.generate_director_millionaire_medium_questions(name, bio, known_for)
            elif self.media_type == "writer":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_medium = ai_handler.generate_writer_millionaire_medium_questions(name, bio, known_for)
            elif self.media_type == "creator":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_medium = ai_handler.generate_creator_millionaire_medium_questions(name, bio, known_for)
            else:
                self.questions_medium = ai_handler.generate_millionaire_medium_questions(
                    media_type=self.media_type,
                    title=self.title,
                    plot=self.plot,
                    genre=self.genre,
                    episode_title=self.episode_title,
                    season_number=self.season_number,
                    episode_number=self.episode_number
                )
            
            xbmc.log(f"[plugin.video.milionar] Generated {len(self.questions_medium)} medium questions", level=xbmc.LOGINFO)
            
            if self.questions_medium:
                self.game_state = "playing_medium"
                self.current_level = "medium"
                self.current_question_index = 0
                self.show_current_question()
            else:
                self.show_generation_error("Nepodařilo se vygenerovat střední otázky.")
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error generating medium questions: {e}", level=xbmc.LOGERROR)
            self.show_generation_error(f"Chyba při generování střední otázek: {str(e)}")
    
    def _generate_hard_questions(self):
        """Generuje těžké otázky v background."""
        try:
            from wcs.ai import AIHandler as ai_handler
            
            if self.media_type == "actor":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_hard = ai_handler.generate_actor_millionaire_hard_questions(name, bio, known_for)
            elif self.media_type == "director":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_hard = ai_handler.generate_director_millionaire_hard_questions(name, bio, known_for)
            elif self.media_type == "writer":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_hard = ai_handler.generate_writer_millionaire_hard_questions(name, bio, known_for)
            elif self.media_type == "creator":
                name = self.actor_data.get('name', '')
                bio = self.actor_data.get('biography', '')
                known_for = [item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data.get('known_for') else []
                self.questions_hard = ai_handler.generate_creator_millionaire_hard_questions(name, bio, known_for)
            else:
                self.questions_hard = ai_handler.generate_millionaire_hard_questions(
                    media_type=self.media_type,
                    title=self.title,
                    plot=self.plot,
                    genre=self.genre,
                    episode_title=self.episode_title,
                    season_number=self.season_number,
                    episode_number=self.episode_number
                )
            
            xbmc.log(f"[plugin.video.milionar] Generated {len(self.questions_hard)} hard questions", level=xbmc.LOGINFO)
            
            if self.questions_hard:
                self.game_state = "playing_hard"
                self.current_level = "hard"
                self.current_question_index = 0
                self.show_current_question()
            else:
                self.show_generation_error("Nepodařilo se vygenerovat těžké otázky.")
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error generating hard questions: {e}", level=xbmc.LOGERROR)
            self.show_generation_error(f"Chyba při generování těžkých otázek: {str(e)}")
    
    def show_final_score(self):
        """Zobrazí finální skóre hry - pouze při dokončení všech 9 otázek."""
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
            master_title = "SERIÁLOVÝ MISTR"
            knowledge_reference = "znalcem seriálů"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
            master_title = "MISTR S HERCEM"
            knowledge_reference = "znalcem herců"
        elif self.media_type == "director":
            game_title = f"MILIONÁŘ S REŽISÉREM"
            master_title = "MISTR S REŽISÉREM"
            knowledge_reference = "znalcem režisérů"
        elif self.media_type == "writer":
            game_title = f"MILIONÁŘ SE SCÉNÁRISTOU"
            master_title = "MISTR SE SCÉNÁRISTOU"
            knowledge_reference = "znalcem scénáristů"
        elif self.media_type == "creator":
            game_title = f"MILIONÁŘ S TVŮRCEM"
            master_title = "MISTR S TVŮRCEM"
            knowledge_reference = "znalcem tvůrců"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            master_title = "FILMOVÝ MISTR"
            knowledge_reference = "znalcem filmů"
            
        final_text = f"""{game_title} - GRATULUJEME!

{master_title}

FINÁLNÍ SKÓRE: 9/9 (100%)

Úžasné! Odpověděli jste správně na všech 9 otázek!

Jste skutečným {knowledge_reference}!

Děkujeme za hraní!"""

        self.game_state = "game_finished"
        self._results_shown = True
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(final_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Nová hra", "Zpět na menu", "Zavřít", "")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()

    def get_sidebar_items(self):
        """Vrací aktuální sidebar položky."""
        game_name = "Seriálový Milionář" if self.media_type == "series" else "Filmový Milionář"
        
        # Pyramida peněžních částek (odspoda nahoru jako v originále)
        money_pyramid = [
            "1 000 000 Kč",      # Otázka 9 (Hard 3/3)
            "500 000 Kč",        # Otázka 8 (Hard 2/3)  
            "100 000 Kč",        # Otázka 7 (Hard 1/3)
            "50 000 Kč",         # Otázka 6 (Medium 3/3)
            "20 000 Kč",         # Otázka 5 (Medium 2/3)
            "10 000 Kč",         # Otázka 4 (Medium 1/3)
            "5 000 Kč",          # Otázka 3 (Easy 3/3)
            "2 000 Kč",          # Otázka 2 (Easy 2/3)
            "1 000 Kč"           # Otázka 1 (Easy 1/3)
        ]
        
        # Speciální stavy s prázdnou pyramidou
        if self.game_state == "millionaire_welcome":
            # Zobrazit prázdnou pyramidu s nadpisem
            pyramid_display = [
                "ŽEBŘÍČEK VÝHER:",
                "================",  # Podtržení nadpisu
                "",  # Prázdný řádek pro oddělení
            ]
            for amount in money_pyramid:
                pyramid_display.append(f"[ ] {amount}")
            return pyramid_display
        elif self.game_state == "millionaire_rules":
            # Zobrazit prázdnou pyramidu s nadpisem
            pyramid_display = [
                "ŽEBŘÍČEK VÝHER:",
                "================",  # Podtržení nadpisu
                "",  # Prázdný řádek pro oddělení
            ]
            for amount in money_pyramid:
                pyramid_display.append(f"[ ] {amount}")
            return pyramid_display
        elif self.game_state in ["generating_easy", "generating_medium", "generating_hard"]:
            level_name = "Snadnou" if "easy" in self.game_state else ("Středni" if "medium" in self.game_state else "Tezkou")
            # Při generování zobrazit pyramidu s označením co se generuje
            pyramid_display = [
                "ŽEBŘÍČEK VÝHER:",
                "================",  # Podtržení nadpisu
                "",  # Prázdný řádek pro oddělení
            ]
            for i, amount in enumerate(money_pyramid):
                question_num = 9 - i
                # Už zodpovězené otázky zůstávají označené jako hotové
                if question_num <= self.score:
                    pyramid_display.append(f"[X] {amount}")
                # Co se aktuálně generuje
                elif "easy" in self.game_state and question_num <= 3:
                    pyramid_display.append(f"[?] {amount}")
                elif "medium" in self.game_state and question_num <= 6 and question_num > 3:
                    pyramid_display.append(f"[?] {amount}")
                elif "hard" in self.game_state and question_num > 6:
                    pyramid_display.append(f"[?] {amount}")
                else:
                    pyramid_display.append(f"[ ] {amount}")
            return pyramid_display
        elif self.game_state == "error":
            media_suggestion = "jiný seriál" if self.media_type == "series" else "jiný film"
            return ["Chyba", "Zkuste znovu", f"nebo zvolte {media_suggestion}"]

        
        # Pro herní stavy zobrazit pyramidu s nadpisem
        pyramid_display = [
            "ŽEBŘÍČEK VÝHER:",
            "================",  # Podtržení nadpisu
            "",  # Prázdný řádek pro oddělení
        ]
        
        # Určení aktuálního čísla otázky podle stavu hry
        if self.game_state in ["playing_easy", "playing_medium", "playing_hard"]:
            current_total_question = self.get_total_question_number()
        elif self.game_state == "answer_pending":
            # Během čekání na vyhodnocení - aktuální otázka
            current_total_question = self.get_total_question_number()
        elif self.game_state == "answer_feedback":
            # Po správné odpovědi - score už je zvýšen, takže score = počet zodpovězených
            current_total_question = self.score
        elif self.game_state == "game_over":
            # Při prohře - score = počet správných, +1 = aktuální nesprávná
            current_total_question = self.score + 1
        elif self.game_state == "game_finished":
            # Při dokončení - všechny zodpovězené
            current_total_question = 10  # Všech 9 otázek zodpovězeno
        else:
            # Pro ostatní stavy (welcome, rules, atd.) - žádná otázka
            current_total_question = 0
        
        for i, amount in enumerate(money_pyramid):
            question_num = 9 - i  # Převod indexu na číslo otázky (9,8,7...1)
            
            if self.game_state == "game_finished":
                # Všechny otázky zodpovězené
                pyramid_display.append(f"[X] {amount}")
            elif self.game_state == "game_over":
                # Nesprávná odpověď - správné do score, pak [!] pro nesprávnou
                if question_num <= self.score:
                    pyramid_display.append(f"[X] {amount}")  # Správně zodpovězené
                elif question_num == current_total_question:
                    pyramid_display.append(f"[!] {amount}")  # Nesprávně zodpovězená
                else:
                    pyramid_display.append(f"[ ] {amount}")  # Nehrané
            elif self.game_state == "answer_feedback":
                # Správná odpověď - všechny otázky až do score jsou zodpovězené
                if question_num <= current_total_question:
                    pyramid_display.append(f"[X] {amount}")
                else:
                    pyramid_display.append(f"[ ] {amount}")
            elif question_num < current_total_question:
                # Zodpovězená otázka
                pyramid_display.append(f"[X] {amount}")
            elif question_num == current_total_question:
                # Aktuální otázka
                pyramid_display.append(f"[>] {amount}")
            else:
                # Nezodpovězená otázka  
                pyramid_display.append(f"[ ] {amount}")
        
        return pyramid_display

    def get_millionaire_ladder_states(self):
        """Vrací stavy jednotlivých řádků grafického žebříčku pro UI."""
        question_to_control_id = {
            1: 111, 2: 112, 3: 113, 4: 114, 5: 115, 6: 116, 7: 107, 8: 108, 9: 109
        }
        question_to_amount = {
            1: "1 000 Kč", 2: "2 000 Kč", 3: "5 000 Kč", 4: "10 000 Kč", 5: "20 000 Kč", 6: "50 000 Kč", 7: "100 000 Kč", 8: "500 000 Kč", 9: "1 000 000 Kč"
        }
        if self.game_state in ["playing_easy", "playing_medium", "playing_hard"]:
            current_total_question = self.get_total_question_number()
        elif self.game_state == "answer_pending":
            current_total_question = self.get_total_question_number()
        elif self.game_state == "answer_feedback":
            current_total_question = self.score
        elif self.game_state == "game_over":
            current_total_question = self.score + 1
        elif self.game_state == "game_finished":
            current_total_question = 10
        else:
            current_total_question = 0
        ladder_states = {}
        for question_num in range(1, 10):
            control_id = question_to_control_id[question_num]
            amount = question_to_amount[question_num]
            # Výchozí barvy
            bg_default = "FF232323"  # tmavá (sidebar/nápověda)
            bg_correct = "FF3CB371"   # decentní zelená
            bg_wrong = "FFB04A4A"     # tlumená červená
            bg_current = "FF335C8F"   # decentní modrá
            # Animované barvy pro generování
            colors = ["AFBBAA33", "BFBBAA33", "CFBBAA33", "DFBBAA33", "EFBBAA33", "FFBBAA33"]
            # Určení stavu a barvy
            if self.game_state in ["millionaire_welcome", "millionaire_rules"]:
                state = "empty"
                text = amount
                color = "white"
                bgcolor = bg_default
            elif self.game_state in ["generating_easy", "generating_medium", "generating_hard"]:
                if question_num <= self.score:
                    state = "completed"
                    text = amount
                    color = "white"
                    bgcolor = bg_correct
                elif ("easy" in self.game_state and question_num <= 3) or \
                     ("medium" in self.game_state and 4 <= question_num <= 6) or \
                     ("hard" in self.game_state and question_num >= 7):
                    state = "generating"
                    text = amount
                    color = "black"
                    try:
                        idx = int(self.ladder_anim_toggle)
                    except Exception:
                        idx = 0
                    bgcolor = colors[idx % len(colors)]
                else:
                    state = "empty"
                    text = amount
                    color = "white"
                    bgcolor = bg_default
            elif self.game_state == "game_finished":
                state = "completed"
                text = amount
                color = "white"
                bgcolor = bg_correct
            elif self.game_state == "game_over":
                if question_num <= self.score:
                    state = "completed"
                    text = amount
                    color = "white"
                    bgcolor = bg_correct
                elif question_num == current_total_question:
                    state = "failed"
                    text = amount
                    color = "white"
                    bgcolor = bg_wrong
                else:
                    state = "empty"
                    text = amount
                    color = "white"
                    bgcolor = bg_default
            elif self.game_state == "answer_feedback":
                if question_num <= current_total_question:
                    state = "completed"
                    text = amount
                    color = "white"
                    bgcolor = bg_correct
                else:
                    state = "empty"
                    text = amount
                    color = "white"
                    bgcolor = bg_default
            elif question_num < current_total_question:
                state = "completed"
                text = amount
                color = "white"
                bgcolor = bg_correct
            elif question_num == current_total_question:
                state = "current"
                text = amount
                color = "white"
                bgcolor = bg_current
            else:
                state = "empty"
                text = amount
                color = "white"
                bgcolor = bg_default
            ladder_states[control_id] = {
                'state': state,
                'text': text,
                'color': color,
                'bgcolor': bgcolor,
                'question_num': question_num
            }
        return ladder_states

    def start_game_selection(self):
        """Návrat na hlavní výběr typu hry."""
        self.reset_lifelines()
        if hasattr(self, 'return_to_menu_callback') and self.return_to_menu_callback:
            self.return_to_menu_callback()
        else:
            if self.close_dialog_callback:
                self.close_dialog_callback()

    def end_game(self):
        """Ukončí hru - zavře dialog."""
        self.reset_lifelines()
        if self.close_dialog_callback:
            self.close_dialog_callback()

    def use_5050_lifeline(self):
        """Aktivuje nápovědu 50:50, pokud je dostupná a ve správném stavu."""
        if self.lifeline_5050_used:
            return  # Již použito
        if self.game_state not in ["playing_easy", "playing_medium", "playing_hard"]:
            return  # Nápověda není dostupná
        if not self.current_question:
            return
        options = self.current_question.get('options', [])
        correct_index = self.current_question.get('correct', 0)
        # Najdi všechny špatné odpovědi
        wrong_indices = [i for i in range(len(options)) if i != correct_index]
        import random
        if not wrong_indices:
            return
        random_wrong = random.choice(wrong_indices)
        # Ulož indexy, které mají zůstat
        self.lifeline_5050_indices = (correct_index, random_wrong)
        self.lifeline_5050_used = True
        # Aktualizuj tlačítka a sidebar
        self.show_current_question()
        
        # Automaticky přesměrovat fokus na první viditelné tlačítko
        if self.set_focus_callback:
            # Najít první neprázdné tlačítko
            keep = set(self.lifeline_5050_indices)
            first_visible_index = next((i for i in range(4) if i in keep), 0)
            self.set_focus_callback(first_visible_index)
            xbmc.log(f"[plugin.video.milionar] 50:50 lifeline used, focus moved to button {first_visible_index}", level=xbmc.LOGINFO)
        
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()

    def use_audience_poll_lifeline(self):
        """Aktivuje nápovědu hlasování diváků - simuluje věrohodné hlasování."""
        if self.lifeline_audience_used:
            return  # Již použito
            
        if self.game_state not in ["playing_easy", "playing_medium", "playing_hard"]:
            return  # Nápověda není dostupná
            
        if not self.current_question:
            return
            
        import random
        correct_index = self.current_question.get('correct', 0)
        options = self.current_question.get('options', [])
        
        # Zohlednit 50:50 nápovědu - pokud je aktivní, hlasovat pouze pro zbývající možnosti
        available_indices = list(range(4))
        if self.lifeline_5050_used and self.lifeline_5050_indices:
            available_indices = list(self.lifeline_5050_indices)
            xbmc.log(f"[plugin.video.milionar] Audience poll with 50:50 active, voting only for indices: {available_indices}", level=xbmc.LOGINFO)
        
        # Určit sílu správné odpovědi podle úrovně obtížnosti
        if self.current_level == "easy":
            correct_confidence = random.randint(60, 85)  # 60-85% pro snadné otázky
        elif self.current_level == "medium":
            correct_confidence = random.randint(45, 70)  # 45-70% pro střední otázky
        else:  # hard
            correct_confidence = random.randint(30, 55)  # 30-55% pro těžké otázky
        
        # Rozdělit zbývající procenta mezi ostatní dostupné odpovědi
        remaining = 100 - correct_confidence
        poll_results = [0] * 4
        poll_results[correct_index] = correct_confidence
        
        # Najít ostatní dostupné odpovědi (kromě správné)
        other_available_indices = [i for i in available_indices if i != correct_index]
        
        # Pokud je správná odpověď mimo dostupné, přesunout ji tam
        if correct_index not in available_indices and available_indices:
            # Najít nejlepší náhradní odpověď (nejméně špatnou)
            best_wrong_index = available_indices[0]
            poll_results[correct_index] = 0  # Vynulovat původní správnou
            poll_results[best_wrong_index] = correct_confidence  # Přesunout na dostupnou
            other_available_indices = [i for i in available_indices if i != best_wrong_index]
        
        # Minimální procento pro každou nesprávnou odpověď
        if other_available_indices:
            min_per_wrong = max(1, remaining // (len(other_available_indices) * 2))
        else:
            min_per_wrong = 1
        
        # Rozdělit zbývající procenta mezi ostatní dostupné odpovědi
        for i, idx in enumerate(other_available_indices):
            if i == len(other_available_indices) - 1:
                # Poslední odpověď dostane zbytek
                poll_results[idx] = max(0, remaining)
            else:
                # Náhodná část zbývajících procent
                max_share = min(remaining - min_per_wrong, remaining // 2)  # Nech něco pro ostatní
                max_share = max(min_per_wrong, max_share)  # Ale minimálně min_per_wrong
                
                if max_share >= min_per_wrong:
                    share = random.randint(min_per_wrong, max_share)
                else:
                    share = min_per_wrong
                    
                poll_results[idx] = share
                remaining -= share
        
        # Ujistit se, že součet je 100% a žádné procento není záporné
        current_sum = sum(poll_results)
        if current_sum != 100:
            # Přidat rozdíl k nejvyššímu procentu
            max_index = poll_results.index(max(poll_results))
            poll_results[max_index] += (100 - current_sum)
            
        # Zajistit, že žádné procento není záporné
        for i in range(4):
            if poll_results[i] < 0:
                poll_results[i] = 0
        
        self.lifeline_audience_results = poll_results
        
        # Označit jako použito PŘED zobrazením výsledků
        self.lifeline_audience_used = True
        
        # Zobrazit výsledky hlasování
        self.show_audience_poll_results()
        
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()

    def show_audience_poll_results(self):
        """Zobrazí výsledky hlasování diváků s animací načítání."""
        if not self.lifeline_audience_results:
            return
            
        # Spustit animaci hlasování
        self.animate_audience_poll()

    def animate_audience_poll(self):
        """Animuje postupné načítání výsledků hlasování během 10 sekund s napínavým efektem."""
        import threading
        import time
        import random
        
        def animation_thread():
            options = self.current_question.get('options', [])
            final_results = self.lifeline_audience_results
            labels = ["A", "B", "C", "D"]
            
            # Napínavé texty pro různé fáze
            status_texts = [
                "Sčítání hlasů...",
                "Zpracování výsledků...", 
                "Analyzování odpovědí...",
                "Kontrola dat...",
                "Finalizace výsledků...",
                "Poslední hlasy...",
                "Téměř hotovo...",
                "Dokončování...",
                "Výsledky připraveny...",
                "Vyberte svou odpověď:"
            ]
            
            # Nastavit odpovědi na tlačítka na začátku a nechat je tam po celou dobu
            if self.update_buttons_callback:
                correct_index = self.current_question.get('correct', 0)
                
                # Kontrola nastavení pro zobrazení "x" u správné odpovědi
                show_correct_marker = self.addon.getSetting('millionaire_show_correct_answer') == 'true'
                
                # 50:50 nápověda - pokud je aktivní, zobraz jen dvě možnosti
                if self.lifeline_5050_used and self.lifeline_5050_indices:
                    keep = set(self.lifeline_5050_indices)
                    button_texts = []
                    for i in range(4):
                        if i in keep:
                            text = options[i]
                            button_texts.append(text)
                        else:
                            button_texts.append("")
                else:
                    button_texts = []
                    for i in range(4):
                        if i < len(options):
                            text = options[i]
                            button_texts.append(text)
                        else:
                            button_texts.append("")
                self.update_buttons_callback(*button_texts)
            
            # 20 kroků animace (0.5 sekundy každý = 10 sekund celkem)
            total_steps = 20
            for step in range(total_steps + 1):
                
                # Nelineární růst - pomalý start, pak rychlejší, pak zase pomalý (S-křivka)
                if step == 0:
                    progress = 0.0
                elif step <= total_steps:
                    # S-křivka pro napínavější efekt
                    t = step / total_steps
                    progress = t * t * (3.0 - 2.0 * t)  # Smooth step funkce
                else:
                    progress = 1.0
                
                # Vypočítat aktuální hodnoty s malým náhodným rozptylem pro napětí
                if step < total_steps:
                    current_results = []
                    for final_val in final_results:
                        base_val = int(final_val * progress)
                        # Přidat malé náhodné kolísání (±2%) pro napětí
                        if step > 2:  # Po prvních krocích
                            wobble = random.randint(-2, 2)
                            wobble_val = max(0, base_val + wobble)
                        else:
                            wobble_val = base_val
                        current_results.append(wobble_val)
                else:
                    # Finální krok - přesné hodnoty
                    current_results = final_results
                
                # Vybrat status text podle kroku
                status_index = min(step // 2, len(status_texts) - 1)
                status_text = status_texts[status_index]
                
                # Sestavit text výsledků
                results_text = f"\nVÝSLEDKY HLASOVÁNÍ DIVÁKŮ:\n"
                results_text += "=" * 35 + "\n\n"
                
                # Zohlednit 50:50 nápovědu při zobrazení výsledků
                available_indices = list(range(4))
                if self.lifeline_5050_used and self.lifeline_5050_indices:
                    available_indices = list(self.lifeline_5050_indices)
                
                for i, (label, option, percentage) in enumerate(zip(labels, options, current_results)):
                    if option and i in available_indices:  # Pouze pokud existuje odpověď a je dostupná
                        results_text += f"{label}: {percentage}%\n"
                
                results_text += f"\n{status_text}"
                
                # Aktualizovat display
                if self.update_display_callback:
                    total_question_number = self.get_total_question_number()
                    money_amount = self.get_money_amount_for_question(total_question_number)
                    if self.media_type == "series":
                        game_title = "SERIÁLOVÝ MILIONÁŘ"
                    elif self.media_type == "actor":
                        game_title = f"MILIONÁŘ S HERCEM"
                    else:
                        game_title = "FILMOVÝ MILIONÁŘ"
                        
                    question_text = f"""{game_title} - OTÁZKA {total_question_number}

OTÁZKA ZA: {money_amount}

{self.current_question['question']}

{results_text}"""
                    
                    self.update_display_callback(self._add_ai_info_header(question_text))
                
                # Pokud není poslední krok, počkat 0.5 sekundy
                if step < total_steps:
                    time.sleep(0.5)

        
        # Spustit animaci v background threadu
        thread = threading.Thread(target=animation_thread)
        thread.daemon = True
        thread.start()

    def use_tip_lifeline(self):
        """Aktivuje nápovědu tip od režiséra - osobní radu od tvůrce."""
        if self.lifeline_tip_used:
            return  # Již použito
            
        if self.game_state not in ["playing_easy", "playing_medium", "playing_hard"]:
            return  # Nápověda není dostupná
            
        if not self.current_question:
            return
            
        # Označit jako použito PŘED voláním AI
        self.lifeline_tip_used = True
        
        # Zobrazit loading zprávu
        self.show_tip_loading()
        
        # Spustit generování tipu v background
        import threading
        thread = threading.Thread(target=self._generate_director_tip)
        thread.daemon = True
        thread.start()
        
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        if self.update_millionaire_ladder_callback:
            self.update_millionaire_ladder_callback()

    def show_tip_loading(self):
        """Zobrazí loading zprávu během generování tipu."""
        if self.media_type == "series":
            role_name = "TVŮRCE SERIÁLU"
        elif self.media_type == "actor":
            role_name = "HEREC"
        elif self.media_type == "director":
            role_name = "REŽISÉR"
        elif self.media_type == "creator":
            role_name = "TVŮRCE"
        else:
            role_name = "REŽISÉR FILMU"
        
        total_question_number = self.get_total_question_number()
        money_amount = self.get_money_amount_for_question(total_question_number)
        
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        elif self.media_type == "director":
            game_title = f"MILIONÁŘ S REŽISÉREM"
        elif self.media_type == "creator":
            game_title = f"MILIONÁŘ S TVŮRCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            
        display_text = f"""{game_title} - OTÁZKA {total_question_number}

OTÁZKA ZA: {money_amount}

{self.current_question['question']}

{role_name} PŘIPRAVUJE TIP...

Chvíli strpení..."""

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(display_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Načítání...", "...", "...", "...")

    def _generate_director_tip(self):
        """Generuje osobní radu od režiséra/tvůrce pomocí AI."""
        try:
            from wcs.ai import AIHandler as ai_handler
            
            # Generovat tip pomocí AI
            director_tip = ai_handler.generate_director_tip(
                media_type=self.media_type,
                title=self.title,
                question=self.current_question['question'],
                options=self.current_question['options'],
                plot=self.plot,
                genre=self.genre,
                episode_title=self.episode_title,
                season_number=self.season_number,
                episode_number=self.episode_number
            )
            
            if director_tip and director_tip.strip():
                self.show_director_tip(director_tip.strip())
            else:
                self.show_tip_error("Nepodařilo se vygenerovat osobní radu od režiséra.")
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error generating director tip: {e}", level=xbmc.LOGERROR)
            self.show_tip_error(f"Chyba při generování tipu: {str(e)}")

    def show_director_tip(self, tip_text):
        """Zobrazí osobní radu od režiséra/tvůrce."""
        if self.media_type == "series":
            role_name = "TVŮRCE"
        elif self.media_type == "actor":
            role_name = "HERCE"
        elif self.media_type == "director":
            role_name = "REŽISÉRA"
        elif self.media_type == "creator":
            role_name = "TVŮRCE"
        else:
            role_name = "REŽISÉRA"
        
        # Aktuální otázka + tip
        total_question_number = self.get_total_question_number()
        money_amount = self.get_money_amount_for_question(total_question_number)
        
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        elif self.media_type == "director":
            game_title = f"MILIONÁŘ S REŽISÉREM"
        elif self.media_type == "creator":
            game_title = f"MILIONÁŘ S TVŮRCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            
        display_text = f"""{game_title} - OTÁZKA {total_question_number}

OTÁZKA ZA: {money_amount}

{self.current_question['question']}

TIP OD {role_name}:
{tip_text}

Vyberte svou odpověď:"""

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(display_text))
            
        # Tlačítka zůstávají stejná (odpovědi)
        options = self.current_question.get('options', [])
        correct_index = self.current_question.get('correct', 0)
        
        # Kontrola nastavení pro zobrazení "x" u správné odpovědi
        show_correct_marker = self.addon.getSetting('millionaire_show_correct_answer') == 'true'
        
        if self.lifeline_5050_used and self.lifeline_5050_indices:
            keep = set(self.lifeline_5050_indices)
            button_texts = []
            for i in range(4):
                if i in keep:
                    text = options[i]
                    button_texts.append(text)
                else:
                    button_texts.append("")
        else:
            button_texts = []
            for i in range(4):
                if i < len(options):
                    text = options[i]
                    button_texts.append(text)
                else:
                    button_texts.append("")
        if self.update_buttons_callback:
            self.update_buttons_callback(*button_texts)

    def show_tip_error(self, error_message):
        """Zobrazí chybu při generování tipu."""
        if self.media_type == "series":
            role_name = "TVŮRCE"
        elif self.media_type == "actor":
            role_name = "HERCE"
        elif self.media_type == "director":
            role_name = "REŽISÉRA"
        elif self.media_type == "creator":
            role_name = "TVŮRCE"
        else:
            role_name = "REŽISÉRA"
        
        total_question_number = self.get_total_question_number()
        money_amount = self.get_money_amount_for_question(total_question_number)
        
        if self.media_type == "series":
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            game_title = f"MILIONÁŘ S HERCEM"
        elif self.media_type == "director":
            game_title = f"MILIONÁŘ S REŽISÉREM"
        elif self.media_type == "creator":
            game_title = f"MILIONÁŘ S TVŮRCEM"
        else:
            game_title = "FILMOVÝ MILIONÁŘ"
            
        display_text = f"""{game_title} - OTÁZKA {total_question_number}

OTÁZKA ZA: {money_amount}

{self.current_question['question']}

CHYBA PŘI GENEROVÁNÍ TIPU OD {role_name}:
{error_message}

Zkuste to znovu později nebo použijte jinou nápovědu.

Vyberte svou odpověď:"""

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(display_text))
            
        # Tlačítka zůstávají stejná (odpovědi)
        options = self.current_question.get('options', [])
        correct_index = self.current_question.get('correct', 0)
        
        # Kontrola nastavení pro zobrazení "x" u správné odpovědi
        show_correct_marker = self.addon.getSetting('millionaire_show_correct_answer') == 'true'
        
        if self.lifeline_5050_used and self.lifeline_5050_indices:
            keep = set(self.lifeline_5050_indices)
            button_texts = []
            for i in range(4):
                if i in keep:
                    text = options[i]
                    button_texts.append(text)
                else:
                    button_texts.append("")
        else:
            button_texts = []
            for i in range(4):
                if i < len(options):
                    text = options[i]
                    button_texts.append(text)
                else:
                    button_texts.append("")
        if self.update_buttons_callback:
            self.update_buttons_callback(*button_texts)

    def reset_lifelines(self):
        """Resetuje všechny nápovědy na začátku nové hry nebo při návratu do menu."""
        self.lifeline_5050_used = False
        self.lifeline_5050_indices = None
        self.lifeline_audience_used = False
        self.lifeline_audience_results = None
        self.lifeline_tip_used = False
        # Rezerva pro další nápovědy (např. self.lifeline_tip_used = False, ...) 

    def animate_generating_ladder(self, control_ids, duration=3.0, interval=0.2):
        import threading, time
        self.ladder_anim_running = True
        colors = ["AFBBAA33", "BFBBAA33", "CFBBAA33", "DFBBAA33", "EFBBAA33", "FFBBAA33"]
        def anim():
            try:
                start = time.time()
                idx = 0
                self.ladder_anim_toggle = 0
                while time.time() - start < duration and self.ladder_anim_running:
                    self.ladder_anim_toggle = idx
                    if self.update_millionaire_ladder_callback:
                        self.update_millionaire_ladder_callback()
                    idx = (idx + 1) % len(colors)
                    time.sleep(interval)
            except Exception as e:
                import xbmc
                xbmc.log(f"[plugin.video.milionar] Ladder animation error: {e}", level=xbmc.LOGERROR)
            finally:
                self.ladder_anim_running = False
                self.ladder_anim_toggle = 0
        t = threading.Thread(target=anim)
        t.daemon = True
        t.start()

    def stop_ladder_animation(self):
        self.ladder_anim_running = False
        self.ladder_anim_toggle = 0