# wcs/my_list_dialog.py
import sys
import os
import xbmc
import xbmcgui
import xbmcplugin
import requests
from urllib.parse import quote_plus
import xbmcaddon
import xbmcvfs
import json
import re
from datetime import datetime

from wcs.metadata.TMDbClient import get_tmdb_api_key
from wcs import user_data
from wcs import utils
from wcs.metadata import TMDbCache as tmdb_cache

class MyListDialog(xbmcgui.WindowXMLDialog):
    """
    Dialog pro zobrazení "Můj seznam" s mřížkou filmů a seriálů.
    """
    def __init__(self, *args, **kwargs):
        super().__init__()
        self.addon = xbmcaddon.Addon()
        self.current_section = "movies"  # Výchozí sekce
        self.movies = []
        self.series = []
        self.current_focus_item = None
        self.movies_position = 0  # Zapamatování pozice v sekci filmů
        self.series_position = 0  # Zapamatování pozice v sekci seriálů
        
        # Nové proměnné pro navigaci seriálů
        self.current_series_id = None
        self.current_series_title = None
        self.seasons = []
        self.current_season_number = None
        self.episodes = []
        self.navigation_level = "series"  # "series", "seasons", "episodes"
        
        # Nové proměnné pro overlay efekt
        self.overlay_active = False
        self.overlay_item = None
        self.overlay_buttons = ["Přehrát", "Detaily", "Hra", "Smazat"]
        self.overlay_selected_button = 0
        
        # Nové proměnné pro overlay efekt epizod
        self.episode_overlay_active = False
        self.episode_overlay_item = None
        self.episode_overlay_buttons = ["Přehrát", "Detaily", "Hra", "Smazat"]
        self.episode_overlay_selected_button = 0
        
    def onInit(self):
        """Inicializace dialogu."""
        xbmc.log("[plugin.video.milionar] Initializing MyListDialog", level=xbmc.LOGINFO)
        
        # Zobrazíme indikátor načítání
        xbmc.executebuiltin("ActivateWindow(busydialognocancel)")
        
        # Načteme data
        self._load_data()
        
        # Skryjeme indikátor načítání
        xbmc.executebuiltin("Dialog.Close(busydialognocancel)")
        
        # Nastavíme výchozí focus na první položku v sekci filmů
        self._set_initial_focus()
        
    def _load_data(self):
        """Načte data filmů a seriálů."""
        try:
            # Načteme filmy a seriály z user_data
            self.movies = user_data.load_tmdb_movies(self.addon)
            self.series = user_data.load_tmdb_series(self.addon)
            
            # Doplníme chybějící data z TMDb API
            self._enhance_movies_data()
            self._enhance_series_data()
            
            # Nastavíme vlastnosti okna
            self._set_window_properties()
            
            # Naplníme seznamy
            self._populate_movies_list()
            self._populate_series_list()
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error loading data: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", "Nepodařilo se načíst data.")
            
    def _enhance_movies_data(self):
        """Doplní chybějící data filmů z TMDb API s cacheováním."""
        for movie in self.movies:
            if not movie.get('genre') or not movie.get('rating'):
                try:
                    tmdb_id = movie.get('id')
                    if tmdb_id:
                        # Cache endpoint pro film
                        endpoint = f'/movie/{tmdb_id}'
                        params = {'language': 'cs-CZ'}
                        
                        def fetch_movie_data():
                            """Funkce pro stažení dat filmu z TMDb"""
                            try:
                                api_key = get_tmdb_api_key(self.addon)
                                url = f'https://api.themoviedb.org/3/movie/{tmdb_id}?api_key={api_key}&language=cs-CZ'
                                response = requests.get(url, timeout=10)
                                if response.status_code == 200:
                                    return response.json()
                                return None
                            except Exception as e:
                                xbmc.log(f"[plugin.video.milionar] Error fetching movie data: {e}", level=xbmc.LOGERROR)
                                return None
                        
                        # Použití cache s 7 dny platnosti
                        data = tmdb_cache.get_or_fetch(endpoint, params, fetch_movie_data, expire_seconds=7*24*3600)
                        
                        if data:
                            movie['genre'] = ', '.join([genre['name'] for genre in data.get('genres', [])[:2]])
                            movie['rating'] = f"{data.get('vote_average', 0):.1f}"
                            movie['backdrop_path'] = data.get('backdrop_path', '')
                            # Doplň další TMDb pole potřebná pro přehrávací dialog
                            movie['overview'] = data.get('overview', '')
                            runtime_val = int(data.get('runtime') or 0)
                            movie['runtime'] = f"{runtime_val} min" if runtime_val > 0 else ''
                            
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Error enhancing movie data: {e}", level=xbmc.LOGERROR)
                    
    def _enhance_series_data(self):
        """Doplní chybějící data seriálů z TMDb API s cacheováním."""
        for series in self.series:
            if not series.get('genre') or not series.get('rating'):
                try:
                    tmdb_id = series.get('id')
                    if tmdb_id:
                        # Cache endpoint pro seriál
                        endpoint = f'/tv/{tmdb_id}'
                        params = {'language': 'cs-CZ'}
                        
                        def fetch_series_data():
                            """Funkce pro stažení dat seriálu z TMDb"""
                            try:
                                api_key = get_tmdb_api_key(self.addon)
                                url = f'https://api.themoviedb.org/3/tv/{tmdb_id}?api_key={api_key}&language=cs-CZ'
                                response = requests.get(url, timeout=10)
                                if response.status_code == 200:
                                    return response.json()
                                return None
                            except Exception as e:
                                xbmc.log(f"[plugin.video.milionar] Error fetching series data: {e}", level=xbmc.LOGERROR)
                                return None
                        
                        # Použití cache s 7 dny platnosti
                        data = tmdb_cache.get_or_fetch(endpoint, params, fetch_series_data, expire_seconds=7*24*3600)
                        
                        if data:
                            series['genre'] = ', '.join([genre['name'] for genre in data.get('genres', [])[:2]])
                            series['rating'] = f"{data.get('vote_average', 0):.1f}"
                            series['backdrop_path'] = data.get('backdrop_path', '')
                            
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Error enhancing series data: {e}", level=xbmc.LOGERROR)
            
    def _set_window_properties(self):
        """Nastaví vlastnosti okna pro navigaci a focus."""
        # Nastavíme aktuální sekci
        self.setProperty("CurrentSection", self.current_section)
        
        # Nastavíme úroveň navigace
        self.setProperty("NavigationLevel", self.navigation_level)
        
        # Nastavíme nadpis dialogu
        self.setProperty("DialogTitle", self.addon.getLocalizedString(32163))
        
        # Nastavíme nadpisy sekcí bez symbolu > (ikony budou v XML)
        self.setProperty("MoviesSectionTitle", self.addon.getLocalizedString(30159))
        self.setProperty("SeriesSectionTitle", self.addon.getLocalizedString(30160))
        
        # Nastavíme fanart první položky s focus efektem
        if self.current_section == "movies" and self.movies:
            first_movie = self.movies[0]
            backdrop_path = first_movie.get('backdrop_path', '')
            if backdrop_path:
                fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}"
                self.setProperty("WCS.MyList.Fanart", fanart)
        elif self.current_section == "series" and self.series:
            first_series = self.series[0]
            backdrop_path = first_series.get('backdrop_path', '')
            if backdrop_path:
                fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}"
                self.setProperty("WCS.MyList.Fanart", fanart)
                
    def _populate_movies_list(self):
        """Naplní seznam filmů."""
        movies_list = self.getControl(100)  # ID pro seznam filmů
        if not movies_list:
            return
            
        movies_list.reset()
        
        # Přidáme tlačítko "Přidat film"
        add_item = xbmcgui.ListItem(label="Přidat film")
        add_item.setArt({'poster': 'DefaultAddSource.png'})
        add_item.setProperty("item_type", "add_movie")
        movies_list.addItem(add_item)
        
        # Přidáme filmy
        for movie in self.movies:
            title = movie.get('title', '')
            year = movie.get('year', '')
            genre = movie.get('genre', '')
            rating = movie.get('rating', '')
            poster_path = movie.get('poster_path', '')
            backdrop_path = movie.get('backdrop_path', '')
            plot = movie.get('overview', '')
            runtime = movie.get('runtime', '')
            
            poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
            fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}" if backdrop_path else poster
            
            item = xbmcgui.ListItem(label=title)
            item.setLabel2(f"{year} • {genre} • {rating}")
            item.setArt({'thumb': poster, 'poster': poster, 'fanart': fanart})
            item.setProperty("item_type", "movie")
            item.setProperty("tmdb_id", str(movie.get('id', '')))
            item.setProperty("title", title)
            item.setProperty("fanart", fanart)
            # Předat TMDb data pro play akci
            item.setProperty("plot", plot or '')
            item.setProperty("rating", rating or '')
            item.setProperty("genre", genre or '')
            item.setProperty("runtime", runtime or '')
            # Director neumíme z tohoto endpointu – ponechá se prázdné
            
            movies_list.addItem(item)
            
    def _populate_series_list(self):
        """Naplní seznam seriálů."""
        try:
            series_list = self.getControl(200)  # ID pro seznam seriálů
            if not series_list:
                xbmc.log(f"[plugin.video.milionar] Error: Series list control not found", level=xbmc.LOGERROR)
                return
                
            xbmc.log(f"[plugin.video.milionar] Populating series list with {len(self.series)} series", level=xbmc.LOGINFO)
            series_list.reset()
            
            # Přidáme tlačítko "Přidat seriál"
            add_item = xbmcgui.ListItem(label="Přidat seriál")
            add_item.setArt({'poster': 'DefaultAddSource.png'})
            add_item.setProperty("item_type", "add_series")
            series_list.addItem(add_item)
            
            # Přidáme seriály
            for series in self.series:
                name = series.get('name', '')
                year = series.get('year', '')
                genre = series.get('genre', '')
                rating = series.get('rating', '')
                poster_path = series.get('poster_path', '')
                backdrop_path = series.get('backdrop_path', '')
                
                poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
                fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}" if backdrop_path else poster
                
                item = xbmcgui.ListItem(label=name)
                item.setLabel2(f"{year} • {genre} • {rating}")
                item.setArt({'thumb': poster, 'poster': poster, 'fanart': fanart})
                item.setProperty("item_type", "series")
                item.setProperty("tmdb_id", str(series.get('id', '')))
                item.setProperty("title", name)
                item.setProperty("fanart", fanart)
                
                series_list.addItem(item)
                
            xbmc.log(f"[plugin.video.milionar] Successfully populated series list with {series_list.size()} items", level=xbmc.LOGINFO)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error populating series list: {e}", level=xbmc.LOGERROR)
            
    def _set_initial_focus(self):
        """Nastaví výchozí focus na první položku v sekci filmů."""
        movies_list = self.getControl(100)
        if movies_list and movies_list.size() > 0:
            movies_list.selectItem(0)
            self.setFocus(movies_list)
            # Nastavíme vlastnost pro focus efekt
            self.setProperty("CurrentSection", "movies")
            # Nastavíme dynamický nadpis pro sekci filmů
            self.setProperty("DialogTitle", self.addon.getLocalizedString(30159))  # "Moje filmy"
            # Nastavíme dynamický header pro první položku
            first_item = movies_list.getSelectedItem()
            if first_item:
                self._update_fanart(first_item)
                
    def _update_section_titles(self, item):
        """Aktualizuje nadpisy sekcí s aktuální položkou."""
        if item:
            item_type = item.getProperty("item_type")
            xbmc.log(f"[plugin.video.milionar] Updating section titles for item type: {item_type}, current_section: {self.current_section}, navigation_level: {self.navigation_level}", level=xbmc.LOGINFO)
            
            if item_type in ["add_movie", "add_series"]:
                # Pro "Přidat" položky - název sekce + návodný text
                if self.current_section == "movies":
                    section_title = self.addon.getLocalizedString(30159)
                    if item_type == "add_movie":
                        self.setProperty("MoviesSectionTitle", f"{section_title} > Přidat film")
                    elif item_type == "add_series":
                        self.setProperty("MoviesSectionTitle", f"{section_title} > Přidat seriál")
                else:
                    section_title = self.addon.getLocalizedString(30160)
                    if item_type == "add_movie":
                        self.setProperty("SeriesSectionTitle", f"{section_title} > Přidat film")
                    elif item_type == "add_series":
                        self.setProperty("SeriesSectionTitle", f"{section_title} > Přidat seriál")
            elif item_type in ["back_to_series", "back_to_seasons"]:
                # Pro "Zpět" položky - hierarchická navigace podle aktuální pozice
                if self.current_section == "movies":
                    section_title = self.addon.getLocalizedString(30159)
                    self.setProperty("MoviesSectionTitle", f"{section_title} > Zpět")
                else:
                    section_title = self.addon.getLocalizedString(30160)
                    if self.navigation_level == "seasons":
                        # Jsem v seznamu sezón, tlačítko "Zpět" vede na seznam seriálů
                        series_title = self.current_series_title or "Neznámý seriál"
                        self.setProperty("SeriesSectionTitle", f"{section_title} > {series_title} > Zpět")
                    elif self.navigation_level == "episodes":
                        # Jsem v seznamu epizod, tlačítko "Zpět" vede na seznam sezón
                        series_title = self.current_series_title or "Neznámý seriál"
                        season_number = self.current_season_number or "?"
                        self.setProperty("SeriesSectionTitle", f"{section_title} > {series_title} > Sezóna {season_number} > Zpět")
                    else:
                        # Jsem v seznamu seriálů, tlačítko "Zpět" vede na hlavní menu
                        self.setProperty("SeriesSectionTitle", f"{section_title} > Zpět")
            elif item_type == "season":
                # Pro sezóny - hierarchická navigace
                if self.current_section == "series" and self.navigation_level == "seasons":
                    section_title = self.addon.getLocalizedString(30160)
                    series_title = self.current_series_title or "Neznámý seriál"
                    season_title = item.getProperty("title") or item.getLabel()
                    self.setProperty("SeriesSectionTitle", f"{section_title} > {series_title} > {season_title}")
                else:
                    # Fallback pro jiné případy
                    section_title = self.addon.getLocalizedString(30160)
                    title = item.getProperty("title") or item.getLabel()
                    self.setProperty("SeriesSectionTitle", f"{section_title} > {title}")
            elif item_type == "episode":
                # Pro epizody - hierarchická navigace
                if self.current_section == "series" and self.navigation_level == "episodes":
                    section_title = self.addon.getLocalizedString(30160)
                    series_title = self.current_series_title or "Neznámý seriál"
                    season_number = self.current_season_number or "?"
                    episode_title = item.getProperty("title") or item.getLabel()
                    self.setProperty("SeriesSectionTitle", f"{section_title} > {series_title} > Sezóna {season_number} > {episode_title}")
                else:
                    # Fallback pro jiné případy
                    section_title = self.addon.getLocalizedString(30160)
                    title = item.getProperty("title") or item.getLabel()
                    self.setProperty("SeriesSectionTitle", f"{section_title} > {title}")
            else:
                # Pro normální položky (filmy, seriály) - název sekce + ">" + název položky + rok
                title = item.getProperty("title") or item.getLabel()
                # Získáme rok z Label2 (první část před " • ")
                year = item.getLabel2().split(" • ")[0] if item.getLabel2() else ""
                
                # Sestavíme název s rokem
                if year:
                    title_with_year = f"{title} ({year})"
                else:
                    title_with_year = title
                
                if self.current_section == "movies":
                    section_title = self.addon.getLocalizedString(30159)
                    self.setProperty("MoviesSectionTitle", f"{section_title} > {title_with_year}")
                else:
                    section_title = self.addon.getLocalizedString(30160)
                    self.setProperty("SeriesSectionTitle", f"{section_title} > {title_with_year}")
            
    def _update_fanart(self, item):
        """Aktualizuje fanart podle aktuální položky nebo nastaví background.png pro 'Přidat film/seriál'."""
        if item:
            item_type = item.getProperty("item_type")
            xbmc.log(f"[plugin.video.milionar] Updating fanart for item type: {item_type}, current_section: {self.current_section}", level=xbmc.LOGINFO)
            
            if item_type in ["add_movie", "add_series", "back_to_series", "back_to_seasons"]:
                # Nastavíme background.png pro položky "Přidat film/seriál" a "Zpět"
                self.setProperty("WCS.MyList.Fanart", "special://home/addons/plugin.video.milionar/resources/media/background.png")
                # Nastavíme správný nadpis podle sekce
                if self.current_section == "movies":
                    self.setProperty("DialogTitle", self.addon.getLocalizedString(30159))  # "Moje filmy"
                else:  # series
                    self.setProperty("DialogTitle", self.addon.getLocalizedString(30160))  # "Moje seriály"
                # Skryjeme název položky v headeru
                self.setProperty("WCS.MyList.CurrentTitle", "")
                
                # Nastavíme návodný text podle typu
                if item_type == "add_movie":
                    self.setProperty("WCS.MyList.CurrentMetadata", "Tímto tlačítkem přidáte film do seznamu")
                elif item_type == "add_series":
                    self.setProperty("WCS.MyList.CurrentMetadata", "Tímto tlačítkem přidáte seriál do seznamu")
                elif item_type == "back_to_series":
                    self.setProperty("WCS.MyList.CurrentMetadata", "Vrátit na seznam seriálů")
                elif item_type == "back_to_seasons":
                    self.setProperty("WCS.MyList.CurrentMetadata", "Vrátit na seznam sezón")
            else:
                # Nastavíme fanart a informace pro normální položky
                fanart = item.getProperty("fanart")
                # Nastavíme nový fanart pouze pokud existuje a není prázdný
                if fanart and fanart.strip():
                    self.setProperty("WCS.MyList.Fanart", fanart)
                # Pokud fanart neexistuje, ponecháme původní fanart beze změny
                
                # Nastavíme dynamický nadpis dialogu s rokem
                title = item.getProperty("title") or item.getLabel()
                # Získáme rok z Label2 (první část před " • ")
                year = item.getLabel2().split(" • ")[0] if item.getLabel2() else ""
                
                # Sestavíme název s rokem
                if year:
                    title_with_year = f"{title} ({year})"
                else:
                    title_with_year = title
                
                self.setProperty("DialogTitle", title_with_year)
                
                # Skryjeme pouze název položky v headeru, ale metadata zůstanou
                self.setProperty("WCS.MyList.CurrentTitle", "")
                
                # Nastavíme metadata (rok, žánr, hodnocení)
                year = item.getLabel2().split(" • ")[0] if item.getLabel2() else ""
                genre = item.getLabel2().split(" • ")[1] if item.getLabel2() and " • " in item.getLabel2() else ""
                rating = item.getLabel2().split(" • ")[2] if item.getLabel2() and item.getLabel2().count(" • ") >= 2 else ""
                
                # Sestavíme metadata string
                metadata_parts = []
                if year:
                    metadata_parts.append(year)
                if genre:
                    metadata_parts.append(genre)
                if rating:
                    metadata_parts.append(f"★ {rating}")
                
                metadata = " • ".join(metadata_parts) if metadata_parts else ""
                self.setProperty("WCS.MyList.CurrentMetadata", metadata)
            
            # Aktualizujeme nadpisy sekcí
            self._update_section_titles(item)
                        
    def _switch_section(self, new_section):
        """Přepne mezi sekcemi."""
        if new_section != self.current_section:
            self.current_section = new_section
            self.setProperty("CurrentSection", self.current_section)
            
            # Nastavíme dynamický nadpis podle sekce
            if new_section == "movies":
                self.setProperty("DialogTitle", self.addon.getLocalizedString(30159))  # "Moje filmy"
            else:  # series
                self.setProperty("DialogTitle", self.addon.getLocalizedString(30160))  # "Moje seriály"
                # Pokud jsme v navigaci sezón/epizod, vrátíme se na seznam seriálů
                # POUZE pokud se přepínáme z filmů na seriály
                if self.navigation_level != "series" and self.current_section == "series":
                    # Necháme uživatele rozhodnout, nevrátíme se automaticky
                    pass
            
            # Vynutíme aktualizaci vlastnosti pro XML engine
            xbmc.sleep(50)  # Krátká pauza pro aktualizaci
            
            # Pouze nastavíme focus na seznam, necháme Kodi aby si sám vybral pozici
            if new_section == "movies":
                movies_list = self.getControl(100)
                if movies_list:
                    self.setFocus(movies_list)
                    if movies_list.getSelectedItem():
                        self._update_fanart(movies_list.getSelectedItem())
            else:  # series
                series_list = self.getControl(200)
                if series_list:
                    self.setFocus(series_list)
                    if series_list.getSelectedItem():
                        self._update_fanart(series_list.getSelectedItem())
                        
    def _open_details_dialog(self, item):
        """Aktivuje overlay efekt místo otevírání dialogu."""
        if not item:
            return
            
        item_type = item.getProperty("item_type")
        tmdb_id = item.getProperty("tmdb_id")
        title = item.getProperty("title")
        
        if item_type == "series" and tmdb_id:
            # Pro seriály načteme sezóny místo overlay efektu
            self._load_series_seasons(tmdb_id, title)
        elif item_type == "movie" and tmdb_id:
            # Pro filmy aktivujeme overlay efekt
            self._activate_overlay(item)
        elif item_type == "episode":
            # Pro epizody aktivujeme overlay efekt
            self._activate_episode_overlay(item)
    
    def _activate_overlay(self, item):
        """Aktivuje overlay efekt na vybrané položce."""
        if not item:
            return
            
        self.overlay_active = True
        self.overlay_item = item
        self.overlay_selected_button = 0
        
        # Zapamatujeme si, ze které sekce byl overlay aktivován
        self.overlay_source_section = self.current_section
        
        # Nastavíme vlastnosti pro overlay filmů
        self.setProperty("MovieOverlayActive", "true")
        self.setProperty("OverlayActive", "true")  # Pro kompatibilitu
        self.setProperty("OverlayItemTitle", item.getProperty("title"))
        self.setProperty("OverlayItemType", item.getProperty("item_type"))
        
        # Aktualizujeme UI
        self._update_overlay_buttons()
        
        # Spustíme animaci zobrazení
        self._animate_overlay_show()
        
        # Nastavíme focus na první tlačítko a vynutíme ho
        self._set_focus_to_overlay_button()
        
        xbmc.log(f"[plugin.video.milionar] Overlay activated for: {item.getProperty('title')}", level=xbmc.LOGINFO)
    
    def _activate_episode_overlay(self, item):
        """Aktivuje overlay efekt na vybrané epizodě."""
        if not item:
            return
            
        self.episode_overlay_active = True
        self.episode_overlay_item = item
        self.episode_overlay_selected_button = 0
        
        # Zapamatujeme si, ze které sekce byl overlay aktivován
        self.episode_overlay_source_section = self.current_section
        
        # Nastavíme vlastnosti pro overlay epizod
        self.setProperty("EpisodeOverlayActive", "true")
        self.setProperty("OverlayActive", "true")  # Pro kompatibilitu
        self.setProperty("OverlayItemTitle", item.getProperty("title"))
        self.setProperty("OverlayItemType", item.getProperty("item_type"))
        
        # Aktualizujeme UI
        self._update_episode_overlay_buttons()
        
        # Spustíme animaci zobrazení
        self._animate_episode_overlay_show()
        
        # Nastavíme focus na první tlačítko a vynutíme ho
        self._set_focus_to_episode_overlay_button()
        
        xbmc.log(f"[plugin.video.milionar] Episode overlay activated for: {item.getProperty('title')}", level=xbmc.LOGINFO)
    
    def _deactivate_overlay(self):
        """Deaktivuje overlay efekt."""
        # Spustíme animaci skrytí
        self._animate_overlay_hide()
        
        self.overlay_active = False
        self.overlay_item = None
        self.overlay_selected_button = 0
        
        # Skryjeme overlay vlastnosti
        self.setProperty("MovieOverlayActive", "false")
        self.setProperty("OverlayActive", "false")
        self.setProperty("OverlayItemTitle", "")
        self.setProperty("OverlayItemType", "")
        
        # Vrátíme focus na původní seznam podle sekce, ze které byl overlay aktivován
        source_section = getattr(self, 'overlay_source_section', self.current_section)
        if source_section == "movies":
            movies_list = self.getControl(100)
            if movies_list:
                self.setFocus(movies_list)
        else:
            series_list = self.getControl(200)
            if series_list:
                self.setFocus(series_list)
        
        xbmc.log(f"[plugin.video.milionar] Overlay deactivated, returning to section: {source_section}", level=xbmc.LOGINFO)
    
    def _deactivate_episode_overlay(self):
        """Deaktivuje overlay efekt epizod."""
        # Spustíme animaci skrytí
        self._animate_episode_overlay_hide()
        
        self.episode_overlay_active = False
        self.episode_overlay_item = None
        self.episode_overlay_selected_button = 0
        
        # Skryjeme overlay vlastnosti
        self.setProperty("EpisodeOverlayActive", "false")
        self.setProperty("OverlayActive", "false")
        self.setProperty("OverlayItemTitle", "")
        self.setProperty("OverlayItemType", "")
        
        # Vrátíme focus na původní seznam podle sekce, ze které byl overlay aktivován
        source_section = getattr(self, 'episode_overlay_source_section', self.current_section)
        if source_section == "movies":
            movies_list = self.getControl(100)
            if movies_list:
                self.setFocus(movies_list)
        else:
            series_list = self.getControl(200)
            if series_list:
                self.setFocus(series_list)
        
        xbmc.log(f"[plugin.video.milionar] Episode overlay deactivated, returning to section: {source_section}", level=xbmc.LOGINFO)
    
    def _animate_overlay_show(self):
        """Animuje zobrazení overlay tlačítek pro filmy."""
        try:
            # Spustíme animaci nastavením vlastnosti
            self.setProperty("OverlayAnimation", "show")
            # Počkáme na dokončení animace
            xbmc.sleep(200)
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error in overlay show animation: {e}", level=xbmc.LOGERROR)
    
    def _animate_overlay_hide(self):
        """Animuje skrytí overlay tlačítek pro filmy."""
        try:
            # Vyčistíme vlastnost animace
            self.setProperty("OverlayAnimation", "")
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error in overlay hide animation: {e}", level=xbmc.LOGERROR)
    
    def _animate_episode_overlay_show(self):
        """Animuje zobrazení overlay tlačítek pro epizody."""
        try:
            # Spustíme animaci nastavením vlastnosti
            self.setProperty("EpisodeOverlayAnimation", "show")
            # Počkáme na dokončení animace
            xbmc.sleep(200)
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error in episode overlay show animation: {e}", level=xbmc.LOGERROR)
    
    def _animate_episode_overlay_hide(self):
        """Animuje skrytí overlay tlačítek pro epizody."""
        try:
            # Vyčistíme vlastnost animace
            self.setProperty("EpisodeOverlayAnimation", "")
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error in episode overlay hide animation: {e}", level=xbmc.LOGERROR)
    
    def _set_focus_to_overlay_button(self):
        """Nastaví focus na správné overlay tlačítko podle vybraného indexu."""
        button_ids = [1334, 1336, 1338, 1340]  # ID tlačítek v pořadí (modré verze)
        if 0 <= self.overlay_selected_button < len(button_ids):
            target_button = self.getControl(button_ids[self.overlay_selected_button])
            if target_button:
                self.setFocus(target_button)
                # Vynutíme focus ještě jednou po krátké pauze
                xbmc.sleep(50)
                self.setFocus(target_button)
                xbmc.log(f"[plugin.video.milionar] Focus set to overlay button {self.overlay_selected_button}", level=xbmc.LOGINFO)
    
    def _set_focus_to_episode_overlay_button(self):
        """Nastaví focus na správné overlay tlačítko epizod podle vybraného indexu."""
        button_ids = [1342, 1344, 1346, 1348]  # ID tlačítek v pořadí (modré verze)
        if 0 <= self.episode_overlay_selected_button < len(button_ids):
            target_button = self.getControl(button_ids[self.episode_overlay_selected_button])
            if target_button:
                self.setFocus(target_button)
                # Vynutíme focus ještě jednou po krátké pauze
                xbmc.sleep(50)
                self.setFocus(target_button)
                xbmc.log(f"[plugin.video.milionar] Focus set to episode overlay button {self.episode_overlay_selected_button}", level=xbmc.LOGINFO)
    
    def _update_overlay_buttons(self):
        """Aktualizuje stav tlačítek v overlay efektu."""
        for i, button in enumerate(self.overlay_buttons):
            is_selected = (i == self.overlay_selected_button)
            self.setProperty(f"OverlayButton{i}Text", button)
            self.setProperty(f"OverlayButton{i}Selected", "true" if is_selected else "false")
            
        # Aktualizujeme vlastnost pro aktuální vybrané tlačítko
        self.setProperty("OverlaySelectedButton", str(self.overlay_selected_button))
            
        xbmc.log(f"[plugin.video.milionar] Overlay buttons updated, selected: {self.overlay_selected_button}", level=xbmc.LOGINFO)
    
    def _update_episode_overlay_buttons(self):
        """Aktualizuje stav tlačítek v overlay efektu epizod."""
        for i, button in enumerate(self.episode_overlay_buttons):
            is_selected = (i == self.episode_overlay_selected_button)
            self.setProperty(f"OverlayButton{i}Text", button)
            self.setProperty(f"OverlayButton{i}Selected", "true" if is_selected else "false")
            
        # Aktualizujeme vlastnost pro aktuální vybrané tlačítko
        self.setProperty("OverlaySelectedButton", str(self.episode_overlay_selected_button))
            
        xbmc.log(f"[plugin.video.milionar] Episode overlay buttons updated, selected: {self.episode_overlay_selected_button}", level=xbmc.LOGINFO)
    
    def _execute_overlay_action(self):
        """Spustí akci vybraného tlačítka v overlay efektu."""
        if not self.overlay_active or not self.overlay_item:
            return
            
        item = self.overlay_item
        action = self.overlay_buttons[self.overlay_selected_button]
        
        xbmc.log(f"[plugin.video.milionar] Executing overlay action: {action}", level=xbmc.LOGINFO)
        
        if action == "Přehrát":
            # Sestavíme název s rokem pro vyhledávání
            year = item.getLabel2().split(" • ")[0] if item.getLabel2() else ""
            title = item.getProperty('title')
            plot = item.getProperty('plot') or ''
            poster = item.getArt('poster') or ''
            fanart = item.getArt('fanart') or ''
            rating = item.getProperty('rating') or ''
            genre = item.getProperty('genre') or ''
            director = item.getProperty('director') or ''
            runtime = item.getProperty('runtime') or ''
            from wcs.utils import play_movie_from_addon
            meta = {
                'title': title,
                'year': year,
                'plot': plot,
                'poster': poster,
                'fanart': fanart,
                'rating': rating,
                'genre': genre,
                'director': director,
                'runtime': runtime,
            }
            play_movie_from_addon(meta, self.addon)
            
        elif action == "Detaily":
            # Otevřeme detailní dialog
            from wcs.ui.dialogs.DialogWcsDetails import DetailsWindow
            detail_dialog = DetailsWindow(
                "details_extras/DialogWcsDetails.xml",
                self.addon.getAddonInfo('path'),
                "Default",
                "1080i",
                media_type="movie",
                tmdb_id=item.getProperty("tmdb_id")
            )
            detail_dialog.doModal()
            del detail_dialog
            
        elif action == "Hra":
            # Spustíme herní dialog
            self._start_game_for_item(item)
            
        elif action == "Smazat":
            # Smazání filmu
            from wcs.ui import UI as ui
            tmdb_id = item.getProperty("tmdb_id")
            title = item.getProperty("title")
            
            # Najdeme index filmu podle tmdb_id
            movies = user_data.load_tmdb_movies(self.addon)
            idx = None
            for i, movie in enumerate(movies):
                if str(movie.get('id', '')) == str(tmdb_id):
                    idx = i
                    break
            if idx is not None:
                # Zavoláme UI funkci a zkontrolujeme výsledek
                result = ui.remove_added_movie({'idx': idx}, self.addon)
                # Obnovíme dialog pouze pokud bylo mazání úspěšné
                if result:
                    self.refresh_data()
            else:
                xbmcgui.Dialog().ok("Chyba", f"Film '{title}' nebyl nalezen v seznamu.")
        
        # Deaktivujeme overlay po provedení akce
        self._deactivate_overlay()
    
    def _execute_episode_overlay_action(self):
        """Spustí akci vybraného tlačítka v overlay efektu epizod."""
        if not self.episode_overlay_active or not self.episode_overlay_item:
            return
            
        item = self.episode_overlay_item
        action = self.episode_overlay_buttons[self.episode_overlay_selected_button]
        
        xbmc.log(f"[plugin.video.milionar] Executing episode overlay action: {action}", level=xbmc.LOGINFO)
        
        episode_number = item.getProperty("episode_number")
        season_number = item.getProperty("season_number")
        episode_title = item.getProperty("title")  # Název epizody z TMDb API
        
        if action == "Přehrát":
            # Použijeme správnou logiku pro epizody
            series_name = self.current_series_title or "Neznámý seriál"
            season_number = int(season_number)
            episode_number = int(episode_number)
            episode_name = episode_title  # Správný název epizody z TMDb API
            
            # Najdeme data aktuálního seriálu pro správná metadata
            current_series_data = None
            for series in self.series:
                if str(series.get('id', '')) == str(self.current_series_id):
                    current_series_data = series
                    break
            
            # Získáme metadata epizody z TMDb
            plot = ""
            poster = ""
            fanart = ""
            rating = ""
            genre = ""
            episode_title = ""
            year = ""
            runtime = ""
            
            try:
                # Načteme data epizody z TMDb pro správná metadata
                api_key = get_tmdb_api_key(self.addon)
                episode_url = f'https://api.themoviedb.org/3/tv/{self.current_series_id}/season/{season_number}/episode/{episode_number}?api_key={api_key}&language=cs-CZ'
                episode_response = requests.get(episode_url, timeout=10)
                if episode_response.status_code == 200:
                    episode_data = episode_response.json()
                    plot = episode_data.get('overview', '')
                    rating = f"{episode_data.get('vote_average', 0):.1f}"
                    # Použijeme název epizody z TMDb API, pokud existuje
                    episode_title = episode_data.get('name', episode_name)
                    # Načteme délku epizody
                    runtime_minutes = episode_data.get('runtime', 0)
                    runtime = f"{runtime_minutes} min" if runtime_minutes > 0 else ""
                
                # Načteme data seriálu pro poster a fanart
                series_url = f'https://api.themoviedb.org/3/tv/{self.current_series_id}?api_key={api_key}&language=cs-CZ'
                series_response = requests.get(series_url, timeout=10)
                if series_response.status_code == 200:
                    series_data = series_response.json()
                    poster_path = series_data.get('poster_path', '')
                    backdrop_path = series_data.get('backdrop_path', '')
                    
                    poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
                    fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}" if backdrop_path else poster
                    genre = ', '.join([g['name'] for g in series_data.get('genres', [])[:2]])
                    year = str(series_data.get('first_air_date', '')[:4]) if series_data.get('first_air_date') else ''
                    
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] Error fetching episode/series metadata: {e}", level=xbmc.LOGERROR)
            
            # Pro přehrávání z MyList předáváme čistý název epizody (bez názvu seriálu),
            # aby se v úvodním epizodním dialogu zobrazovalo jen jméno epizody
            ep_name_for_dialog = episode_title if episode_title else f"Epizoda {episode_number}"
            
            # Spuštění přes jednotnou utils funkci – bez duplicit dialogů
            from wcs.utils import play_episode_from_addon
            meta = {
                'series_name': series_name,
                'season_number': season_number,
                'episode_number': episode_number,
                'episode_title': ep_name_for_dialog,
                'poster': poster,                       # poster seriálu pro UI
                'fanart': fanart,
                'episode_thumb': item.getProperty("fanart") or poster,  # still epizody
                'plot': plot,
                'rating': rating,
                'genre': genre,
                'year': year,
                'runtime': runtime,
                'tmdb_id': self.current_series_id,
            }
            play_episode_from_addon(meta, self.addon)
            
        elif action == "Detaily":
            # Otevřeme detailní dialog pro epizodu
            from wcs.ui.dialogs.DialogWcsDetails import DetailsWindow
            detail_dialog = DetailsWindow(
                "details_extras/DialogWcsDetails.xml",
                self.addon.getAddonInfo('path'),
                "Default",
                "1080i",
                media_type="tv",
                tmdb_id=self.current_series_id,
                season_number=season_number,
                episode_number=episode_number
            )
            detail_dialog.doModal()
            del detail_dialog
            
        elif action == "Hra":
            # Spustíme herní dialog pro epizodu - použijeme stejnou logiku jako kontextové menu
            series_name = self.current_series_title or "Neznámý seriál"
            episode_name = title
            season_number = int(season_number)
            episode_number = int(episode_number)
            
            # Sestavíme URL pro play_text_game akci stejně jako v kontextovém menu
            import urllib.parse as _urllib_parse
            safe_series_name = _urllib_parse.quote_plus(series_name)
            safe_episode_title = _urllib_parse.quote_plus(episode_name)
            safe_plot = _urllib_parse.quote_plus("")  # Pro epizody nemáme plot
            safe_poster = _urllib_parse.quote_plus(item.getProperty("fanart") or "")
            
            play_game_url = (f"plugin://{self.addon.getAddonInfo('id')}?"
                           f"action=play_text_game&"
                           f"media_type=series&"
                           f"tmdb_id={self.current_series_id}&"
                           f"title={safe_series_name}&"
                           f"plot={safe_plot}&"
                           f"poster={safe_poster}&"
                           f"episode_title={safe_episode_title}&"
                           f"season_number={season_number}&"
                           f"episode_number={episode_number}")
            
            xbmc.log(f"[plugin.video.milionar] Playing game for episode: {play_game_url}", level=xbmc.LOGINFO)
            xbmc.executebuiltin(f'RunPlugin("{play_game_url}")')
            
        elif action == "Smazat":
            # Pro epizody smažeme celý seriál
            from wcs.ui import UI as ui
            # Najdeme index seriálu podle current_series_id
            series = user_data.load_tmdb_series(self.addon)
            idx = None
            for i, series_item in enumerate(series):
                if str(series_item.get('id', '')) == str(self.current_series_id):
                    idx = i
                    break
            if idx is not None:
                # Zavoláme UI funkci a zkontrolujeme výsledek
                result = ui.remove_added_series({'idx': idx}, self.addon)
                # Obnovíme dialog pouze pokud bylo mazání úspěšné
                if result:
                    self.refresh_data()
            else:
                xbmcgui.Dialog().ok("Chyba", f"Seriál '{self.current_series_title}' nebyl nalezen v seznamu.")
        
        # Deaktivujeme overlay po provedení akce
        self._deactivate_episode_overlay()
            
    def _start_game_for_item(self, item):
        """Spustí herní dialog pro vybranou položku."""
        if not item:
            return
            
        item_type = item.getProperty("item_type")
        tmdb_id = item.getProperty("tmdb_id")
        title = item.getProperty("title")
        
        if item_type not in ["movie", "series"] or not tmdb_id:
            xbmcgui.Dialog().ok("Chyba", "Nelze spustit hru pro tuto položku.")
            return
        
        # Získáme metadata o filmu/seriálu
        media_type = "movie" if item_type == "movie" else "tv"
        plot = ""
        genre = ""
        fanart = item.getProperty("fanart") or ""
        
        # Zkusíme získat dodatečné informace z TMDb
        try:
            if media_type == "movie":
                from wcs.metadata import TMDbClient as tmdb_handler
                movie_details = tmdb_handler.get_movie_detailed_info(tmdb_id)
                if movie_details:
                    plot = movie_details.get('overview', '')
                    genres = movie_details.get('genres', [])
                    if genres:
                        genre = ', '.join([g.get('name', '') for g in genres[:3]])
                    backdrop_path = movie_details.get('backdrop_path', '')
                    if backdrop_path:
                        fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}"
            else:  # series
                import requests
                api_key = tmdb_handler.get_tmdb_api_key(self.addon)
                url = f'https://api.themoviedb.org/3/tv/{tmdb_id}'
                response = requests.get(url, params={'api_key': api_key, 'language': 'cs-CZ'}, timeout=10)
                response.raise_for_status()
                series_details = response.json()
                if series_details:
                    plot = series_details.get('overview', '')
                    genres = series_details.get('genres', [])
                    if genres:
                        genre = ', '.join([g.get('name', '') for g in genres[:3]])
                    backdrop_path = series_details.get('backdrop_path', '')
                    if backdrop_path:
                        fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}"
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error getting details for game: {e}", level=xbmc.LOGERROR)
        
        # Spustíme herní dialog
        from wcs.games.DialogWcsTextGame import TextGameWindow
        dialog = TextGameWindow(
            "games/DialogWcsTextGame.xml",
            self.addon.getAddonInfo('path'),
            "Default",
            title=title,
            fanart=fanart,
            plot=plot,
            genre=genre,
            media_type=media_type
        )
        dialog.doModal()
        del dialog
            
    def _add_new_item(self, item_type):
        """Otevře dialog pro přidání nové položky."""
        if item_type == "add_movie":
            # Otevřeme dialog pro přidání filmu
            search_query = xbmcgui.Dialog().input(self.addon.getLocalizedString(30152), type=xbmcgui.INPUT_ALPHANUM)
            if search_query:
                try:
                    from wcs.metadata.TMDbClient import get_tmdb_api_key
                    import requests
                    from urllib.parse import quote_plus
                    
                    url = f'https://api.themoviedb.org/3/search/movie?api_key={get_tmdb_api_key(self.addon)}&language=cs-CZ&query={quote_plus(search_query)}'
                    response = requests.get(url)
                    response.raise_for_status()
                    results = response.json().get('results', [])
                    
                    if results:
                        choices = [f"{item['title']} ({item.get('release_date', '')[:4]})" for item in results]
                        selected_index = xbmcgui.Dialog().select(self.addon.getLocalizedString(30154), choices)
                        if selected_index >= 0:
                            selected_item = results[selected_index]
                            from wcs.metadata import TMDbClient as tmdb_handler
                            tmdb_handler.add_movie_to_list({'tmdb_id': selected_item['id'], 'title': selected_item['title']}, self.addon)
                            # Obnovíme dialog po přidání
                            self.refresh_data()
                    else:
                        xbmcgui.Dialog().notification(self.addon.getLocalizedString(30153), search_query, xbmcgui.NOTIFICATION_INFO)
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Error adding movie: {e}", level=xbmc.LOGERROR)
                    xbmcgui.Dialog().ok("Chyba", f"Nepodařilo se přidat film: {e}")
        elif item_type == "add_series":
            # Otevřeme dialog pro přidání seriálu
            from wcs.ui import UI as ui
            ui.add_series({'new': '1', 'add_to_my_series': '1'}, self.addon, None)
            # Obnovíme dialog po přidání
            self.refresh_data()
            
    def refresh_data(self):
        """Obnoví data dialogu."""
        self._load_data()
        self._populate_movies_list()
        self._populate_series_list()
            
    def onClick(self, controlId):
        """Zpracuje kliknutí na kontrolky."""
        if self.overlay_active and controlId in [1334, 1336, 1338, 1340]:  # ID overlay tlačítek filmů
            # Mapování ID tlačítek na index
            button_map = {1334: 0, 1336: 1, 1338: 2, 1340: 3}
            self.overlay_selected_button = button_map.get(controlId, 0)
            self._execute_overlay_action()
        elif self.episode_overlay_active and controlId in [1342, 1344, 1346, 1348]:  # ID overlay tlačítek epizod
            # Mapování ID tlačítek na index
            button_map = {1342: 0, 1344: 1, 1346: 2, 1348: 3}
            self.episode_overlay_selected_button = button_map.get(controlId, 0)
            self._execute_episode_overlay_action()
        else:
            # Normální logika pro kliknutí je přesunuta do onAction pro vyhnutí se duplicitnímu volání
            pass
                        
    def onAction(self, action):
        """Zpracuje akce (klávesy, dálkové ovládání)."""
        action_id = action.getId()
        
        if action_id == xbmcgui.ACTION_NAV_BACK or action_id == xbmcgui.ACTION_PREVIOUS_MENU:
            if self.overlay_active:
                # Pokud je aktivní overlay filmů, deaktivujeme ho místo zavření dialogu
                self._deactivate_overlay()
            elif self.episode_overlay_active:
                # Pokud je aktivní overlay epizod, deaktivujeme ho místo zavření dialogu
                self._deactivate_episode_overlay()
            else:
                self.close()
            
        elif action_id == xbmcgui.ACTION_SELECT_ITEM:
            # Enter - otevření detailů nebo přidání
            if self.overlay_active:
                # Pokud je aktivní overlay filmů, spustíme vybranou akci
                self._execute_overlay_action()
            elif self.episode_overlay_active:
                # Pokud je aktivní overlay epizod, spustíme vybranou akci
                self._execute_episode_overlay_action()
            else:
                # Normální logika pro kliknutí na položku
                focused_control = self.getFocus()
                if focused_control:
                    selected_item = focused_control.getSelectedItem()
                    if selected_item:
                        item_type = selected_item.getProperty("item_type")
                        xbmc.log(f"[plugin.video.milionar] Clicked on item type: {item_type}", level=xbmc.LOGINFO)
                        
                        if item_type in ["add_movie", "add_series"]:
                            self._add_new_item(item_type)
                        elif item_type == "movie":
                            self._open_details_dialog(selected_item)
                        elif item_type == "series":
                            xbmc.log(f"[plugin.video.milionar] Opening series details for: {selected_item.getProperty('title')}", level=xbmc.LOGINFO)
                            self._open_details_dialog(selected_item)
                        elif item_type == "season":
                            # Načteme epizody sezóny
                            season_number = selected_item.getProperty("season_number")
                            xbmc.log(f"[plugin.video.milionar] Loading episodes for season: {season_number}", level=xbmc.LOGINFO)
                            if season_number:
                                self._load_season_episodes(int(season_number))
                        elif item_type == "episode":
                            # Otevřeme overlay efekt pro epizodu
                            xbmc.log(f"[plugin.video.milionar] Opening episode overlay for: {selected_item.getProperty('title')}", level=xbmc.LOGINFO)
                            self._open_details_dialog(selected_item)
                        elif item_type == "back_to_series":
                            # Vrátíme se na seznam seriálů
                            xbmc.log(f"[plugin.video.milionar] Going back to series list", level=xbmc.LOGINFO)
                            self._go_back_to_series()
                        elif item_type == "back_to_seasons":
                            # Vrátíme se na seznam sezón
                            xbmc.log(f"[plugin.video.milionar] Going back to seasons list", level=xbmc.LOGINFO)
                            self._go_back_to_seasons()
                        
        # Navigace v overlay efektu - odstraněno, navigace řešena XML
        # Aktualizace fanart při navigačních akcích (pouze pokud není aktivní overlay)
        elif not self.overlay_active and not self.episode_overlay_active and action_id in [xbmcgui.ACTION_MOVE_LEFT, xbmcgui.ACTION_MOVE_RIGHT, 
                                                      xbmcgui.ACTION_MOVE_UP, xbmcgui.ACTION_MOVE_DOWN]:
            # Krátká pauza pro aktualizaci seznamu
            xbmc.sleep(50)
            # Aktualizujeme fanart podle aktuální položky
            focused_control = self.getFocus()
            if focused_control:
                selected_item = focused_control.getSelectedItem()
                if selected_item:
                    self._update_fanart(selected_item)
        
        # Blokujeme všechny ostatní akce když je aktivní overlay
        elif self.overlay_active:
            # Povolujeme pouze Back, Select a navigační akce pro overlay filmů
            if action_id in [xbmcgui.ACTION_MOVE_UP, xbmcgui.ACTION_MOVE_DOWN]:
                # Navigace mezi overlay tlačítky
                if action_id == xbmcgui.ACTION_MOVE_UP:
                    # Předchozí tlačítko
                    self.overlay_selected_button = (self.overlay_selected_button - 1) % len(self.overlay_buttons)
                elif action_id == xbmcgui.ACTION_MOVE_DOWN:
                    # Další tlačítko
                    self.overlay_selected_button = (self.overlay_selected_button + 1) % len(self.overlay_buttons)
                
                # Aktualizujeme UI a nastavíme focus na správné tlačítko
                self._update_overlay_buttons()
                self._set_focus_to_overlay_button()
                
            elif action_id in [xbmcgui.ACTION_MOVE_LEFT, xbmcgui.ACTION_MOVE_RIGHT]:
                # Blokujeme navigaci doleva/doprava v seznamech
                return
                
            elif action_id not in [xbmcgui.ACTION_NAV_BACK, xbmcgui.ACTION_PREVIOUS_MENU, xbmcgui.ACTION_SELECT_ITEM]:
                return  # Blokujeme ostatní akce
                
        elif self.episode_overlay_active:
            # Povolujeme pouze Back, Select a navigační akce pro overlay epizod
            if action_id in [xbmcgui.ACTION_MOVE_UP, xbmcgui.ACTION_MOVE_DOWN]:
                # Navigace mezi overlay tlačítky
                if action_id == xbmcgui.ACTION_MOVE_UP:
                    # Předchozí tlačítko
                    self.episode_overlay_selected_button = (self.episode_overlay_selected_button - 1) % len(self.episode_overlay_buttons)
                elif action_id == xbmcgui.ACTION_MOVE_DOWN:
                    # Další tlačítko
                    self.episode_overlay_selected_button = (self.episode_overlay_selected_button + 1) % len(self.episode_overlay_buttons)
                
                # Aktualizujeme UI a nastavíme focus na správné tlačítko
                self._update_episode_overlay_buttons()
                self._set_focus_to_episode_overlay_button()
                
            elif action_id in [xbmcgui.ACTION_MOVE_LEFT, xbmcgui.ACTION_MOVE_RIGHT]:
                # Blokujeme navigaci doleva/doprava v seznamech
                return
                
            elif action_id not in [xbmcgui.ACTION_NAV_BACK, xbmcgui.ACTION_PREVIOUS_MENU, xbmcgui.ACTION_SELECT_ITEM]:
                return  # Blokujeme ostatní akce
                        
    def onFocus(self, controlId):
        """Zpracuje změnu focus efektu."""
        # Pokud je aktivní overlay, vrátíme focus na overlay tlačítka
        if self.overlay_active:
            self._set_focus_to_overlay_button()
            return
        elif self.episode_overlay_active:
            self._set_focus_to_episode_overlay_button()
            return
            
        if controlId == 100:  # Seznam filmů
            # Aktualizujeme vlastnost pouze pokud není aktivní overlay
            if not self.overlay_active and not self.episode_overlay_active:
                self.current_section = "movies"
                self.setProperty("CurrentSection", "movies")
            movies_list = self.getControl(100)
            if movies_list:
                # Zapamatujeme si aktuální pozici
                self.movies_position = movies_list.getSelectedPosition()
                selected_item = movies_list.getSelectedItem()
                if selected_item:
                    self._update_fanart(selected_item)
                    
        elif controlId == 200:  # Seznam seriálů
            # Aktualizujeme vlastnost pouze pokud není aktivní overlay
            if not self.overlay_active and not self.episode_overlay_active:
                self.current_section = "series"
                self.setProperty("CurrentSection", "series")
            series_list = self.getControl(200)
            if series_list:
                # Zapamatujeme si aktuální pozici
                self.series_position = series_list.getSelectedPosition()
                selected_item = series_list.getSelectedItem()
                if selected_item:
                    self._update_fanart(selected_item)
    
    def _load_series_seasons(self, series_id, series_title):
        """Načte sezóny seriálu a zobrazí je v sekci seriálů."""
        try:
            xbmc.log(f"[plugin.video.milionar] Loading seasons for series {series_id}: {series_title}", level=xbmc.LOGINFO)
            
            # Nastavíme stav navigace
            self.current_series_id = series_id
            self.current_series_title = series_title
            self.navigation_level = "seasons"
            self.setProperty("NavigationLevel", self.navigation_level)
            
            # Zobrazíme indikátor načítání
            xbmc.executebuiltin("ActivateWindow(busydialognocancel)")
            
            # Cache endpoint pro sezóny seriálu
            endpoint = f'/tv/{series_id}'
            params = {'language': 'cs-CZ', 'append_to_response': 'seasons'}
            
            def fetch_series_data():
                """Funkce pro stažení dat seriálu z TMDb"""
                try:
                    api_key = get_tmdb_api_key(self.addon)
                    url = f'https://api.themoviedb.org/3/tv/{series_id}?api_key={api_key}&language=cs-CZ&append_to_response=seasons'
                    xbmc.log(f"[plugin.video.milionar] Fetching series data from: {url}", level=xbmc.LOGINFO)
                    response = requests.get(url, timeout=10)
                    if response.status_code == 200:
                        data = response.json()
                        xbmc.log(f"[plugin.video.milionar] Successfully fetched series data, seasons count: {len(data.get('seasons', []))}", level=xbmc.LOGINFO)
                        return data
                    else:
                        xbmc.log(f"[plugin.video.milionar] HTTP error {response.status_code} when fetching series data", level=xbmc.LOGERROR)
                        return None
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Error fetching series data: {e}", level=xbmc.LOGERROR)
                    return None
            
            # Použití cache s 7 dny platnosti
            data = tmdb_cache.get_or_fetch(endpoint, params, fetch_series_data, expire_seconds=7*24*3600)
            
            if data and 'seasons' in data:
                self.seasons = data['seasons']
                xbmc.log(f"[plugin.video.milionar] Successfully loaded {len(self.seasons)} seasons", level=xbmc.LOGINFO)
                
                # Skryjeme indikátor načítání
                xbmc.executebuiltin("Dialog.Close(busydialognocancel)")
                
                # Zobrazíme sezóny v sekci seriálů
                self._populate_seasons_list()
                
                # Nastavíme focus na první sezónu (tlačítko "Zpět")
                series_list = self.getControl(200)
                if series_list and series_list.size() > 0:
                    # Nastavíme focus na první položku (tlačítko "Zpět")
                    series_list.selectItem(0)
                    self.setFocus(series_list)
                    # Zajistíme, že jsme v sekci seriálů
                    self.current_section = "series"
                    self.setProperty("CurrentSection", "series")
                    # Aktualizujeme fanart pro první položku
                    if series_list.getSelectedItem():
                        self._update_fanart(series_list.getSelectedItem())
                else:
                    xbmc.log(f"[plugin.video.milionar] Warning: Series list is empty after populating seasons", level=xbmc.LOGWARNING)
            else:
                xbmc.executebuiltin("Dialog.Close(busydialognocancel)")
                xbmc.log(f"[plugin.video.milionar] Failed to load seasons data", level=xbmc.LOGERROR)
                xbmcgui.Dialog().ok("Chyba", "Nepodařilo se načíst sezóny seriálu.")
                # Vrátíme se na seznam seriálů
                self._go_back_to_series()
                
        except Exception as e:
            xbmc.executebuiltin("Dialog.Close(busydialognocancel)")
            xbmc.log(f"[plugin.video.milionar] Error loading series seasons: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Nepodařilo se načíst sezóny: {e}")
            # Vrátíme se na seznam seriálů
            self._go_back_to_series()
    
    def _load_season_episodes(self, season_number):
        """Načte epizody sezóny a zobrazí je v sekci seriálů."""
        try:
            # Nastavíme stav navigace
            self.current_season_number = season_number
            self.navigation_level = "episodes"
            self.setProperty("NavigationLevel", self.navigation_level)
            
            # Zobrazíme indikátor načítání
            xbmc.executebuiltin("ActivateWindow(busydialognocancel)")
            
            # Cache endpoint pro epizody sezóny
            endpoint = f'/tv/{self.current_series_id}/season/{season_number}'
            params = {'language': 'cs-CZ'}
            
            def fetch_episodes_data():
                """Funkce pro stažení dat epizod z TMDb"""
                try:
                    api_key = get_tmdb_api_key(self.addon)
                    url = f'https://api.themoviedb.org/3/tv/{self.current_series_id}/season/{season_number}?api_key={api_key}&language=cs-CZ'
                    response = requests.get(url, timeout=10)
                    if response.status_code == 200:
                        return response.json()
                    return None
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Error fetching episodes data: {e}", level=xbmc.LOGERROR)
                    return None
            
            # Použití cache s 7 dny platnosti
            data = tmdb_cache.get_or_fetch(endpoint, params, fetch_episodes_data, expire_seconds=7*24*3600)
            
            if data and 'episodes' in data:
                self.episodes = data['episodes']
                # Skryjeme indikátor načítání
                xbmc.executebuiltin("Dialog.Close(busydialognocancel)")
                
                # Zobrazíme epizody v sekci seriálů
                self._populate_episodes_list()
                
                # Nastavíme focus na první epizodu
                series_list = self.getControl(200)
                if series_list and series_list.size() > 0:
                    series_list.selectItem(0)
                    self.setFocus(series_list)
                    if series_list.getSelectedItem():
                        self._update_fanart(series_list.getSelectedItem())
            else:
                xbmc.executebuiltin("Dialog.Close(busydialognocancel)")
                xbmcgui.Dialog().ok("Chyba", "Nepodařilo se načíst epizody sezóny.")
                
        except Exception as e:
            xbmc.executebuiltin("Dialog.Close(busydialognocancel)")
            xbmc.log(f"[plugin.video.milionar] Error loading season episodes: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Nepodařilo se načíst epizody: {e}")
    
    def _populate_seasons_list(self):
        """Naplní seznam sezón."""
        try:
            series_list = self.getControl(200)
            if not series_list:
                xbmc.log(f"[plugin.video.milionar] Error: Series list control not found", level=xbmc.LOGERROR)
                return
                
            xbmc.log(f"[plugin.video.milionar] Populating seasons list with {len(self.seasons)} seasons", level=xbmc.LOGINFO)
            series_list.reset()
            
            # Přidáme tlačítko "Zpět"
            back_item = xbmcgui.ListItem(label="Zpět")
            back_item.setArt({'poster': 'DefaultFolderBack.png'})
            back_item.setProperty("item_type", "back_to_series")
            series_list.addItem(back_item)
            
            # Bezpečné načtení nastavení 'show_specials'
            show_specials_setting = self.addon.getSetting('show_specials')  # Načíst jako string
            show_specials = show_specials_setting == 'true'
            
            # Přidáme sezóny
            for season in self.seasons:
                season_number = season.get('season_number', 0)
                
                # Přeskočíme "Specials" (season_number 0), pokud nemají epizody
                # nebo pokud to explicitně nenastavíme v addonu
                if season_number == 0 and not show_specials:
                    continue
                
                name = season.get('name', f'Sezóna {season_number}')
                air_date = season.get('air_date', '')
                year = air_date[:4] if air_date else ''
                poster_path = season.get('poster_path', '')
                
                poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
                
                item = xbmcgui.ListItem(label=name)
                item.setLabel2(f"{year} • Sezóna {season_number}")
                item.setArt({'thumb': poster, 'poster': poster})
                item.setProperty("item_type", "season")
                item.setProperty("season_number", str(season_number))
                item.setProperty("title", name)
                item.setProperty("fanart", poster)
                
                series_list.addItem(item)
                
            xbmc.log(f"[plugin.video.milionar] Successfully populated seasons list with {series_list.size()} items (show_specials: {show_specials})", level=xbmc.LOGINFO)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error populating seasons list: {e}", level=xbmc.LOGERROR)
            # Vrátíme se na seznam seriálů
            self._go_back_to_series()
    
    def _populate_episodes_list(self):
        """Naplní seznam epizod."""
        series_list = self.getControl(200)
        if not series_list:
            return
            
        series_list.reset()
        
        # Přidáme tlačítko "Zpět"
        back_item = xbmcgui.ListItem(label="Zpět")
        back_item.setArt({'poster': 'DefaultFolderBack.png'})
        back_item.setProperty("item_type", "back_to_seasons")
        series_list.addItem(back_item)
        
        # Přidáme epizody
        for episode in self.episodes:
            episode_number = episode.get('episode_number', 0)
            name = episode.get('name', f'Epizoda {episode_number}')
            air_date = episode.get('air_date', '')
            year = air_date[:4] if air_date else ''
            rating = episode.get('vote_average', 0)
            poster_path = episode.get('still_path', '')
            
            poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
            
            item = xbmcgui.ListItem(label=name)
            item.setLabel2(f"{year} • Epizoda {episode_number} • ★ {rating:.1f}")
            item.setArt({'thumb': poster, 'poster': poster})
            item.setProperty("item_type", "episode")
            item.setProperty("episode_number", str(episode_number))
            item.setProperty("season_number", str(self.current_season_number))
            item.setProperty("title", name)
            item.setProperty("fanart", poster)
            
            series_list.addItem(item)
    
    def _go_back_to_series(self):
        """Vrátí se na seznam seriálů."""
        try:
            xbmc.log(f"[plugin.video.milionar] Going back to series list", level=xbmc.LOGINFO)
            
            self.navigation_level = "series"
            self.setProperty("NavigationLevel", self.navigation_level)
            self.current_series_id = None
            self.current_series_title = None
            self.seasons = []
            self.current_season_number = None
            self.episodes = []
            
            # Obnovíme původní seznam seriálů
            self._populate_series_list()
            
            # Nastavíme focus na první položku
            series_list = self.getControl(200)
            if series_list and series_list.size() > 0:
                series_list.selectItem(0)
                self.setFocus(series_list)
                if series_list.getSelectedItem():
                    self._update_fanart(series_list.getSelectedItem())
                    
            xbmc.log(f"[plugin.video.milionar] Successfully returned to series list", level=xbmc.LOGINFO)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error going back to series list: {e}", level=xbmc.LOGERROR)
    
    def _go_back_to_seasons(self):
        """Vrátí se na seznam sezón."""
        self.navigation_level = "seasons"
        self.setProperty("NavigationLevel", self.navigation_level)
        self.current_season_number = None
        self.episodes = []
        
        # Zobrazíme sezóny
        self._populate_seasons_list()
        
        # Nastavíme focus na první položku
        series_list = self.getControl(200)
        if series_list and series_list.size() > 0:
            series_list.selectItem(0)
            self.setFocus(series_list)
            if series_list.getSelectedItem():
                self._update_fanart(series_list.getSelectedItem())
    
 