---
trigger: always_on
description: Python syntax and code style rules for Kodi addon development
---

# Python Syntax for Kodi

## Critical Rules
- **Python 3 only** -- Kodi 19+ uses Python 3
- **No pip installs** -- only standard library + Kodi-provided modules
- **UTF-8 encoding** -- all source files

## Available Kodi Modules
```python
import xbmc          # Core: logging, conditions, player
import xbmcgui       # GUI: dialogs, windows, controls
import xbmcplugin    # Plugin: directory listings, resolved URLs
import xbmcaddon     # Addon: settings, info, paths
import xbmcvfs       # VFS: file operations, translatePath
```

## File Structure Template
```python
# Standard library imports
import os
import json

# Kodi imports
import xbmc
import xbmcgui
import xbmcaddon

# Local imports
from .utils import helper_function


def main_function():
    """Docstring for complex functions."""
    pass
```

## Logging
```python
import xbmc

# Use xbmc.log() -- NEVER print()
xbmc.log("[AddonName] message", xbmc.LOGINFO)
xbmc.log("[AddonName] error detail", xbmc.LOGERROR)

# Debug logs -- prefix [DEBUG] for easy cleanup
xbmc.log("[DEBUG] variable: {}".format(var), xbmc.LOGINFO)
```

## Path Handling
```python
import xbmcvfs
import xbmcaddon

# WRONG -- special:// paths don't work with os.path
profile = xbmcaddon.Addon().getAddonInfo('profile')
os.path.exists(profile)  # Always False!

# RIGHT -- translate first
profile = xbmcvfs.translatePath(
    xbmcaddon.Addon().getAddonInfo('profile')
)
os.path.exists(profile)  # Works correctly
```

## Error Handling
```python
# ALWAYS wrap external calls
try:
    response = fetch_data(url)
except Exception as e:
    xbmc.log("[AddonName] fetch failed: {}".format(e), xbmc.LOGERROR)
    return None
```

## String Formatting
```python
# Prefer .format() or f-strings (Python 3)
msg = "Found {} results".format(count)
msg = f"Found {count} results"

# NEVER use % formatting for new code
```

## Settings Access
```python
addon = xbmcaddon.Addon()
value = addon.getSetting('setting_id')       # Always returns string
enabled = addon.getSettingBool('setting_id')  # Returns bool
number = addon.getSettingInt('setting_id')    # Returns int
```

## Checklist
- Kodi modules imported correctly (xbmc, xbmcgui, xbmcvfs, xbmcaddon, xbmcplugin)
- Paths translated with `xbmcvfs.translatePath()`
- Error handling on all external calls
- Logging via `xbmc.log()`, never `print()`
- No pip dependencies
- UTF-8 source files
- Clean import order (stdlib, kodi, local)