# wcs/ai/ChannelHistory.py
"""
Správa historie sledování pro jednotlivé MyTV/MyCinema kanály.
Každý kanál má vlastní JSON soubor s progressem seriálů/filmů.

Supported channel_type values:
- 'tv' (default): MyTV channels with series (episodes progress)
- 'cinema': MyCinema channels with movies (watched status)
"""

import os
import json
import xbmc
import xbmcvfs
import xbmcaddon


def _get_history_dir(channel_type='tv'):
    """
    Vrátí cestu k adresáři s historiemi kanálů.
    
    Args:
        channel_type: 'tv' pro MyTV, 'cinema' pro MyCinema
    """
    profile = xbmcvfs.translatePath(xbmcaddon.Addon().getAddonInfo('profile'))
    if channel_type == 'cinema':
        history_dir = os.path.join(profile, 'mycinema_history')
    else:
        history_dir = os.path.join(profile, 'mytv_history')
    if not os.path.exists(history_dir):
        xbmcvfs.mkdirs(history_dir)
    return history_dir


def _get_history_file(channel_id, channel_type='tv'):
    """Vrátí cestu k JSON souboru historie pro daný kanál."""
    return os.path.join(_get_history_dir(channel_type), f'{channel_id}.json')


def load_channel_history(channel_id, channel_type='tv'):
    """
    Načte historii sledování pro daný kanál.
    
    Args:
        channel_id: ID kanálu
        channel_type: 'tv' pro MyTV, 'cinema' pro MyCinema
    
    Returns:
        dict: Pro TV: {series_id: {"season": int, "episode": int}}
              Pro Cinema: {movie_id: {"watched": bool}}
    """
    file_path = _get_history_file(channel_id, channel_type)
    if not os.path.exists(file_path):
        return {}
    
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            return json.load(f)
    except (IOError, json.JSONDecodeError) as e:
        log_prefix = '[MyCinema]' if channel_type == 'cinema' else '[MyTV]'
        xbmc.log(f'{log_prefix} Error loading channel history {channel_id}: {e}', xbmc.LOGWARNING)
        return {}


def save_channel_history(channel_id, history, channel_type='tv'):
    """
    Uloží historii sledování pro daný kanál.
    
    Args:
        channel_id: ID kanálu
        history: dict s historií
        channel_type: 'tv' pro MyTV, 'cinema' pro MyCinema
    """
    file_path = _get_history_file(channel_id, channel_type)
    try:
        with open(file_path, 'w', encoding='utf-8') as f:
            json.dump(history, f, ensure_ascii=False, indent=2)
    except IOError as e:
        log_prefix = '[MyCinema]' if channel_type == 'cinema' else '[MyTV]'
        xbmc.log(f'{log_prefix} Error saving channel history {channel_id}: {e}', xbmc.LOGERROR)


def update_series_progress(channel_id, series_id, season, episode, channel_type='tv'):
    """
    Aktualizuje progress seriálu v kanálu (pouze pro MyTV).
    
    Args:
        channel_id: ID kanálu
        series_id: TMDb ID seriálu (bude převedeno na string)
        season: Číslo sezóny
        episode: Číslo epizody
        channel_type: 'tv' pro MyTV (default)
    """
    history = load_channel_history(channel_id, channel_type)
    history[str(series_id)] = {
        'season': int(season),
        'episode': int(episode)
    }
    save_channel_history(channel_id, history, channel_type)
    xbmc.log(f'[MyTV] Updated progress: channel={channel_id}, series={series_id}, S{season:02d}E{episode:02d}', xbmc.LOGINFO)


def mark_movie_watched(channel_id, movie_id, watched=True):
    """
    Označí film jako zhlédnutý v MyCinema kanálu.
    
    Args:
        channel_id: ID kanálu/kina
        movie_id: TMDb ID filmu
        watched: True pokud zhlédnuto
    """
    history = load_channel_history(channel_id, 'cinema')
    history[str(movie_id)] = {'watched': watched}
    save_channel_history(channel_id, history, 'cinema')
    xbmc.log(f'[MyCinema] Marked movie {movie_id} as watched in channel {channel_id}', xbmc.LOGINFO)


def is_movie_watched(channel_id, movie_id):
    """
    Zjistí, zda byl film zhlédnut v daném MyCinema kanálu.
    
    Args:
        channel_id: ID kanálu/kina
        movie_id: TMDb ID filmu
    
    Returns:
        bool: True pokud byl film zhlédnut
    """
    history = load_channel_history(channel_id, 'cinema')
    movie_data = history.get(str(movie_id), {})
    return movie_data.get('watched', False)


def get_series_progress(channel_id, series_id, channel_type='tv'):
    """
    Vrátí aktuální progress seriálu v kanálu.
    
    Args:
        channel_id: ID kanálu
        series_id: TMDb ID seriálu
        channel_type: 'tv' pro MyTV (default)
    
    Returns:
        tuple: (season, episode) nebo (1, 0) pokud neexistuje
    """
    history = load_channel_history(channel_id, channel_type)
    progress = history.get(str(series_id), {})
    return (
        progress.get('season', 1),
        progress.get('episode', 0)
    )


def delete_channel_history(channel_id, channel_type='tv'):
    """
    Smaže historii kanálu (volat při mazání kanálu).
    
    Args:
        channel_id: ID kanálu
        channel_type: 'tv' pro MyTV, 'cinema' pro MyCinema
    """
    file_path = _get_history_file(channel_id, channel_type)
    log_prefix = '[MyCinema]' if channel_type == 'cinema' else '[MyTV]'
    if os.path.exists(file_path):
        try:
            os.remove(file_path)
            xbmc.log(f'{log_prefix} Deleted channel history: {channel_id}', xbmc.LOGINFO)
            return True
        except OSError as e:
            xbmc.log(f'{log_prefix} Error deleting channel history {channel_id}: {e}', xbmc.LOGERROR)
    return False


def migrate_from_global_history(channel_id, series_ids, addon):
    """
    Migruje progress seriálů z globální historie do kanálové.
    Volat pouze jednou při prvním použití kanálu.
    
    Args:
        channel_id: ID kanálu
        series_ids: Seznam TMDb ID seriálů v kanálu
        addon: Kodi addon instance
    """
    from wcs import user_data
    
    # Načíst globální historii
    global_history = user_data.load_recently_played(addon)
    if not global_history:
        return
    
    # Načíst existující kanálovou historii
    channel_history = load_channel_history(channel_id)
    
    # Převést series_ids na stringy pro porovnání
    series_ids_str = {str(sid) for sid in series_ids}
    
    migrated_count = 0
    for item in global_history:
        item_id = str(item.get('tmdb_id', ''))
        media_type = item.get('media_type', '')
        
        # Zkontrolovat, zda je tento seriál v kanálu
        if item_id in series_ids_str and media_type in ('tv', 'episode', 'series'):
            # Mít season a episode
            if 'season' in item and 'episode' in item:
                # Přidat pouze pokud ještě neexistuje (nebo je novější)
                existing = channel_history.get(item_id, {})
                existing_s = existing.get('season', 0)
                existing_e = existing.get('episode', 0)
                new_s = int(item['season'])
                new_e = int(item['episode'])
                
                # Porovnat - novější = vyšší sezóna nebo vyšší epizoda ve stejné sezóně
                if new_s > existing_s or (new_s == existing_s and new_e > existing_e):
                    channel_history[item_id] = {
                        'season': new_s,
                        'episode': new_e
                    }
                    migrated_count += 1
    
    if migrated_count > 0:
        save_channel_history(channel_id, channel_history)
        xbmc.log(f'[MyTV] Migrated {migrated_count} series progress to channel {channel_id}', xbmc.LOGINFO)


def check_and_migrate_if_needed(channel_id, series_list, addon, channel_type='tv'):
    """
    Zkontroluje, zda kanál potřebuje migraci a provede ji.
    Volat při načítání programu kanálu.
    
    Args:
        channel_id: ID kanálu
        series_list: Seznam seriálů v kanálu (list of dicts s 'id')
        addon: Kodi addon instance
        channel_type: 'tv' pro MyTV (default) - migrace pouze pro TV
    """
    # Migrace pouze pro TV, cinema nemá globální historii
    if channel_type != 'tv':
        return
    
    history_file = _get_history_file(channel_id, channel_type)
    
    # Migrace pouze pokud soubor neexistuje (nový kanál nebo první spuštění)
    if not os.path.exists(history_file):
        series_ids = [s.get('id') for s in series_list if s.get('id')]
        if series_ids:
            migrate_from_global_history(channel_id, series_ids, addon)
