---
name: kodi-image-generator
description: Generating consistent images/icons for Kodi UI. Use when user wants to generate graphics for addon (icons, cards, banners). DO NOT use for programmatic mask/texture creation in Python code.
trigger: generate_image, create_icon, card_graphics, button_texture, banner_design, kodi_artwork, image, icon, background, fanart, banner, graphics
---

# Kodi Image Generator Skill

Generating graphics for Kodi addon using AI.

## When to Use
- User wants to **generate images/icons** for Kodi UI (cards, buttons, banners)
- Addon icon (`icon.png`), fanart (`fanart.jpg`), banners
- Output goes to `resources/media/` or XML textures

## When NOT to Use
- Creating masks/textures using PIL in Python code -> use `kodi-texture-generator`
- Editing existing images without generating new ones

---

## Standard Resolutions

| Type | Dimensions | Format |
|------|-----------|--------|
| Icon | 256x256 or 512x512 | PNG |
| Fanart | 1920x1080 | JPG |
| Banner | 758x140 | PNG |
| Poster | 1000x1500 | JPG |

## Paths in Addon
```
resources/
├── icon.png      # Main icon 256x256
├── fanart.jpg    # Background 1920x1080
└── media/        # Additional graphics
    ├── banner.png
    └── ...
```

## Rules
- No WebP (limited Kodi support)
- Icons must be readable at small sizes
- Fanart must not contain text (will be covered by UI)
- Transparency only PNG

---

## Workflow

### 1. Target Analysis
1. Determine **target dimensions** from XML (`<width>`, `<height>`)
2. Calculate **target aspect ratio** (e.g. 340x200 = 1.7)
3. Generator returns **1024x1024** (1:1) - always!

### 2. Style Discussion
**NEVER generate in bulk!**

1. **Ask** about desired visual style (Netflix, Apple TV, Flat, Neon...)
2. **Wait for answer** and clarify details
3. Only then proceed

### 3. Master Image
1. Generate **ONLY ONE** image (master)
2. **Show to user** and wait for approval
3. If not approved -> adjust prompt and repeat
4. **Prompt tips:**
   - Content must be **centered** (top/bottom will be cropped)
   - Style: flat 2D, no 3D effects
   - Background: dark, uniform

### 4. Generating Additional Images
After master image approval:

1. Generate **ONE AT A TIME** (not in parallel!)
2. **MANDATORY pass master image as `ImagePaths` parameter:**
   ```
   generate_image(
       ImagePaths=["/path/to/master_image.png"],
       Prompt="SAME STYLE AS INPUT IMAGE, ..."
   )
   ```
3. In prompt **ALWAYS** include: `"SAME STYLE AS INPUT IMAGE"`
4. Change only the subject, style transfers from master reference
5. Wait for approval after each image

### 5. Crop to Target Format
**CRITICAL:** Generator returns 1:1 (1024x1024). You must crop, NOT resize!

**Crop height calculation:**
```
target_height = 1024 * (target_height / target_width)
```
Example for 340x200:
```
1024 * (200 / 340) = 602px
```

**Crop command (macOS):**
```bash
sips -c [height] [width] input.png -o output.png
```
Example:
```bash
sips -c 602 1024 master.png -o card.png
```

**What it does:**
- Keeps **full width** (1024px)
- Crops **symmetrically top and bottom** to 602px
- Quality stays **100%** (no downscale)
- Kodi then scales to target dimensions (340x200)

### 6. Deployment
```bash
cp card_*.png "resources/media/"
```

XML reference:
```xml
<texturefocus>special://home/addons/[addon]/resources/media/card_name.png</texturefocus>
```

---

## Flow Summary
```
ANALYSIS -> STYLE DISCUSSION -> MASTER (1x) -> APPROVAL ->
NEXT (1 at a time with master as ref) -> CROP (sips -c) -> DEPLOYMENT
```

## Common Mistakes
| Mistake | Correct |
|---------|---------|
| `sips -z` (resize) | `sips -c` (crop) |
| Generate all at once | One at a time, with master as ref |
| Prompt without "centered" | Always centered composition |
| Resize to small px | Crop to format, quality preserved |
