# wcs/download_dialog.py
import xbmc
import xbmcgui
import xbmcaddon
from wcs.ui import UI as ui

# Import pro format_quality_with_color funkci
def format_quality_with_color(quality):
    """Formátuje kvalitu s barevným kódováním pro Kodi."""
    if not quality:
        return ""
    
    # Barevné schéma pro kvality
    color_map = {
        '4K': 'FFFF4444',      # Červená pro 4K
        'REMUX': 'FFFF00FF',   # Magenta pro REMUX
        '1080p': 'FF44AAFF',   # Modrá pro 1080p
        '720p': 'FFFF9944',    # Oranžová pro 720p
        '480p': 'FFDDDD44',    # Žlutá pro 480p
        '360p': 'FFAA8844',    # Hnědá pro 360p
        'SD': 'FF999999'       # Šedá pro SD
    }
    
    # Najdi barvu pro kvalitu
    color = color_map.get(quality, 'FFFFFFFF')  # Bílá jako fallback
    
    # Vrať kvalitu s tečkou a barevným formátováním
    return f"• [COLOR {color}]{quality}[/COLOR]"

class DownloadDialog(xbmcgui.WindowXMLDialog):
    """
    Dialogové okno pro výběr souboru ke stažení.
    Používá search/DialogWcsDownload.xml.
    """
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        super().__init__(xml_filename, addon_path, "Default", "1080i")
        
        # Uložíme si naše kwargs
        self.addon = xbmcaddon.Addon()
        
        # Data z search dialogu
        self.search_results = kwargs.get('search_results', [])
        self.movie_title = kwargs.get('movie_title', '')
        self.movie_year = kwargs.get('movie_year', '')
        self.poster_url = kwargs.get('poster_url', '')
        self.rating = kwargs.get('rating', '')
        self.genre = kwargs.get('genre', '')
        self.director = kwargs.get('director', '')
        self.runtime = kwargs.get('runtime', '')
        self.plot = kwargs.get('plot', '')
        self.fanart_url = kwargs.get('fanart_url', '')
        
        # Metadata pro epizody
        self.is_episode = kwargs.get('is_episode', False)
        self.series_name = kwargs.get('series_name', '')
        self.season_number = kwargs.get('season_number', None)
        self.episode_number = kwargs.get('episode_number', None)
        self.episode_title = kwargs.get('episode_title', '')
        
        # Vybraný výsledek
        self.selected_result = None
        
    def onInit(self):
        """
        Volá se při inicializaci okna. Zde nastavíme metadata a naplníme seznam.
        """
        xbmc.log(f"[plugin.video.milionar] Initializing DownloadDialog for {len(self.search_results)} files", level=xbmc.LOGINFO)
        
        if not self.search_results:
            xbmc.log("[plugin.video.milionar] Error: No search results for DownloadDialog.", level=xbmc.LOGERROR)
            ui.popinfo(self.addon.getLocalizedString(40005), "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)
            self.close()
            return

        # Nastavíme texty v dialogu
        dialog_title = f"Výběr souboru ke stažení"
        self.setProperty('WCS.Download.Title', dialog_title)

        info_text = f"Vyberte soubor ke stažení:"
        self.setProperty('WCS.Download.Info', info_text)

        # Nastavíme poster do dialogu
        if self.poster_url:
            self.setProperty('WCS.Download.Poster', self.poster_url)
        else:
            self.setProperty('WCS.Download.Poster', '')

        # Nastavíme název filmu/seriálu
        if self.movie_title:
            self.setProperty('WCS.Download.MovieTitle', self.movie_title)
        else:
            self.setProperty('WCS.Download.MovieTitle', '')

        # Nastavíme fanart
        if self.fanart_url:
            self.setProperty('WCS.Download.Fanart', self.fanart_url)
        else:
            self.setProperty('WCS.Download.Fanart', '')

        # Nastavíme metadata filmu
        if self.rating:
            try:
                rating_val = float(self.rating)
                rating_str = f"{rating_val:.1f}/10"
            except Exception:
                rating_str = str(self.rating)
            self.setProperty('WCS.Download.Rating', rating_str)
        else:
            self.setProperty('WCS.Download.Rating', '')

        if self.genre:
            self.setProperty('WCS.Download.Genre', self.genre)
        else:
            self.setProperty('WCS.Download.Genre', '')

        if self.director:
            self.setProperty('WCS.Download.Director', self.director)
        else:
            self.setProperty('WCS.Download.Director', '')

        if self.runtime:
            self.setProperty('WCS.Download.Runtime', self.runtime)
        else:
            self.setProperty('WCS.Download.Runtime', '')

        if self.movie_year:
            self.setProperty('WCS.Download.Year', str(self.movie_year))
        else:
            self.setProperty('WCS.Download.Year', '')

        # Naplníme seznam výsledků
        self._populate_results_list()
        
        xbmc.log("[plugin.video.milionar] DownloadDialog initialized successfully", level=xbmc.LOGINFO)

    def _populate_results_list(self):
        """Naplní seznam výsledků vyhledávání."""
        try:
            results_list = self.getControl(50)
            if not results_list:
                xbmc.log("[plugin.video.milionar] Error: Could not get results list control", level=xbmc.LOGERROR)
                return

            # Vyčistíme seznam
            results_list.reset()

            # Importujeme potřebné funkce
            from wcs.utils import (
                extract_clean_title, 
                extract_quality, 
                extract_language, 
                extract_format, 
                format_size, 
                calculate_rating
            )

            # Naplníme seznam výsledky
            for result in self.search_results:
                filename = result.get('name', '')
                
                # Vytvoření ListItem - stejný styl jako v search dialogu
                list_item = xbmcgui.ListItem(label=f"[B][COLOR FF00D9FF]webshare.cz[/COLOR][/B] [COLOR FFFFFFFF]{filename}[/COLOR]")
                
                # Extrahujeme vlastnosti pomocí stejných funkcí jako v search dialogu
                clean_title = extract_clean_title(filename)
                quality = extract_quality(filename)
                quality_colored = format_quality_with_color(quality)  # Barevná kvalita
                language = extract_language(filename)
                file_format = extract_format(filename)
                size_formatted = format_size(result.get('size', 0))
                user_rating = calculate_rating(result)
                
                # Nastavíme properties pro zobrazení v seznamu - STEJNÉ jako v search dialogu
                list_item.setProperty('ident', result.get('ident', ''))
                list_item.setProperty('filename', filename)
                list_item.setProperty('quality', quality_colored)  # Používáme barevnou verzi
                list_item.setProperty('language', language)
                list_item.setProperty('format', file_format)
                list_item.setProperty('size_formatted', size_formatted)
                list_item.setProperty('user_rating', user_rating)
                
                # Původní Webshare metadata
                list_item.setProperty('positive_votes', str(result.get('positive_votes', '0')))
                list_item.setProperty('negative_votes', str(result.get('negative_votes', '0')))
                
                # Přidáme položku do seznamu
                results_list.addItem(list_item)



            xbmc.log(f"[plugin.video.milionar] Populated results list with {len(self.search_results)} items", level=xbmc.LOGINFO)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error populating results list: {e}", level=xbmc.LOGERROR)

    def onClick(self, controlId):
        """Obsluha kliknutí na kontrolky."""
        if controlId == 50:  # Seznam výsledků 
            results_list = self.getControl(50)
            selected_item = results_list.getSelectedItem()
            if selected_item:
                ident = selected_item.getProperty('ident')
                name = selected_item.getProperty('filename')
                self.selected_result = {'ident': ident, 'name': name}
                xbmc.log(f"[plugin.video.milionar] File selected for download: {name}", level=xbmc.LOGINFO)
                
                # IHNED spustit stahování vybraného souboru
                self._download_selected_file()
                
        elif controlId == 100:  # Tlačítko "Stáhnout nejlepší"
            self._download_best_file()
            
        elif controlId == 101:  # Tlačítko "Nastavit adresář"
            self._open_download_settings()
            
        elif controlId == 102:  # Tlačítko "Zrušit"
            xbmc.log("[plugin.video.milionar] Download cancelled by user", level=xbmc.LOGINFO)
            self.close()
            


    def _download_selected_file(self):
        """Stáhne vybraný soubor."""
        try:
            # Získat vybranou položku ze seznamu
            results_list = self.getControl(50)
            selected_item = results_list.getSelectedItem()
            
            if not selected_item:
                ui.popinfo(self.addon.getLocalizedString(40006), "Chyba", icon=xbmcgui.NOTIFICATION_WARNING)
                return
            
            # Získat Webshare ident a název souboru
            ident = selected_item.getProperty('ident')
            filename = selected_item.getProperty('filename')
            
            if not ident:
                ui.popinfo(self.addon.getLocalizedString(40007), "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)
                return
            
            xbmc.log(f"[plugin.video.milionar] Starting download for file: {filename}", level=xbmc.LOGINFO)
            
            # Zavřít všechny otevřené dialogy
            xbmc.executebuiltin("Dialog.Close(all)")
            
            # Také zavřít aktuální dialog pro jistotu
            self.close()
            
            # Spustit stahování
            from wcs.webshare import DownloadManager as download
            download.download_file(ident, self.addon)
            
            xbmc.log("[plugin.video.milionar] Download started successfully", level=xbmc.LOGINFO)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error starting download: {e}", level=xbmc.LOGERROR)
            ui.popinfo(f"Chyba při stahování: {e}", "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)

    def _download_best_file(self):
        """Stáhne nejlepší (první) soubor ze seznamu."""
        try:
            if not self.search_results:
                ui.popinfo("Žádné soubory k dispozici", "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)
                return
            
            # Vezmeme první (nejlepší) soubor
            best_file = self.search_results[0]
            ident = best_file.get('ident', '')
            filename = best_file.get('name', '')
            
            if not ident:
                ui.popinfo("Chybí identifikátor souboru", "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)
                return
            
            xbmc.log(f"[plugin.video.milionar] Starting download of best file: {filename}", level=xbmc.LOGINFO)
            
            # Zavřít všechny otevřené dialogy
            xbmc.executebuiltin("Dialog.Close(all)")
            
            # Také zavřít aktuální dialog pro jistotu
            self.close()
            
            # Spustit stahování
            from wcs.webshare import DownloadManager as download
            download.download_file(ident, self.addon)
            
            xbmc.log("[plugin.video.milionar] Best file download started successfully", level=xbmc.LOGINFO)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error starting best file download: {e}", level=xbmc.LOGERROR)
            ui.popinfo(f"Chyba při stahování nejlepšího souboru: {e}", "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)

    def _open_download_settings(self):
        """Otevře dialog pro nastavení složky pro stahování."""
        try:
            xbmc.log("[plugin.video.milionar] Opening download folder dialog", level=xbmc.LOGINFO)
            
            # Otevřít dialog pro výběr složky pro stahování
            from wcs.login.DialogDownloadFolder import show_download_folder_dialog
            show_download_folder_dialog(self.addon)
            
            # Dialog se zavře automaticky po výběru složky nebo zavření
            # Download dialog zůstává otevřený pro pokračování
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error opening download folder dialog: {e}", level=xbmc.LOGERROR)
            ui.popinfo(f"Chyba při otevírání dialogu pro výběr složky: {e}", "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)



    def onAction(self, action):
        """Obsluha akcí (např. tlačítko Zpět)."""
        if action.getId() in (xbmcgui.ACTION_PREVIOUS_MENU, xbmcgui.ACTION_NAV_BACK):
            self.close()

def show_download_dialog(search_results, **kwargs):
    """
    Zobrazí dialog pro výběr souboru ke stažení.
    
    Args:
        search_results (list): Seznam výsledků vyhledávání
        **kwargs: Další parametry (movie_title, poster_url, atd.)
        
    Returns:
        dict or None: Vybraný výsledek nebo None pokud uživatel nic nevybral
    """
    addon = xbmcaddon.Addon()
    
    if not search_results:
        xbmcgui.Dialog().notification("Chyba", addon.getLocalizedString(40005), xbmcgui.NOTIFICATION_ERROR)
        return None
        
    # Vytvoříme dialog
    dialog = DownloadDialog(
        "search/DialogWcsDownload.xml", 
        addon.getAddonInfo('path'),
        search_results=search_results,
        **kwargs
    )
    
    # Zobrazíme dialog
    dialog.doModal()
    
    # Získáme výsledek
    selected_result = dialog.selected_result
    
    # Uvolníme dialog
    del dialog
    
    return selected_result
