---
name: kodi-texture-generator
description: Programmatic generation of masks and textures using PIL/Pillow. Use for creating rounded corners, pill shapes, scrollbars and other geometric UI elements in Python code. DO NOT use for generating photos or complex graphics.
trigger: pil_mask, rounded_corners, pill_shape, scrollbar_texture, alpha_mask, generate_texture_code, python_image
---

# Kodi Texture Generator (PIL)

## When to Use
- Creating **masks** for rounded corners
- Generating **geometric shapes** (pill, rounded rect, scrollbar)
- Programmatic textures with **transparency** (alpha channel)
- Python code using **PIL/Pillow**

## When NOT to Use
- Generating photos, icons, illustrations -> use `kodi-image-generator`
- Complex graphics requiring AI generator

---

## Code Pattern

### Basic Structure
```python
#!/usr/bin/env python3
from PIL import Image, ImageDraw

# Dimensions
WIDTH, HEIGHT = 100, 100
RADIUS = 25

# Transparent background
img = Image.new('RGBA', (WIDTH, HEIGHT), (0, 0, 0, 0))
draw = ImageDraw.Draw(img)

# Draw shape (white fill for colordiffuse)
draw.rounded_rectangle(
    [(0, 0), (WIDTH - 1, HEIGHT - 1)],
    radius=RADIUS,
    fill=(255, 255, 255, 255)
)

# Save
img.save('output.png', 'PNG')
```

### Antialiasing (for smooth edges)
```python
SCALE = 4  # Upscale factor
img = Image.new('RGBA', (WIDTH * SCALE, HEIGHT * SCALE), (0, 0, 0, 0))
# ... draw at high resolution ...
img = img.resize((WIDTH, HEIGHT), Image.LANCZOS)  # Downscale
```

---

## Texture Types

| Type | Description | Radius |
|------|-------------|--------|
| Rounded rect | Rounded rectangle | `radius = 25` (or as needed) |
| Pill | Fully rounded ends | `radius = HEIGHT // 2` |
| Circle | Circle | `radius = WIDTH // 2` |

## Rules

1. **Always white fill** `(255, 255, 255, 255)` - color is set by Kodi via `colordiffuse`
2. **Transparent background** `(0, 0, 0, 0)`
3. **PNG format** - preserves alpha channel
4. **Script location** - `resources/media/generate_*.py`
5. **Output** - `resources/media/*.png`

## Common Use Cases

- `btn_rounded.png` - button with rounded corners
- `btn_pill.png` - pill-shaped button
- `scrollbar_rounded.png` - scrollbar with rounded ends
- `card_mask.png` - mask for cropping posters
- `still_mask_rounded.png` - mask for images
