# Stream Cinema Native macOS: Current State and Handover

This document is the continuation baseline. It captures architecture, behavior, and operational details so future work can continue without re-reading the full codebase.

## Goal and parity scope

Target parity with Kodi plugin for:

1. menu browsing
2. `kra.sk` login (and later additional providers)
3. stream selection
4. stream playback

## Current implementation status

Implemented:

1. native macOS app shell (`sc-desktop`) with login, catalog navigation, stream list, playback panel
2. backend clients in Swift (`StreamCinemaCore`) for kra + stream-cinema API flow
3. keychain persistence for credentials and tokens
4. token/session reuse across app restarts
5. automatic auth refresh on `401 authorization required` and retry
6. dual playback backend:
   - primary `AVPlayer`
   - VLC fallback (`VLCKitSPM`) with automatic and manual failover
7. CLI tooling (`sc-cli`) for menu/play/inspect and auth-parity checks

Not implemented yet:

1. full 1:1 stream autoselect scoring parity from Kodi
2. full settings UI parity with Kodi settings.xml
3. provider abstraction beyond kra
4. formal automated tests

## Reverse-engineered API flow

### 1) kra login

- `POST https://api.kra.sk/api/user/login`
- body:

```json
{
  "data": {
    "username": "...",
    "password": "..."
  }
}
```

- response includes `session_id`

### 2) stream-cinema auth token

- `POST https://stream-cinema.online/kodi/auth/token?krt=<kra_session_id>`
- request includes default query params and headers (`X-Uuid`, `User-Agent`)
- response includes `token` used as `X-AUTH-TOKEN`

### 3) catalog/menu

- base: `https://stream-cinema.online/kodi`
- common params: `ver`, `uid`, `skin`, `lang`, `HDR`, `DV`, `old`
- menu payload in `menu`, stream payload in `strms`

### 4) play resolution

1. request `/Play/...` => receive stream candidates in `strms`
2. pick candidate, if provider is `kraska`, call stream provider path (`/ws2/...`) to get `ident`
3. call kra file download endpoint with `ident`
4. final direct URL returned by kra is used by player engine

## Auth/token lifecycle (important behavior)

Kodi plugin behavior reuses stored tokens and refreshes when needed. Native app now follows the same model.

### sc-desktop

1. On first login, app stores in Keychain:
   - username
   - password
   - kra session token
   - stream-cinema auth token
2. On launch, app restores persisted state and tries menu request immediately.
3. If request fails due to auth (`401` / `authorization required`), app re-authenticates, persists refreshed tokens, retries request once.

### sc-cli

1. Uses env credentials if provided.
2. If env credentials are missing, uses persisted Keychain credentials.
3. Reuses persisted tokens first.
4. If reused token/session fails, re-authenticates once, persists new tokens, retries command once.

## Playback behavior

### Engine strategy

1. default playback attempt via `AVPlayer`
2. if `AVPlayerItem.status == .failed`, fallback to VLC automatically
3. user can also force fallback with `Play with VLC`

### Why fallback is required

Stream URLs are often MKV files, frequently with codecs/container combinations and HTTP response styles (`application/octet-stream`, attachment disposition) that may fail in `AVPlayer` even when URL is valid.

## Runtime launch notes

Because VLC is linked as framework, launching the raw binary directly may fail with dyld path issues.

Use one of:

1. `swift run sc-desktop`
2. `./run-sc-desktop` (preferred for direct launch from built artifacts)

`run-sc-desktop` sets `DYLD_FRAMEWORK_PATH` for `VLCKit.framework`.

## CLI runbook

### Basic commands

```bash
cd StreamCinemaNative
swift build
./.build/debug/sc-cli menu /
./.build/debug/sc-cli play /Play/<movie-id>
./.build/debug/sc-cli inspect /Play/<movie-id>
```

### Useful with explicit creds

```bash
SC_USERNAME="<user>" SC_PASSWORD="<pass>" ./.build/debug/sc-cli menu /
```

### Useful with macOS Keychain

```bash
SC_USERNAME="JaLeKo" \
SC_PASSWORD="$(security find-generic-password -a "JaLeKo" -s "streamcinema-kra" -w)" \
./.build/debug/sc-cli menu /
```

## Key files map

### Core

1. `StreamCinemaNative/Sources/StreamCinemaCore/Clients/KraskaClient.swift`
2. `StreamCinemaNative/Sources/StreamCinemaCore/Clients/StreamCinemaClient.swift`
3. `StreamCinemaNative/Sources/StreamCinemaCore/Services/PlaybackResolver.swift`
4. `StreamCinemaNative/Sources/StreamCinemaCore/Persistence/KeychainStore.swift`
5. `StreamCinemaNative/Sources/StreamCinemaCore/Persistence/AuthStateStore.swift`
6. `StreamCinemaNative/Sources/StreamCinemaCore/Models/StreamCinemaModels.swift`

### Desktop app

1. `StreamCinemaNative/Sources/StreamCinemaDesktop/main.swift`

### CLI

1. `StreamCinemaNative/Sources/StreamCinemaCLI/main.swift`

### Package/deps

1. `StreamCinemaNative/Package.swift`
2. `StreamCinemaNative/run-sc-desktop`

## Known issues and constraints

1. first build with VLC dependency downloads large artifact (~780MB)
2. some streams still may fail in both engines depending on source/network/provider-side issues
3. advanced Kodi-specific features are intentionally not ported yet (Trakt services, background tasks, full filter UX parity)

## Next recommended feature work

1. complete stream autoselect/scoring parity with Kodi logic
2. add settings screen for language/quality/HDR/DV preferences
3. add provider abstraction protocol for non-kra sources
4. add integration tests:
   - auth reuse
   - 401 refresh retry
   - `/FMovies/Latest` browse + `/Play/...` resolve
