# wcs/login/DialogKraskaLogin.py
"""KRA.sk login dialog - plný XML dialog jako Webshare."""

import xbmcgui
import xbmcaddon
import xbmc
from wcs.ui.dialogs.guide.DialogKraskaGuide import KraskaGuideDialog


class KraskaLoginDialog(xbmcgui.WindowXMLDialog):
    """Dialog pro přihlášení ke KRA.sk s možností zadání údajů nebo zobrazení návodu."""
    
    def __init__(self, *args, **kwargs):
        self.on_success_callback = kwargs.pop('on_success_callback', None)
        super().__init__(*args, **kwargs)
        self.addon = xbmcaddon.Addon()
        self.credentials_entered = False
        self.dialog_closed = False
        
    def onInit(self):
        """Inicializace dialogu."""
        xbmc.log("[plugin.video.milionar] KraskaLoginDialog: Dialog initialized", level=xbmc.LOGINFO)
        
    def onClick(self, controlId):
        """Obsluha kliknutí na tlačítka."""
        if controlId == 7001:  # Zadat údaje
            xbmc.log("[plugin.video.milionar] KraskaLoginDialog: Enter credentials clicked", level=xbmc.LOGINFO)
            self._handle_enter_credentials()
            
        elif controlId == 7002:  # Návod
            xbmc.log("[plugin.video.milionar] KraskaLoginDialog: Guide clicked", level=xbmc.LOGINFO)
            self._handle_show_guide()
            
        elif controlId == 7003:  # Zavřít
            xbmc.log("[plugin.video.milionar] KraskaLoginDialog: Close clicked", level=xbmc.LOGINFO)
            self.dialog_closed = True
            self.close()
            
    def _handle_enter_credentials(self):
        """Zpracuje zadávání přihlašovacích údajů a ihned ověří jejich platnost."""
        from wcs.kraska import KraskaClient as kraska_api
        
        # Krok 1: Zadání uživatelského jména
        keyboard = xbmc.Keyboard('', self.addon.getLocalizedString(33021))  # Zadejte uživatelské jméno pro KRA.sk
        keyboard.doModal()
        if not keyboard.isConfirmed():
            return  # Uživatel zrušil
        username = keyboard.getText().strip()
        if not username:
            xbmcgui.Dialog().ok("Chyba zadání", "Uživatelské jméno nemůže být prázdné.")
            return
            
        # Krok 2: Zadání hesla
        keyboard = xbmc.Keyboard('', self.addon.getLocalizedString(33022), True)  # True = skrýt znaky
        keyboard.doModal()
        if not keyboard.isConfirmed():
            return  # Uživatel zrušil
        password = keyboard.getText().strip()
        if not password:
            xbmcgui.Dialog().ok("Chyba zadání", "Heslo nemůže být prázdné.")
            return
            
        # Ověření platnosti údajů přes KRA.sk API
        def silent_popinfo(*args, **kwargs):
            pass  # Ignorujeme, budeme řešit sami
        
        token = kraska_api.login(username, password, self.addon, silent_popinfo, open_settings_on_fail=False)
        
        if token:
            xbmc.log(f"[plugin.video.milionar] KraskaLoginDialog: Login successful", level=xbmc.LOGINFO)
            
            # Aktualizovat subscription info
            try:
                kraska_api.refresh_subscription_info(self.addon, token_override=token, show_notification=True)
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] KraskaLoginDialog: Error refreshing subscription: {e}", level=xbmc.LOGWARNING)
            
            # Získat SC auth token pro vyhledávání
            try:
                from wcs.streamcinema import SCClient
                sc_token = SCClient.get_auth_token(self.addon, token, force_refresh=True)
                if sc_token:
                    xbmc.log("[plugin.video.milionar] KraskaLoginDialog: SC auth token obtained", level=xbmc.LOGINFO)
                else:
                    xbmc.log("[plugin.video.milionar] KraskaLoginDialog: Failed to get SC auth token", level=xbmc.LOGWARNING)
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] KraskaLoginDialog: Error getting SC token: {e}", level=xbmc.LOGWARNING)
            
            self.credentials_entered = True
            if self.on_success_callback:
                xbmc.log("[plugin.video.milionar] KraskaLoginDialog: Calling on_success_callback", level=xbmc.LOGINFO)
                self.on_success_callback()
            self.close()
        else:
            # Údaje jsou neplatné, nabídneme zadání znovu nebo zavření
            confirm = xbmcgui.Dialog().yesno(
                self.addon.getLocalizedString(33024),  # Neplatné přihlašovací údaje ke KRA.sk
                "Zadané údaje ke KRA.sk jsou neplatné. Zkontrolujte prosím, že jste je zadali správně.",
                yeslabel="Zadat znovu",
                nolabel="Zavřít"
            )
            if confirm:
                self._handle_enter_credentials()
                return
            else:
                return
            
    def _handle_show_guide(self):
        """Zobrazí návod pro registraci na KRA.sk."""
        try:
            guide_dialog = KraskaGuideDialog(
                'guide/DialogKraskaGuide.xml',
                self.addon.getAddonInfo('path')
            )
            guide_dialog.doModal()
            del guide_dialog
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] KraskaLoginDialog: Error showing guide: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Došlo k chybě při zobrazení návodu: {e}")
            
    def get_result(self):
        """Vrátí výsledek dialogu."""
        return {
            'credentials_entered': self.credentials_entered,
            'dialog_closed': self.dialog_closed
        }


def show_kraska_login_dialog(addon, on_success_callback=None):
    """Zobrazí dialog pro přihlášení ke KRA.sk. Pokud je předán callback, zavolá jej po úspěšném přihlášení."""
    try:
        dialog = KraskaLoginDialog('login/DialogKraskaLogin.xml', addon.getAddonInfo('path'), on_success_callback=on_success_callback)
        dialog.doModal()
        return dialog.get_result()
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] show_kraska_login_dialog: Error: {e}", level=xbmc.LOGERROR)
        return {'credentials_entered': False, 'dialog_closed': True}
