# wcs/kraska/KraskaClient.py
"""
KRA.sk API client pro Milionář addon.
Poskytuje login, resolve a správu předplatného.
"""

import requests
import xbmc
import xbmcgui


BASE = 'https://api.kra.sk'
UA = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
HEADERS = {'User-Agent': UA, 'Content-Type': 'application/json'}

_session = requests.Session()
_session.headers.update(HEADERS)

# Cache pro user info
_user_info_cache = None


def _log(msg, level=xbmc.LOGINFO):
    """Logování s prefixem."""
    xbmc.log(f"[plugin.video.milionar] KraskaClient: {msg}", level=level)


def _get_data(endpoint, data=None, token=None):
    """
    Volání KRA.sk API.
    
    Args:
        endpoint: API endpoint (např. '/api/user/login')
        data: dict s daty pro request
        token: session token (volitelné)
    
    Returns:
        dict s odpovědí nebo False při chybě
    """
    if data is None:
        data = {}
    if token:
        data['session_id'] = token
    
    try:
        _log(f"API call: {endpoint}")
        response = _session.post(BASE + endpoint, json=data, timeout=15)
        return response.json()
    except Exception as e:
        _log(f"API error: {e}", level=xbmc.LOGERROR)
        return False


def login(username, password, addon, popinfo, open_settings_on_fail=True):
    """
    Přihlášení ke KRA.sk.
    
    Args:
        username: uživatelské jméno
        password: heslo
        addon: xbmcaddon instance
        popinfo: callback pro notifikace
        open_settings_on_fail: otevřít nastavení při chybě
    
    Returns:
        session_id token nebo None
    """
    if not username or not password:
        popinfo(addon.getLocalizedString(33024), addon.getAddonInfo('name'), sound=True)
        if open_settings_on_fail:
            addon.openSettings()
        return None
    
    _log(f"Login attempt for user: {username}")
    
    data = {
        'data': {
            'username': username,
            'password': password
        }
    }
    
    response = _get_data('/api/user/login', data)
    
    if response and 'session_id' in response:
        token = response['session_id']
        addon.setSetting('kraska_token', token)
        addon.setSetting('kraska_user', username)
        addon.setSetting('kraska_pass', password)
        # Verify token was saved
        saved = addon.getSetting('kraska_token')
        _log("Login successful")
        return token
    else:
        error_msg = response.get('msg', 'Unknown error') if response else 'Connection failed'
        _log(f"Login failed: {error_msg}", level=xbmc.LOGWARNING)
        popinfo(addon.getLocalizedString(33024), addon.getAddonInfo('name'), icon='error', sound=True)
        if open_settings_on_fail:
            addon.openSettings()
        return None


def revalidate(addon, popinfo):
    """
    Ověření a obnova tokenu.
    
    Returns:
        Platný token nebo None
    """
    token = addon.getSetting('kraska_token')
    
    if not token:
        # Zkusit přihlásit se uloženými údaji
        username = addon.getSetting('kraska_user')
        password = addon.getSetting('kraska_pass')
        if username and password:
            return login(username, password, addon, popinfo, open_settings_on_fail=False)
        return None
    
    # Ověřit platnost tokenu
    info = get_user_info(addon, popinfo, token_override=token)
    if info:
        return token
    
    # Token neplatný, zkusit znovu přihlásit
    _log("Token invalid, attempting re-login")
    addon.setSetting('kraska_token', '')
    username = addon.getSetting('kraska_user')
    password = addon.getSetting('kraska_pass')
    if username and password:
        return login(username, password, addon, popinfo, open_settings_on_fail=False)
    
    return None


def get_user_info(addon, popinfo, token_override=None):
    """
    Získání informací o uživateli.
    
    Returns:
        dict s user info nebo None
    """
    global _user_info_cache
    
    token = token_override or addon.getSetting('kraska_token')
    if not token:
        return None
    
    response = _get_data('/api/user/info', {}, token=token)
    
    if response and 'data' in response:
        _user_info_cache = response['data']
        return _user_info_cache
    
    return None


def get_days_left(addon, popinfo):
    """
    Získání počtu dnů do expirace předplatného.
    
    Returns:
        int počet dnů nebo 0
    """
    info = get_user_info(addon, popinfo)
    if not info:
        return 0
    
    return info.get('days_left', 0)


def resolve(ident, addon, server=None):
    """
    Resolve ident na stream URL.
    
    Args:
        ident: identifikátor souboru
        addon: xbmcaddon instance
        server: volitelný custom server
    
    Returns:
        Stream URL nebo None
    """
    from wcs.ui import UI as ui
    
    token = revalidate(addon, ui.popinfo)
    if not token:
        _log("Resolve failed: no valid token")
        return None
    
    # Kontrola předplatného
    days_left = get_days_left(addon, ui.popinfo)
    if days_left <= 0:
        _log("Resolve failed: no active subscription")
        xbmcgui.Dialog().notification(
            addon.getAddonInfo('name'),
            addon.getLocalizedString(33020),  # Žádné aktivní předplatné
            xbmcgui.NOTIFICATION_WARNING,
            5000
        )
        return None
    
    # Varování na blížící se expiraci
    try:
        warn_days = int(addon.getSetting('kraska_warn_days') or '14')
    except Exception:
        warn_days = 14
    
    if days_left <= warn_days:
        xbmcgui.Dialog().notification(
            addon.getAddonInfo('name'),
            addon.getLocalizedString(33026).format(days=days_left),
            xbmcgui.NOTIFICATION_WARNING,
            5000
        )
    
    # Resolve souboru
    _log(f"Resolving ident: {ident}")
    
    data = {
        'data': {
            'ident': ident
        }
    }
    
    response = _get_data('/api/file/download', data, token=token)
    
    if response and 'data' in response:
        file_data = response['data']
        if 'link' in file_data:
            url = file_data['link']
            _log(f"Resolved URL: {url}")
            return url
    
    error_msg = response.get('msg', 'Unknown error') if response else 'Connection failed'
    _log(f"Resolve failed: {error_msg}", level=xbmc.LOGWARNING)
    return None


def refresh_subscription_info(addon, token_override=None, show_notification=False):
    """
    Obnovení informací o předplatném v settings.
    
    Args:
        addon: xbmcaddon instance
        token_override: volitelný token
        show_notification: zobrazit notifikaci
    
    Returns:
        True pokud úspěšné
    """
    from wcs.ui import UI as ui
    
    info = get_user_info(addon, ui.popinfo, token_override=token_override)
    
    if not info:
        addon.setSetting('kraska_status', addon.getLocalizedString(33017))  # Nepřihlášen
        addon.setSetting('kraska_days_left', '')
        return False
    
    username = info.get('username', '')
    days_left = info.get('days_left', 0)
    subscribed_until = info.get('subscribed_until', '')
    
    # Nastavit status
    if username:
        addon.setSetting('kraska_status', addon.getLocalizedString(33018).format(name=username))
    else:
        addon.setSetting('kraska_status', addon.getLocalizedString(33017))
    
    # Nastavit dny do expirace
    if days_left and days_left > 0:
        addon.setSetting('kraska_days_left', addon.getLocalizedString(33019).format(days=days_left))
    elif subscribed_until:
        addon.setSetting('kraska_days_left', f"Do: {subscribed_until}")
    else:
        addon.setSetting('kraska_days_left', addon.getLocalizedString(33020))  # Žádné předplatné
    
    if show_notification:
        xbmcgui.Dialog().notification(
            addon.getAddonInfo('name'),
            addon.getLocalizedString(33023),  # Přihlášení úspěšné
            xbmcgui.NOTIFICATION_INFO,
            3000
        )
    
    _log(f"Subscription info updated: {days_left} days left")
    return True


def delete_account(addon):
    """
    Smazání uložených údajů KRA.sk.
    """
    addon.setSetting('kraska_user', '')
    addon.setSetting('kraska_pass', '')
    addon.setSetting('kraska_token', '')
    addon.setSetting('kraska_status', addon.getLocalizedString(33017))  # Nepřihlášen
    addon.setSetting('kraska_days_left', '')
    
    global _user_info_cache
    _user_info_cache = None
    
    xbmcgui.Dialog().notification(
        addon.getAddonInfo('name'),
        addon.getLocalizedString(33025),  # Účet smazán
        xbmcgui.NOTIFICATION_INFO,
        3000
    )
    _log("Account data deleted")
