"""
DialogAIRecommendationHub.py

Fullscreen dialog s rozcestníkem AI doporučení.
Nabízí karty různých typů doporučení a textový vstup.
"""

import xbmc
import xbmcgui
import xbmcaddon
import threading
import time
from wcs.ai.DialogAIChatRecommendation import AIChatRecommendationDialog, AIRecommendationPromptBuilder
from wcs import user_data


class AIRecommendationHubDialog(AIChatRecommendationDialog):
    """
    Fullscreen dialog s kartami možností AI doporučení.
    Po výběru karty nebo zadání textu spustí AI inference.
    """
    
    # Definice karet doporučení
    RECOMMENDATION_CARDS = [
        {'id': 'movies', 'label': 'Doporuč filmy', 'icon': 'ai_card_movies.png', 
         'description': 'Nechej si doporučit zajímavé filmy', 'context_type': 'my_movies', 'media_type': 'movie'},
        {'id': 'series', 'label': 'Doporuč seriály', 'icon': 'ai_card_series.png',
         'description': 'Nechej si doporučit seriály k sledování', 'context_type': 'my_series', 'media_type': 'series'},
        {'id': 'mix', 'label': 'Doporuč mix', 'icon': 'ai_card_mix.png',
         'description': 'Smíšená doporučení filmů i seriálů', 'context_type': 'my_mixed_collection', 'media_type': 'movie'},
        {'id': 'actors', 'label': 'Podle herců', 'icon': 'ai_card_actors.png',
         'description': 'Doporučení podle oblíbených herců', 'context_type': 'favorite_actors', 'media_type': 'movie'},
        {'id': 'directors', 'label': 'Podle režisérů', 'icon': 'ai_card_directors.png',
         'description': 'Doporučení podle oblíbených režisérů', 'context_type': 'favorite_directors', 'media_type': 'movie'},
        {'id': 'collections', 'label': 'Podle sbírky', 'icon': 'ai_card_collections.png',
         'description': 'Doporučení filmových kolekcí a ság', 'context_type': 'my_collections', 'media_type': 'collection'},
        {'id': 'history', 'label': 'Podle historie', 'icon': 'ai_card_history.png',
         'description': 'Doporučení podle historie přehrávání', 'context_type': 'recently_played', 'media_type': 'movie'},
    ]
    
    # Informační obsah pro každou kartu - texty, hinty, badges
    CARD_CONTENT = {
        'movies': {
            'title': 'Doporuč filmy',
            'ai_info': 'AI ti navrhne filmy na základě tvých preferencí a aktuálních trendů. Můžeš upřesnit žánr, rok, zemi původu nebo náladu.',
            'context_type': 'film',
            'section_title': 'MOJE FILMY',
            'empty_hint': 'Začni sledovat a AI se naučí tvůj vkus!',
            'badge_suffix': 've sbírce',
            'data_source': 'tmdb_movies',
        },
        'series': {
            'title': 'Doporuč seriály',
            'ai_info': 'AI najde seriály odpovídající tvému vkusu. Specifikuj žánr, délku epizod nebo platformu.',
            'context_type': 'serial',
            'section_title': 'MOJE SERIÁLY',
            'empty_hint': 'Sleduj seriály a AI pozná tvůj vkus!',
            'badge_suffix': 've sbírce',
            'data_source': 'tmdb_series',
        },
        'mix': {
            'title': 'Mix filmů a seriálů',
            'ai_info': 'AI analyzuje filmy a seriály z tvé sbírky a najde podobné tituly obou typů.',
            'context_type': 'titul',
            'section_title': 'Z TVÉ SBÍRKY',
            'empty_hint': 'Přidej filmy a seriály do sbírky pro personalizovaná doporučení.',
            'badge_suffix': 've sbírce',
            'data_source': 'mixed',
        },
        'actors': {
            'title': 'Podle oblíbených herců',
            'ai_info': 'AI dostane seznam tvých oblíbených herců a jejich filmografii. Na základě toho najde filmy s podobnými herci, žánry a stylem.',
            'context_type': 'herec',
            'section_title': 'TVOJI OBLÍBENÍ HERCI',
            'empty_hint': 'Zatím nemáš oblíbené herce. Přidej je v Filmy → Herci.',
            'badge_suffix': '',
            'data_source': 'favorite_actors',
        },
        'directors': {
            'title': 'Podle oblíbených režisérů',
            'ai_info': 'AI analyzuje styl tvých oblíbených režisérů - jejich typické žánry, vizuální styl a tematická zaměření. Najde filmy od podobných tvůrců.',
            'context_type': 'reziser',
            'section_title': 'TVOJI OBLÍBENÍ REŽISÉŘI',
            'empty_hint': 'Zatím nemáš oblíbené režiséry. Přidej je v Filmy → Režiséři.',
            'badge_suffix': '',
            'data_source': 'favorite_directors',
        },
        'collections': {
            'title': 'Filmové kolekce a ságy',
            'ai_info': 'AI navrhne filmové série a kolekce, které by tě mohly zajímat. Např. Marvel, DC, Nolanovy filmy, ságy jako LOTR.',
            'context_type': 'kolekce',
            'section_title': 'TVÉ SBÍRKY',
            'empty_hint': 'Přidej filmy z kolekcí do sbírky.',
            'badge_suffix': '',
            'data_source': 'collections',
        },
        'history': {
            'title': 'Z historie přehrávání',
            'ai_info': 'AI analyzuje tvých posledních přehraných titulů a najde podobné filmy/seriály na základě žánrů, herců a hodnocení.',
            'context_type': 'titul',
            'section_title': 'NAPOSLEDY PŘEHRANÉ',
            'empty_hint': 'Zatím nemáš historii přehrávání.',
            'badge_suffix': 'v historii',
            'data_source': 'recently_played',
        },
    }
    
    # Česká pluralizace pro kontext
    CZECH_PLURALS = {
        'film': ('film', 'filmy', 'filmů'),           # 1 film, 2-4 filmy, 5+ filmů
        'serial': ('seriál', 'seriály', 'seriálů'),   # 1 seriál, 2-4 seriály, 5+ seriálů
        'titul': ('titul', 'tituly', 'titulů'),       # 1 titul, 2-4 tituly, 5+ titulů
        'herec': ('herec', 'herci', 'herců'),         # 1 herec, 2-4 herci, 5+ herců
        'reziser': ('režisér', 'režiséři', 'režisérů'), # 1 režisér, 2-4 režiséři, 5+ režisérů
        'kolekce': ('kolekce', 'kolekce', 'kolekcí'), # 1 kolekce, 2-4 kolekce, 5+ kolekcí
    }

    
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        # Extrahovat auto_action pred volanim parent
        self._auto_action = kwargs.pop('auto_action', None)
        # Parent zpracovava show_nav_sidebar, nav_position, context
        super(AIRecommendationHubDialog, self).__init__(xml_filename, addon_path, *args, **kwargs)
        # Identifikator sekce pro navigacni sidebar
        self._nav_section_id = 'ai_recommend'
        self._is_hub_mode = True  # Flag pro rozliseni hub vs. vysledky

    def onInit(self):
        """Pretizena inicializace - zobrazi karty moznosti nebo provede auto_action."""
        # Resetovat stav (vyuziva parent metodu)
        self._reset_state()
        self.media_type = 'movie'
        self._is_hub_mode = True
        
        # Nastavit titulek
        self.setProperty('WCS.AIChat.Title', 'AI Doporučení')
        self.setProperty('WCS.AIChat.Status', 'Vyberte typ doporučení')
        
        # Nastavit fanart pozadi specificke pro AI Hub
        self.setProperty('WCS.AIPreview.Fanart', 'special://home/addons/plugin.video.milionar/resources/media/ai_hub_fanart.png')

        
        # Zobrazit uvitaci zpravu
        self.append_chat_message("AI", "Vyberte typ doporučení nebo napište vlastní dotaz.")
        
        # Nastavit tlacitka pro rychly vyber
        self._setup_hub_buttons()
        
        # Naplnit poster flow kartami
        self._populate_hub_cards()
        
        # Pokud mame auto_action, provest ji a preskocit zobrazeni hubu
        if self._auto_action:
            self._execute_auto_action()
            return
        
        # Pokud byl dialog spusten z navigacniho sidebaru, zobrazit ho bez animace
        if self._show_nav_on_init:
            self._show_nav_sidebar(animate=False)
        else:
            # Skryt chat sidebar, fokus na karty s kratkym zpozdenim
            self.clearProperty('WCS.AIChat.Visible')
            self._delayed_focus(2000, 0.3)
    
    def _execute_auto_action(self):
        """Provede automatickou akci pri startu dialogu (z menu 'Poradit se s AI')."""
        action = self._auto_action
        if not action:
            return
        
        # Zpracovat hub_select: akce (bez kontextu)
        if action.startswith('hub_select:'):
            card_id = action.split(':')[1]
            self._start_button_loading("Nacitam...")
            self._handle_card_selection(card_id)
            return
        
        # Zpracovat hub_context: akce (s kontextem)
        if action.startswith('hub_context:'):
            context_type = action.split(':')[1]
            self._start_button_loading("Nacitam...")
            self._handle_context_recommendation(context_type)
            return
    
    def _setup_hub_buttons(self):
        """Nastavi tlacitka v AI chat sidebar."""
        buttons = [
            {"label": "Doporuč filmy", "value": "hub_context:default_movie"},
            {"label": "Doporuč seriály", "value": "hub_context:default_series"},
            {"label": "Doporuč mix", "value": "hub_context:hub_mix"},
            {"label": "Podle sbírky filmu", "value": "hub_context:my_movies"},
            {"label": "Podle sbírky serialu", "value": "hub_context:my_series"},
            {"label": "Mix podle celé sbírky", "value": "hub_context:my_mixed_collection"},
        ]
        self.update_buttons(buttons)

    def _handle_context_recommendation(self, context_key):
        """
        Zpracuje požadavek na spuštění AI s konkrétním kontextem.
        Voláno z tlačítek 'hub_context:...'
        """
        # 1. Načíst data (pokud jsou třeba)
        # _load_context_data většinou očekává klíč shodný s context_key (my_movies atd.)
        # Pro default_movie/series vrátí None, což je správně.
        data = self._load_context_data(context_key)
        
        # 2. Určit úvodní zprávu pro chat
        user_msg = "Doporuc mi neco zajimaveho."
        
        if context_key == 'default_movie':
            user_msg = "Doporuc mi zajimave filmy k sledovani."
        elif context_key == 'default_series':
            user_msg = "Doporuc mi zajimave serialy k sledovani."
        elif context_key == 'hub_mix':
            user_msg = "Chci mix filmu a serialu - pul na pul. Doporuc mi 6 skvelych filmu a 6 skvelych serialu."
        elif context_key == 'my_movies':
            user_msg = "Doporuc mi filmy na zaklade me sbirky."
        elif context_key == 'my_series':
            user_msg = "Doporuc mi serialy na zaklade me sbirky."
        elif context_key == 'my_mixed_collection':
            user_msg = "Doporuc mi filmy a serialy na zaklade me kompletni sbirky."
            
        # 3. GUI Update
        # Skrýt overlaye
        self.clearProperty('WCS.AIPreview.ShowActions')
        self._clear_hub_properties()
        self._is_hub_mode = False
        
        # Zobrazit chat sidebar s loading animací
        self.setProperty('WCS.AIChat.Visible', 'true')
        self._start_button_loading("Analyzuji...")
        
        # 4. Start AI
        self.start_ai_with_context(context_key, data=data, user_text=user_msg, is_initial=True)
    
    def _format_czech_context(self, count, context_type):
        """
        Formátuje text kontextu s českou pluralizací.
        
        Pravidla:
        - 1: singular (film, seriál, herec)
        - 2-4: plural1 (filmy, seriály, herci)
        - 5+: plural2/genitive (filmů, seriálů, herců)
        """
        forms = self.CZECH_PLURALS.get(context_type, ('položka', 'položky', 'položek'))
        
        if count == 1:
            noun = forms[0]
        elif 2 <= count <= 4:
            noun = forms[1]
        else:
            noun = forms[2]
        
        return f"Kontext  •  {count} {noun}"
    
    def _format_czech_badge(self, count, context_type, suffix=''):
        """
        Formátuje badge text s českou pluralizací.
        
        Příklady:
        - 11 filmů v sbírce
        - 3 seriály v sbírce
        - 1 titul v historii
        """
        forms = self.CZECH_PLURALS.get(context_type, ('položka', 'položky', 'položek'))
        
        if count == 1:
            noun = forms[0]
        elif 2 <= count <= 4:
            noun = forms[1]
        else:
            noun = forms[2]
        
        if suffix:
            return f"{count} {noun} {suffix}"
        return f"{count} {noun}"
    
    def _populate_hub_cards(self):
        """Naplní poster flow kartami možností doporučení."""
        list_control = self.getControl(2000)
        list_control.reset()
        
        for card in self.RECOMMENDATION_CARDS:
            item = xbmcgui.ListItem(label=card['label'])
            item.setArt({
                'poster': f"special://home/addons/plugin.video.milionar/resources/media/{card['icon']}",
                'icon': card['icon']
            })
            item.setProperty('card_id', card['id'])
            item.setProperty('description', card['description'])
            item.setProperty('is_hub_card', 'true')
            item.setProperty('context_type', card['context_type'] or '')
            item.setProperty('media_type', card['media_type'])
            list_control.addItem(item)
        
        # Explicitně vybrat první položku a aktualizovat preview
        list_control.selectItem(0)
        self._update_hub_preview()
    
    def _update_hub_preview(self):
        """Aktualizuje preview podle vybrané karty - používá stejnou logiku jako filmy."""
        try:
            list_control = self.getControl(2000)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            card_id = item.getProperty('card_id')
            if not card_id:
                # Není to hub karta, použít standardní preview
                self._update_preview_for_current_item()
                # Skrýt hub content
                self.clearProperty('WCS.Hub.IsActive')
                return
            
            # Aktivovat hub mode
            self.setProperty('WCS.Hub.IsActive', 'true')
            
            # Reset flagu pro zobrazování akcí položek (aby se nenačítaly při procházení)
            self._showing_item_actions = False
            
            # Načíst content pro kartu
            content = self.CARD_CONTENT.get(card_id, {})
            
            # Načíst data pro kartu (potřebujeme count pro Metadata)
            items, count = self._load_hub_content_data(card_id)
            
            # === SESTAVIT DATA DICT PRO _apply_cached_preview_data (STEJNĚ JAKO PRO FILMY) ===
            
            # Tagline - Provider • Model
            try:
                provider_name = self.addon.getSetting('ai_provider') or 'mistral'
                model_name = self.addon.getSetting(f'{provider_name}_model') or ''
                provider_display = provider_name.capitalize() if provider_name else 'AI'
                tagline_text = f"{provider_display}  •  {model_name}" if model_name else provider_display
            except:
                tagline_text = 'AI Engine'
            
            # Metadata - Kontext • Počet (s českou pluralizací)
            context_type = content.get('context_type', '')
            if context_type and count > 0:
                metadata_text = self._format_czech_context(count, context_type)
            elif context_type:
                metadata_text = f"Kontext  •  0 {self.CZECH_PLURALS.get(context_type, ('', '', ''))[2]}"
            else:
                metadata_text = ''
            
            # Sestavit data dict (stejná struktura jako pro filmy)
            data = {
                'fmt_tagline': f"[I][COLOR FFB0B0B0]{tagline_text}[/COLOR][/I]" if tagline_text else '',
                'fmt_metadata': f"[COLOR grey]{metadata_text}[/COLOR]" if metadata_text else '',
                'fmt_extended': '',  # Prázdné - nepoužíváme pro Hub
                'plot': content.get('ai_info', ''),
            }
            
            # Nastavit Title (není v _apply_cached_preview_data)
            self.setProperty('WCS.AIPreview.Title', content.get('title', ''))
            
            # Zavolat EXISTUJÍCÍ metodu pro nastavení properties
            self._apply_cached_preview_data(data)
            
            # Nastavit Hub-specifické properties
            self.setProperty('WCS.Hub.Title', content.get('title', ''))
            self.setProperty('WCS.Hub.AIInfo', content.get('ai_info', ''))
            self.setProperty('WCS.Hub.SectionTitle', content.get('section_title', ''))
            self.setProperty('WCS.Hub.EmptyHint', content.get('empty_hint', ''))
            
            # Nastavit badge (s českou pluralizací)
            context_type = content.get('context_type', '')
            badge_suffix = content.get('badge_suffix', '')
            if context_type and count > 0:
                badge_text = self._format_czech_badge(count, context_type, badge_suffix)
                self.setProperty('WCS.Hub.Badge', badge_text)
            else:
                self.clearProperty('WCS.Hub.Badge')
            
            # Naplnit content preview (control 2500)
            self._populate_content_preview(items)
            
            # Nastavit visibility stavy
            if items:
                self.setProperty('WCS.Hub.HasContent', 'true')
                self.clearProperty('WCS.Hub.ShowEmpty')
            else:
                self.clearProperty('WCS.Hub.HasContent')
                self.setProperty('WCS.Hub.ShowEmpty', 'true')
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Hub preview error: {e}", xbmc.LOGERROR)

    
    def _populate_preview_actions(self, media_type, tmdb_id, title, year):
        """Přetíženo - Smart Actions logika pro Hub."""
        # Pokud je Hub mode a NEMÁME flag _showing_item_actions -> smazat tlačítka
        if self._is_hub_mode and not getattr(self, '_showing_item_actions', False):
            self.getControl(6100).reset()
            return

        if self._is_hub_mode and getattr(self, '_showing_item_actions', False):
            # === SMART ACTIONS LOGIKA (PRO HUB ITEMS) ===
            try:
                # 1. Zjistit kontext karty
                list_control = self.getControl(2000)
                item = list_control.getSelectedItem()
                card_id = item.getProperty('card_id') if item else None
                
                context_key = None
                
                # 2. Určit sadu tlačítek podle typu a karty
                if media_type == 'series' and card_id == 'series':
                    # Pro "Doporuč Seriály" (My Series logika) - Smart Start/Continue
                    if self._is_in_history(tmdb_id):
                        context_key = 'col_series_continue'
                    else:
                        context_key = 'col_series_start'
                
                elif media_type == 'movie' and card_id == 'movies':
                    # Pro "Doporuč Filmy" (My Movies logika)
                    context_key = 'col_movie'
                
                elif card_id == 'history':
                     # Pro Historii
                     if media_type == 'series':
                         context_key = 'hist_series'
                     else:
                         context_key = 'hist_movie'
                
                else:
                    # Fallback (Generic AI recommendation)
                    if media_type == 'series':
                        context_key = 'ai_series'
                    else:
                        context_key = 'ai_movie'

                # 3. Vygenerovat položky
                items = self.build_action_items(context_key)
                
                # 4. Naplnit list
                list_actions = self.getControl(6100)
                list_actions.reset()
                for item in items:
                    item.setProperty('tmdb_id', str(tmdb_id))
                    item.setProperty('title', title)
                    item.setProperty('year', str(year))
                    item.setProperty('media_type', media_type)
                    list_actions.addItem(item)
                    
                return # Hotovo, nevolat super
                
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] Smart Actions error: {e}", xbmc.LOGERROR)

        # === NON-HUB MODE (když zobrazujeme AI doporučení) ===
        # Zde explicitně definujeme akce, abychom měli jistotu, že tam je "Similar"
        try:
            list_control = self.getControl(6100)
            list_control.reset()
            
            # Určit kontext podle media_type
            if media_type == 'movie':
                context = 'ai_movie'
            elif media_type in ['series', 'tv']:
                context = 'ai_series'
            else:
                context = 'ai_col'
                
            # Získat itemy z registru (používáme metody z base class)
            items = self.build_action_items(context)
            
            for item in items:
                item.setProperty('tmdb_id', str(tmdb_id))
                item.setProperty('title', title)
                item.setProperty('year', str(year))
                item.setProperty('media_type', media_type)
                list_control.addItem(item)
                
        except Exception as e:
             xbmc.log(f"[plugin.video.milionar] Hub non-hub actions error: {e}", xbmc.LOGERROR)


    def _is_in_history(self, tmdb_id):
        """Zjistí, zda je titul v historii sledování."""
        try:
            # Načíst historii (optimálně by mělo být cachované, ale pro button click to nevadí)
            history = user_data.load_recently_played(self.addon)
            return any(str(i.get('tmdb_id')) == str(tmdb_id) for i in history)
        except Exception:
            return False

    def _handle_continue_series(self, title, tmdb_id):
         """Spustí přehrávání dalšího dílu (Next Episode) nebo začne od začátku."""
         try:
             # Zavřít dialog
             self.close()
             
             from wcs import utils
             
             # 1. Zjistit poslední sledovanou epizodu z historie
             last_season = 0
             last_episode = 0
             
             try:
                 history = user_data.load_recently_played(self.addon)
                 # Najít záznam
                 # Poznámka: Historie může mít více záznamů pro seriál? Obvykle ne, update overwrituje.
                 # Hledáme podle tmdb_id
                 found = next((i for i in history if str(i.get('tmdb_id')) == str(tmdb_id)), None)
                 
                 if found:
                     # Pokud nalezeno, použít season/episode z historie
                     # Historie ukládá 'season' a 'episode' (viz user_data.add_recently_played_item)
                     # POZOR: user_data.add_recently_played_item ukládá to, co bylo PRÁVĚ přehrané.
                     # play_next_episode_with_metadata očekává 'current_season/episode' (poslední shlédnuté),
                     # aby vypočítal DALŠÍ.
                     last_season = int(found.get('season', 0))
                     last_episode = int(found.get('episode', 0))
             except Exception as h_err:
                 xbmc.log(f"[plugin.video.milionar] Hub history check error: {h_err}", xbmc.LOGWARNING)

             # 2. Sestavit parametry
             params = {
                 'tmdb_id': tmdb_id, # [FIX] Použít správný klíč 'tmdb_id', ne 'series_id'
                 'series_name': title, # [FIX] Použít správný klíč 'series_name', ne 'name'
                 'current_season': last_season,
                 'current_episode': last_episode
             }
             
             xbmc.log(f"[plugin.video.milionar] Hub starting series '{title}' (ID {tmdb_id}) from S{last_season}E{last_episode}", level=xbmc.LOGINFO)
             
             utils.play_next_episode_with_metadata(params, self.addon)
         except Exception as e:
             xbmc.log(f"[plugin.video.milionar] Continue Series error: {e}", xbmc.LOGERROR)

    
    def _load_hub_content_data(self, card_id):
        """
        Načte data pro danou kartu z uživatelských dat.
        
        Returns:
            tuple: (list of dicts s 'poster' a 'label' klíči, count)
        """
        addon = xbmcaddon.Addon()
        items = []
        
        try:
            if card_id == 'actors':
                # Oblíbení herci
                actors = user_data.load_favorite_actors(addon)
                for actor in actors:  # Všechny položky
                    profile_path = actor.get('profile_path', '')
                    poster_url = f"https://image.tmdb.org/t/p/w342{profile_path}" if profile_path else ''
                    items.append({
                        'poster': poster_url,
                        'label': actor.get('name', ''),
                        'fallback': 'DefaultActor.png'
                    })
                return items, len(actors)
            
            elif card_id == 'directors':
                # Oblíbení režiséři
                directors = user_data.load_favorite_directors(addon)
                for director in directors:
                    profile_path = director.get('profile_path', '')
                    poster_url = f"https://image.tmdb.org/t/p/w342{profile_path}" if profile_path else ''
                    items.append({
                        'poster': poster_url,
                        'label': director.get('name', ''),
                        'fallback': 'DefaultActor.png'
                    })
                return items, len(directors)
            
            elif card_id == 'history':
                # Historie přehrávání
                history = user_data.load_recently_played(addon)
                for item in history:
                    items.append({
                        'poster': item.get('poster', ''),
                        'label': item.get('title', ''),
                        'fallback': 'DefaultVideo.png'
                    })
                return items, len(history)
            
            elif card_id == 'mix':
                # Mix filmů a seriálů - všechny položky
                movies = user_data.load_tmdb_movies(addon) or []
                series = user_data.load_tmdb_series(addon) or []
                
                # Střídavě přidávat filmy a seriály (všechny)
                all_items = []
                max_len = max(len(movies), len(series))
                for i in range(max_len):
                    if i < len(movies):
                        m = movies[i]
                        poster_path = m.get('poster_path', '')
                        all_items.append({
                            'poster': f"https://image.tmdb.org/t/p/w342{poster_path}" if poster_path else '',
                            'label': m.get('title', ''),
                            'fallback': 'DefaultVideo.png',
                            'overview': m.get('overview', ''),
                            'year': m.get('year', ''),
                            'id': m.get('id'),
                            'vote_average': m.get('vote_average'),
                            'media_type': 'movie'
                        })
                    if i < len(series):
                        s = series[i]
                        poster_path = s.get('poster_path', '')
                        all_items.append({
                            'poster': f"https://image.tmdb.org/t/p/w342{poster_path}" if poster_path else '',
                            'label': s.get('name', ''),
                            'fallback': 'DefaultVideo.png',
                            'overview': s.get('overview', ''),
                            'year': s.get('year', ''),
                            'id': s.get('id'),
                            'vote_average': s.get('vote_average'),
                            'media_type': 'series'
                        })
                return all_items, len(movies) + len(series)
            
            elif card_id == 'movies':
                # Moje filmy (nebo trending pokud prázdné)
                movies = user_data.load_tmdb_movies(addon) or []
                items = []
                for m in movies:
                    poster_path = m.get('poster_path', '')
                    items.append({
                        'poster': f"https://image.tmdb.org/t/p/w342{poster_path}" if poster_path else '',
                        'label': m.get('title', ''),
                        'fallback': 'DefaultVideo.png',
                        'overview': m.get('overview', ''),
                        'year': m.get('year', ''),
                        'id': m.get('id'),
                        'vote_average': m.get('vote_average'),
                        'media_type': 'movie'
                    })
                return items, len(movies)
            
            elif card_id == 'series':
                # Moje seriály
                series = user_data.load_tmdb_series(addon) or []
                items = []
                for s in series:
                    poster_path = s.get('poster_path', '')
                    items.append({
                        'poster': f"https://image.tmdb.org/t/p/w342{poster_path}" if poster_path else '',
                        'label': s.get('name', ''),
                        'fallback': 'DefaultVideo.png',
                        'overview': s.get('overview', ''),
                        'year': s.get('year', ''),
                        'id': s.get('id'),
                        'vote_average': s.get('vote_average'),
                        'media_type': 'series'
                    })
                return items, len(series)
            
            elif card_id == 'collections':
                # Kolekce z filmů uživatele (bez API volání - jen pokud už máme data)
                # Pro jednoduchost použijeme filmy jako placeholder
                movies = user_data.load_tmdb_movies(addon) or []
                items = []
                for m in movies:
                    poster_path = m.get('poster_path', '')
                    items.append({
                        'poster': f"https://image.tmdb.org/t/p/w342{poster_path}" if poster_path else '',
                        'label': m.get('title', ''),
                        'fallback': 'DefaultVideo.png',
                        'overview': m.get('overview', ''),
                        'year': m.get('year', ''),
                        'id': m.get('id'),
                        'vote_average': m.get('vote_average'),
                        'media_type': 'movie'
                    })
                return items, len(movies)
                
            elif card_id == 'history':
                # Historie přehrávání
                history = user_data.load_recently_played(addon)
                items = []
                for item in history:
                    items.append({
                        'poster': item.get('poster', ''),
                        'label': item.get('title', ''),
                        'fallback': 'DefaultVideo.png',
                        'overview': item.get('plot', ''),
                        'year': item.get('year', ''),
                        'id': item.get('tmdb_id'),
                        'vote_average': item.get('rating'),
                        'media_type': item.get('media_type', 'movie')
                    })
                return items, len(history)

        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error loading hub content for {card_id}: {e}", xbmc.LOGERROR)
        
        return items, 0
    
    def _populate_content_preview(self, items):
        """
        Naplní content preview control (id 2500) postery z uživatelských dat.
        
        Args:
            items: List of dicts s 'poster', 'label' a 'fallback' klíči
        """
        try:
            preview_control = self.getControl(2500)
            preview_control.reset()
            
            for item_data in items:
                list_item = xbmcgui.ListItem(label=item_data.get('label', ''))
                poster_url = item_data.get('poster', '')
                fallback = item_data.get('fallback', 'DefaultVideo.png')
                
                list_item.setArt({
                    'poster': poster_url if poster_url else fallback,
                    'thumb': poster_url if poster_url else fallback,
                    'icon': fallback
                })
                
                # Ulozit metadata
                if item_data.get('id'): list_item.setProperty('tmdb_id', str(item_data['id']))
                if item_data.get('year'): list_item.setProperty('year', str(item_data['year']))
                if item_data.get('overview'): list_item.setProperty('overview', item_data['overview'])
                if item_data.get('media_type'): list_item.setProperty('media_type', item_data['media_type'])
                if item_data.get('vote_average'): list_item.setProperty('vote_average', str(item_data['vote_average']))
                
                preview_control.addItem(list_item)
                
        except Exception as e:
            # Control nemusí existovat v XML - to je OK
            xbmc.log(f"[plugin.video.milionar] Content preview control not available: {e}", xbmc.LOGDEBUG)
    
    def _load_context_data(self, context_type):
        """Nacte data pro dany context_type. Centralni metoda pro vsechny kontextove operace."""
        addon = xbmcaddon.Addon()
        
        if context_type == 'my_movies':
            return user_data.load_tmdb_movies(addon)
        elif context_type == 'my_series':
            return user_data.load_tmdb_series(addon)
        elif context_type == 'my_mixed_collection':
            movies = user_data.load_tmdb_movies(addon)
            series = user_data.load_tmdb_series(addon)
            return {'movies': movies, 'series': series}
        elif context_type == 'favorite_actors':
            return user_data.load_favorite_actors(addon)
        elif context_type == 'favorite_directors':
            return user_data.load_favorite_directors(addon)
        elif context_type == 'recently_played':
            return user_data.load_recently_played(addon)
        elif context_type == 'my_collections':
            tmdb_movies = user_data.load_tmdb_movies(addon)
            colls = {}
            if tmdb_movies:
                from wcs.metadata.TMDbClient import get_movie_detailed_info
                for movie in tmdb_movies:
                    mid = movie.get('id')
                    if mid:
                        details = get_movie_detailed_info(mid)
                        if details and details.get('belongs_to_collection'):
                            c_info = details['belongs_to_collection']
                            cid = c_info.get('id')
                            if cid and cid not in colls:
                                colls[cid] = c_info
            return list(colls.values())
        return None
    
    def handle_recommendation_click(self):
        """Přetížená metoda - zpracuje klik na kartu nebo výsledek."""
        try:
            list_control = self.getControl(2000)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            # Kontrola zda je to hub karta
            if item.getProperty('is_hub_card') == 'true':
                card_id = item.getProperty('card_id')
                self._handle_card_selection(card_id)
            else:
                # Standardní zpracování výsledku
                super().handle_recommendation_click()
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Hub click error: {e}", xbmc.LOGERROR)
    
    
    def _update_miniposter_preview(self):
        """Aktualizuje preview panel (pravy sloupec) podle vybraneho miniposteru (item z listu 2500)."""
        try:
            list_control = self.getControl(2500)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            tmdb_id = item.getProperty('tmdb_id')
            media_type = item.getProperty('media_type') or 'movie'
            title = item.getLabel()
            year = item.getProperty('year') or ''
            poster = item.getArt('poster')
            plot = item.getProperty('overview') or item.getProperty('plot') or ''
            
            # Nastavit Hub specifické properties (protože _apply_new_preview se stará o AIPreview)
            # Tyto se používají v dolní kartě (Group 2400)
            self.setProperty('WCS.Hub.Title', title)
            self.setProperty('WCS.Hub.AIInfo', plot)
            
            # Pokud máme ID, použijeme standardní logiku z DialogAIChatRecommendation
            # která zajistí konzistentní zobrazení (včetně Tagline, Metadata, Workeru, Cache...)
            # Stejně jako to dělá DialogMyMovies
            if tmdb_id:
                # [CRITICAL] Použití zděděné metody pro plnou konzistenci s MyMovies
                self._apply_new_preview(tmdb_id, media_type, title, year, poster)
            else:
                # Fallback pro položky bez ID (custom položky)
                self.setProperty('WCS.AIPreview.Title', title)
                self.setProperty('WCS.AIPreview.Year', year)
                self.setProperty('WCS.AIPreview.Poster', poster)
                self.setProperty('WCS.AIPreview.Plot', plot)
                self.clearProperty('WCS.AIPreview.Tagline')
                self.clearProperty('WCS.AIPreview.Metadata')
                self.clearProperty('WCS.AIPreview.Extended')

        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Miniposter preview error: {e}", xbmc.LOGERROR)
    
    def _clear_hub_properties(self):
        """Vymaze vsechny Hub-specificke properties kdyz se opousti hub mode."""
        self.clearProperty('WCS.Hub.IsActive')
        self.clearProperty('WCS.Hub.HasContent')
        self.clearProperty('WCS.Hub.ShowEmpty')
        self.clearProperty('WCS.Hub.Title')
        self.clearProperty('WCS.Hub.AIInfo')
        self.clearProperty('WCS.Hub.SectionTitle')
        self.clearProperty('WCS.Hub.EmptyHint')
        self.clearProperty('WCS.Hub.Badge')
    
    def _handle_card_selection(self, card_id):
        """Zpracuje vyber karty a spusti AI inference - nacita data stejne jako router.py."""
        

        # Specialni pripad: collections_simple = kolekce bez kontextu
        if card_id == 'collections_simple':
            # Skryt overlay a focus
            self._clear_hub_properties()
            self._is_hub_mode = False
            
            user_msg = "Doporuc mi zajimave filmove kolekce a sagy k prozkoumani."
            
            # Unified Start
            self.setProperty('WCS.AIChat.Visible', 'true')
            self._start_button_loading("Analyzuji...")
            self.start_ai_with_context('hub_col', data=None, user_text=user_msg)
            return

        # Najit kartu
        card = None
        for c in self.RECOMMENDATION_CARDS:
            if c['id'] == card_id:
                card = c
                break
        
        if not card:
            return
        
        # Nastavit media type
        self.media_type = card['media_type']
        self._clear_hub_properties()
        self._is_hub_mode = False
        
        # Vytvorit kontext s nactenymi daty
        context_type = card['context_type']
        if context_type:
            self.context = {'type': context_type}
            self.context['data'] = self._load_context_data(context_type)
        elif card_id == 'mix':
            # Mix bez kontextu - nastavit simple_mix pro spravny system prompt
            self.context = {'type': 'simple_mix'}
        else:
            self.context = None
        
        # Inicializovat prompt builder s nactenymi daty
        self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
        
        # Vytvorit zpravu podle typu - pouzit context instruction pokud je dostupna
        context_instruction = self._get_context_instruction() if self.context else None
        
        if context_instruction:
            user_msg = context_instruction
        elif card_id == 'movies':
            user_msg = "Doporuc mi zajimave filmy k sledovani."
        elif card_id == 'series':
            user_msg = "Doporuc mi zajimave serialy k sledovani."
        elif card_id == 'mix':
            user_msg = "Chci mix filmu a serialu - pul na pul. Doporuc mi 6 skvelych filmu a 6 skvelych serialu."
        elif card_id == 'actors':
            user_msg = "Doporuc mi filmy podle mych oblibenych hercu."
        elif card_id == 'directors':
            user_msg = "Doporuc mi filmy podle mych oblibenych reziseru."
        elif card_id == 'collections':
            user_msg = "Doporuc mi filmove kolekce a sagy."
        elif card_id == 'history':
            user_msg = "Doporuc mi filmy a serialy na zaklade me historie sledovani."
        else:
            user_msg = "Doporuc mi neco zajimaveho."
        
        # Přidat do historie
        self.history.append({"role": "user", "content": user_msg})
        self.append_chat_message("Ty", user_msg)
        
        # Zobrazit chat sidebar s loading
        self.setProperty('WCS.AIChat.Visible', 'true')
        self._start_button_loading("Analyzuji...")
        
        # Spustit AI konverzaci
        self._start_ai_conversation(is_initial_context=True)

    
    def onClick(self, controlId):
        """Pretizena metoda - zpracuje hub_select: a hub_context: tlacitka pred delegaci na parent."""
        if controlId == 4000:
            list_control = self.getControl(4000)
            item = list_control.getSelectedItem()
            if item:
                val = item.getProperty('value')
                label = item.getLabel()
                
                # Zpracovat hub_select: prikazy (Doporuc filmy/serialy/mix)
                if val.startswith('hub_select:'):
                    card_id = val.split(':')[1]
                    self._start_button_loading(label)
                    self._handle_card_selection(card_id)
                    return
                
                # Zpracovat hub_context: prikazy (Z mych filmu/serialu)
                if val.startswith('hub_context:'):
                    context_type = val.split(':')[1]
                    self._start_button_loading(label)
                    self._handle_context_recommendation(context_type)
                    return

        elif controlId == 2500:
            # Klik na miniposter (2500) - Zobrazit akční tlačítka (Přehrát, Detaily, atd.)
            list_control = self.getControl(2500)
            item = list_control.getSelectedItem()
            if item:
                # Nastavit flag, že chceme zobrazit akce pro tuto položku
                self._showing_item_actions = True
                
                # Získat metadata
                media_type = item.getProperty('media_type') or item.getProperty('type') or 'movie'
                tmdb_id = item.getProperty('tmdb_id')
                title = item.getLabel() # Title je v labelu, ale lepsi by bylo z property 'title' pokud existuje? Label muze byt formatovany.
                # Base class _populate_preview_actions bere title. Většinou posíláme 'title' property v _update_miniposter_preview.
                # Zde ale item v 2500 je xbmcgui.ListItem.
                # Musíme zavolat populate.
                # Poznámka: item.getLabel() může obsahovat formátování.
                # Zkusme získat 'title' property, kterou jsme tam uložili.
                raw_title = item.getProperty('title') or item.getLabel() 
                year = item.getProperty('year')
                
                # Vynutit naplnění tlačítek (override nyní projde díky flagu)
                self._populate_preview_actions(media_type, tmdb_id, raw_title, year)
                
                # Zobrazit akce a nastavit fokus
                self.setProperty('WCS.AIPreview.ShowActions', 'true')
                self.setFocusId(6100)
                return

        # Vše ostatni delegovat na parent
        super().onClick(controlId)
    
    def onAction(self, action):
        """Přetížená akce pro hub mode.
        
        Hub mode má specifické chování:
        - UP na seznamu zobrazí hub-specifické akce (textový vstup + vybraná karta)
        - LEFT/RIGHT aktualizuje hub preview místo standardního
        - Vše ostatní (včetně nav sidebar) deleguje na parent
        """
        action_id = action.getId()
        
        # V hub mode - šipka nahoru zobrazí akční overlay s textovým vstupem
        if self._is_hub_mode and action_id == xbmcgui.ACTION_MOVE_UP:
            if self.getFocusId() == 2000:
                # Nastavit Hub-specifické akce
                self._setup_hub_actions()
                self.setProperty('WCS.AIPreview.ShowActions', 'true')
                self.setFocusId(6100)
                return
        
        # V hub mode - šipka DOLU z miniposterů (2500) zpět na akce (6100)
        # Protože 6100 je skryté (v onFocus 2500 jsme property smazali), XML navigace <ondown> nefunguje
        if self._is_hub_mode and action_id == xbmcgui.ACTION_MOVE_DOWN:
            if self.getFocusId() == 2500:
                # [CRITICAL] Obnovit Hub-specifické akce (Generic/Collection)
                # Protože focus na miniposter (2500) je přepsal standardními akcemi položky
                self._setup_hub_actions()
                
                self.setProperty('WCS.AIPreview.ShowActions', 'true')
                self.setFocusId(6100)
                # Obnovit hub preview (nadpis karty atd.) misto miniposter preview
                self._update_hub_preview()
                return
        
        # V hub mode - LEFT/RIGHT aktualizuje hub preview místo standardního
        # DŮLEŽITÉ: Necháme parent zpracovat VŠECHNO včetně nav sidebar logiky
        if self._is_hub_mode and action_id in (xbmcgui.ACTION_MOVE_LEFT, xbmcgui.ACTION_MOVE_RIGHT):
            if self.getFocusId() == 2000:
                # Uložit pozici před delegací na parent
                list_control = self.getControl(2000)
                current_pos = list_control.getSelectedPosition()
                
                # LEFT na první položce - delegovat na parent (může otevřít nav sidebar)
                if action_id == xbmcgui.ACTION_MOVE_LEFT and self._last_list_position == 0:
                    super().onAction(action)
                    return
                
                # Aktualizovat pozici pro příští akci
                self._last_list_position = current_pos
                
                # Aktualizovat hub-specifický preview
                time.sleep(0.05)
                self._update_hub_preview()
                return

            if self.getFocusId() == 2500:
                # Aktualizovat preview podle vybraneho miniposteru
                time.sleep(0.05) 
                self._update_miniposter_preview()
                # Zadne dalsi akce (nepredavat parentovi)
                return
        
        # Standardní zpracování (včetně nav sidebar, chat, atd.)
        super().onAction(action)
    
    def onFocus(self, controlId):
        """Pretizena metoda onFocus pro Hub mode.
        
        V hub mode pouziva _update_hub_preview() misto parent _update_preview_for_current_item(),
        protoze hub karty nemaji tmdb_id a parent by vymazal vsechny Hub properties.
        """
        if controlId == 2000:
            # Focus na panel karet/doporuceni
            if self._last_focus_id in [4000, 3000, 3001, 1001, 1002]:
                self._returning_from_chat = True
            
            self.clearProperty('WCS.AIPreview.ShowActions')
            self.clearProperty('WCS.AIChat.Visible')
            self.clearProperty('WCS.NavSidebar.Visible')
            
            # KLICOVA ZMENA: V hub mode pouzit hub-specificky preview
            if self._is_hub_mode:
                self.clearProperty('WCS.AIPreview.Fanart')
                self._update_hub_preview()
            else:
                self._update_preview_for_current_item()
        
        elif controlId == 6100:
            # Uzivateleske akce - obnovit property pro viditelnost (aby nezmizely pri ztrate focusu)
            self.setProperty('WCS.AIPreview.ShowActions', 'true')
            
        elif controlId == 2500:
            # Minipostery - skryt akcni tlacitka (menu "zmizi", aby se pri navratu dolu "znovu objevilo")
            self.clearProperty('WCS.AIPreview.ShowActions')
            # Aktualizovat preview hned pri vstupu
            if self._is_hub_mode:
                self._update_miniposter_preview()
        
        elif controlId == 7100:
            # Navigacni sidebar - skryt ostatni overlays
            self.clearProperty('WCS.AIPreview.ShowActions')
            self.clearProperty('WCS.AIChat.Visible')
        
        elif controlId in [4000, 3000, 3001, 1001, 1002]:
            # Chat overlay - skryt akcni tlacitka, ALE nechat preview (pozadi)
            self.clearProperty('WCS.AIPreview.ShowActions')
        
        else:
            # Cokoliv jineho - delegovat na parent
            super().onFocus(controlId)
            return
        
        # Ulozit aktualni focus ID pro pristi kontrolu
        self._last_focus_id = controlId
    
    # Konfigurace akcí pro Hub karty
    HUB_CARD_ACTIONS = {
        'movies': [
            {'label': 'Doporuč filmy', 'action': 'hub_context:default_movie'},
            {'label': 'Podle sbírky', 'action': 'hub_context:my_movies'},
        ],
        'series': [
            {'label': 'Doporuč seriály', 'action': 'hub_context:default_series'},
            {'label': 'Podle sbírky', 'action': 'hub_context:my_series'},
        ],
        'mix': [
            {'label': 'Doporuč mix', 'action': 'hub_context:hub_mix'}, 
            {'label': 'Podle sbírky', 'action': 'hub_context:my_mixed_collection'},
        ],
        'collections': [
            {'label': 'Filmové kolekce', 'action': 'hub_select:collections_simple'},
            {'label': 'Podle sbírky', 'action': 'hub_select:collections'},
        ],
        # Ostatni karty (history, actors, directors) maji jen defaultni 'recommend'
    }

    def _setup_hub_actions(self):
        """Nastavi akcni tlacitka pro Hub (Data-Driven Refactoring)."""
        # Ziskat vybranou kartu
        try:
            list_control = self.getControl(2000)
            item = list_control.getSelectedItem()
            card_id = item.getProperty('card_id') if item else None
            card_label = item.getLabel() if item else "Dotaz"
        except:
            card_id = None
            card_label = "Dotaz"
        
        # 1. Nacist konfiguraci pro kartu (default: jen recommend)
        card_config = self.HUB_CARD_ACTIONS.get(card_id, [{'type': 'recommend', 'action': 'hub_select_current'}])
        
        actions = []
        
        # 2. Vygenerovat akce podle konfigurace
        for conf in card_config:
            if conf.get('type') == 'recommend':
                # Dynamicky label pro hlavni doporuceni
                actions.append({'label': card_label, 'action': conf['action']})
            else:
                # Staticky label
                actions.append({'label': conf['label'], 'action': conf['action']})
                
        # 3. Vzdy pridat Vlastni dotaz na konec
        actions.append({"label": "Vlastní dotaz", "action": "hub_text_input"})
        
        # Naplnit action list (6100)
        action_list = self.getControl(6100)
        action_list.reset()
        for act in actions:
            item = xbmcgui.ListItem(label=act['label'])
            item.setProperty('action', act['action'])
            action_list.addItem(item)

    
    def _handle_preview_action_click(self):
        """Přetížená metoda pro zpracování akčních tlačítek."""
        if not self._is_hub_mode:
            # === SPECIFICKÁ LOGIKA PRO HUB ITEMS (když jsme ve výsledcích vyhledávání) ===
            # V Hubu musíme zajistit, aby "Similar" fungovalo správně
            # Base class používá self.context, ale v Hubu se kontexty mění.
            # Zde delegujeme na super(), protože `_populate_preview_actions`
            # už nastavilo správné akce (ai_movie/ai_series).
            super()._handle_preview_action_click()
            return

        # Hub mode - vlastni zpracovani
        try:
            action_list = self.getControl(6100)
            item = action_list.getSelectedItem()
            if not item:
                return
            
            action = item.getProperty('action')
            label = item.getLabel()
            
            if action == 'hub_text_input':
                # Otevrit klavesnici a odeslat dotaz
                self._handle_hub_text_input()
            elif action == 'hub_select_current':
                # Vybrat aktualni kartu
                self.setFocusId(2000)
                self.handle_recommendation_click()
            elif action.startswith('hub_select:'):
                # Primy vyber karty (napr. mix_simple)
                card_id = action.split(':')[1]
                self.clearProperty('WCS.AIPreview.ShowActions')
                self._start_button_loading(label)
                self._handle_card_selection(card_id)
            elif action.startswith('hub_context:'):
                # Kontextove doporuceni (my_movies, my_series, my_mixed_collection)
                context_type = action.split(':')[1]
                self.clearProperty('WCS.AIPreview.ShowActions')
                self._start_button_loading(label)
                self._handle_context_recommendation(context_type)
            elif action in ['continue', 'next_episode']:
                # Smart Continue / Start Series
                tmdb_id = item.getProperty('tmdb_id')
                title = item.getProperty('title') 
                self._handle_continue_series(title, tmdb_id)
            elif action == 'remove_from_list':
                tmdb_id = item.getProperty('tmdb_id')
                title = item.getProperty('title')
                year = item.getProperty('year')
                media_type = item.getProperty('media_type')
                self._handle_remove_from_list(tmdb_id, media_type, title, year)
            elif action == 'remove_from_history':
                tmdb_id = item.getProperty('tmdb_id')
                title = item.getProperty('title')
                self._handle_remove_from_history(tmdb_id, title)
            else:
                # Ostatní akce (play, details, similar...) delegovat na parent
                super()._handle_preview_action_click()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Hub action error: {e}", xbmc.LOGERROR)


    def handle_user_input(self, user_text):
        """Přetížená metoda pro zpracování vstupu z sidebar chatu (nebo odkudkoliv)."""
        # Pokud jsme v HUB MODE (vyběr karet), chceme, aby chat fungoval jako "Vlastní dotaz"
        # ALE POZOR: Pokud píšeme do sidebar chatu (tato metoda), chceme DEFAULTNĚ MIX.
        # Uživatel: "na začátku to bude vždycky mix"
        if self._is_hub_mode:
            # Skrýt akční overlay
            self.clearProperty('WCS.AIPreview.ShowActions')
            self._clear_hub_properties()
            self._is_hub_mode = False
            
            # Sidebar input = Vždy začít jako MIX (pokud není specifikováno jinak)
            # Ignorujeme vybranou kartu, protože sidebar je "globální" chat
            context_key = 'hub_mix'
            
            xbmc.log(f"[plugin.video.milionar] Starting sidebar query (Hub Mode) with context: {context_key}", xbmc.LOGINFO)
            self.start_ai_with_context(context_key, data=None, user_text=user_text, is_initial=False)
        else:
            # Pokud už máme nějaký kontext/konverzaci (už nejsme v Hub mode), 
            # chováme se standardně a držíme kontext
            super().handle_user_input(user_text)

    def _handle_hub_text_input(self):
        """Zpracuje textový vstup od uživatele (z akčního tlačítka)."""
        # Otevřít klávesnici
        keyboard = xbmc.Keyboard('', 'Napište svůj dotaz pro AI')
        keyboard.doModal()
        
        if keyboard.isConfirmed():
            user_text = keyboard.getText().strip()
            if user_text:
                self._handle_hub_text_input_logic(user_text)

    def _handle_hub_text_input_logic(self, user_text):
        """Společná logika pro zpracování textu v Hub módu."""
        # 1. Zjistit aktuálně vybranou kartu (pro určení kontextu)
        context_key = 'default_movie' # Fallback default
        
        # Pokusíme se zjistit kontext z vybrané karty, pokud nějaká je
        try:
            list_control = self.getControl(2000)
            item = list_control.getSelectedItem()
            if item:
                card_id = item.getProperty('card_id')
                if card_id == 'movies':
                    context_key = 'default_movie'
                elif card_id == 'series':
                    context_key = 'default_series'
                elif card_id == 'mix':
                    context_key = 'hub_mix' # Použije simple_mix kontext
                elif card_id == 'collections':
                    context_key = 'hub_col' # Použije simple_collections kontext
                elif card_id in ['actors', 'directors', 'history']:
                    context_key = 'hub_mix' # Pro ostatní použít mix
        except:
            pass
            
        # Skrýt akční overlay
        self.clearProperty('WCS.AIPreview.ShowActions')
        
        # Přepnout z hub mode do normálního režimu
        self._clear_hub_properties()
        self._is_hub_mode = False
        
        # Pokud je to sidebar input (voláno přímo), musíme zajistit zobrazení chatu
        self.setProperty('WCS.AIChat.Visible', 'true')
        
        # Unified Start s dynamickým kontextem
        xbmc.log(f"[plugin.video.milionar] Starting custom query with context: {context_key}", xbmc.LOGINFO)
        self.start_ai_with_context(context_key, data=None, user_text=user_text, is_initial=False)



    
    def _handle_no_recommendations(self):
        """Přetížená metoda - vrátit se zpět do hub mode."""
        self.append_system_message("Model nedoporučil žádné tituly.")
        self._is_hub_mode = True
        self._populate_hub_cards()
        self._setup_hub_buttons()


def show_ai_recommendation_hub_dialog(addon, show_nav_sidebar=False, nav_position=0, auto_action=None):
    """Zobrazi dialog AI Doporuceni Hub.
    
    Args:
        addon: Kodi addon instance
        show_nav_sidebar: Zobrazit navigacni sidebar
        nav_position: Pozice v navigaci
        auto_action: Automaticka akce pri startu (napr. 'hub_select:movies', 'hub_context:my_movies')
    """
    addon_path = addon.getAddonInfo('path')
    
    dialog = AIRecommendationHubDialog(
        "ai/DialogAIChatRecommendation.xml",
        addon_path,
        "Default",
        "1080i",
        show_nav_sidebar=show_nav_sidebar,
        nav_position=nav_position,
        auto_action=auto_action
    )
    dialog.doModal()
    del dialog
