"""
ChannelNameGenerator - Automatické generování názvů kanálů podle žánrů.

Podporuje dvě varianty:
1. Náhodný výběr z předdefinovaných kreativních názvů podle dominantního žánru
2. AI generování pomocí LLM API
"""

import random
import xbmc
import xbmcaddon

from wcs.ai.AIProvider import AIProvider


class ChannelNameGenerator:
    """
    Generator kreativnich nazvu kanalu/kin podle zanru.
    
    Podporuje dva typy:
    - 'tv': MyTV kanaly se serialy
    - 'cinema': MyCinema kina s filmy
    """
    
    # TMDb TV genre ID -> list of creative Czech channel names
    GENRE_NAMES = {
        10759: [  # Action & Adventure
            "Adrenalín", "Akční jízda", "Plný plyn", "Na ostří nože", "Nebezpečná zóna",
            "Hrdinsky", "V akci", "Napětí na maximum", "Bez limitu", "Výzva přijata"
        ],
        16: [  # Animation
            "Animáček", "Kreslené světy", "Barevné příběhy", "Animovaná magie", "Pohádkový svět",
            "Kresba a fantazie", "Živé kresby", "Animované dobrodružství", "Svět animace", "Kouzelná tužka"
        ],
        35: [  # Comedy
            "Smíchov", "Dobrá nálada", "Úsměv na rtech", "Veselé večery", "Lehké srdce",
            "Humor bez hranic", "Salvy smíchu", "Komediální mix", "Úlet dne", "Pro zasmání"
        ],
        80: [  # Crime
            "Stopy zločinu", "Noční hlídka", "Detektivní klub", "Temná strana", "Vyšetřování",
            "Za mřížemi", "Policejní stanice", "Zákon a spravedlnost", "Kriminální příběhy", "Stín podezření"
        ],
        99: [  # Documentary
            "Okno do světa", "Pohled zblízka", "Skutečné příběhy", "Za kulisami", "Dokument dne",
            "Poznání", "Svět kolem nás", "Fakta a objevy", "Realita", "Pod lupou"
        ],
        18: [  # Drama
            "Lidské osudy", "Hlubiny emocí", "Životní příběhy", "Drama večer", "Silné momenty",
            "Srdce na dlani", "Citová bouře", "Osudy lidí", "Mezi řádky", "Pravdivé drama"
        ],
        10751: [  # Family
            "Rodinné večery", "Pro celou rodinu", "Společné chvíle", "Rodinná pohoda", "Domácí kino",
            "U krbu", "Generace spolu", "Rodinný čas", "Pro malé i velké", "Všichni spolu"
        ],
        10762: [  # Kids
            "Dětský svět", "Pro nejmenší", "Pohádkový koutek", "Dětská radost", "Malí diváci",
            "Svět dětí", "Veselé dobrodružství", "Pro kluky a holky", "Dětský ráj", "Školáček"
        ],
        9648: [  # Mystery
            "Záhadné příběhy", "Tajemno", "Neznámé teritorium", "Mystérium", "Skryté pravdy",
            "Za závojem", "Nevysvětlitelné", "Temné tajemství", "Enigma", "Hádanka dne"
        ],
        10763: [  # News
            "Zpravodaj", "Aktuálně", "Ze světa", "Denní přehled", "Info kanál",
            "Události dne", "Novinky", "Zprávy 24", "Report", "Aktuální dění"
        ],
        10764: [  # Reality
            "Skutečný život", "Reality TV", "Bez scénáře", "Pravda a nic než pravda", "Za kamerou",
            "Autentické příběhy", "Realita show", "Bez přípravy", "Živě", "Nepředvídatelné"
        ],
        10765: [  # Sci-Fi & Fantasy
            "Za horizontem", "Budoucnost volá", "Jiné světy", "Fantazie a věda", "Vesmírná odysea",
            "Portál do neznáma", "Dimenze X", "Magie a technologie", "Nadpřirozeno", "Sci-Fi klub"
        ],
        10766: [  # Soap
            "Srdcové záležitosti", "Denní drama", "Láska a intriky", "Seriálové odpoledne", "Osudy hrdinů",
            "Emoce non-stop", "Rodinné vztahy", "Příběhy bez konce", "Srdce v plamenech", "Touhy a sny"
        ],
        10767: [  # Talk
            "Rozhovory", "Na slovíčko", "Talk show večer", "Hosté večera", "Debatní klub",
            "Na kávě s...", "Interview", "Povídání", "Otevřeně", "Face to face"
        ],
        10768: [  # War & Politics
            "Válečné příběhy", "Politická aréna", "Zákopy historie", "Moc a válka", "Bojové pole",
            "Historické konflikty", "Za frontou", "Politické drama", "Svět ve válce", "Boj o moc"
        ],
        37: [  # Western
            "Divoký západ", "Kovbojské příběhy", "Prairie", "Westernový klub", "Pod žhnoucím sluncem",
            "Revolveroví hrdinové", "Pouštní stopy", "Západ slunce", "Šerifova hvězda", "Cesta na západ"
        ]
    }
    
    # Fallback names for mixed genres or unknown genres
    FALLBACK_NAMES = [
        "Večerní program", "Můj mix", "Televize pro mě", "Oblíbené", "Moje volba",
        "Seriálový večer", "Bez názvu", "Nový kanál", "Kanál dle výběru", "Mix dne"
    ]
    
    # Genre ID to short Czech name (for combination display)
    GENRE_SHORT_NAMES = {
        10759: "Akce",
        16: "Animovany",
        35: "Komedie",
        80: "Krimi",
        99: "Dokument",
        18: "Drama",
        10751: "Rodinny",
        10762: "Detsky",
        9648: "Zahady",
        10763: "Zpravy",
        10764: "Reality",
        10765: "Sci-Fi",
        10766: "Telenovela",
        10767: "Talk Show",
        10768: "Valecny",
        37: "Western"
    }
    
    # ========== CINEMA (Movie) GENRE NAMES ==========
    # TMDb Movie genre IDs are different from TV genre IDs
    CINEMA_GENRE_NAMES = {
        28: [  # Action
            "Akce bez kompromisu", "Adrenalinka", "Bojove sceny", "Hrdinove v akci", "Exploze a honicky",
            "Misto cinu", "Na ostri noze", "Akcni maraton", "Bez milosti", "Pln plyn"
        ],
        12: [  # Adventure
            "Dobrodruzne vyprvy", "Za horizontem", "Neznama zeme", "Objevitele", "Velka cesta",
            "Expedition", "Za pokladem", "Dobrodruzna jizda", "Novy svet", "Vyprvy za sluncem"
        ],
        16: [  # Animation
            "Animovana magie", "Kreslenek", "Barevne svety", "Zive kresby", "Animovany sen",
            "Pohadkovy film", "Pixelova zabava", "Animace pro vsechny", "Kouzelny svet", "Tuzka a fantazie"
        ],
        35: [  # Comedy
            "Smichov", "Dobra nalada", "Usmev na rtech", "Vesele vecery", "Komedialni maraton",
            "Pro zasmani", "Salvy smichu", "Ulet filmu", "Humor bez hranic", "Lehky film"
        ],
        80: [  # Crime
            "Stopy zlocinu", "Zlocin a trest", "Temna strana", "Mafia", "Organizovany zlocin",
            "Svet podsveti", "Gangsterky", "Nocni misto", "Zlocinecke osudy", "Bez zakona"
        ],
        99: [  # Documentary
            "Okno do sveta", "Pohled zblizka", "Skutecn pribhy", "Za kulisami", "Dokument dne",
            "Realita", "Pod lupou", "Fakta a objevy", "Svet kolem nas", "Poznavaci film"
        ],
        18: [  # Drama
            "Lidske osudy", "Hlubiny emoci", "Zivotn pribhy", "Silne momenty", "Drama vecera",
            "Srdce na dlani", "Citova boure", "Mezi radky", "Pravdive drama", "Emocn film"
        ],
        10751: [  # Family
            "Pro celou rodinu", "Rodinne vecery", "Spolecne chvile", "Rodinna pohoda", "Domaci kino",
            "U krbu", "Generace spolu", "Pro male i velke", "Vsichni spolu", "Rodinny film"
        ],
        14: [  # Fantasy
            "Kouzelne svety", "Fantasy maraton", "Magie a mec", "Jine svety", "Portal do neznama",
            "Nadprirozeno", "Mytick pribhy", "Svt fantazie", "Zaklet kino", "Legendy ozvaji"
        ],
        36: [  # History
            "Historicke ody", "Z minulosti", "Velike dejiny", "Epoch", "Kronika dejin",
            "Staroveke pribhy", "Historicky maraton", "Minulost ozvaja", "Dejinne momenty", "Cas kralu"
        ],
        27: [  # Horror
            "Hruzostrn kino", "Nocni mury", "Temne stiny", "Strach v oci", "Krev a desv",
            "Hororova noc", "Nesnevni filmy", "Paranormalni", "Zlo cihajici", "Peklo v kine"
        ],
        10402: [  # Music
            "Hudebni filmy", "Melodie kina", "Hudb sceny", "Koncertn sal", "Ryt v obraze",
            "Zpivani pod hvezdami", "Muzikalov vecery", "Tance a pisne", "Hudebn magic", "Zvuk filmu"
        ],
        9648: [  # Mystery
            "Zahadn pribhy", "Tajemno", "Nezname teritorium", "Mysterium", "Skryte pravdy",
            "Za zavojem", "Nevysvetliteln", "Temne tajemstvi", "Enigma", "Hadanka vecera"
        ],
        10749: [  # Romance
            "Romanticke vecery", "Laska na platen", "Srdcove zale", "Romanticka komedie", "Lasky pribhy",
            "Pro zamilovane", "Valentynske kino", "Svet lask", "Romantick maraton", "Poliby na platen"
        ],
        878: [  # Science Fiction
            "Za horizontem", "Budoucnost vola", "Jin svety", "Vesmir odyssea", "Sci-Fi maraton",
            "Galaktick kino", "Techno sny", "Kosmicka stanice", "Futuristicke vize", "Dimenze X"
        ],
        10770: [  # TV Movie
            "Televizni film", "Domaci kino", "Vecerni film", "Televize pro me", "Filmovy vecer",
            "Prima film", "Domaci projekce", "TV premiera", "Vecerni pohoda", "Domc filmy"
        ],
        53: [  # Thriller
            "Napeti do posledni chvile", "Thrillerova noc", "Na ostri noze", "Srazejici dech", "Napinavy vecr",
            "Adrenalin v zilach", "Nebezpecna zona", "Temne tajemstvi", "Plot without escape", "Bez uniku"
        ],
        10752: [  # War
            "Valecne pribhy", "Zakopy historie", "Bojove pole", "Za frontou", "Svet ve valce",
            "Hrdinove valky", "Valecn epos", "Boj o moc", "Historicke konflikty", "Den D"
        ],
        37: [  # Western
            "Divoky zapad", "Kovbojske pribhy", "Prairie", "Westernovy maraton", "Pod zhnoucim sluncem",
            "Revolverovi hrdinove", "Poustni stopy", "Zapad slunce", "Serifova hvezda", "Cesta na zapad"
        ]
    }
    
    # Fallback names for MyCinema (movies)
    CINEMA_FALLBACK_NAMES = [
        "Moje kino", "Filmovy vecer", "Filmovy maraton", "Oblibene filmy", "Moje volba",
        "Vecerni projekce", "Nove kino", "Filmov mix", "Projekce dne", "Kino pro me"
    ]
    
    # Cinema genre short names (same IDs as CINEMA_GENRE_NAMES)
    CINEMA_GENRE_SHORT_NAMES = {
        28: "Akce",
        12: "Dobrodruzny",
        16: "Animovany",
        35: "Komedie",
        80: "Krimi",
        99: "Dokument",
        18: "Drama",
        10751: "Rodinny",
        14: "Fantasy",
        36: "Historicky",
        27: "Horor",
        10402: "Hudebni",
        9648: "Zahady",
        10749: "Romanticky",
        878: "Sci-Fi",
        10770: "TV Film",
        53: "Thriller",
        10752: "Valecny",
        37: "Western"
    }
    
    def __init__(self, addon=None):
        """
        Initialize the generator.
        
        Args:
            addon: xbmcaddon.Addon instance (optional, needed for AI mode)
        """
        self.addon = addon or xbmcaddon.Addon()
    
    def generate_from_genres(self, media_list, media_type='tv'):
        """
        Generate channel/cinema name based on dominant genres from media list.
        
        Uses random selection from predefined creative names.
        
        Args:
            media_list: List of media dicts with 'genre_ids' or 'genres' field
            media_type: 'tv' for MyTV channels, 'cinema' for MyCinema
            
        Returns:
            str: Generated name
        """
        log_prefix = '[MyCinema]' if media_type == 'cinema' else '[MyTV]'
        fallback_names = self.CINEMA_FALLBACK_NAMES if media_type == 'cinema' else self.FALLBACK_NAMES
        genre_names_map = self.CINEMA_GENRE_NAMES if media_type == 'cinema' else self.GENRE_NAMES
        genre_short_names = self.CINEMA_GENRE_SHORT_NAMES if media_type == 'cinema' else self.GENRE_SHORT_NAMES
        
        xbmc.log(f"{log_prefix} generate_from_genres called with {len(media_list) if media_list else 0} items", xbmc.LOGINFO)
        
        if not media_list:
            fallback = random.choice(fallback_names)
            xbmc.log(f"{log_prefix} No items, using fallback: {fallback}", xbmc.LOGINFO)
            return fallback
        
        # Collect all genre IDs from all media items
        genre_counts = {}
        for item in media_list:
            item_name = item.get('name') or item.get('title', 'Unknown')
            genre_ids = self._extract_genre_ids(item)
            for gid in genre_ids:
                genre_counts[gid] = genre_counts.get(gid, 0) + 1
        
        if not genre_counts:
            fallback = random.choice(fallback_names)
            xbmc.log(f"{log_prefix} No genres found, using fallback: {fallback}", xbmc.LOGINFO)
            return fallback
        
        # Sort by frequency (descending)
        sorted_genres = sorted(genre_counts.items(), key=lambda x: x[1], reverse=True)
        
        # Get top 1-2 genres
        top_genre_id = sorted_genres[0][0]
        top_genre_name = genre_short_names.get(top_genre_id, f'ID:{top_genre_id}')
        
        # Check if there's a second significant genre (at least 50% of top genre count)
        if len(sorted_genres) > 1:
            second_genre_id = sorted_genres[1][0]
            second_count = sorted_genres[1][1]
            top_count = sorted_genres[0][1]
            
            # If second genre is significant and different, use combination
            if second_count >= top_count * 0.5 and second_genre_id != top_genre_id:
                combined_name = self._generate_combined_name(top_genre_id, second_genre_id, media_type)
                xbmc.log(f"{log_prefix} Using combined name: {combined_name}", xbmc.LOGINFO)
                return combined_name
        
        # Single dominant genre - random selection
        single_name = self._generate_single_genre_name(top_genre_id, media_type)
        xbmc.log(f"{log_prefix} Using single genre name: {single_name}", xbmc.LOGINFO)
        return single_name
    
    def generate_from_ai(self, media_list, media_type='tv'):
        """
        Generate channel/cinema name using AI/LLM.
        
        Sends media names and genres to AI and asks for creative name.
        
        Args:
            media_list: List of media dicts
            media_type: 'tv' for MyTV channels, 'cinema' for MyCinema
            
        Returns:
            str: AI-generated name, or fallback if AI fails
        """
        fallback_names = self.CINEMA_FALLBACK_NAMES if media_type == 'cinema' else self.FALLBACK_NAMES
        genre_short_names = self.CINEMA_GENRE_SHORT_NAMES if media_type == 'cinema' else self.GENRE_SHORT_NAMES
        
        if not media_list:
            return random.choice(fallback_names)
        
        try:
            # Build context - media names and genres
            media_info = []
            for item in media_list:
                name = item.get('name') or item.get('title', '')
                genres = self._extract_genre_names(item, media_type)
                if name:
                    if genres:
                        media_info.append(f"- {name} ({genres})")
                    else:
                        media_info.append(f"- {name}")
            
            if not media_info:
                return random.choice(fallback_names)
            
            # Build prompt based on media type
            if media_type == 'cinema':
                prompt = f"""Jsi kreativni copywriter specializujici se na filmovy branding.

Tvym ukolem je vymyslet originalni a zapamatovatelny nazev pro nove soukrome kino. Toto kino bude promitat nasledujici filmy:

{chr(10).join(media_info)}

Zamysli se nad tim, co maji tyto filmy spolecneho:
- Jaka je jejich prevladajici atmosfera? (napinava, temna, odlehcena, dramaticka, futuristicka...)
- Jake emoce vyvolavaji? (strach, smich, napeti, dojeti, vzruseni...)
- Jaky typ divaka by si je vybral?
- Jake jsou jejich hlavni zanrove prvky?

Na zaklade teto analyzy vytvor nazev kina, ktery:
- Je v cestine
- Ma maximalne 3 slova
- Vystihuje podstatu a atmosferu nabizeneho obsahu
- Zni profesionalne jako skutecne kino
- Je snadno zapamatovatelny a originalni
- Muze obsahovat slovo "kino" ale nemusi

Odpovez POUZE nazvem kina, nic dalsiho nepridavej.

Nazev:"""
            else:
                prompt = f"""Jsi kreativni copywriter specializujici se na televizni branding.

Tvym ukolem je vymyslet originalni a zapamatovatelny nazev pro novy televizni kanal. Tento kanal bude vysilat nasledujici serialy:

{chr(10).join(media_info)}

Zamysli se nad tim, co maji tyto serialy spolecneho:
- Jaka je jejich prevladajici atmosfera? (napinava, temna, odlehcena, dramaticka, futuristicka...)
- Jake emoce vyvolavaji? (strach, smich, napeti, dojeti, vzruseni...)
- Jaky typ divaka by si je vybral?
- Jake jsou jejich hlavni zanrove prvky?

Na zaklade teto analyzy vytvor nazev kanalu, ktery:
- Je v cestine
- Ma maximalne 3 slova
- Vystihuje podstatu a atmosferu nabizeneho obsahu
- Zni profesionalne jako skutecny televizni kanal
- Je snadno zapamatovatelny a originalni
- Evokuje spravne ocekavani u divaka

Odpovez POUZE nazvem kanalu, nic dalsiho nepridavej.

Nazev:"""
            
            # Call AI
            ai_provider = AIProvider()
            response, usage = ai_provider.send_prompt(prompt)
            
            # Track AI usage
            if usage:
                try:
                    from wcs.ai.UsageTracker import track_ai_usage
                    provider = ai_provider.get_current_provider() or "AI"
                    model = ai_provider.get_current_model() or "Default"
                    usage_type = "cinema_name" if media_type == 'cinema' else "channel_name"
                    track_ai_usage(provider, model, usage, usage_type)
                except Exception as e:
                    xbmc.log(f"[ChannelNameGenerator] Usage tracking error: {e}", xbmc.LOGWARNING)
            
            if response:
                # Clean up response - remove quotes, extra whitespace, newlines
                name = response.strip().strip('"\'').strip()
                # Limit length
                if len(name) <= 30 and name:
                    log_prefix = '[MyCinema]' if media_type == 'cinema' else '[MyTV]'
                    xbmc.log(f"{log_prefix} AI generated name: {name}", xbmc.LOGINFO)
                    return name
            
            # Fallback to genre-based if AI failed or returned garbage
            return self.generate_from_genres(media_list, media_type)
            
        except Exception as e:
            xbmc.log(f"[ChannelNameGenerator] AI generation failed: {e}", xbmc.LOGERROR)
            return self.generate_from_genres(media_list, media_type)
    
    def _extract_genre_ids(self, series):
        """Extract genre IDs from a series dict."""
        # Try genre_ids first (TMDb API format)
        if 'genre_ids' in series and series['genre_ids']:
            return series['genre_ids']
        
        # Try genres (detailed format with objects)
        if 'genres' in series:
            genres = series['genres']
            if isinstance(genres, list):
                return [g.get('id') for g in genres if isinstance(g, dict) and g.get('id')]
        
        return []
    
    def _extract_genre_names(self, media, media_type='tv'):
        """Extract genre names as comma-separated string."""
        genre_short_names = self.CINEMA_GENRE_SHORT_NAMES if media_type == 'cinema' else self.GENRE_SHORT_NAMES
        
        genre_ids = self._extract_genre_ids(media)
        if genre_ids:
            names = [genre_short_names.get(gid, '') for gid in genre_ids[:2]]
            return ', '.join([n for n in names if n])
        
        # Try genre string directly
        if 'genre' in media and media['genre']:
            return media['genre']
        
        return ''
    
    def _generate_single_genre_name(self, genre_id, media_type='tv'):
        """Generate name for a single dominant genre.
        
        Format: "(genre name) - (creative name)"
        Example: "Drama - Lidske osudy"
        """
        genre_names_map = self.CINEMA_GENRE_NAMES if media_type == 'cinema' else self.GENRE_NAMES
        genre_short_names = self.CINEMA_GENRE_SHORT_NAMES if media_type == 'cinema' else self.GENRE_SHORT_NAMES
        fallback_names = self.CINEMA_FALLBACK_NAMES if media_type == 'cinema' else self.FALLBACK_NAMES
        
        genre_name = genre_short_names.get(genre_id, '')
        
        if genre_id in genre_names_map:
            creative_name = random.choice(genre_names_map[genre_id])
            if genre_name:
                full_name = f"{genre_name} - {creative_name}"
            else:
                full_name = creative_name
            return full_name
        
        # Fallback - no creative names for this genre
        fallback = random.choice(fallback_names)
        if genre_name:
            full_name = f"{genre_name} - {fallback}"
        else:
            full_name = fallback
        return full_name
    
    def _generate_combined_name(self, genre_id_1, genre_id_2, media_type='tv'):
        """Generate combined name for two dominant genres.
        
        Format: "(genre1) / (genre2) - (creative name from genre1)"
        Example: "Sci-Fi / Drama - Jine svety"
        """
        genre_names_map = self.CINEMA_GENRE_NAMES if media_type == 'cinema' else self.GENRE_NAMES
        genre_short_names = self.CINEMA_GENRE_SHORT_NAMES if media_type == 'cinema' else self.GENRE_SHORT_NAMES
        fallback_names = self.CINEMA_FALLBACK_NAMES if media_type == 'cinema' else self.FALLBACK_NAMES
        
        name1 = genre_short_names.get(genre_id_1, '')
        name2 = genre_short_names.get(genre_id_2, '')
        
        if name1 and name2:
            # Get creative name from the first (top) genre
            if genre_id_1 in genre_names_map:
                creative_name = random.choice(genre_names_map[genre_id_1])
            elif genre_id_2 in genre_names_map:
                creative_name = random.choice(genre_names_map[genre_id_2])
            else:
                creative_name = random.choice(fallback_names)
            
            full_name = f"{name1} / {name2} - {creative_name}"
            return full_name
        elif name1:
            return self._generate_single_genre_name(genre_id_1, media_type)
        elif name2:
            return self._generate_single_genre_name(genre_id_2, media_type)
        else:
            return random.choice(fallback_names)
