"""
ChannelNameGenerator - Automatické generování názvů kanálů podle žánrů.

Podporuje dvě varianty:
1. Náhodný výběr z předdefinovaných kreativních názvů podle dominantního žánru
2. AI generování pomocí LLM API
"""

import random
import xbmc
import xbmcaddon

from wcs.ai.AIProvider import AIProvider


class ChannelNameGenerator:
    """Generátor kreativních názvů kanálů podle žánrů seriálů."""
    
    # TMDb TV genre ID -> list of creative Czech channel names
    GENRE_NAMES = {
        10759: [  # Action & Adventure
            "Adrenalín", "Akční jízda", "Plný plyn", "Na ostří nože", "Nebezpečná zóna",
            "Hrdinsky", "V akci", "Napětí na maximum", "Bez limitu", "Výzva přijata"
        ],
        16: [  # Animation
            "Animáček", "Kreslené světy", "Barevné příběhy", "Animovaná magie", "Pohádkový svět",
            "Kresba a fantazie", "Živé kresby", "Animované dobrodružství", "Svět animace", "Kouzelná tužka"
        ],
        35: [  # Comedy
            "Smíchov", "Dobrá nálada", "Úsměv na rtech", "Veselé večery", "Lehké srdce",
            "Humor bez hranic", "Salvy smíchu", "Komediální mix", "Úlet dne", "Pro zasmání"
        ],
        80: [  # Crime
            "Stopy zločinu", "Noční hlídka", "Detektivní klub", "Temná strana", "Vyšetřování",
            "Za mřížemi", "Policejní stanice", "Zákon a spravedlnost", "Kriminální příběhy", "Stín podezření"
        ],
        99: [  # Documentary
            "Okno do světa", "Pohled zblízka", "Skutečné příběhy", "Za kulisami", "Dokument dne",
            "Poznání", "Svět kolem nás", "Fakta a objevy", "Realita", "Pod lupou"
        ],
        18: [  # Drama
            "Lidské osudy", "Hlubiny emocí", "Životní příběhy", "Drama večer", "Silné momenty",
            "Srdce na dlani", "Citová bouře", "Osudy lidí", "Mezi řádky", "Pravdivé drama"
        ],
        10751: [  # Family
            "Rodinné večery", "Pro celou rodinu", "Společné chvíle", "Rodinná pohoda", "Domácí kino",
            "U krbu", "Generace spolu", "Rodinný čas", "Pro malé i velké", "Všichni spolu"
        ],
        10762: [  # Kids
            "Dětský svět", "Pro nejmenší", "Pohádkový koutek", "Dětská radost", "Malí diváci",
            "Svět dětí", "Veselé dobrodružství", "Pro kluky a holky", "Dětský ráj", "Školáček"
        ],
        9648: [  # Mystery
            "Záhadné příběhy", "Tajemno", "Neznámé teritorium", "Mystérium", "Skryté pravdy",
            "Za závojem", "Nevysvětlitelné", "Temné tajemství", "Enigma", "Hádanka dne"
        ],
        10763: [  # News
            "Zpravodaj", "Aktuálně", "Ze světa", "Denní přehled", "Info kanál",
            "Události dne", "Novinky", "Zprávy 24", "Report", "Aktuální dění"
        ],
        10764: [  # Reality
            "Skutečný život", "Reality TV", "Bez scénáře", "Pravda a nic než pravda", "Za kamerou",
            "Autentické příběhy", "Realita show", "Bez přípravy", "Živě", "Nepředvídatelné"
        ],
        10765: [  # Sci-Fi & Fantasy
            "Za horizontem", "Budoucnost volá", "Jiné světy", "Fantazie a věda", "Vesmírná odysea",
            "Portál do neznáma", "Dimenze X", "Magie a technologie", "Nadpřirozeno", "Sci-Fi klub"
        ],
        10766: [  # Soap
            "Srdcové záležitosti", "Denní drama", "Láska a intriky", "Seriálové odpoledne", "Osudy hrdinů",
            "Emoce non-stop", "Rodinné vztahy", "Příběhy bez konce", "Srdce v plamenech", "Touhy a sny"
        ],
        10767: [  # Talk
            "Rozhovory", "Na slovíčko", "Talk show večer", "Hosté večera", "Debatní klub",
            "Na kávě s...", "Interview", "Povídání", "Otevřeně", "Face to face"
        ],
        10768: [  # War & Politics
            "Válečné příběhy", "Politická aréna", "Zákopy historie", "Moc a válka", "Bojové pole",
            "Historické konflikty", "Za frontou", "Politické drama", "Svět ve válce", "Boj o moc"
        ],
        37: [  # Western
            "Divoký západ", "Kovbojské příběhy", "Prairie", "Westernový klub", "Pod žhnoucím sluncem",
            "Revolveroví hrdinové", "Pouštní stopy", "Západ slunce", "Šerifova hvězda", "Cesta na západ"
        ]
    }
    
    # Fallback names for mixed genres or unknown genres
    FALLBACK_NAMES = [
        "Večerní program", "Můj mix", "Televize pro mě", "Oblíbené", "Moje volba",
        "Seriálový večer", "Bez názvu", "Nový kanál", "Kanál dle výběru", "Mix dne"
    ]
    
    # Genre ID to short Czech name (for combination display)
    GENRE_SHORT_NAMES = {
        10759: "Akce",
        16: "Animovaný",
        35: "Komedie",
        80: "Krimi",
        99: "Dokument",
        18: "Drama",
        10751: "Rodinný",
        10762: "Dětský",
        9648: "Záhady",
        10763: "Zprávy",
        10764: "Reality",
        10765: "Sci-Fi",
        10766: "Telenovela",
        10767: "Talk Show",
        10768: "Válečný",
        37: "Western"
    }
    
    def __init__(self, addon=None):
        """
        Initialize the generator.
        
        Args:
            addon: xbmcaddon.Addon instance (optional, needed for AI mode)
        """
        self.addon = addon or xbmcaddon.Addon()
    
    def generate_from_genres(self, series_list):
        """
        Generate channel name based on dominant genres from series list.
        
        Uses random selection from predefined creative names.
        
        Args:
            series_list: List of series dicts with 'genre_ids' or 'genres' field
            
        Returns:
            str: Generated channel name
        """
        xbmc.log(f"[ChannelNameGenerator] generate_from_genres called with {len(series_list) if series_list else 0} series", xbmc.LOGWARNING)
        
        if not series_list:
            fallback = random.choice(self.FALLBACK_NAMES)
            xbmc.log(f"[ChannelNameGenerator] No series, using fallback: {fallback}", xbmc.LOGWARNING)
            return fallback
        
        # Collect all genre IDs from all series
        genre_counts = {}
        for series in series_list:
            series_name = series.get('name', 'Unknown')
            genre_ids = self._extract_genre_ids(series)
            xbmc.log(f"[ChannelNameGenerator] Series '{series_name}' has genre_ids: {genre_ids}", xbmc.LOGWARNING)
            for gid in genre_ids:
                genre_counts[gid] = genre_counts.get(gid, 0) + 1
        
        xbmc.log(f"[ChannelNameGenerator] Genre counts: {genre_counts}", xbmc.LOGWARNING)
        
        if not genre_counts:
            fallback = random.choice(self.FALLBACK_NAMES)
            xbmc.log(f"[ChannelNameGenerator] No genres found, using fallback: {fallback}", xbmc.LOGWARNING)
            return fallback
        
        # Sort by frequency (descending)
        sorted_genres = sorted(genre_counts.items(), key=lambda x: x[1], reverse=True)
        xbmc.log(f"[ChannelNameGenerator] Sorted genres: {sorted_genres}", xbmc.LOGWARNING)
        
        # Get top 1-2 genres
        top_genre_id = sorted_genres[0][0]
        top_genre_name = self.GENRE_SHORT_NAMES.get(top_genre_id, f'ID:{top_genre_id}')
        xbmc.log(f"[ChannelNameGenerator] Top genre: {top_genre_name} (ID: {top_genre_id})", xbmc.LOGWARNING)
        
        # Check if there's a second significant genre (at least 50% of top genre count)
        if len(sorted_genres) > 1:
            second_genre_id = sorted_genres[1][0]
            second_count = sorted_genres[1][1]
            top_count = sorted_genres[0][1]
            second_genre_name = self.GENRE_SHORT_NAMES.get(second_genre_id, f'ID:{second_genre_id}')
            
            xbmc.log(f"[ChannelNameGenerator] Second genre: {second_genre_name} (ID: {second_genre_id}), count: {second_count}/{top_count}", xbmc.LOGWARNING)
            
            # If second genre is significant and different, use combination
            if second_count >= top_count * 0.5 and second_genre_id != top_genre_id:
                combined_name = self._generate_combined_name(top_genre_id, second_genre_id)
                xbmc.log(f"[ChannelNameGenerator] Using combined name: {combined_name}", xbmc.LOGWARNING)
                return combined_name
        
        # Single dominant genre - random selection
        single_name = self._generate_single_genre_name(top_genre_id)
        xbmc.log(f"[ChannelNameGenerator] Using single genre name: {single_name}", xbmc.LOGWARNING)
        return single_name
    
    def generate_from_ai(self, series_list):
        """
        Generate channel name using AI/LLM.
        
        Sends series names and genres to AI and asks for creative channel name.
        
        Args:
            series_list: List of series dicts
            
        Returns:
            str: AI-generated channel name, or fallback if AI fails
        """
        if not series_list:
            return random.choice(self.FALLBACK_NAMES)
        
        try:
            # Build context - series names and genres
            series_info = []
            for series in series_list:
                name = series.get('name', '')
                genres = self._extract_genre_names(series)
                if name:
                    if genres:
                        series_info.append(f"- {name} ({genres})")
                    else:
                        series_info.append(f"- {name}")
            
            if not series_info:
                return random.choice(self.FALLBACK_NAMES)
            
            # Build prompt
            prompt = f"""Jsi kreativní copywriter specializující se na televizní branding.

Tvým úkolem je vymyslet originální a zapamatovatelný název pro nový televizní kanál. Tento kanál bude vysílat následující seriály:

{chr(10).join(series_info)}

Zamysli se nad tím, co mají tyto seriály společného:
- Jaká je jejich převládající atmosféra? (napínavá, temná, odlehčená, dramatická, futuristická...)
- Jaké emoce vyvolávají? (strach, smích, napětí, dojetí, vzrušení...)
- Jaký typ diváka by si je vybral?
- Jaké jsou jejich hlavní žánrové prvky?

Na základě této analýzy vytvoř název kanálu, který:
- Je v češtině
- Má maximálně 3 slova
- Vystihuje podstatu a atmosféru nabízeného obsahu
- Zní profesionálně jako skutečný televizní kanál
- Je snadno zapamatovatelný a originální
- Evokuje správné očekávání u diváka

Odpověz POUZE názvem kanálu, nic dalšího nepřidávej.

Název:"""
            
            # Call AI
            ai_provider = AIProvider()
            response, usage = ai_provider.send_prompt(prompt)
            
            # Track AI usage
            if usage:
                try:
                    from wcs.ai.UsageTracker import track_ai_usage
                    provider = ai_provider.get_current_provider() or "AI"
                    model = ai_provider.get_current_model() or "Default"
                    track_ai_usage(provider, model, usage, "channel_name")
                except Exception as e:
                    xbmc.log(f"[ChannelNameGenerator] Usage tracking error: {e}", xbmc.LOGWARNING)
            
            if response:
                # Clean up response - remove quotes, extra whitespace, newlines
                name = response.strip().strip('"\'').strip()
                # Limit length
                if len(name) <= 30 and name:
                    xbmc.log(f"[ChannelNameGenerator] AI generated name: {name}", xbmc.LOGINFO)
                    return name
            
            # Fallback to genre-based if AI failed or returned garbage
            return self.generate_from_genres(series_list)
            
        except Exception as e:
            xbmc.log(f"[ChannelNameGenerator] AI generation failed: {e}", xbmc.LOGERROR)
            return self.generate_from_genres(series_list)
    
    def _extract_genre_ids(self, series):
        """Extract genre IDs from a series dict."""
        # Try genre_ids first (TMDb API format)
        if 'genre_ids' in series and series['genre_ids']:
            return series['genre_ids']
        
        # Try genres (detailed format with objects)
        if 'genres' in series:
            genres = series['genres']
            if isinstance(genres, list):
                return [g.get('id') for g in genres if isinstance(g, dict) and g.get('id')]
        
        return []
    
    def _extract_genre_names(self, series):
        """Extract genre names as comma-separated string."""
        genre_ids = self._extract_genre_ids(series)
        if genre_ids:
            names = [self.GENRE_SHORT_NAMES.get(gid, '') for gid in genre_ids[:2]]
            return ', '.join([n for n in names if n])
        
        # Try genre string directly
        if 'genre' in series and series['genre']:
            return series['genre']
        
        return ''
    
    def _generate_single_genre_name(self, genre_id):
        """Generate name for a single dominant genre.
        
        Format: "(genre name) - (creative name)"
        Example: "Drama - Lidské osudy"
        """
        genre_name = self.GENRE_SHORT_NAMES.get(genre_id, '')
        
        if genre_id in self.GENRE_NAMES:
            creative_name = random.choice(self.GENRE_NAMES[genre_id])
            if genre_name:
                full_name = f"{genre_name} - {creative_name}"
            else:
                full_name = creative_name
            xbmc.log(f"[ChannelNameGenerator] _generate_single_genre_name({genre_id}) -> {full_name}", xbmc.LOGWARNING)
            return full_name
        
        # Fallback - no creative names for this genre
        fallback = random.choice(self.FALLBACK_NAMES)
        if genre_name:
            full_name = f"{genre_name} - {fallback}"
        else:
            full_name = fallback
        xbmc.log(f"[ChannelNameGenerator] _generate_single_genre_name({genre_id}) -> fallback: {full_name}", xbmc.LOGWARNING)
        return full_name
    
    def _generate_combined_name(self, genre_id_1, genre_id_2):
        """Generate combined name for two dominant genres.
        
        Format: "(genre1) / (genre2) - (creative name from genre1)"
        Example: "Sci-Fi / Drama - Jiné světy"
        """
        name1 = self.GENRE_SHORT_NAMES.get(genre_id_1, '')
        name2 = self.GENRE_SHORT_NAMES.get(genre_id_2, '')
        
        if name1 and name2:
            # Get creative name from the first (top) genre
            if genre_id_1 in self.GENRE_NAMES:
                creative_name = random.choice(self.GENRE_NAMES[genre_id_1])
            elif genre_id_2 in self.GENRE_NAMES:
                creative_name = random.choice(self.GENRE_NAMES[genre_id_2])
            else:
                creative_name = random.choice(self.FALLBACK_NAMES)
            
            full_name = f"{name1} / {name2} - {creative_name}"
            xbmc.log(f"[ChannelNameGenerator] _generate_combined_name({genre_id_1}, {genre_id_2}) -> {full_name}", xbmc.LOGWARNING)
            return full_name
        elif name1:
            return self._generate_single_genre_name(genre_id_1)
        elif name2:
            return self._generate_single_genre_name(genre_id_2)
        else:
            return random.choice(self.FALLBACK_NAMES)
