# wcs/ai/ChannelHistory.py
"""
Správa historie sledování pro jednotlivé MyTV kanály.
Každý kanál má vlastní JSON soubor s progressem seriálů.
"""

import os
import json
import xbmc
import xbmcvfs
import xbmcaddon


def _get_history_dir():
    """Vrátí cestu k adresáři s historiemi kanálů."""
    profile = xbmcvfs.translatePath(xbmcaddon.Addon().getAddonInfo('profile'))
    history_dir = os.path.join(profile, 'mytv_history')
    if not os.path.exists(history_dir):
        xbmcvfs.mkdirs(history_dir)
    return history_dir


def _get_history_file(channel_id):
    """Vrátí cestu k JSON souboru historie pro daný kanál."""
    return os.path.join(_get_history_dir(), f'{channel_id}.json')


def load_channel_history(channel_id):
    """
    Načte historii sledování pro daný kanál.
    
    Returns:
        dict: Mapa {series_id: {"season": int, "episode": int}}
    """
    file_path = _get_history_file(channel_id)
    if not os.path.exists(file_path):
        return {}
    
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            return json.load(f)
    except (IOError, json.JSONDecodeError) as e:
        xbmc.log(f'[MyTV] Error loading channel history {channel_id}: {e}', xbmc.LOGWARNING)
        return {}


def save_channel_history(channel_id, history):
    """
    Uloží historii sledování pro daný kanál.
    
    Args:
        channel_id: ID kanálu
        history: dict {series_id: {"season": int, "episode": int}}
    """
    file_path = _get_history_file(channel_id)
    try:
        with open(file_path, 'w', encoding='utf-8') as f:
            json.dump(history, f, ensure_ascii=False, indent=2)
    except IOError as e:
        xbmc.log(f'[MyTV] Error saving channel history {channel_id}: {e}', xbmc.LOGERROR)


def update_series_progress(channel_id, series_id, season, episode):
    """
    Aktualizuje progress seriálu v kanálu.
    
    Args:
        channel_id: ID kanálu
        series_id: TMDb ID seriálu (bude převedeno na string)
        season: Číslo sezóny
        episode: Číslo epizody
    """
    history = load_channel_history(channel_id)
    history[str(series_id)] = {
        'season': int(season),
        'episode': int(episode)
    }
    save_channel_history(channel_id, history)
    xbmc.log(f'[MyTV] Updated progress: channel={channel_id}, series={series_id}, S{season:02d}E{episode:02d}', xbmc.LOGINFO)


def get_series_progress(channel_id, series_id):
    """
    Vrátí aktuální progress seriálu v kanálu.
    
    Returns:
        tuple: (season, episode) nebo (1, 0) pokud neexistuje
    """
    history = load_channel_history(channel_id)
    progress = history.get(str(series_id), {})
    return (
        progress.get('season', 1),
        progress.get('episode', 0)
    )


def delete_channel_history(channel_id):
    """Smaže historii kanálu (volat při mazání kanálu)."""
    file_path = _get_history_file(channel_id)
    if os.path.exists(file_path):
        try:
            os.remove(file_path)
            xbmc.log(f'[MyTV] Deleted channel history: {channel_id}', xbmc.LOGINFO)
            return True
        except OSError as e:
            xbmc.log(f'[MyTV] Error deleting channel history {channel_id}: {e}', xbmc.LOGERROR)
    return False


def migrate_from_global_history(channel_id, series_ids, addon):
    """
    Migruje progress seriálů z globální historie do kanálové.
    Volat pouze jednou při prvním použití kanálu.
    
    Args:
        channel_id: ID kanálu
        series_ids: Seznam TMDb ID seriálů v kanálu
        addon: Kodi addon instance
    """
    from wcs import user_data
    
    # Načíst globální historii
    global_history = user_data.load_recently_played(addon)
    if not global_history:
        return
    
    # Načíst existující kanálovou historii
    channel_history = load_channel_history(channel_id)
    
    # Převést series_ids na stringy pro porovnání
    series_ids_str = {str(sid) for sid in series_ids}
    
    migrated_count = 0
    for item in global_history:
        item_id = str(item.get('tmdb_id', ''))
        media_type = item.get('media_type', '')
        
        # Zkontrolovat, zda je tento seriál v kanálu
        if item_id in series_ids_str and media_type in ('tv', 'episode', 'series'):
            # Mít season a episode
            if 'season' in item and 'episode' in item:
                # Přidat pouze pokud ještě neexistuje (nebo je novější)
                existing = channel_history.get(item_id, {})
                existing_s = existing.get('season', 0)
                existing_e = existing.get('episode', 0)
                new_s = int(item['season'])
                new_e = int(item['episode'])
                
                # Porovnat - novější = vyšší sezóna nebo vyšší epizoda ve stejné sezóně
                if new_s > existing_s or (new_s == existing_s and new_e > existing_e):
                    channel_history[item_id] = {
                        'season': new_s,
                        'episode': new_e
                    }
                    migrated_count += 1
    
    if migrated_count > 0:
        save_channel_history(channel_id, channel_history)
        xbmc.log(f'[MyTV] Migrated {migrated_count} series progress to channel {channel_id}', xbmc.LOGINFO)


def check_and_migrate_if_needed(channel_id, series_list, addon):
    """
    Zkontroluje, zda kanál potřebuje migraci a provede ji.
    Volat při načítání programu kanálu.
    
    Args:
        channel_id: ID kanálu
        series_list: Seznam seriálů v kanálu (list of dicts s 'id')
        addon: Kodi addon instance
    """
    history_file = _get_history_file(channel_id)
    
    # Migrace pouze pokud soubor neexistuje (nový kanál nebo první spuštění)
    if not os.path.exists(history_file):
        series_ids = [s.get('id') for s in series_list if s.get('id')]
        if series_ids:
            migrate_from_global_history(channel_id, series_ids, addon)
