# wcs/user_data.py

import os
import io
import json
import traceback
import xbmcgui
import xbmc
import xbmcaddon
import datetime

# Získáme profilovou cestu zde, aby byla dostupná v celém modulu
_profile = xbmcaddon.Addon().getAddonInfo('profile')

# Správa uživatelských dat, nastavení, historie a oblíbených seriálů (bude implementováno v dalších krocích) 

# Načtení historie hledání

def load_search_history_setting(addon, key):
    data = addon.getSetting(key)
    if data:
        try:
            return json.loads(data)
        except Exception:
            return []
    return []

def store_search_history_setting(addon, key, what, max_size):
    if not what:
        return
    history = load_search_history_setting(addon, key)
    if what in history:
        history.remove(what)
    history = [what] + history
    if len(history) > max_size:
        history = history[:max_size]
    addon.setSetting(key, json.dumps(history))

def remove_search_history_setting(addon, key, what):
    if not what:
        return
    history = load_search_history_setting(addon, key)
    if what in history:
        history.remove(what)
        addon.setSetting(key, json.dumps(history))

def clear_search_history_setting(addon, key):
    addon.setSetting(key, json.dumps([]))

# Původní funkce přesměrovat na nové (kvůli kompatibilitě s UI)
def load_search_history(profile_path, search_history_file):
    addon = xbmcaddon.Addon()
    if 'universal' in search_history_file:
        key = 'search_history_universal'
    elif 'tmdb' in search_history_file:
        key = 'search_history_tmdb'
    else:
        key = 'search_history_tv'
    return load_search_history_setting(addon, key)

def store_search_history(what, profile_path, search_history_file, max_size):
    addon = xbmcaddon.Addon()
    if 'universal' in search_history_file:
        key = 'search_history_universal'
    elif 'tmdb' in search_history_file:
        key = 'search_history_tmdb'
    else:
        key = 'search_history_tv'
    store_search_history_setting(addon, key, what, max_size)

def remove_search_history(what, profile_path, search_history_file):
    addon = xbmcaddon.Addon()
    if 'universal' in search_history_file:
        key = 'search_history_universal'
    elif 'tmdb' in search_history_file:
        key = 'search_history_tmdb'
    else:
        key = 'search_history_tv'
    remove_search_history_setting(addon, key, what)

def clear_search_history(profile_path, search_history_file):
    addon = xbmcaddon.Addon()
    if 'universal' in search_history_file:
        key = 'search_history_universal'
    elif 'tmdb' in search_history_file:
        key = 'search_history_tmdb'
    else:
        key = 'search_history_tv'
    clear_search_history_setting(addon, key)

# Správa TMDb seriálů

def load_tmdb_series(addon):
    series_list = addon.getSetting('tmdb_series')
    if series_list:
        return json.loads(series_list)
    return []

def store_tmdb_series(addon, series_list):
    addon.setSetting('tmdb_series', json.dumps(series_list))

def remove_tmdb_series_by_index(addon, idx):
    """Odstraní seriál podle indexu ze seznamu 'Moje seriály'. Vrací True při úspěchu, jinak False."""
    series = load_tmdb_series(addon)
    if not (0 <= idx < len(series)):
        return False
    series.pop(idx)
    store_tmdb_series(addon, series)
    return True

# Správa TMDb filmů

def load_tmdb_movies(addon):
    movies_list = addon.getSetting('tmdb_movies')
    if movies_list:
        return json.loads(movies_list)
    return []

def store_tmdb_movies(addon, movies_list):
    addon.setSetting('tmdb_movies', json.dumps(movies_list))

def remove_tmdb_movie_by_index(addon, idx):
    """Odstraní film podle indexu ze seznamu 'Moje filmy'. Vrací True při úspěchu, jinak False."""
    movies = load_tmdb_movies(addon)
    if not (0 <= idx < len(movies)):
        return False
    movies.pop(idx)
    store_tmdb_movies(addon, movies)
    return True

# Správa oblíbených herců

def load_favorite_actors(addon):
    actors_list = addon.getSetting('favorite_actors')
    if actors_list:
        return json.loads(actors_list)
    return []

def store_favorite_actors(addon, actors_list):
    addon.setSetting('favorite_actors', json.dumps(actors_list))

def check_actor_exists(actor_id, actors_list):
    """Kontroluje, zda herec s daným ID již existuje v seznamu."""
    for actor in actors_list:
        if actor.get('id') == actor_id:
            return True
    return False

def remove_favorite_actor_by_index(addon, idx):
    """Odstraní herce podle indexu ze seznamu oblíbených. Vrací True při úspěchu, jinak False."""
    actors = load_favorite_actors(addon)
    if not (0 <= idx < len(actors)):
        return False
    actors.pop(idx)
    store_favorite_actors(addon, actors)
    return True

# Správa oblíbených režisérů

def load_favorite_directors(addon):
    directors_list = addon.getSetting('favorite_directors')
    if directors_list:
        return json.loads(directors_list)
    return []

def store_favorite_directors(addon, directors_list):
    addon.setSetting('favorite_directors', json.dumps(directors_list))

def check_director_exists(director_id, directors_list):
    """Kontroluje, zda režisér s daným ID již existuje v seznamu."""
    for director in directors_list:
        if director.get('id') == director_id:
            return True
    return False

def remove_favorite_director_by_index(addon, idx):
    """Odstraní režiséra podle indexu ze seznamu oblíbených. Vrací True při úspěchu, jinak False."""
    directors = load_favorite_directors(addon)
    if not (0 <= idx < len(directors)):
        return False
    directors.pop(idx)
    store_favorite_directors(addon, directors)
    return True

def _check_credentials_and_prompt(addon, on_success_callback=None):
    """
    Zkontroluje přihlašovací údaje. Pokud chybí, zobrazí dialog.
    Vždy zavolá `on_success_callback` po úspěšném přihlášení nebo pokud je uživatel již přihlášen.
    Vrací `True`, pokud je uživatel přihlášen (nebo se právě přihlásil), jinak `False`.
    """
    ws_user = addon.getSetting('wsuser')
    ws_pass = addon.getSetting('wspass')

    if ws_user and ws_pass:
        if on_success_callback:
            on_success_callback()
        return True
    
    from wcs.login.DialogWebshareLogin import show_webshare_login_dialog
    
    # Předáme callback přímo do dialogu.
    # Dialog se postará o jeho zavolání po úspěšném přihlášení.
    show_webshare_login_dialog(addon, on_success_callback)
    
    # Funkce nyní nevrací stav přihlášení, o to se postará callback.
    # Vracíme False, protože v momentě volání není jisté, zda se uživatel přihlásí.
    # Logika pokračování je plně v rukou callbacku.
    return False

# --- Oblíbení Scenáristé ---

FAVORITE_WRITERS_FILE = os.path.join(_profile, 'favorite_writers.json')

def load_favorite_writers(addon):
    """Načte seznam oblíbených scénáristů z JSON souboru."""
    try:
        if os.path.exists(FAVORITE_WRITERS_FILE):
            with open(FAVORITE_WRITERS_FILE, 'r', encoding='utf-8') as f:
                return json.load(f)
        else:
            return []
    except (IOError, ValueError, json.JSONDecodeError) as e:
        xbmc.log(f'Error loading favorite writers: {e}', level=xbmc.LOGERROR)
        return []

def store_favorite_writers(addon, writers):
    """Uloží seznam oblíbených scénáristů do JSON souboru."""
    try:
        if not os.path.exists(_profile):
            os.makedirs(_profile)
        with open(FAVORITE_WRITERS_FILE, 'w', encoding='utf-8') as f:
            json.dump(writers, f, ensure_ascii=False, indent=4)
    except (IOError, TypeError) as e:
        xbmc.log(f'Error storing favorite writers: {e}', level=xbmc.LOGERROR)

def check_writer_exists(writer_id, writers_list):
    """Zkontroluje, zda scénárista s daným ID již existuje v seznamu."""
    return any(writer['id'] == writer_id for writer in writers_list)

def remove_favorite_writer_by_index(addon, idx):
    """Odstraní scénáristu podle indexu ze seznamu oblíbených. Vrací True při úspěchu, jinak False."""
    writers = load_favorite_writers(addon)
    if not (0 <= idx < len(writers)):
        return False
    writers.pop(idx)
    store_favorite_writers(addon, writers)
    return True

# --- Funkce pro obecné ukládání/načítání (pro budoucí použití) --- 



# --- Naposledy přehrané ---

# Starý JSON soubor pro migraci
_RECENTLY_PLAYED_FILE_OLD = os.path.join(_profile, 'recently_played.json')

def load_recently_played(addon):
    """Načte seznam naposledy přehraných položek z Kodi settings."""
    # Nejdřív zkusit načíst z nového úložiště (settings)
    data = addon.getSetting('recently_played')
    if data:
        try:
            return json.loads(data)
        except (ValueError, json.JSONDecodeError) as e:
            xbmc.log(f'[plugin.video.milionar] Error parsing recently played from settings: {e}', level=xbmc.LOGERROR)
            return []
    
    # Fallback: Migrace ze starého JSON souboru
    if os.path.exists(_RECENTLY_PLAYED_FILE_OLD):
        try:
            with open(_RECENTLY_PLAYED_FILE_OLD, 'r', encoding='utf-8') as f:
                old_data = json.load(f)
            if old_data:
                # Migrovat do nového úložiště
                store_recently_played(addon, old_data)
                # Smazat starý soubor
                try:
                    os.remove(_RECENTLY_PLAYED_FILE_OLD)
                    xbmc.log('[plugin.video.milionar] Migrated recently_played.json to settings', level=xbmc.LOGINFO)
                except:
                    pass
                return old_data
        except (IOError, ValueError, json.JSONDecodeError) as e:
            xbmc.log(f'[plugin.video.milionar] Error loading old recently played JSON: {e}', level=xbmc.LOGERROR)
    
    return []

def store_recently_played(addon, items):
    """Uloží seznam naposledy přehraných položek do Kodi settings."""
    try:
        addon.setSetting('recently_played', json.dumps(items, ensure_ascii=False))
    except (TypeError) as e:
        xbmc.log(f'[plugin.video.milionar] Error storing recently played: {e}', level=xbmc.LOGERROR)

def add_recently_played_item(addon, item):
    """
    Přidá položku na začátek seznamu naposledy přehraných.
    Omezuje velikost seznamu na 50 položek.
    Duplicitní položky (stejné ID/titul) přesune na začátek.
    """
    items = load_recently_played(addon)
    
    # Identifikace položky (pro duplicitu) - stačí porovnat tmdb_id a typ, nebo titul pokud ID chybí
    tmdb_id = item.get('tmdb_id')
    title = item.get('title')
    media_type = item.get('media_type', 'movie')
    
    # Odstranění existující stejné položky
    new_items = []
    for existing in items:
        ex_id = existing.get('tmdb_id')
        ex_title = existing.get('title')
        ex_type = existing.get('media_type', 'movie')
        
        # Shoda ID a typu
        if tmdb_id and ex_id and str(tmdb_id) == str(ex_id) and media_type == ex_type:
            continue
        # Shoda titulu (fallback pro položky bez ID, např. manuální hledání)
        if (not tmdb_id or not ex_id) and title == ex_title:
            continue
            
        new_items.append(existing)
    
    # Přidání nové na začátek (včetně timestampu)
    item['timestamp'] = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    new_items.insert(0, item)
    
    # Limit na 50 (odpovídá max nastavení pro AI kontext)
    if len(new_items) > 50:
        new_items = new_items[:50]
        
    store_recently_played(addon, new_items)

# Starý AI usage systém byl odstraněn - používáme nyní ai_usage_tracker.py 