"""
Google AI (Gemini) API handler
Kompletní implementace pro Google AI Gemini API podle oficiální dokumentace
"""

import os
import requests
import xbmcaddon
import xbmc
import json
import time
from wcs.ai.DialogAIDebugRequest import show_api_request_debug, show_api_response_debug, show_api_error_debug

# Načtení nastavení addonu
addon = xbmcaddon.Addon()
API_BASE_URL = "https://generativelanguage.googleapis.com"
API_VERSION = "v1beta"

def get_current_model():
    """Vrací aktuálně vybraný Google AI model z nastavení."""
    addon = xbmcaddon.Addon()
    selected_model = addon.getSetting("google_model")
    if not selected_model:
        selected_model = "gemini-2.5-flash"  # Výchozí model
    return selected_model

def send_prompt(prompt, model=None, **kwargs) -> (str, dict):
    """
    Odešle prompt přes Google AI (Gemini) API.
    
    Args:
        prompt (str): Text prompt k odeslání
        model (str, optional): Název modelu. Pokud není specifikován, použije se model z nastavení.
    
    Returns:
        tuple: (response_text: str, usage: dict)
    """
    addon = xbmcaddon.Addon()
    api_key = addon.getSetting("google_api_key")
    
    if not api_key:
        raise ValueError("Google AI API klíč není nastaven.")
    
    # Extra data pro debug
    context_titles = kwargs.get('context_titles')
    context_limit = kwargs.get('context_limit')
    system_prompt = kwargs.get('system_prompt')
    
    if not model:
        model = get_current_model()
    
    # Načtení parametrů z nastavení
    temperature = float(addon.getSetting("google_temperature") or "1.0")
    top_p = float(addon.getSetting("google_top_p") or "0.95")
    top_k = int(addon.getSetting("google_top_k") or "40")
    max_tokens = int(addon.getSetting("google_max_tokens") or "2048")
    
    # Sestavení URL pro Google AI API
    url = f"{API_BASE_URL}/{API_VERSION}/models/{model}:generateContent"
    
    headers = {
        "Content-Type": "application/json"
    }
    
    # === SESTAVENÍ STANDARDNÍCH MESSAGES ===
    messages = []
    
    messages_list = kwargs.get('messages', None)
    if messages_list:
        # Zkontrolovat jestli už messages obsahují system roli
        has_system = any(msg.get('role') == 'system' for msg in messages_list)
        
        # Přidat system prompt jako první, pokud není v messages a máme ho
        if system_prompt and not has_system:
            messages.append({"role": "system", "content": system_prompt})
            
        for msg in messages_list:
            messages.append({"role": msg['role'], "content": msg['content']})
    else:
        # Jednoduchý prompt bez multi-turn
        if system_prompt:
            messages.append({"role": "system", "content": system_prompt})
        messages.append({"role": "user", "content": prompt})
    
    # === TRANSFORMACE NA GOOGLE API FORMÁT ===
    # Google používá contents pole s role: "user" nebo "model"
    contents = []
    for msg in messages:
        if msg['role'] == 'system':
            continue  # System prompt jde do systemInstruction
        role = "model" if msg['role'] == 'assistant' else "user"
        contents.append({
            "role": role,
            "parts": [{"text": msg['content']}]
        })
    
    # Získát system prompt z messages (první system message)
    final_system_prompt = None
    for msg in messages:
        if msg['role'] == 'system':
            final_system_prompt = msg['content']
            break
    
    data = {
        "contents": contents,
        "generationConfig": {
            "temperature": temperature,
            "topP": top_p,
            "topK": top_k,
            "maxOutputTokens": max_tokens,
            "candidateCount": 1
        }
    }
    
    # Přidat system prompt jako systemInstruction pokud je definován
    if final_system_prompt:
        data["systemInstruction"] = {
            "parts": [{"text": final_system_prompt}]
        }
    
    # JSON mode - vynutit JSON response formát
    json_mode = kwargs.get('json_mode', False)
    if json_mode:
        data["generationConfig"]["responseMimeType"] = "application/json"
    
    # Přidání API klíče do URL parametrů
    url_with_key = f"{url}?key={api_key}"
    
    xbmc.log(f"[wcs][google_api] Model: {model}", xbmc.LOGINFO)
    
    # === DEBUG PAYLOAD - messages už obsahují system pokud existuje ===
    debug_payload = {
        "messages": messages,
        "model": model,
        "temperature": temperature,
        "max_tokens": max_tokens
    }
    
    # Debug: Zobrazí standardní request před odesláním
    if not show_api_request_debug("Google", model, debug_payload, context_titles=context_titles, context_limit=context_limit):
        raise Exception("Odeslání API requestu bylo zrušeno uživatelem.")
    
    start_time = time.time()
    response = requests.post(url_with_key, headers=headers, json=data, timeout=120)
    end_time = time.time()
    latency = end_time - start_time
    
    xbmc.log(f"[wcs][google_api] Status: {response.status_code}", xbmc.LOGINFO)
    
    # Debug: Zobrazí raw response po obdržení
    try:
        response_data = response.json()
        show_api_response_debug("Google", model, response_data, latency=latency)
    except:
        show_api_response_debug("Google", model, {"raw_text": response.text}, latency=latency)
    
    response.raise_for_status()
    result = response.json()
    
    # Extract usage data (Google format -> standard format)
    usage = {}
    if 'usageMetadata' in result:
        meta = result['usageMetadata']
        usage = {
            'prompt_tokens': meta.get('promptTokenCount', 0),
            'completion_tokens': meta.get('candidatesTokenCount', 0),
            'total_tokens': meta.get('totalTokenCount', 0)
        }
    
    # Extract content
    content = ""
    if 'candidates' in result and len(result['candidates']) > 0:
        candidate = result['candidates'][0]
        if 'content' in candidate and 'parts' in candidate['content']:
            parts = candidate['content']['parts']
            if len(parts) > 0 and 'text' in parts[0]:
                content = parts[0]['text']
    
    if kwargs.get('include_raw', False):
        return content, usage, result, latency
    
    return content, usage

def validate_api_key():
    """
    Ověří platnost API klíče.
    
    Returns:
        bool: True pokud je klíč platný, False jinak
    """
    addon = xbmcaddon.Addon()
    api_key = addon.getSetting("google_api_key")
    
    if not api_key:
        return False
    
    try:
        # Jednoduchý test
        url = f"{API_BASE_URL}/{API_VERSION}/models"
        response = requests.get(f"{url}?key={api_key}", timeout=10)
        response.raise_for_status()
        return True
    except:
        return False

def get_available_models():
    """Vrací seznam dostupných Google AI modelů."""
    return [
        "gemini-2.5-flash",
        "gemini-2.5-pro", 
        "gemini-2.0-flash",
        "gemini-2.0-flash-lite",
        "gemini-1.5-pro",
        "gemini-1.5-flash",
        "gemini-1.5-flash-8b"
    ]

def get_model_info(model_name):
    """Vrací informace o konkrétním modelu."""
    model_info = {
        "gemini-2.5-flash": {"description": "Nejnovější rychlý model s vylepšenou kvalitou", "context_length": "1M tokens"},
        "gemini-2.5-pro": {"description": "Nejnovější vysoce výkonný model", "context_length": "2M tokens"},
        "gemini-2.0-flash": {"description": "Rychlý multimodální model", "context_length": "1M tokens"},
        "gemini-2.0-flash-lite": {"description": "Lehčí verze pro rychlé úkoly", "context_length": "32K tokens"},
        "gemini-1.5-pro": {"description": "Vysoce výkonný model", "context_length": "2M tokens"},
        "gemini-1.5-flash": {"description": "Rychlý a efektivní model", "context_length": "1M tokens"},
        "gemini-1.5-flash-8b": {"description": "Kompaktní model", "context_length": "1M tokens"}
    }
    return model_info.get(model_name, {"description": "Neznámý model", "context_length": "Neznámá"})