import xbmc
import xbmcaddon
import random
from wcs.ai import AIHandler as ai_handler
from wcs import user_data
from wcs.ai.AIProvider import ai_provider


class QuizGameController:
    """
    Kontrolér specificky pro kvízové hry.
    Obsahuje logiku pro otázky a odpovědi na téma filmů/seriálů.
    """
    
    def __init__(self, title="", plot="", genre="", media_type="movie", episode_title="", season_number=None, episode_number=None, actor_data=None):
        self.addon = xbmcaddon.Addon()
        
        # Informace o médiu
        self.title = title
        self.plot = plot
        self.genre = genre
        self.media_type = media_type  # "movie" nebo "series"
        self.episode_title = episode_title  # Pro seriály
        self.season_number = season_number  # Pro seriály
        self.episode_number = episode_number  # Pro seriály
        
        # Data o herci
        self.actor_data = actor_data or {}
        
        # Základní herní stav
        self.game_state = "intro"
        self.game_type = "quiz"
        
        # UI callbacks
        self.update_display_callback = None
        self.update_buttons_callback = None
        self.update_sidebar_callback = None
        self.show_error_callback = None
        self.set_focus_callback = None
        self.start_countdown_callback = None
        self.close_dialog_callback = None
        
        # Quiz-specifické vlastnosti
        self.quiz_score = 0
        self.quiz_question_number = 0
        self.quiz_total_questions = 5
        self.current_correct_answer = None
        self.current_question_text = ""
        self.selected_answer = ""
        self.selected_answer_index = -1
        
        # Quiz batch - všechny otázky najednou
        self.quiz_questions = []  # Seznam všech otázek načtených najednou
        self.quiz_loaded = False  # Zda jsou otázky již načtené
        self.current_choices = []  # Pro kompatibilitu
    
    def set_ui_callbacks(self, update_display, update_buttons, update_sidebar, show_error, set_focus=None, start_countdown=None, close_dialog=None):
        """Nastaví callback funkce pro komunikaci s UI."""
        self.update_display_callback = update_display
        self.update_buttons_callback = update_buttons
        self.update_sidebar_callback = update_sidebar
        self.show_error_callback = show_error
        self.set_focus_callback = set_focus
        self.start_countdown_callback = start_countdown
        self.close_dialog_callback = close_dialog

    def set_return_to_menu_callback(self, callback):
        """Nastaví callback pro návrat na hlavní menu."""
        self.return_to_menu_callback = callback

    def _add_ai_info_header(self, text: str) -> str:
        """Přidá informaci o AI modelu na začátek textu velmi malým písmem."""
        # AI info odstraněno na požádání uživatele
        return text
        
    def _track_ai_usage(self, usage, usage_type):
        """Společná metoda pro sledování AI usage."""
        if usage:
            from wcs.ai.UsageTracker import track_ai_usage
            provider = ai_provider.get_current_provider()
            model = ai_provider.get_current_model()
            track_ai_usage(provider, model, usage, usage_type)
        
    def start_specific_game(self):
        """Spustí kvízovou hru s welcome screenem."""
        xbmc.log("[plugin.video.milionar] QuizGameController.start_specific_game called", level=xbmc.LOGINFO)
        self.game_state = "quiz_welcome"
        self.quiz_score = 0
        self.quiz_question_number = 0  # Začínáme od 0 pro welcome screen
        
        # Reset batch stavu
        self.quiz_questions = []
        self.quiz_loaded = False
        
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
            
        # Zobrazit welcome screen
        self.show_quiz_welcome()

    def show_quiz_welcome(self):
        """Zobrazí uvítací obrazovku kvízu."""
        # Sestavení textu podle typu média
        if self.media_type == "series" and self.episode_title:
            media_info = f'"{self.title}" - {self.episode_title}'
            if self.season_number and self.episode_number:
                media_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
            quiz_headline = f"Kvízová hra na základě epizody seriálu {media_info}"
            quiz_type_line = f"• 5 otázek různé obtížnosti o epizodě seriálu"
            challenge_line = "Jste připraveni na kvízovou výzvu ze světa seriálů?"
        elif self.media_type == "series":
            media_info = f'"{self.title}"'
            quiz_headline = f"Kvízová hra na základě seriálu {media_info}"
            quiz_type_line = f"• 5 otázek různé obtížnosti o seriálu"
            challenge_line = "Jste připraveni na kvízovou výzvu ze světa seriálů?"
        elif self.media_type == "actor":
            media_info = f'"{self.title}"'
            quiz_headline = f"Kvízová hra na základě herce {media_info}"
            quiz_type_line = f"• 5 otázek různé obtížnosti o herci"
            challenge_line = "Jste připraveni na kvízovou výzvu ze světa herců?"
        elif self.media_type == "director":
            media_info = f'"{self.title}"'
            quiz_headline = f"Kvízová hra na základě režiséra {media_info}"
            quiz_type_line = f"• 5 otázek různé obtížnosti o režiséru"
            challenge_line = "Jste připraveni na kvízovou výzvu ze světa režie?"
        elif self.media_type == "writer":
            media_info = f'"{self.title}"'
            quiz_headline = f"Kvízová hra na základě scénáristy {media_info}"
            quiz_type_line = f"• 5 otázek různé obtížnosti o scénáristovi"
            challenge_line = "Jste připraveni na kvízovou výzvu ze světa scénáristů?"
        elif self.media_type == "creator":
            media_info = f'"{self.title}"'
            quiz_headline = f"Kvízová hra na základě tvůrce {media_info}"
            quiz_type_line = f"• 5 otázek různé obtížnosti o tvůrci"
            challenge_line = "Jste připraveni na kvízovou výzvu ze světa seriálových tvůrců?"
        else:
            media_info = f'"{self.title}"'
            quiz_headline = f"Kvízová hra na základě filmu {media_info}"
            quiz_type_line = f"• 5 otázek různé obtížnosti o filmu"
            challenge_line = "Jste připraveni na kvízovou výzvu ze světa filmu?"

        # Přidání žánru pokud je dostupný
        genre_text = f" • Žánr: {self.genre}" if self.genre else ""

        # Krátký popis děje pokud je dostupný
        plot_preview = ""
        if self.plot and len(self.plot) > 50:
            plot_preview = f"\nDĚJ: {self.plot[:150]}{'...' if len(self.plot) > 150 else ''}\n"

        welcome_text = f"""CO VÁS ČEKÁ:
• {quiz_headline}
{quiz_type_line}
• Testování znalostí děje, postav a trivia
• Získejte co nejvyšší skóre!{genre_text}
{plot_preview}Připravte se prokázat své znalosti!\n\n{challenge_line}"""

        # Nastavit property pro welcome sidebar
        self._set_quiz_film_info()

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(welcome_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Začít kvíz", "Pravidla", "Zpět na menu", "Zavřít")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        # Nastavit fokus na tlačítko "Začít kvíz" (index 0)
        if self.set_focus_callback:
            self.set_focus_callback(0)

    def _set_quiz_film_info(self):
        """Nastaví property s informacemi o filmu pro welcome sidebar."""
        try:
            # Získat reference na hlavní window pro nastavení property
            import xbmcgui
            window = xbmcgui.Window(xbmcgui.getCurrentWindowId())
            
            # Sestavit informace o filmu podle typu média
            if self.media_type == "series" and self.episode_title:
                film_info = f"Seriál: {self.title}\nEpizoda: {self.episode_title}"
                if self.season_number and self.episode_number:
                    film_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
            elif self.media_type == "series":
                film_info = f"Seriál: {self.title}"
            elif self.media_type == "actor":
                film_info = f"Herec: {self.title}"
            elif self.media_type == "director":
                film_info = f"Režisér: {self.title}"
            elif self.media_type == "writer":
                film_info = f"Scénárista: {self.title}"
            elif self.media_type == "creator":
                film_info = f"Tvůrce: {self.title}"
            else:
                film_info = f"Film: {self.title}"
            
            # Přidat žánr pokud je dostupný
            if self.genre:
                film_info += f"\nŽánr: {self.genre}"
            
            window.setProperty('WCS.Quiz.FilmInfo', film_info)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error setting quiz film info: {e}", level=xbmc.LOGERROR)

    def show_quiz_instructions(self):
        """Zobrazí návod ke kvízu."""
        if self.media_type == "series" and self.episode_title:
            media_info = f'{self.title} - {self.episode_title}'
            if self.season_number and self.episode_number:
                media_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
            intro_line = f"Odpovídáte na 5 otázek o epizodě seriálu: {media_info}"
            ai_line = "AI vytvoří otázky na základě epizody"
            final_line = "Jste připraveni prokázat své seriálové znalosti?"
        elif self.media_type == "series":
            media_info = f'{self.title}'
            intro_line = f"Odpovídáte na 5 otázek o seriálu: {media_info}"
            ai_line = "AI vytvoří otázky na základě seriálu"
            final_line = "Jste připraveni prokázat své seriálové znalosti?"
        elif self.media_type == "actor":
            media_info = f'{self.title}'
            intro_line = f"Odpovídáte na 5 otázek o herci: {media_info}"
            ai_line = "AI vytvoří otázky na základě herce"
            final_line = "Jste připraveni prokázat své znalosti o hercích?"
        elif self.media_type == "director":
            media_info = f'{self.title}'
            intro_line = f"Odpovídáte na 5 otázek o režiséru: {media_info}"
            ai_line = "AI vytvoří otázky na základě režiséra"
            final_line = "Jste připraveni prokázat své znalosti o režii?"
        elif self.media_type == "creator":
            media_info = f'{self.title}'
            intro_line = f"Odpovídáte na 5 otázek o tvůrci: {media_info}"
            ai_line = "AI vytvoří otázky na základě tvůrce"
            final_line = "Jste připraveni prokázat své znalosti o tvůrcích seriálů?"
        else:
            media_info = f'{self.title}'
            intro_line = f"Odpovídáte na 5 otázek o filmu: {media_info}"
            ai_line = "AI vytvoří otázky na základě filmu"
            final_line = "Jste připraveni prokázat své filmové znalosti?"

        instructions_text = f"""PRŮBĚH HRY:
• {intro_line}
• Každá správná odpověď = 1 bod
• Celkem 3 možnosti odpovědí na každou otázku
• Otázky pokrývají různé oblasti:
  - Děj a postavy
  - Herce a režii
  - Zajímavosti a trivia
  - Technické aspekty

HERNÍ MECHANIKA:
• {ai_line}
• Vždy máte 3 možnosti volby (A, B, C)
• Po každé odpovědi uvidíte výsledek
• Finální hodnocení na konci

{final_line}"""

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(instructions_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Začít kvíz", "Zpět", "Zpět na menu", "Zavřít")

    def load_next_quiz_question(self):
        """Načte další kvízovou otázku z předpřipravené sady."""
        if self.quiz_question_number > self.quiz_total_questions:
            self.end_quiz()
            return
        
        # Pokud nejsou otázky načtené, načti všechny najednou
        if not self.quiz_loaded:
            self.load_all_quiz_questions()
            return
        
        # RESET PŘEDCHOZÍCH DAT pro čisté načtení nové otázky
        self.selected_answer = ""
        self.selected_answer_index = -1
        self.game_state = "quiz_playing"
        
        # Načti otázku z předpřipravené sady
        current_question_data = self.quiz_questions[self.quiz_question_number - 1]
        
        # Uložení správné odpovědi a textu otázky
        self.current_correct_answer = current_question_data['correct_index']
        self.current_choices = current_question_data['options']
        self.current_question_text = current_question_data['question']
        
        # Zobrazit pouze aktuální otázku (čisté okno)
        display_text = f"OTÁZKA {self.quiz_question_number}/{self.quiz_total_questions}\n\n{current_question_data['question']}"
        
        # Aktualizovat zobrazení s pouze aktuální otázkou
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(display_text))
        
        # Obnovení tlačítek s možnostmi odpovědí
        if self.update_buttons_callback:
            options = current_question_data['options']
            self.update_buttons_callback(
                f"{options[0]}" if len(options) > 0 else "",
                f"{options[1]}" if len(options) > 1 else "",
                f"{options[2]}" if len(options) > 2 else "",
                "Zavřít"
            )
        if self.update_sidebar_callback:
            self.update_sidebar_callback()

    def load_all_quiz_questions(self):
        """Načte všech 5 otázek najednou pomocí batch funkce."""
        # Zobrazit loading pro celý batch
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(f"AI připravuje kvízovou hru...\n\nTéma: {self.title}\n\nGeneruji {self.quiz_total_questions} různých otázek...\n\nChvíli strpení..."))
        if self.update_buttons_callback:
            self.update_buttons_callback("Načítání...", "...", "...", "...")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        
        try:
            if self.media_type == 'actor':
                questions, usage = ai_handler.generate_actor_quiz_batch(
                    actor_name=self.title,
                    actor_bio=self.plot,
                    known_for=[item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data else [],
                    total_questions=self.quiz_total_questions
                )
            elif self.media_type == 'director':
                questions, usage = ai_handler.generate_director_quiz_batch(
                    director_name=self.title,
                    director_bio=self.plot,
                    known_for=[item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data else [],
                    total_questions=self.quiz_total_questions
                )
            elif self.media_type == 'writer':
                questions, usage = ai_handler.generate_writer_quiz_batch(
                    writer_name=self.title,
                    writer_bio=self.plot,
                    known_for=[item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data else [],
                    total_questions=self.quiz_total_questions
                )
            elif self.media_type == 'creator':
                questions, usage = ai_handler.generate_creator_quiz_batch(
                    creator_name=self.title,
                    creator_bio=self.plot,
                    known_for=[item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data else [],
                    total_questions=self.quiz_total_questions
                )
            else:
                questions, usage = ai_handler.generate_quiz_batch(
                    media_type=self.media_type,
                    title=self.title,
                    plot=self.plot,
                    genre=self.genre,
                    episode_title=self.episode_title,
                    season_number=self.season_number,
                    episode_number=self.episode_number,
                    total_questions=self.quiz_total_questions
                )
            
            # Uložení usage statistik
            self._track_ai_usage(usage, "quiz")
            
            # Uložení všech otázek
            self.quiz_questions = questions
            self.quiz_loaded = True
            
            # Teď načti první otázku
            self.load_next_quiz_question()
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error loading quiz batch: {e}", level=xbmc.LOGERROR)
            if self.show_error_callback:
                self.show_error_callback("Chyba při načítání kvízových otázek. Zkuste to znovu.")

    def answer_quiz_question(self, answer_index):
        """Zpracuje odpověď na kvízovou otázku - spustí countdown."""
        if answer_index >= len(self.current_choices):
            return
            
        # Uložit vybranou odpověď
        self.selected_answer = self.current_choices[answer_index]
        self.selected_answer_index = answer_index
        
        # OKAMŽITĚ aktualizovat text s vybranou odpovědí PŘED změnou stavu
        display_text = f"OTÁZKA {self.quiz_question_number}/{self.quiz_total_questions}\n\n"
        display_text += f"{self.current_question_text}\n\n"
        display_text += f"VAŠE ODPOVĚĎ:\n{self.selected_answer}\n\n"
        display_text += "Vyhodnocení odpovědi..."
        
        # IHNED aktualizovat UI - to musí být první věc!
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(display_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Vyhodnocuje se...", "...", "...", "...")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        
        # Teprve nyní změnit stav
        self.game_state = "quiz_answer_pending"
        
        # Spustit countdown (3 sekundy)
        if self.start_countdown_callback:
            self.start_countdown_callback(3, self.finish_answer_evaluation)
        else:
            # Fallback pokud není callback - ihned vyhodnotit
            self.finish_answer_evaluation()

    def finish_answer_evaluation(self):
        """Dokončí vyhodnocení odpovědi po countdown."""
        is_correct = (self.selected_answer_index == self.current_correct_answer)
        
        if is_correct:
            self.quiz_score += 1
            result_text = "SPRÁVNĚ!"
        else:
            correct_option = self.current_choices[self.current_correct_answer]
            result_text = f"ŠPATNĚ! Správná odpověď: {correct_option}"
        
        # Příprava na další otázku
        self.quiz_question_number += 1
        
        # Vždy zobrazit transition screen (i pro poslední otázku)
        self.game_state = "quiz_transition"
        self.show_quiz_transition(is_correct, self.selected_answer, result_text)

    def show_quiz_transition(self, is_correct, selected_option, result_text):
        """Zobrazí výsledek na stejné stránce s otázkou."""
        # Rozlišit motivační texty podle pozice otázky
        is_last_question = self.quiz_question_number > self.quiz_total_questions
        
        if is_correct:
            if is_last_question:
                motivational_texts = [
                    "Skvělé! Končíte na vysoké notě!",
                    "Výborně! Poslední otázka zvládnuta!",
                    "Perfektní! Dokončili jste skvěle!",
                    "Správně! Silný závěr!",
                    "Excelentně! Kvíz je u konce!"
                ]
            else:
                motivational_texts = [
                    "Výborně! Pokračujte dále!",
                    "Správně! Jste na dobré cestě!",
                    "Skvěle! Ukazujete své znalosti!",
                    "Přesně tak! Jedete skvěle!",
                    "Perfektní! Jen tak dál!"
                ]
        else:
            if is_last_question:
                motivational_texts = [
                    "Nevadí! Kvíz jste dokončili!",
                    "Stává se! Důležité je, že jste to dokázali!",
                    "V pohodě! Každý kvíz je zkušenost!",
                    "To je OK! Hlavně jste si zahráli!",
                    "Nic se neděje! Kvíz je u konce!"
                ]
            else:
                motivational_texts = [
                    "Nevadí! Další otázka vás čeká!",
                    "To je v pohodě! Zkuste to dál!",
                    "Každý se učí! Pokračujme!",
                    "Další šance přichází!",
                    "Stává se! Jdeme na další!"
                ]
        
        motivation = random.choice(motivational_texts)
        
        # Zobrazit otázku + odpověď + výsledek na stejné stránce
        previous_question = self.quiz_question_number - 1
        current_score = self.quiz_score
        
        transition_text = f"OTÁZKA {previous_question}/{self.quiz_total_questions}\n\n"
        transition_text += f"{self.current_question_text}\n\n"
        transition_text += f"VAŠE ODPOVĚĎ:\n{selected_option}\n\n"
        transition_text += f"{result_text}\n\n"
        transition_text += f"{motivation}\n\n"
        transition_text += f"Skóre: {current_score}/{previous_question}\n\n"
        
        # Rozlišit mezi meziotázkami a poslední otázkou
        if self.quiz_question_number <= self.quiz_total_questions:
            # Další otázka čeká
            transition_text += "Připravený na další otázku?"
            button1_text = "Další otázka"
        else:
            # To byla poslední otázka
            transition_text += "To byla poslední otázka! Jste připravený na výsledky?"
            button1_text = "Zobrazit výsledky"
        
        # Aktualizovat zobrazení s výsledkem na stejné stránce
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(transition_text))
        if self.update_buttons_callback:
            self.update_buttons_callback(button1_text, "Nová hra", "Zpět na menu", "Zavřít")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        # Nastavit fokus na první tlačítko (index 0)
        if self.set_focus_callback:
            self.set_focus_callback(0)

    def show_final_quiz_results(self):
        """Zobrazí dramatické finální výsledky kvízu."""
        percentage = (self.quiz_score / self.quiz_total_questions) * 100
        
        # Dramatické hodnocení
        if percentage == 100:
            grade = "ABSOLUTNÍ ZNALEC!"
            comment = "Perfektní skóre! Znáte tento film/seriál jako svoje vlastní boty. Jste skutečný mistr!"
        elif percentage >= 80:
            grade = "VÝBORNÝ ZNALEC"
            comment = "Většinu otázek jste zvládli bez problémů. Máte opravdu dobré znalosti!"
        elif percentage >= 60:
            grade = "DOBRÝ ZNALEC"
            comment = "Základní znalosti máte v pořádku, ale je zde prostor pro zlepšení!"
        elif percentage >= 40:
            grade = "ZAČÁTEČNÍK"
            comment = "Něco víte, ale doporučujeme se podívat znovu a více si zapamatovat!"
        else:
            grade = "ZKUSTE TO ZNOVU"
            comment = "Každý začíná. Pusťte si film/seriál znovu a zkuste kvíz později!"
        
        # Zobrazit pouze finální výsledky (čisté okno)
        final_text = f"""VÝSLEDKY KVÍZU

Správných odpovědí: {self.quiz_score}/{self.quiz_total_questions}
Úspěšnost: {percentage:.1f}%

HODNOCENÍ: {grade}

{comment}

Děkujeme za hraní!"""
        
        self.game_state = "ended"
        
        # Aktualizovat zobrazení s pouze finálními výsledky
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(final_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Hrát znovu", "Zpět na menu", "Zavřít", "Zavřít")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()

    def end_quiz(self):
        """Ukončí kvíz a zobrazí výsledky."""
        self.show_final_quiz_results()

    def end_game(self):
        """Ukončí hru - zavře dialog."""
        if self.close_dialog_callback:
            self.close_dialog_callback()

    def start_game_selection(self):
        """Návrat na hlavní výběr typu hry."""
        xbmc.log("[plugin.video.milionar] QuizGameController.start_game_selection called", level=xbmc.LOGINFO)
        if hasattr(self, 'return_to_menu_callback') and self.return_to_menu_callback:
            xbmc.log("[plugin.video.milionar] QuizGameController calling return_to_menu_callback", level=xbmc.LOGINFO)
            self.return_to_menu_callback()
        else:
            xbmc.log("[plugin.video.milionar] QuizGameController fallback - closing dialog", level=xbmc.LOGINFO)
            # Fallback pro zpětnou kompatibilitu
            if self.close_dialog_callback:
                self.close_dialog_callback()

    def make_choice(self, choice_index):
        """Hlavní metoda pro zpracování voleb."""
        self.handle_game_choice(choice_index)
    
    def handle_game_choice(self, choice_index):
        """Zpracuje volbu specifickou pro kvízovou hru."""
        xbmc.log(f"[plugin.video.milionar] QuizGameController.handle_game_choice: choice_index={choice_index}, game_state={self.game_state}", level=xbmc.LOGINFO)
        
        if self.game_state == "quiz_welcome":
            if choice_index == 0:  # Začít kvíz
                self.game_state = "quiz_playing"
                self.quiz_question_number = 1
                
                # Zobrazit loading podobně jako u textové adventury
                if self.update_display_callback:
                    self.update_display_callback(self._add_ai_info_header(f"AI připravuje kvízovou hru...\n\nTéma: {self.title}\n\nGeneruji {self.quiz_total_questions} otázek různé obtížnosti:\n• Děj a postavy\n• Herce a režii\n• Zajímavosti a trivia\n\nChvíli strpení..."))
                if self.update_buttons_callback:
                    self.update_buttons_callback("Načítání...", "...", "...", "...")
                if self.update_sidebar_callback:
                    self.update_sidebar_callback()
                
                # Načíst první otázku po krátké pauze
                self.load_next_quiz_question()
            elif choice_index == 1:  # Návod
                self.game_state = "quiz_instructions"
                self.show_quiz_instructions()
            elif choice_index == 2:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "quiz_instructions":
            if choice_index == 0:  # Začít kvíz
                self.game_state = "quiz_playing"
                self.quiz_question_number = 1
                
                # Zobrazit loading podobně jako u textové adventury
                if self.update_display_callback:
                    self.update_display_callback(self._add_ai_info_header(f"AI připravuje kvízovou hru...\n\nTéma: {self.title}\n\nGeneruji {self.quiz_total_questions} otázek různé obtížnosti:\n• Děj a postavy\n• Herce a režii\n• Zajímavosti a trivia\n\nChvíli strpení..."))
                if self.update_buttons_callback:
                    self.update_buttons_callback("Načítání...", "...", "...", "...")
                if self.update_sidebar_callback:
                    self.update_sidebar_callback()
                
                # Načíst první otázku po krátké pauze
                self.load_next_quiz_question()
            elif choice_index == 1:  # Zpět
                self.game_state = "quiz_welcome"
                self.show_quiz_welcome()
            elif choice_index == 2:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "quiz_transition":
            if choice_index == 0:  # Další otázka / Zobrazit výsledky
                if self.quiz_question_number <= self.quiz_total_questions:
                    # Další otázka
                    self.game_state = "quiz_playing"
                    self.load_next_quiz_question()
                else:
                    # Zobrazit finální výsledky
                    self.show_final_quiz_results()
            elif choice_index == 1:  # Nová hra
                self.start_specific_game()
            elif choice_index == 2:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "quiz_playing":
            if choice_index == 3:  # Zavřít tlačítko během kvízu  
                self.end_game()
            else:
                self.answer_quiz_question(choice_index)
        elif self.game_state == "quiz_answer_pending":
            # Během countdown nelze nic dělat
            pass
        elif self.game_state == "ended":
            if choice_index == 0:  # Hrát znovu
                self.start_specific_game()
            elif choice_index == 1:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 2:  # Zavřít
                self.end_game()
            elif choice_index == 3:  # Zavřít (duplicitní)
                self.end_game()


    def get_game_specific_sidebar_items(self):
        """Vrací sidebar položky specifické pro kvízovou hru."""
        if self.game_state == "quiz_welcome":
            return ["Kvízová výzva", f"Film: {self.title[:20]}", "5 otázek čeká"]
        elif self.game_state == "quiz_instructions":
            return ["Návod ke kvízu", "Pravidla hry", f"Film: {self.title[:15]}"]
        elif self.game_state == "quiz_playing":
            progress = f"{self.quiz_question_number}/{self.quiz_total_questions}"
            score = f"{self.quiz_score}/{self.quiz_question_number}" if self.quiz_question_number > 0 else "0/0"
            return [f"Otázka: {progress}", f"Skóre: {score}", "Kvíz probíhá"]
        elif self.game_state == "quiz_answer_pending":
            progress = f"{self.quiz_question_number}/{self.quiz_total_questions}"
            score = f"{self.quiz_score}/{self.quiz_question_number}" if self.quiz_question_number > 0 else "0/0"
            return [f"Otázka: {progress}", f"Skóre: {score}", "Vyhodnocuje se..."]
        elif self.game_state == "quiz_transition":
            progress = f"{self.quiz_question_number}/{self.quiz_total_questions}"
            score = f"{self.quiz_score}/{self.quiz_question_number-1}"
            return [f"Další: {progress}", f"Skóre: {score}", "Přechod na další"]
        elif self.game_state == "ended":
            return [
                f"Dokončeno: {self.quiz_total_questions}/{self.quiz_total_questions}",
                f"Skóre: {self.quiz_score}/{self.quiz_total_questions}",
                "Kvízová hra ukončena"
            ]

        else:
            return super().get_sidebar_items()

    def get_sidebar_items(self):
        """Vrací aktuální sidebar položky."""
        return self.get_game_specific_sidebar_items() 