# wcs/router.py

import sys
from urllib.parse import parse_qsl, quote_plus
import xbmcaddon
import xbmcplugin
import xbmcgui
import xbmc
import datetime
from functools import wraps
import os
import requests
import xbmcvfs
import xml.etree.ElementTree as ET

# Lokální importy modulů doplňku
from wcs import user_data, utils, localization
from wcs.ui import UI as ui
from wcs.metadata import TMDbClient as tmdb_handler, CSFDClient as csfd_handler
from wcs.webshare import WebshareClient as webshare_api, DownloadManager as download
from wcs.search import MovieSearch as search_movies, SeriesSearch as search_series
from wcs.library import LibraryManager as library
from wcs.ai import AIHandler as ai_handler
from wcs.ui.dialogs.DialogWcsDetails import DetailsWindow
from wcs.ui.dialogs.DialogWcsSearch import show_search_dialog
from wcs.utils import search_and_play_movie
from wcs.games.DialogWcsTextGame import TextGameWindow
from wcs.playback.DialogMediaInfoWithGame import get_autoplay_manager
from wcs.playback.DialogMediaInfoWithGame import get_episode_info_manager
from wcs.playback.PlayerMonitor import start_autoplay_monitoring, stop_autoplay_monitoring
from wcs.login.DialogTMDbLogin import show_tmdb_login_dialog
from wcs.ui.dialogs.guide.DialogTMDbGuide import TMDbGuideDialog
from wcs.ui.dialogs.guide.DialogWebshareGuide import WebshareGuideDialog
from wcs.ui.dialogs.guide.DialogAIGuide import AIGuideDialog
from wcs.ui.dialogs.guide.DialogLibraryGuide import LibraryGuideDialog
from wcs.ui.dialogs.guide.DialogRemoteGuide import RemoteGuideDialog

# Globální proměnné doplňku
_addon = xbmcaddon.Addon()
_handle = int(sys.argv[1])
_profile = xbmcaddon.Addon().getAddonInfo('profile')

def router(paramstring):
    """Hlavní směrovací funkce pluginu."""
    params = dict(parse_qsl(paramstring))
    action = params.get('action')

    # Získání handle - buď z URL parametrů nebo z sys.argv[1]
    handle_from_url = params.get('handle')
    if handle_from_url:
        try:
            _handle = int(handle_from_url)
        except (ValueError, TypeError):
            _handle = int(sys.argv[1])
    else:
        _handle = int(sys.argv[1])

    # Kontrola a přidání do knihovny
    if action == 'check_library_settings_and_add':
         library.check_settings_and_add_item(params, _addon)
         return

    # Základní akce pluginu
    if action == 'search':
        # Detekce typu obsahu podle hledaného výrazu
        what = params.get('what', '')
        if what and utils.extract_season_episode(what) != (None, None):
            # Obsahuje epizodu - použij search_series
            search_series.perform_search_and_display(_addon, params, _handle)
        else:
            # Film nebo obecné vyhledávání - použij search_movies
            search_movies.perform_search_and_display(_addon, params, _handle)
        return
    elif action == 'manual_webshare_search':
        # Manuální vyhledávání na Webshare
        search_movies.perform_manual_search(_addon, params, _handle)
        return
    elif action == 'queue':
        ui.display_queue(_addon, params, _handle)
        return
    elif action == 'history':
        ui.display_history(_addon, params, _handle)
        return
    elif action == 'recently_played':
        ui.display_recently_played(_addon, _handle)
        return
    elif action == 'continue_watching_dialog':
        # Fullscreen dialog "Pokračovat ve sledování"
        from wcs.ai.DialogContinueWatching import show_continue_watching_dialog
        show_nav = params.get('show_nav_sidebar') == '1'
        nav_pos = int(params.get('nav_position', '0'))
        show_continue_watching_dialog(_addon, show_nav_sidebar=show_nav, nav_position=nav_pos)
        return
    elif action == 'my_movies_dialog':
        # Fullscreen dialog "Moje filmy"
        from wcs.ai.DialogMyMovies import show_my_movies_dialog
        show_nav = params.get('show_nav_sidebar') == '1'
        nav_pos = int(params.get('nav_position', '0'))
        show_my_movies_dialog(_addon, show_nav_sidebar=show_nav, nav_position=nav_pos)
        return
    elif action == 'my_series_dialog':
        # Fullscreen dialog "Moje seriály"
        from wcs.ai.DialogMySeries import show_my_series_dialog
        show_nav = params.get('show_nav_sidebar') == '1'
        nav_pos = int(params.get('nav_position', '0'))
        show_my_series_dialog(_addon, show_nav_sidebar=show_nav, nav_position=nav_pos)
        return
    elif action == 'my_tv_dialog':
        # Fullscreen dialog "Moje televize"
        from wcs.ai.DialogMyTV import show_my_tv_dialog
        show_nav = params.get('show_nav_sidebar') == '1'
        nav_pos = int(params.get('nav_position', '0'))
        show_my_tv_dialog(_addon, show_nav_sidebar=show_nav, nav_position=nav_pos)
        return


    elif action == 'settings':
        # --- Dynamická aktualizace stavu přihlášení k Webshare ---
        ws_user = _addon.getSetting('wsuser')
        token = _addon.getSetting('token')
        status = 'Nepřihlášen'
        token_value = ''
        if ws_user and token:
            # Ověřím platnost tokenu (bez notifikace)
            valid_token = webshare_api.revalidate(_addon, lambda *a, **k: None)
            if valid_token:
                status = f'Přihlášen jako: {ws_user}'
                token_value = token
                # Tichá obnova VIP informace pomocí jednotné funkce
                try:
                    webshare_api.refresh_vip_info(_addon, token_override=token, show_notification=False, log_context="settings")
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] settings VIP refresh error: {e}", level=xbmc.LOGWARNING)
        _addon.setSetting('webshare_status', status)
        _addon.setSetting('wsuser', ws_user or '')
        _addon.setSetting('wspass', _addon.getSetting('wspass') or '')
        _addon.setSetting('token', token_value)
        
        # --- Dynamická aktualizace stavu přihlášení k TMDb ---
        tmdb_api_key = _addon.getSetting('tmdb_api_key')
        tmdb_status = 'Nepřihlášen'
        if tmdb_api_key:
            # Ověřím platnost API klíče přes TMDbClient (centralizovaně)
            is_valid = tmdb_handler.check_tmdb_api_key_validity(_addon)
            tmdb_status = 'Přihlášen k TMDb' if is_valid else 'Neplatný API klíč'
        _addon.setSetting('tmdb_status', tmdb_status)
        
        # --- Dynamická aktualizace stavu přihlášení k AI poskytovatelům ---
        # OpenAI
        openai_api_key = _addon.getSetting('openai_api_key')
        openai_status = 'Nepřihlášen'
        if openai_api_key:
            openai_status = 'Přihlášen k OpenAI'
        _addon.setSetting('openai_status', openai_status)
        
        # Anthropic
        anthropic_api_key = _addon.getSetting('anthropic_api_key')
        anthropic_status = 'Nepřihlášen'
        if anthropic_api_key:
            anthropic_status = 'Přihlášen k Anthropic'
        _addon.setSetting('anthropic_status', anthropic_status)
        
        # Google
        google_api_key = _addon.getSetting('google_api_key')
        google_status = 'Nepřihlášen'
        if google_api_key:
            google_status = 'Přihlášen k Google'
        _addon.setSetting('google_status', google_status)
        
        # Mistral
        mistral_api_key = _addon.getSetting('mistral_api_key')
        mistral_status = 'Nepřihlášen'
        if mistral_api_key:
            mistral_status = 'Přihlášen k Mistral'
        _addon.setSetting('mistral_status', mistral_status)
        

        
        ui.open_settings(_addon)
        # Po zavření nastavení refreshneme menu, aby se zobrazilo správně podle stavu TMDb
        xbmc.executebuiltin('Container.Refresh')
        xbmcplugin.setResolvedUrl(_handle, False, xbmcgui.ListItem())
        return # Ukončíme zde, protože openSettings nevrací seznam
    elif action == 'info':
        ident = params.get('ident')
        if ident:
             ui.display_file_info(ident, _addon)
        else:
             ui.popinfo("Chybí identifikátor souboru pro zobrazení informací.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
        return
    elif action == 'play_movie_from_addon':
        # Přehrání filmu z doplňku přes kontextové menu
        title = params.get('title', '')
        year = params.get('year', '')
        plot = params.get('plot', '')
        poster = params.get('poster', '')
        
        if not title:
            ui.popinfo("Chybí název filmu pro přehrání.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
            return
            
        # Doplnění chybějících metadat z knihovny před voláním play_movie_from_addon
        fanart = ''
        rating = ''
        genre = ''
        director = ''
        runtime = ''
        
        try:
            from wcs.library.LibraryManager import get_movie_metadata_from_library
            metadata = get_movie_metadata_from_library(title, year) or {}
            plot = plot or metadata.get('plot', '')
            poster = poster or metadata.get('poster', '') or metadata.get('thumb', '')
            fanart = fanart or metadata.get('fanart', '')
            rating = rating or str(metadata.get('rating', ''))
            genre = genre or metadata.get('genre', '')
            director = director or metadata.get('director', '')
            runtime = runtime or metadata.get('runtime', '')
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Router: Error getting metadata: {e}", level=xbmc.LOGWARNING)
        
        # Vytvoření meta slovníku s doplněnými metadaty
        meta = {
            'title': title,
            'year': year,
            'plot': plot,
            'poster': poster,
            'fanart': fanart,
            'rating': rating,
            'genre': genre,
            'director': director,
            'runtime': runtime,
            'tmdb_id': params.get('tmdb_id')  # Pro uložení do historie
        }
        
        # Volání centrální funkce pro přehrání s kompletními metadaty
        from wcs.utils import play_movie_from_addon
        play_movie_from_addon(meta, _addon)
        return
    elif action == 'play':
        ident = params.get('ident')
        name = params.get('name', '')
        
        # Kontrola povinných parametrů
        if not ident:
            ui.popinfo("Chybí identifikátor souboru pro přehrání.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
            xbmcplugin.setResolvedUrl(_handle, False, xbmcgui.ListItem())
            return
        
        # OPRAVA: Zjednodušená logika - použít webshare_api.resolve_playable_item
        xbmc.log(f"[plugin.video.milionar] Router: play action - ident: {ident}", level=xbmc.LOGINFO)
        webshare_api.resolve_playable_item(ident, name, _addon, params, _handle)
        return
    elif action == 'download_file':
        ident = params.get('ident')
        if not ident:
             ui.popinfo("Chybí identifikátor souboru pro stažení.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
             return
        download.download_file(ident, _addon)
        return

    elif action == 'my_files':
        ui.display_my_files(_addon, params, _handle)
        return
    elif action == 'my_added_items':
        ui.display_my_added_items(_addon, _handle)
        return
    elif action == 'my_list_dialog':
        # Otevření dialogu "Můj seznam"
        from wcs.ui.dialogs.DialogWcsMyList import MyListDialog
        dialog = MyListDialog(
            "details_extras/DialogWcsMyList.xml",
            _addon.getAddonInfo('path'),
            "Default",
            "1080i"
        )
        dialog.doModal()
        del dialog
        return

    elif action == 'update_library':
        ui.trigger_kodi_library_update(_addon)
        return
    elif action == 'db':
        ui.display_db_placeholder_info(_addon, _handle)
        return

    # Akce TMDb menu a procházení
    elif action == 'tmdb_menu':
        # Kontrola zda je nastaven TMDb API klíč
        if ui.has_tmdb_api_key(_addon):
            ui.display_main_menu(_addon, _handle)
        else:
            ui.display_limited_menu(_addon, _handle)
        return
    elif action == 'tmdb_series':
        ui.display_series_menu(_addon, _handle)
        return
    elif action == 'tmdb_movies':
        ui.display_movies_menu(_addon, _handle)
        return
    elif action == 'tmdb_browse_categories':
        ui.browse_movie_categories(_addon, _handle)
        return
    elif action == 'tmdb_browse_series_categories':
        ui.browse_series_categories(_addon, _handle)
        return
    elif action == 'tmdb_browse_genres':
        ui.browse_movie_genres(params, _addon, _handle)
        return
    elif action == 'tmdb_browse_series_genres':
        ui.browse_series_genres(params, _addon, _handle)
        return
    elif action == 'tmdb_browse_countries':
        ui.browse_movie_countries(_addon, _handle)
        return
    elif action == 'tmdb_browse_series_countries':
        ui.browse_series_countries(_addon, _handle)
        return
    elif action == 'tmdb_combo_genres':
        ui.combo_select_genres(params, _addon, _handle)
        return
    elif action == 'tmdb_combo_countries':
        ui.combo_select_countries(params, _addon, _handle)
        return
    elif action == 'tmdb_combo_years':
        ui.combo_select_years(params, _addon, _handle)
        return

    # Akce TMDb - Zobrazení seznamů a detailů
    elif action == 'tmdb_list_movies':
        tmdb_handler.list_movies(params, _addon, _handle)
        # Tato funkce volá endOfDirectory sama
    elif action == 'tmdb_list_series':
        tmdb_handler.list_series(params, _addon, _handle)
        # Tato funkce volá endOfDirectory sama
    elif action == 'tmdb_series_detail' and 'series_id' in params:
        tmdb_handler.display_series_seasons(params, _addon, _handle)
        return
    elif action == 'tmdb_season_detail' and 'series_id' in params and 'season_number' in params:
        tmdb_handler.display_season_episodes(params, _addon, _handle)
        return
    elif action == 'tmdb_movie_detail' and 'idx' in params:
        tmdb_handler.display_movie_text_details(params, _addon, _handle)
        return
    elif action == 'tmdb_show_item_details':
        tmdb_handler.show_item_details(params, _addon)
        return
    elif action == 'csfd_top_movies':
        csfd_handler.list_csfd_movies(params, _addon, _handle)
        return
    elif action == 'open_csfd_url':
        csfd_handler.open_csfd_url(params, _addon)
        return
    elif action == 'reset_csfd_cache':
        csfd_handler.reset_csfd_cache(params, _addon)
        return
    elif action == 'reset_csfd_tmdb_cache':
        csfd_handler.reset_csfd_tmdb_cache(params, _addon)
        return
    elif action == 'show_wcs_details':
        media_type = params.get('media_type')
        tmdb_id = params.get('tmdb_id')
        if media_type and tmdb_id:
            # Název XML souboru
            xml_filename = "details_extras/DialogWcsDetails.xml"
            # Kořenová cesta addonu
            addon_path = _addon.getAddonInfo('path') 

            # Vytvoření instance okna s povinnými argumenty + našimi kwargs
            # Předpokládáme, že XML je v resources/skins/Default/1080i/ (výchozí hodnoty skinu a res)
            window = DetailsWindow(xml_filename, addon_path, 
                                   media_type=media_type, tmdb_id=tmdb_id)
            window.doModal() # Zobrazí okno a čeká na jeho zavření
            del window # Uvolní paměť
        else:
            # Zobrazit chybu, pokud chybí parametry
            ui.popinfo("Chyba: Chybí parametry pro zobrazení WCS detailů.", _addon.getAddonInfo('name'))
        return # Tato akce nevrací seznam do Kodi
    elif action == 'tmdb_added_movies':
        ui.display_added_movies(_addon, _handle)
        return
    elif action == 'tmdb_added_series':
        ui.display_added_series(_addon, _handle)
        return

    # NOVÁ AKCE pro zobrazení podobných filmů
    elif action == 'show_similar_movies' and 'tmdb_id' in params:
        tmdb_handler.display_similar_movies(params, _addon, _handle)
        # Funkce display_similar_movies volá endOfDirectory sama
        return

    # Akce TMDb - Přidání/Odebrání/Vyhledání
    elif action == 'tmdb_add_series':
        ui.add_series(params, _addon, _handle)
        return
    elif action == 'tmdb_add_movie':
        ui.add_movie(params, _addon, _handle)
        return
    elif action == 'tmdb_universal_search':
        ui.universal_search(params, _addon, _handle)
        return
    elif action == 'search_history_select':
        ui.search_history_select(params, _addon, _handle)
        return
    elif action == 'tmdb_add_movie_to_list':
        tmdb_handler.add_movie_to_list(params, _addon)
        return
    elif action == 'tmdb_add_series_to_list':
        tmdb_id = params.get('tmdb_id')
        name = params.get('name')
        ui.add_series_item_to_list(_addon, tmdb_id, name)
        return
    elif action == 'tmdb_movie_remove':
        ui.remove_added_movie(params, _addon)
        return
    elif action == 'tmdb_remove_all_movies': # NOVÁ AKCE
        ui.remove_all_tmdb_movies(_addon)
        return
    elif action == 'tmdb_remove_all_series':
        ui.remove_all_tmdb_series(_addon)
        return
    elif action == 'remove_all_my_items':
        ui.remove_all_my_items(_addon)
        return
    elif action == 'remove_recently_played':
        ui.remove_recently_played_history(_addon)
        return
    elif action == 'remove_recently_played_item':
        ui.remove_recently_played_item(params, _addon)
        return
    elif action == 'tmdb_series_remove':
        ui.remove_added_series(params, _addon)
        return
    elif action == 'tmdb_episode_search' and all(k in params for k in ['series_name','season','episode','ep_name']):
        search_series.search_and_handle_episode(params, _addon, _handle)
        return
    elif action == 'tmdb_movie_search' and 'title' in params:
        search_movies.search_and_handle_movie(params, _addon, _handle)
        return
    elif action == 'tmdb_movie_search_from_browse' and 'title' in params:
        search_movies.search_and_handle_movie(params, _addon, _handle)
        return
    elif action == 'tmdb_series_search' and 'name' in params:
        search_name = params.get('name', '')
        # Přesměrování na obecné hledání
        xbmc.executebuiltin(f'Container.Update(plugin://{_addon.getAddonInfo("id")}?action=search&what={quote_plus(search_name)})')
        return
    elif action == 'tmdb_combo_results' and 'media_type' in params:
        media_type = params.get('media_type')
        if media_type == 'movie':
            tmdb_handler.list_movies(params, _addon, _handle)
        elif media_type == 'tv':
            tmdb_handler.list_series(params, _addon, _handle)
        else:
            ui.popinfo("Chyba: Neznámý typ média pro výsledky kombinovaného výběru.", _addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(_handle)
            return # Ukončit, pokud je chyba
        # list_movies/list_series volají endOfDirectory

    # Akce přehrávání z knihovny
    elif action == 'play_movie_from_library' and 'title' in params:
        library.play_movie_from_library(params, _addon, _handle)
        return
    elif action == 'play_episode_from_library' and 'series_name' in params and 'season' in params and 'episode' in params:
        library.resolve_episode_from_library(params, _addon, _handle)
        return
    elif action == 'play_next_episode_with_metadata':
        from wcs.utils import play_next_episode_with_metadata
        play_next_episode_with_metadata(params, _addon)
        return
    elif action == 'play_episode_from_addon':
        from wcs import utils
        series_name = params.get('series_name')
        season = params.get('season_number')
        episode = params.get('episode_number')
        ep_name = params.get('episode_title')
        year = params.get('year', '')
        poster = params.get('poster', '')
        tmdb_id = params.get('tmdb_id')
        
        if series_name and season and episode:
            # Převedeme na int, pokud jsou stringy
            try:
                season = int(season)
                episode = int(episode)
            except: pass
            
            utils.search_and_play_episode(
                series_name=series_name, 
                season=season, 
                episode=episode, 
                ep_name=ep_name, 
                poster=poster, 
                year=year,
                tmdb_id=tmdb_id,
                addon=_addon
            )

        return
    elif action == 'play_episode_from_addon_playlist':
        # Akce pro postupné přehrávání v playlistu (řeší URL a volá setResolvedUrl)
        from wcs import utils
        series_name = params.get('series_name')
        season = params.get('season_number')
        episode = params.get('episode_number')
        ep_name = params.get('episode_title')
        year = params.get('year', '')
        poster = params.get('poster', '')
        tmdb_id = params.get('tmdb_id')
        plot = params.get('plot', '')
        rating = params.get('rating', '')
        genre = params.get('genre', '')
        
        if series_name and season and episode:
            try:
                season = int(season)
                episode = int(episode)
            except: pass
            
            stream_url = utils.resolve_episode_for_playlist(
                series_name=series_name,
                season=season, 
                episode=episode, 
                ep_name=ep_name, 
                plot=plot,
                poster=poster, 
                rating=rating,
                genre=genre,
                year=year,
                tmdb_id=tmdb_id,
                addon=_addon
            )
            
            if stream_url:
                li = xbmcgui.ListItem(path=stream_url)
                # Nastavíme i základní info pro player
                li.setInfo('video', {'title': ep_name or f"{series_name} S{season}E{episode}"})
                xbmcplugin.setResolvedUrl(_handle, True, li)
            else:
                xbmcplugin.setResolvedUrl(_handle, False, xbmcgui.ListItem())
        else:
             xbmcplugin.setResolvedUrl(_handle, False, xbmcgui.ListItem())
        return

    # Akce pro správu oblíbených herců
    elif action == 'favorite_actors':
        ui.display_favorite_actors(_addon, _handle)
        return
    elif action == 'tmdb_search_actor':
        tmdb_handler.search_actor(_addon, _handle)
        return
    elif action == 'tmdb_remove_favorite_actor':
        ui.remove_favorite_actor(params, _addon)
        return
    elif action == 'tmdb_remove_all_favorite_actors': # NOVÁ AKCE
        ui.remove_all_favorite_actors(_addon)
        return
    elif action == 'tmdb_search_actor_movies':
        tmdb_handler.search_actor_movies(params, _addon, _handle)
        return
    elif action == 'tmdb_add_actor_to_favorites':
        actor_id = params.get('actor_id')
        actor_name = params.get('actor_name', '')
        if actor_id:
            tmdb_handler.add_actor_to_favorites(_addon, actor_id, actor_name)
        else:
            ui.popinfo("Chybí ID herce pro přidání do oblíbených.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
        return
    # Akce pro dialog herce
    elif action == 'display_actor_dialog':
        actor_id = params.get('actor_id')
        media_type = params.get('media_type', 'movie')
        person_type = params.get('person_type', 'actor')
        if actor_id:
            from wcs.ui.dialogs.DialogActorDetails import ActorDialog
            dialog = ActorDialog("details_extras/DialogActorDetails.xml", _addon.getAddonInfo('path'), "Default", actor_id=actor_id, media_type=media_type, person_type=person_type)
            dialog.doModal()
            del dialog
        else:
            ui.popinfo("Chyba: Chybí ID herce.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_ERROR)
        return
        
    # Akce pro správu oblíbených režisérů
    elif action == 'favorite_directors':
        ui.display_favorite_directors(_addon, _handle)
        return
    elif action == 'tmdb_search_director':
        tmdb_handler.search_director(_addon, _handle)
        return
    elif action == 'tmdb_remove_favorite_director':
        ui.remove_favorite_director(params, _addon)
        return
    elif action == 'tmdb_remove_all_favorite_directors': # NOVÁ AKCE
        ui.remove_all_favorite_directors(_addon)
        return
    elif action == 'tmdb_search_director_movies':
        tmdb_handler.search_director_movies(params, _addon, _handle)
        return
    elif action == 'tmdb_add_director_to_favorites':
        director_id = params.get('director_id')
        director_name = params.get('director_name', '')
        if director_id:
            tmdb_handler.add_director_to_favorites(_addon, director_id, director_name)
        else:
            ui.popinfo("Chybí ID režiséra pro přidání do oblíbených.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
        return
    elif action == 'display_director_dialog':
        director_id = params.get('director_id')
        if director_id:
            from wcs.ui.dialogs.DialogDirectorDetails import DirectorDialog
            dialog = DirectorDialog("details_extras/DialogDirectorDetails.xml", _addon.getAddonInfo('path'), "Default", director_id=director_id)
            dialog.doModal()
            del dialog
        else:
            ui.popinfo("Chyba: Chybí ID režiséra.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_ERROR)
        return

    # Akce pro správu oblíbených scénáristů
    elif action == 'favorite_writers':
        ui.display_favorite_writers(_addon, _handle)
        return
    elif action == 'tmdb_search_writer':
        tmdb_handler.search_writer(_addon, _handle)
        return
    elif action == 'tmdb_add_writer_to_favorites':
        writer_id = params.get('writer_id')
        writer_name = params.get('writer_name', '')
        if writer_id:
            tmdb_handler.add_writer_to_favorites(_addon, writer_id, writer_name)
        else:
            ui.popinfo("Chybí ID scénáristy pro přidání do oblíbených.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
        return
    elif action == 'tmdb_remove_favorite_writer':
        ui.remove_favorite_writer(params, _addon)
        return
    elif action == 'tmdb_remove_all_favorite_writers': # NOVÁ AKCE
        ui.remove_all_favorite_writers(_addon)
        return
    elif action == 'tmdb_search_writer_movies':
        tmdb_handler.search_writer_movies(params, _addon, _handle)
        return
    elif action == 'display_writer_dialog':
        tmdb_handler.display_writer_dialog(params, _addon)
        return

    # Akce pro mazání uživatelských dat
    elif action == 'delete_all':
        dialog = xbmcgui.Dialog()
        if dialog.yesno(_addon.getLocalizedString(30187), _addon.getLocalizedString(30188)):
            # Smazat vše kromě přihlašovacích údajů
            user_data.store_favorite_actors(_addon, [])
            user_data.store_favorite_directors(_addon, [])
            user_data.store_tmdb_movies(_addon, [])
            user_data.store_tmdb_series(_addon, [])
            
            # Smazat soubory historie vyhledávání
            search_history_files = ['search_history.json', 'search_history_tmdb.json']
            for file in search_history_files:
                try:
                    file_path = os.path.join(_profile, file)
                    if os.path.exists(file_path):
                        os.remove(file_path)
                except Exception:
                    pass
                    
            ui.popinfo(_addon.getLocalizedString(30186), _addon.getAddonInfo('name'))
        return

    # Akce pro AI účet a nastavení
    elif action == 'ai_account':
        ui.display_ai_account_menu(_addon, _handle)
        return
    elif action == 'ai_usage_stats':
        from wcs.ai.UsageUI import display_ai_usage_menu
        display_ai_usage_menu(_addon, _handle)
        return
    elif action == 'ai_settings_beta':
        # Otevření nastavení doplňku na kategorii AI (index 5)
        _addon.openSettings()
        xbmc.executebuiltin('SetFocus(10, 5, true)')  # Přepne na kategorie
        xbmc.executebuiltin('SetFocus(11, 0, true)')  # Vybere první položku (beta heslo)
        return
    elif action == 'ai_stats':
        # Pouze pro zobrazení statistik, které jsou už v popisku položky v menu
        xbmcplugin.endOfDirectory(_handle)
        return
    elif action == 'ai_usage_menu':
        from wcs.ai.UsageUI import display_ai_usage_menu
        display_ai_usage_menu(_addon, _handle)
        return
    elif action == 'ai_usage_details':
        from wcs.ai.UsageUI import display_ai_usage_details
        display_ai_usage_details(params, _addon, _handle)
        return
    elif action == 'ai_usage_reset_menu':
        from wcs.ai.UsageUI import display_ai_usage_reset_menu
        display_ai_usage_reset_menu(_addon, _handle)
        return
    elif action == 'ai_usage_reset':
        from wcs.ai.UsageUI import handle_ai_usage_reset
        handle_ai_usage_reset(params, _addon)
        return
    elif action == 'ai_usage_export':
        from wcs.ai.UsageUI import handle_ai_usage_export
        handle_ai_usage_export(_addon)
        return
    elif action == 'ai_usage_model_detail':
        from wcs.ai.UsageUI import display_ai_usage_model_detail
        display_ai_usage_model_detail(params, _addon, _handle)
        return
    # Nová akce: Reset statistik AI (předá se novému systému)
    elif action == 'ai_reset_stats':
        # Resetuje statistiky využití AI API pomocí nového systému
        try:
            from wcs.ai.UsageTracker import reset_usage_data
            if reset_usage_data("all"):
                ui.popinfo("Všechny AI statistiky byly úspěšně resetovány.", _addon.getAddonInfo('name'))
            else:
                ui.popinfo("Chyba při resetování AI statistik.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_ERROR)
        except Exception as e:
            ui.popinfo(f"Chyba při resetování AI statistik: {e}", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_ERROR)
        xbmcplugin.endOfDirectory(_handle, succeeded=False, updateListing=True, cacheToDisc=False)
        # Obnovíme menu Účet AI, aby se zobrazily aktualizované statistiky
        xbmc.executebuiltin(f'Container.Update(plugin://{_addon.getAddonInfo("id")}?action=ai_account)')
    elif action == 'ai_chat_recommendation':
        # Pouzit Hub dialog s auto_action podle context_type
        from wcs.ai.DialogAIRecommendationHub import show_ai_recommendation_hub_dialog
        
        show_nav = params.get('show_nav_sidebar') == '1'
        nav_pos = int(params.get('nav_position', '0'))
        context_type = params.get('context_type')
        media_type = params.get('media_type', 'movie')
        
        # Urcit auto_action podle context_type
        auto_action = None
        
        if context_type == 'section_root':
            # Hlavni menu Filmy/Serialy -> bez kontextu
            if media_type == 'tv' or media_type == 'series':
                auto_action = 'hub_select:series'
            else:
                auto_action = 'hub_select:movies'
        elif context_type == 'my_movies':
            auto_action = 'hub_context:my_movies'
        elif context_type == 'my_series':
            auto_action = 'hub_context:my_series'
        elif context_type == 'my_collections':
            auto_action = 'hub_context:my_collections'
        elif context_type == 'favorite_actors':
            auto_action = 'hub_context:favorite_actors'
        elif context_type == 'favorite_directors':
            auto_action = 'hub_context:favorite_directors'
        elif context_type == 'favorite_writers':
            auto_action = 'hub_context:favorite_writers'
        elif context_type == 'recently_played':
            auto_action = 'hub_context:recently_played'
        elif context_type == 'my_mixed_collection':
            auto_action = 'hub_context:my_mixed_collection'
        # Pokud neni context_type, otevre se normalne Hub bez auto_action
        
        show_ai_recommendation_hub_dialog(_addon, show_nav_sidebar=show_nav, nav_position=nav_pos, auto_action=auto_action)
        return

    # --- Zobrazení kolekcí --- 
    elif action == 'show_movie_collection':
        tmdb_id = params.get('tmdb_id')
        if tmdb_id:
            tmdb_handler.display_movie_collection(tmdb_id, _addon, _handle) # Zavoláme novou funkci
        else:
            xbmcplugin.endOfDirectory(_handle)
            ui.popinfo("Chybí TMDB ID pro zobrazení kolekce.", _addon.getAddonInfo('name'))

    # Akce pro dialog kolekce
    elif action == 'display_collection_dialog':
        collection_id = params.get('collection_id')
        if collection_id:
            from wcs.ui.dialogs.DialogCollectionDetails import CollectionDialog
            dialog = CollectionDialog("details_extras/DialogCollectionDetails.xml", _addon.getAddonInfo('path'), "Default", collection_id=collection_id)
            dialog.doModal()
            del dialog
        else:
            ui.popinfo("Chyba: Chybí ID kolekce.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_ERROR)
        return
        
    # Akce pro zobrazení filmů v kolekci
    elif action == 'list_collection_movies':
        tmdb_handler.list_collection_movies(params, _addon, _handle)
        # Tato funkce volá endOfDirectory sama
        
    # Akce pro přidání všech filmů z kolekce do Moje filmy (voláno z kontext. menu seznamu kolekcí)
    elif action == 'add_collection_to_my_movies':
        collection_id = params.get('collection_id')
        if collection_id:
            # Voláme interní funkci, která již existuje a řeší přidání
            tmdb_handler.add_collection_movies_to_list(collection_id, _addon)
        else:
            ui.popinfo("Chybí ID kolekce pro přidání filmů.", _addon.getAddonInfo('name'))
        # Tato akce nevrací seznam, jen notifikace
        return 

    elif action == 'add_collection_to_library':
        collection_id = params.get('collection_id')
        if not collection_id:
             ui.popinfo("Chybí ID kolekce pro přidání do knihovny.", _addon.getAddonInfo('name'))
             return

        # 1. Načíst detaily kolekce (seznam filmů)
        coll_details = tmdb_handler.get_collection_details(collection_id)
        if not coll_details or 'parts' not in coll_details:
             ui.popinfo("Nepodařilo se načíst filmy z kolekce.", _addon.getAddonInfo('name'))
             return
        
        parts = coll_details['parts']
        count = 0

        # Progress dialog pro lepší UX při přidávání více filmů
        p_dialog = xbmcgui.DialogProgress()
        p_dialog.create(_addon.getAddonInfo('name'), "Přidávám filmy z kolekce do knihovny...")
        
        for i, movie in enumerate(parts):
            if p_dialog.iscanceled():
                break
            
            mid = movie.get('id')
            title = movie.get('title', 'Neznámý film')
            p_dialog.update(int((i / len(parts)) * 100), f"Zpracovávám: {title}")
            
            if mid:
                # Použijeme existující funkci z library manageru
                # Ta si řeší kontrolu složky a vytvoření .strm
                # POZOR: check_settings_and_add_item může vyvolat dialog pro výběr složky,
                # což v cyklu není ideální, ale pokud je složka nastavená, projde to hladce.
                # Pro jistotu zkontrolujeme nastavení složky PŘED cyklem? 
                # check_settings_and_add_item to dělá uvnitř.
                # Zjednodušení: zavoláme to pro první, a pak doufejme že je nastaveno.
                library.check_settings_and_add_item({'media_type': 'movie', 'movie_id': mid}, _addon)
                count += 1
                
        p_dialog.close()
        ui.popinfo(f"Přidáno {count} filmů z kolekce do knihovny.", _addon.getAddonInfo('name'))
        return 

    # NOVÁ AKCE pro zobrazení uživatelských filmových kolekcí
    elif action == 'display_my_movie_collections':
        tmdb_handler.display_my_movie_collections(_addon, _handle)
        # Tato funkce volá endOfDirectory sama

    elif action == 'display_similar_movies_dialog':
        tmdb_id = params.get('tmdb_id')
        original_title = params.get('original_title', '')
        if tmdb_id:
            from wcs.ui.dialogs.DialogSimilarMovies import SimilarMoviesDialog
            dialog = SimilarMoviesDialog("details_extras/DialogSimilarMovies.xml", _addon.getAddonInfo('path'), "Default", tmdb_id=tmdb_id, original_title=original_title)
            dialog.doModal()
            del dialog
        else:
            ui.popinfo("Chyba: Chybí TMDb ID pro zobrazení podobných filmů.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_ERROR)
        return

    elif action == 'display_similar_series_dialog':
        tmdb_id = params.get('tmdb_id')
        original_title = params.get('original_title', '')
        if tmdb_id:
            from wcs.ui.dialogs.DialogSimilarSeries import SimilarSeriesDialog
            dialog = SimilarSeriesDialog("details_extras/DialogSimilarMovies.xml", _addon.getAddonInfo('path'), "Default", tmdb_id=tmdb_id, original_title=original_title)
            dialog.doModal()
            del dialog
        else:
            ui.popinfo("Chyba: Chybí TMDb ID pro zobrazení podobných seriálů.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_ERROR)
        return

    elif action == 'show_search_dialog':
        # Nová akce pro zobrazení search dialogu přes RunPlugin
        search_query = params.get('search_query', '')
        movie_title = params.get('movie_title', '')
        movie_year = params.get('movie_year', '')
        poster_url = params.get('poster_url', '')
        rating = params.get('rating', '')
        genre = params.get('genre', '')
        director = params.get('director', '')
        runtime = params.get('runtime', '')
        plot = params.get('plot', '')
        is_episode = params.get('is_episode', 'false') == 'true'
        series_name = params.get('series_name', '')
        season_number = params.get('season_number')
        episode_number = params.get('episode_number')
        tmdb_id = params.get('tmdb_id')
        preferred_quality = params.get('preferred_quality')
        from_library = params.get('from_library', 'false') == 'true'
        
        # Konverze na int pokud jsou k dispozici
        if season_number:
            try:
                season_number = int(season_number)
            except (ValueError, TypeError):
                season_number = None
        if episode_number:
            try:
                episode_number = int(episode_number)
            except (ValueError, TypeError):
                episode_number = None
        if tmdb_id:
            try:
                tmdb_id = int(tmdb_id)
            except (ValueError, TypeError):
                tmdb_id = None
        
        if search_query:
            from wcs.ui.dialogs.DialogWcsSearch import show_search_dialog
            result = show_search_dialog(
                search_query=search_query,
                movie_title=movie_title,
                movie_year=movie_year,
                poster_url=poster_url,
                rating=rating,
                genre=genre,
                director=director,
                runtime=runtime,
                plot=plot,
                is_episode=is_episode,
                series_name=series_name,
                season_number=season_number,
                episode_number=episode_number,
                tmdb_id=tmdb_id,
                preferred_quality=preferred_quality,
                from_library=from_library
            )
            
            # Pokud byl vybrán soubor, přehraj ho
            if result and 'ident' in result and 'name' in result:
                from urllib.parse import quote_plus
                play_url = (f"plugin://{_addon.getAddonInfo('id')}?action=play"
                           f"&ident={result['ident']}"
                           f"&name={quote_plus(result['name'])}"
                           f"&tmdb_title={quote_plus(movie_title)}"
                           f"&tmdb_year={quote_plus(movie_year) if movie_year else ''}"
                           f"&tmdb_plot={quote_plus(plot) if plot else ''}"
                           f"&tmdb_thumb={quote_plus(poster_url) if poster_url else ''}"
                           f"&tmdb_fanart={quote_plus(poster_url) if poster_url else ''}"
                           f"&tmdb_rating={quote_plus(rating) if rating else ''}"
                           f"&tmdb_genre={quote_plus(genre) if genre else ''}"
                           f"&tmdb_director={quote_plus(director) if director else ''}"
                           f"&tmdb_runtime={quote_plus(runtime) if runtime else ''}")
                
                # Nastavit flag že se přehrává film z vyhledávání
                _addon.setSetting('movie_playing_from_search', 'true')
                
                xbmc.executebuiltin('Dialog.Close(all, true)')
                xbmc.executebuiltin(f'PlayMedia({play_url})')
        else:
            xbmcgui.Dialog().notification("Chyba", "Prázdný vyhledávací dotaz", xbmcgui.NOTIFICATION_ERROR)
        return

    elif action == 'display_series_creators_dialog':
        tmdb_id = params.get('tmdb_id')
        title = params.get('title', '')
        if tmdb_id:
            from wcs.ui.dialogs.DialogSeriesCreators import SeriesCreatorsDialog
            dialog = SeriesCreatorsDialog("details_extras/DialogActorDetails.xml", _addon.getAddonInfo('path'), "Default", tmdb_id=tmdb_id, title=title)
            dialog.doModal()
            del dialog
        else:
            ui.popinfo("Chyba: Chybí TMDb ID pro zobrazení tvůrců seriálu.", _addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_ERROR)
        return

        # Akce 'my_library' byla nahrazena 'my_added_items'
        return
    elif action == 'play_text_game':
        title = params.get('title', '')
        fanart = params.get('fanart', '')
        poster = params.get('poster', '')  # Nový parametr z kontextového menu
        tmdb_id = params.get('tmdb_id', '')
        media_type = params.get('media_type', 'movie')  # Default na film pro zpětnou kompatibilitu
        plot = params.get('plot', '')  # Nový parametr z kontextového menu
        
        # Nové parametry pro epizody
        episode_title = params.get('episode_title', '')
        season_number = params.get('season_number')
        episode_number = params.get('episode_number')
        
        # Konverze na int pokud jsou k dispozici
        if season_number:
            try:
                season_number = int(season_number)
            except (ValueError, TypeError):
                season_number = None
        if episode_number:
            try:
                episode_number = int(episode_number)
            except (ValueError, TypeError):
                episode_number = None
        
        # Získání dodatečných dat o filmu/seriálu z TMDb (pouze pokud nejsou předány)
        genre = ''
        banner_fanart = fanart or poster  # Fallback na původní fanart nebo poster
        actor_data = None  # Pro herce
        
        # Pokud nemáme plot nebo fanart, zkusíme je získat z TMDb
        if not plot or not fanart:
            if tmdb_id:
                try:
                    if media_type in ['actor', 'director', 'writer']:
                        # Pro všechny osoby použijeme univerzální funkci get_person_details
                        person_details = tmdb_handler.get_person_details(tmdb_id, media_type)
                        if person_details:
                            if not plot:
                                plot = person_details.get('biography', '')
                            if not title:
                                title = person_details.get('name', '')
                            # Získání banner fanartu z profilového obrázku
                            profile_path = person_details.get('profile_path', '')
                            if profile_path:
                                banner_fanart = f"https://image.tmdb.org/t/p/w780{profile_path}"
                            # Sestavení actor_data pro herní systém
                            actor_data = {
                                'name': person_details.get('name', ''),
                                'biography': person_details.get('biography', ''),
                                'known_for_department': person_details.get('known_for_department', ''),
                                'profile_path': person_details.get('profile_path', ''),
                                'known_for': person_details.get('known_for', [])
                            }
                    elif media_type == 'series' or media_type == 'tv':
                        # Pro seriály použijeme TV API endpoint
                        api_params = {
                            'api_key': tmdb_handler.get_tmdb_api_key(_addon),
                            'language': 'cs-CZ'
                        }
                        url = f'https://api.themoviedb.org/3/tv/{tmdb_id}'
                        response = requests.get(url, params=api_params, timeout=10)
                        response.raise_for_status()
                        series_details = response.json()
                        if series_details:
                            if not plot:
                                plot = series_details.get('overview', '')
                            genres = series_details.get('genres', [])
                            if genres:
                                genre = ', '.join([g.get('name', '') for g in genres[:3]])  # Max 3 žánry
                            # Získání banner fanartu
                            backdrop_path = series_details.get('backdrop_path', '')
                            if backdrop_path:
                                banner_fanart = f"https://image.tmdb.org/t/p/w780{backdrop_path}"
                    else:
                        # Pro filmy použijeme existující funkci
                        movie_details = tmdb_handler.get_movie_detailed_info(tmdb_id)
                        if movie_details:
                            if not plot:
                                plot = movie_details.get('overview', '')
                            genres = movie_details.get('genres', [])
                            if genres:
                                genre = ', '.join([g.get('name', '') for g in genres[:3]])  # Max 3 žánry
                            # Získání banner fanartu
                            backdrop_path = movie_details.get('backdrop_path', '')
                            if backdrop_path:
                                banner_fanart = f"https://image.tmdb.org/t/p/w780{backdrop_path}"
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Error getting {media_type} details for text game: {e}", level=xbmc.LOGERROR)
        
                    xml_filename = "games/DialogWcsTextGame.xml"
        # Předám media_type, banner_fanart, actor_data a parametry epizod do TextGameWindow
        from wcs.games.DialogWcsTextGame import TextGameWindow
        if 'xml_filename' not in locals():
            xml_filename = "games/DialogWcsTextGame.xml"
        dialog = TextGameWindow(
            xml_filename, 
            _addon.getAddonInfo('path'), 
            "Default", 
            title=title, 
            fanart=fanart or poster, 
            plot=plot, 
            genre=genre, 
            media_type=media_type, 
            banner_fanart=banner_fanart, 
            actor_data=actor_data,
            episode_title=episode_title,
            season_number=season_number,
            episode_number=episode_number
        )
        dialog.doModal()
        del dialog
        return
    elif action == 'show_tmdb_login':
        # Zobrazí dialog pro zadání TMDb API klíče
        result = show_tmdb_login_dialog()
        if result['api_key_entered']:
            # API klíč byl zadán, můžeme obnovit menu
            xbmc.executebuiltin('Container.Refresh')
        return
    elif action == 'tmdb_login':
        # Zobrazí dialog pro přihlášení k TMDb
        result = show_tmdb_login_dialog()
        if result['api_key_entered']:
            # API klíč byl zadán, můžeme obnovit nastavení
            xbmc.executebuiltin('Container.Refresh')
        return
    elif action == 'show_tmdb_guide':
        # Zobrazí dialog s návodem pro získání TMDb API klíče (z nastavení)
        xml_filename = "guide/DialogTMDbGuide.xml"
        addon_path = _addon.getAddonInfo('path')
        
        window = TMDbGuideDialog(xml_filename, addon_path)
        window.doModal()
        del window
        return
    elif action == 'show_webshare_guide':
        # Zobrazí dialog s návodem pro registraci na Webshare.cz
        xml_filename = "guide/DialogWebshareGuide.xml"
        addon_path = _addon.getAddonInfo('path')
        
        window = WebshareGuideDialog(xml_filename, addon_path)
        window.doModal()
        del window
        return
    elif action == 'webshare_login':
        from wcs.login.DialogWebshareLogin import show_webshare_login_dialog
        show_webshare_login_dialog(_addon)
        return
    elif action == 'show_ai_guide':
        # Zobrazí dialog s návodem pro získání AI API klíčů
        xml_filename = "guide/DialogAIGuide.xml"
        addon_path = _addon.getAddonInfo('path')
        
        window = AIGuideDialog(xml_filename, addon_path)
        window.doModal()
        del window
        return
    elif action == 'show_library_guide':
        # Zobrazí dialog s návodem pro nastavení knihovny Kodi
        xml_filename = "guide/DialogLibraryGuide.xml"
        addon_path = _addon.getAddonInfo('path')
        
        window = LibraryGuideDialog(xml_filename, addon_path)
        window.doModal()
        del window
        return
    elif action == 'show_remote_guide':
        # Zobrazí dialog s návodem pro nastavení dálkového ovládání
        xml_filename = "guide/DialogRemoteGuide.xml"
        addon_path = _addon.getAddonInfo('path')
        
        window = RemoteGuideDialog(xml_filename, addon_path)
        window.doModal()
        del window
        return
    elif action == 'csfd_top_series':
        csfd_handler.list_csfd_series(params, _addon, _handle)
        return
    elif action == 'csfd_favorite_series':
        csfd_handler.list_csfd_favorite_series(params, _addon, _handle)
        return
    elif action == 'csfd_favorite_movies':
        csfd_handler.list_csfd_favorite_movies(params, _addon, _handle)
        return
    elif action == 'csfd_controversial_movies':
        csfd_handler.list_csfd_controversial_movies(params, _addon, _handle)
        return
    elif action == 'csfd_worst_movies':
        csfd_handler.list_csfd_worst_movies(params, _addon, _handle)
        return
    elif action == 'csfd_controversial_series':
        csfd_handler.list_csfd_controversial_series(params, _addon, _handle)
        return
    elif action == 'csfd_worst_series':
        csfd_handler.list_csfd_worst_series(params, _addon, _handle)
        return
    elif action == 'csfd_menu':
        ui.display_csfd_menu(_addon, _handle)
        return
    elif action == 'csfd_movies_menu':
        ui.display_csfd_movies_menu(_addon, _handle)
        return
    elif action == 'csfd_series_menu':
        ui.display_csfd_series_menu(_addon, _handle)
        return
    elif action == 'csfd_series_seasons':
        series_csfd_id = params.get('series_csfd_id')
        series_name = params.get('series_name', 'Seriál')
        if series_csfd_id:
            ui.display_csfd_series_seasons(_addon, _handle, series_csfd_id, series_name)
        else:
            ui.popinfo("Chyba: Chybí ČSFD ID seriálu pro zobrazení sezón.", _addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(_handle)
        return
    elif action == 'csfd_series_episodes':
        series_csfd_id = params.get('series_csfd_id')
        season_csfd_id = params.get('season_csfd_id')
        season_name = params.get('season_name', 'Sezóna')
        series_name = params.get('series_name', 'Seriál')
        if series_csfd_id and season_csfd_id:
            ui.display_csfd_season_episodes(_addon, _handle, series_csfd_id, season_csfd_id, season_name, series_name)
        else:
            ui.popinfo("Chyba: Chybí ČSFD ID seriálu nebo sezóny pro zobrazení epizod.", _addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(_handle)
        return
    elif action == 'csfd_episode_detail':
        series_csfd_id = params.get('series_csfd_id')
        episode_csfd_id = params.get('episode_csfd_id')
        episode_name = params.get('episode_name', 'Epizoda')
        if series_csfd_id and episode_csfd_id:
            # Pro nyní přesměrujeme na vyhledávání
            # V budoucnu lze implementovat detailní stránku epizody
            episode_url = f"https://www.csfd.cz/film/{series_csfd_id}/{episode_csfd_id}/"
            csfd_handler.open_csfd_url({'url': episode_url}, _addon)
        else:
            ui.popinfo("Chyba: Chybí ČSFD ID seriálu nebo epizody pro zobrazení detailu.", _addon.getAddonInfo('name'))
        return
    elif action == 'display_series_seasons' and 'series_id' in params:
        tmdb_handler.display_series_seasons(params, _addon, _handle)
        return
    elif action == 'delete_webshare_account':
        dialog = xbmcgui.Dialog()
        if dialog.yesno(_addon.getLocalizedString(32080), _addon.getLocalizedString(32081)):
            _addon.setSetting('wsuser', '')
            _addon.setSetting('wspass', '')
            _addon.setSetting('token', '')
            _addon.setSetting('webshare_status', 'Nepřihlášen')
            _addon.setSetting('webshare_vip_info', '')
            dialog.notification(_addon.getAddonInfo('name'), _addon.getLocalizedString(32082), xbmcgui.NOTIFICATION_INFO, 3000)
        return
    elif action == 'delete_ai_account':
        dialog = xbmcgui.Dialog()
        provider = params.get('provider', _addon.getSetting('ai_provider'))
        provider_names = {
            'openai': 'OpenAI',
            'anthropic': 'Anthropic', 
            'google': 'Google',
            'mistral': 'Mistral',
            'groq': 'Groq'
        }
        selected_provider = provider_names.get(provider, 'AI')
        
        if dialog.yesno(f'Smazat účet {selected_provider}', f'Opravdu chcete smazat účet pro {selected_provider}? Tato akce je nevratná.'):
            # Smažeme pouze zadaného poskytovatele
            if provider == 'openai':
                _addon.setSetting('openai_api_key', '')
                _addon.setSetting('openai_status', 'Nepřihlášen')
            elif provider == 'anthropic':
                _addon.setSetting('anthropic_api_key', '')
                _addon.setSetting('anthropic_status', 'Nepřihlášen')
            elif provider == 'google':
                _addon.setSetting('google_api_key', '')
                _addon.setSetting('google_status', 'Nepřihlášen')
            elif provider == 'mistral':
                _addon.setSetting('mistral_api_key', '')
                _addon.setSetting('mistral_status', 'Nepřihlášen')
            elif provider == 'groq':
                _addon.setSetting('groq_api_key', '')
                _addon.setSetting('groq_status', 'Nepřihlášen')
            
            dialog.notification(_addon.getAddonInfo('name'), f'Účet {selected_provider} byl smazán.', xbmcgui.NOTIFICATION_INFO, 3000)
        return
    elif action == 'delete_tmdb_account':
        dialog = xbmcgui.Dialog()
        if dialog.yesno(_addon.getLocalizedString(32086), _addon.getLocalizedString(32087)):
            _addon.setSetting('tmdb_api_key', '')
            _addon.setSetting('tmdb_status', 'Nepřihlášen')
            dialog.notification(_addon.getAddonInfo('name'), _addon.getLocalizedString(32088), xbmcgui.NOTIFICATION_INFO, 3000)
        return
    elif action == 'reset_library_folders':
        dialog = xbmcgui.Dialog()
        if dialog.yesno(_addon.getLocalizedString(32090), _addon.getLocalizedString(32091)):

            # Cesta k uživatelskému settings.xml
            addon_profile = xbmcaddon.Addon().getAddonInfo('profile')
            user_settings_path = os.path.join(addon_profile, 'settings.xml')
            if xbmcvfs.exists(user_settings_path):
                try:
                    # Načti XML
                    f = xbmcvfs.File(user_settings_path)
                    xml_data = f.read().decode('utf-8') if hasattr(f, 'decode') else f.read()
                    f.close()
                    root = ET.fromstring(xml_data)
                    # Najdi a odstraň příslušné <setting>
                    for setting_id in ('library_movies_folder', 'library_series_folder'):
                        for elem in root.findall(f".//setting[@id='{setting_id}']"):
                            root.remove(elem)
                    # Ulož zpět
                    new_xml = ET.tostring(root, encoding='utf-8', method='xml').decode('utf-8')
                    f = xbmcvfs.File(user_settings_path, 'w')
                    f.write(new_xml)
                    f.close()
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Chyba při mazání složek knihovny v settings.xml: {e}", xbmc.LOGERROR)
            dialog.notification(_addon.getAddonInfo('name'), _addon.getLocalizedString(32092), xbmcgui.NOTIFICATION_INFO, 3000)
            xbmc.executebuiltin('Dialog.Close(settings)')
            xbmcgui.Dialog().ok(_addon.getAddonInfo('name'), _addon.getLocalizedString(32093))
            xbmc.executebuiltin('Dialog.Close(all,true)')
        return
    elif action == 'reset_download_folder':
        dialog = xbmcgui.Dialog()
        if dialog.yesno(_addon.getLocalizedString(32134), _addon.getLocalizedString(32135)):

            # Cesta k uživatelskému settings.xml
            addon_profile = xbmcaddon.Addon().getAddonInfo('profile')
            user_settings_path = os.path.join(addon_profile, 'settings.xml')
            if xbmcvfs.exists(user_settings_path):
                try:
                    # Načti XML
                    f = xbmcvfs.File(user_settings_path)
                    xml_data = f.read().decode('utf-8') if hasattr(f, 'decode') else f.read()
                    f.close()
                    root = ET.fromstring(xml_data)
                    # Najdi a odstraň příslušné <setting>
                    for setting_id in ('dfolder',):
                        for elem in root.findall(f".//setting[@id='{setting_id}']"):
                            root.remove(elem)
                    # Ulož zpět
                    new_xml = ET.tostring(root, encoding='utf-8', method='xml').decode('utf-8')
                    f = xbmcvfs.File(user_settings_path, 'w')
                    f.write(new_xml)
                    f.close()
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Chyba při mazání složky pro stahování v settings.xml: {e}", xbmc.LOGERROR)
            dialog.notification(_addon.getAddonInfo('name'), _addon.getLocalizedString(32136), xbmcgui.NOTIFICATION_INFO, 3000)
            xbmc.executebuiltin('Dialog.Close(settings)')
            xbmcgui.Dialog().ok(_addon.getAddonInfo('name'), _addon.getLocalizedString(32093))
            xbmc.executebuiltin('Dialog.Close(all,true)')
        return

    # --- Testovací tlačítko pro AI hlas Milionáře ---
    elif action == 'milionar_tts_test_button':
        try:
            tts_voice = _addon.getSetting('milionar_tts_voice') or 'alloy'
            tts_style_value = _addon.getSetting('milionar_tts_style')
            from wcs.ai import TTSClient as openai_tts
            test_text = 'Kdo ve filmu Terminátor 2: Den zúčtování hraje hlavní roli kyborga T-800?'
            audio_path = openai_tts.generate_tts(test_text, voice=tts_voice, style=tts_style_value)
            xbmc.executebuiltin(f'PlayMedia("{audio_path}", noresume)')
            xbmcgui.Dialog().notification(_addon.getAddonInfo('name'), 'Testovací AI hlas přehrán.', xbmcgui.NOTIFICATION_INFO, 3000)
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] TTS test chyba: {e}", level=xbmc.LOGWARNING)
            xbmcgui.Dialog().notification(_addon.getAddonInfo('name'), f'Chyba TTS: {e}', xbmcgui.NOTIFICATION_ERROR, 4000)
        return

    # Výchozí akce (zobrazení hlavního menu)
    else:
        if ui.has_tmdb_api_key(_addon):
            ui.display_main_menu(_addon, _handle)
        else:
            ui.display_limited_menu(_addon, _handle)