# wcs/ai/DialogMySeries.py
"""
Dialog "Moje seriály" - Fullscreen dialog pro uložené seriály uživatele.
Dědí z AIChatRecommendationDialog a sdílí jeho XML grafiku.
"""

import xbmc
import xbmcgui
import xbmcaddon
import threading
from urllib.parse import quote_plus

from wcs.ai.DialogAIChatRecommendation import AIChatRecommendationDialog, AIRecommendationPromptBuilder
from wcs import user_data
from wcs import utils as wcs_utils


class MySeriesDialog(AIChatRecommendationDialog):
    """
    Dialog pro uložené seriály uživatele.
    Dědí všechny funkce z AI Chat dialogu, ale přetěžuje:
    - onInit() - načítá z user_data místo AI inference
    - _populate_preview_actions() - upravené akční tlačítka (Detaily, Sezóny, Podobné, Smazat)
    """
    
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        # Parent zpracovává show_nav_sidebar, nav_position, context
        super(MySeriesDialog, self).__init__(xml_filename, addon_path, *args, **kwargs)
        self._series_items = []  # Cache pro položky seriálů
        self._current_item_index = 0  # Index aktuální položky pro akce
        # Identifikátor sekce pro navigační sidebar
        self._nav_section_id = 'my_series'


    
    def onInit(self):
        """Přetížená inicializace - načte položky z uložených seriálů."""
        # Resetovat stav (využívá parent metodu)
        self._reset_state()
        self.media_type = 'series'
        self._is_showing_collection = True  # Flag pro rozlišení mezi sbírkou a AI doporučeními
        
        # Načíst seriály
        self._series_items = user_data.load_tmdb_series(self.addon)
        
        # Nastavit titulek
        self.setProperty('WCS.AIChat.Title', 'Moje seriály')
        self.setProperty('WCS.AIChat.Status', 'Připraven')
        
        if not self._series_items:
            # Prázdná sbírka - zobrazit sidebar s možností přidat
            self.append_chat_message("AI", "Vaše sbírka seriálů je prázdná. Přidejte první seriál pomocí tlačítka níže.")
            self._setup_empty_state_buttons()
            # Zobrazit chat sidebar
            self.setProperty('WCS.AIChat.Visible', 'true')
            self.setFocusId(4000)
            return
        
        # Zobrazit uvítací zprávu v chatu
        self.append_chat_message("AI", f"Ve vaší sbírce je {len(self._series_items)} seriálů. Vyberte seriál nebo si nechte doporučit podobné.")
        
        # Nastavit AI tlačítka
        self._setup_initial_buttons()
        
        # Naplnit poster flow ze seriálů
        self._populate_from_items()
        
        # Pokud byl dialog spuštěn z navigačního sidebaru, zobrazit ho bez animace
        if self._show_nav_on_init:
            self._show_nav_sidebar(animate=False)
        else:
            # Skrýt chat sidebar, fokus na postery s krátkým zpožděním
            self.clearProperty('WCS.AIChat.Visible')
            self._delayed_focus(2000, 0.5)

    
    def _setup_empty_state_buttons(self):
        """Nastaví tlačítka pro prázdný stav."""
        buttons = [
            {"label": "Přidat seriál", "value": "add_item"},
            {"label": "Doporučit nový seriál", "value": "recommend_new"}
        ]
        self.update_buttons(buttons)
    
    def _setup_initial_buttons(self):
        """Nastaví počáteční AI tlačítka."""
        buttons = [
            {"label": "Přidat seriál", "value": "add_item"},
            {"label": "Doporučit na základě sbírky", "value": "recommend_from_collection"},
            {"label": "Doporučit nový seriál", "value": "recommend_new"}
        ]
        self.update_buttons(buttons)
    
    def _populate_from_items(self):
        """Naplní poster flow položkami ze seriálů."""
        list_control = self.getControl(2000)
        list_control.reset()
        
        for item in self._series_items:
            title = item.get('name') or item.get('title', '')
            year = item.get('year', '') or item.get('first_air_date', '')[:4] if item.get('first_air_date') else ''
            poster_path = item.get('poster_path', '')
            poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
            tmdb_id = item.get('id', '')
            
            display_title = f"{title} ({year})" if year else title
            
            list_item = xbmcgui.ListItem(label=display_title)
            
            if poster:
                list_item.setArt({'poster': poster})
            
            # Uložit metadata pro akce
            list_item.setProperty('tmdb_id', str(tmdb_id) if tmdb_id else '')
            list_item.setProperty('media_type', 'series')
            list_item.setProperty('title', title)
            list_item.setProperty('year', str(year) if year else '')
            
            list_control.addItem(list_item)
        
        # Pokud máme položky, nastavit preview pro první
        if list_control.size() > 0:
            self._update_preview_for_current_item()
    
    def _populate_preview_actions(self, media_type, tmdb_id, title, year):
        """Přetíženo - upravené akční tlačítka pro Moje seriály."""
        try:
            list_control = self.getControl(6100)
            list_control.reset()
            
            # Získat aktuální index
            poster_list = self.getControl(2000)
            current_index = poster_list.getSelectedPosition()
            
            # Rozlišit mezi položkami ze sbírky a AI doporučeními
            is_collection_item = getattr(self, '_is_showing_collection', False) and current_index >= 0 and current_index < len(self._series_items)
            
            if is_collection_item:
                # ===== SCÉNÁŘ 1: Položky ze sbírky =====
                actions = [
                    {'label': 'Detaily', 'action': 'details'},
                    {'label': 'Sezóny', 'action': 'seasons'},
                    {'label': 'Podobné', 'action': 'similar'},
                    {'label': 'Smazat', 'action': 'remove_from_list'},
                ]
            else:
                # ===== SCÉNÁŘ 2: AI doporučení (kopie z AI chat dialogu) =====
                actions = [
                    {'label': 'Detaily', 'action': 'details'},
                    {'label': 'Sezóny', 'action': 'seasons'},
                    {'label': 'Moje seriály', 'action': 'add_list'},
                    {'label': 'Knihovna', 'action': 'add_library'},
                    {'label': 'Podobné', 'action': 'similar'},
                ]
            
            for act in actions:
                item = xbmcgui.ListItem(label=act['label'])
                item.setProperty('action', act['action'])
                item.setProperty('tmdb_id', str(tmdb_id))
                item.setProperty('title', title)
                item.setProperty('year', year)
                item.setProperty('media_type', media_type)
                item.setProperty('collection_index', str(current_index))
                list_control.addItem(item)
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] MySeries preview actions error: {e}", xbmc.LOGERROR)
    
    def _handle_preview_action_click(self):
        """Přetíženo - zpracování akcí specifických pro Moje seriály."""
        try:
            # Reset automatického zobrazení tlačítek (jakákoliv akce ukončí delay)
            self.clearProperty('WCS.AIPreview.ShowActions')
            
            list_control = self.getControl(6100)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            action = item.getProperty('action')
            tmdb_id = item.getProperty('tmdb_id')
            title = item.getProperty('title')
            year = item.getProperty('year')
            media_type = item.getProperty('media_type')
            collection_index = item.getProperty('collection_index')
            
            if action == 'remove_from_list':
                self._handle_remove_from_list(collection_index)
            elif action == 'similar':
                self._handle_similar(title, year)
            elif action == 'seasons':
                self._handle_seasons(tmdb_id, title)
            else:
                # Delegovat na parent pro standardní akce (details, add_list, atd.)
                super(MySeriesDialog, self)._handle_preview_action_click()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] MySeries action error: {e}", xbmc.LOGERROR)
    
    def _handle_seasons(self, tmdb_id, title):
        """Zobrazí sezóny seriálu."""
        try:
            # Zavřít dialog
            self.close()
            
            # Přejít na sezóny seriálu
            import xbmc
            xbmc.executebuiltin(f"Container.Update(plugin://{self.addon.getAddonInfo('id')}?action=tmdb_series_detail&series_id={tmdb_id}&name={quote_plus(title)})")
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Seasons action error: {e}", xbmc.LOGERROR)
    
    def _handle_similar(self, title, year):
        """Spustí AI doporučení podobných seriálů."""
        try:
            self.media_type = 'series'
            inspiration_text = f"Seriál '{title}' ({year}) mě nadchl. Hledám podobný seriál - může mít jinou atmosféru, ale ať mě to vtáhne stejně."
            
            # Vymazat starý kontext (jde o prostý dotaz bez kontextu)
            self.context = None
            
            # Reinicializovat prompt builder s novým media_type a bez kontextu
            self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
            
            # Skrýt overlay a fokus na chat
            self._clear_preview_properties()
            self.setFocusId(4000)
            
            # Spustit načítací animaci
            self._start_button_loading("Hledám podobné...")
            self.handle_user_input(inspiration_text)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Similar action error: {e}", xbmc.LOGERROR)
    
    def fetch_and_display_recommendations(self, recommendations):
        """Přetíženo - nastaví flag že už nezobrazujeme sbírku."""
        # Po AI doporučení už nezobrazujeme sbírku
        self._is_showing_collection = False
        # Volat parent implementaci
        super(MySeriesDialog, self).fetch_and_display_recommendations(recommendations)
    
    def _handle_no_recommendations(self):
        """Přetíženo - obnoví původní tlačítka když AI nevrátí nic."""
        self.append_system_message("Model nedoporučil žádné tituly.")
        # Obnovit původní tlačítka
        self._setup_initial_buttons()

    
    def _handle_remove_from_list(self, collection_index):
        """Smaže položku ze sbírky."""
        try:
            idx = int(collection_index) if collection_index else 0
            if idx < 0 or idx >= len(self._series_items):
                return
            
            removed_item = self._series_items[idx]
            item_title = removed_item.get('name') or removed_item.get('title', 'Neznámý')
            
            # Potvrzovací dialog
            dialog = xbmcgui.Dialog()
            if not dialog.yesno("Smazat ze sbírky", f"Opravdu chcete smazat '{item_title}' ze sbírky?"):
                return
            
            # Smazat položku
            self._series_items.pop(idx)
            user_data.store_tmdb_series(self.addon, self._series_items)
            
            # Notifikace
            xbmcgui.Dialog().notification(
                self.addon.getAddonInfo('name'),
                f"'{item_title}' smazáno ze sbírky",
                xbmcgui.NOTIFICATION_INFO
            )
            
            # Obnovit poster flow
            if self._series_items:
                self._populate_from_items()
            else:
                # Sbírka je prázdná - přepnout na prázdný stav
                self.getControl(2000).reset()
                self._setup_empty_state_buttons()
                self.setProperty('WCS.AIChat.Visible', 'true')
                self.setFocusId(4000)
                self.append_chat_message("AI", "Vaše sbírka seriálů je prázdná. Přidejte první seriál pomocí tlačítka níže.")
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Remove from list error: {e}", xbmc.LOGERROR)
    
    def _handle_add_series(self):
        """Otevře dialog pro přidání nového seriálu - BEZ zavření našeho dialogu."""
        try:
            import requests
            from wcs.metadata.TMDbClient import get_tmdb_api_key
            
            # 1. Zobrazit klávesnici pro vyhledávání
            search_query = xbmcgui.Dialog().input(
                self.addon.getLocalizedString(31062),  # "Hledat seriál"
                type=xbmcgui.INPUT_ALPHANUM
            )
            
            if not search_query:
                return  # Uživatel zrušil
            
            # 2. Vyhledat na TMDb
            try:
                url = f'https://api.themoviedb.org/3/search/tv?api_key={get_tmdb_api_key()}&language=cs-CZ&query={quote_plus(search_query)}'
                response = requests.get(url, timeout=10)
                response.raise_for_status()
                results = response.json().get('results', [])
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] TMDb search error: {e}", xbmc.LOGERROR)
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    "Chyba při vyhledávání",
                    xbmcgui.NOTIFICATION_ERROR
                )
                return
            
            if not results:
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    f"Nenalezen žádný seriál: {search_query}",
                    xbmcgui.NOTIFICATION_INFO
                )
                return
            
            # 3. Zobrazit výběrový dialog
            choices = [f"{item['name']} ({item.get('first_air_date', '')[:4]})" for item in results]
            selected_index = xbmcgui.Dialog().select(
                self.addon.getLocalizedString(31064),  # "Vyberte seriál"
                choices
            )
            
            if selected_index < 0:
                return  # Uživatel zrušil
            
            selected_item = results[selected_index]
            
            # 4. Zkontrolovat, zda seriál už není ve sbírce
            if any(s.get('id') == selected_item['id'] for s in self._series_items):
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    f"'{selected_item['name']}' už je ve sbírce",
                    xbmcgui.NOTIFICATION_INFO
                )
                return
            
            # 5. Přidat seriál do sbírky
            new_series = {
                'id': selected_item['id'],
                'name': selected_item['name'],
                'year': selected_item.get('first_air_date', '')[:4],
                'overview': selected_item.get('overview', ''),
                'poster_path': selected_item.get('poster_path', ''),
                'backdrop_path': selected_item.get('backdrop_path', '')
            }
            self._series_items.insert(0, new_series)
            user_data.store_tmdb_series(self.addon, self._series_items)
            
            # 6. Notifikace o úspěchu
            xbmcgui.Dialog().notification(
                self.addon.getAddonInfo('name'),
                f"'{selected_item['name']}' přidán do sbírky",
                xbmcgui.NOTIFICATION_INFO
            )
            
            # 7. Obnovit poster flow a nastavit flag
            self._is_showing_collection = True
            self._populate_from_items()
            
            # Skrýt sidebar a fokusovat na postery
            self.clearProperty('WCS.AIChat.Visible')
            self.setFocusId(2000)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Add series error: {e}", xbmc.LOGERROR)

    
    def onClick(self, controlId):
        """Přetíženo pro handling speciálních tlačítek."""
        # AI tlačítka
        if controlId == 4000:
            list_control = self.getControl(4000)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            val = item.getProperty('value')
            label = item.getLabel()
            
            if val == 'add_item':
                self._handle_add_series()
            elif val == 'recommend_from_collection':
                self._start_ai_from_collection()
            elif val == 'recommend_new':
                self._start_ai_without_context()
            else:
                # Delegovat na parent
                super(MySeriesDialog, self).onClick(controlId)
        else:
            # Delegovat na parent pro ostatní controls
            super(MySeriesDialog, self).onClick(controlId)
    
    def _start_ai_from_collection(self):
        """Spustí AI doporučení na základě sbírky seriálů."""
        if not self._series_items:
            self.append_system_message("Vaše sbírka seriálů je prázdná. Nejdříve přidejte nějaké seriály.")
            return
        
        self.media_type = 'series'
        
        # Nastavit kontext
        self.context = {
            'type': 'my_series',
            'media_type': 'series',
            'data': self._series_items
        }
        
        # Inicializovat prompt builder
        self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
        
        # Vytvořit zprávu pro uživatele
        user_msg = "Doporuč mi seriály na základě mé sbírky."
        
        # Přidat do historie a spustit
        self.history.append({"role": "user", "content": user_msg})
        self.append_chat_message("Ty", user_msg)
        
        # Spustit loading animaci
        self._start_button_loading("Analyzuji sbírku...")
        
        # Zavolat AI
        self._start_ai_conversation(is_initial_context=True)
    
    def _start_ai_without_context(self):
        """Spustí AI doporučení bez kontextu (nové seriály)."""
        self.media_type = 'series'
        self.context = None
        
        # Inicializovat prompt builder
        self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
        
        # Vytvořit zprávu
        user_msg = "Doporuč mi nové zajímavé seriály."
        
        # Přidat do historie a spustit
        self.history.append({"role": "user", "content": user_msg})
        self.append_chat_message("Ty", user_msg)
        
        # Spustit loading animaci
        self._start_button_loading("Hledám seriály...")
        
        # Zavolat AI
        self._start_ai_conversation(is_initial_context=False)


def show_my_series_dialog(addon, show_nav_sidebar=False, nav_position=0):
    """Zobrazí dialog Moje seriály."""
    addon_path = addon.getAddonInfo('path')
    
    # Použijeme SDÍLENÝ XML z AI Chat dialogu
    dialog = MySeriesDialog(
        "ai/DialogAIChatRecommendation.xml",
        addon_path,
        "Default",
        "1080i",
        show_nav_sidebar=show_nav_sidebar,
        nav_position=nav_position
    )
    dialog.doModal()
    del dialog


