"""
DialogAIRecommendationHub.py

Fullscreen dialog s rozcestníkem AI doporučení.
Nabízí karty různých typů doporučení a textový vstup.
"""

import xbmc
import xbmcgui
import xbmcaddon
import threading
import time
from wcs.ai.DialogAIChatRecommendation import AIChatRecommendationDialog, AIRecommendationPromptBuilder
from wcs import user_data


class AIRecommendationHubDialog(AIChatRecommendationDialog):
    """
    Fullscreen dialog s kartami možností AI doporučení.
    Po výběru karty nebo zadání textu spustí AI inference.
    """
    
    # Definice karet doporučení
    RECOMMENDATION_CARDS = [
        {'id': 'movies', 'label': 'Doporuč filmy', 'icon': 'ai_card_movies.png', 
         'description': 'Nechej si doporučit zajímavé filmy', 'context_type': None, 'media_type': 'movie'},
        {'id': 'series', 'label': 'Doporuč seriály', 'icon': 'ai_card_series.png',
         'description': 'Nechej si doporučit seriály k sledování', 'context_type': None, 'media_type': 'series'},
        {'id': 'mix', 'label': 'Doporuč mix', 'icon': 'ai_card_mix.png',
         'description': 'Smíšená doporučení filmů i seriálů', 'context_type': 'my_mixed_collection', 'media_type': 'movie'},
        {'id': 'actors', 'label': 'Podle herců', 'icon': 'ai_card_actors.png',
         'description': 'Doporučení podle oblíbených herců', 'context_type': 'favorite_actors', 'media_type': 'movie'},
        {'id': 'directors', 'label': 'Podle režisérů', 'icon': 'ai_card_directors.png',
         'description': 'Doporučení podle oblíbených režisérů', 'context_type': 'favorite_directors', 'media_type': 'movie'},
        {'id': 'collections', 'label': 'Filmové kolekce', 'icon': 'ai_card_collections.png',
         'description': 'Doporučení filmových kolekcí a ság', 'context_type': 'my_collections', 'media_type': 'collection'},
        {'id': 'history', 'label': 'Z historie', 'icon': 'ai_card_history.png',
         'description': 'Doporučení podle historie přehrávání', 'context_type': 'recently_played', 'media_type': 'movie'},
    ]

    
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        # Parent zpracovává show_nav_sidebar, nav_position, context
        super(AIRecommendationHubDialog, self).__init__(xml_filename, addon_path, *args, **kwargs)
        # Identifikátor sekce pro navigační sidebar
        self._nav_section_id = 'ai_recommend'
        self._is_hub_mode = True  # Flag pro rozlišení hub vs. výsledky

    def onInit(self):
        """Přetížená inicializace - zobrazí karty možností."""
        # Resetovat stav (využívá parent metodu)
        self._reset_state()
        self.media_type = 'movie'
        self._is_hub_mode = True
        
        # Nastavit titulek
        self.setProperty('WCS.AIChat.Title', 'AI Doporučení')
        self.setProperty('WCS.AIChat.Status', 'Vyberte typ doporučení')
        
        # Nastavit fanart pozadí specifické pro AI Hub
        self.setProperty('WCS.AIPreview.Fanart', 'special://home/addons/plugin.video.milionar/resources/media/ai_hub_fanart.png')

        
        # Zobrazit uvítací zprávu
        self.append_chat_message("AI", "Vyberte typ doporučení nebo napište vlastní dotaz.")
        
        # Nastavit tlačítka pro rychlý výběr
        self._setup_hub_buttons()
        
        # Naplnit poster flow kartami
        self._populate_hub_cards()
        
        # Pokud byl dialog spuštěn z navigačního sidebaru, zobrazit ho bez animace
        if self._show_nav_on_init:
            self._show_nav_sidebar(animate=False)
        else:
            # Skrýt chat sidebar, fokus na karty s krátkým zpožděním
            self.clearProperty('WCS.AIChat.Visible')
            self._delayed_focus(2000, 0.3)
    
    def _setup_hub_buttons(self):
        """Nastaví tlačítka v AI chat sidebar."""
        buttons = [
            {"label": "Doporuč filmy", "value": "hub_select:movies"},
            {"label": "Doporuč seriály", "value": "hub_select:series"},
            {"label": "Doporuč na základě mé sbírky", "value": "hub_select:mix"},
        ]
        self.update_buttons(buttons)
    
    def _populate_hub_cards(self):
        """Naplní poster flow kartami možností doporučení."""
        list_control = self.getControl(2000)
        list_control.reset()
        
        for card in self.RECOMMENDATION_CARDS:
            item = xbmcgui.ListItem(label=card['label'])
            item.setArt({
                'poster': f"special://home/addons/plugin.video.milionar/resources/media/{card['icon']}",
                'icon': card['icon']
            })
            item.setProperty('card_id', card['id'])
            item.setProperty('description', card['description'])
            item.setProperty('is_hub_card', 'true')
            item.setProperty('context_type', card['context_type'] or '')
            item.setProperty('media_type', card['media_type'])
            list_control.addItem(item)
        
        # Aktualizovat preview pro první položku
        self._update_hub_preview()
    
    def _update_hub_preview(self):
        """Aktualizuje preview podle vybrané karty."""
        try:
            list_control = self.getControl(2000)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            card_id = item.getProperty('card_id')
            if not card_id:
                # Není to hub karta, použít standardní preview
                self._update_preview_for_current_item()
                return
            
            # Najít kartu
            for card in self.RECOMMENDATION_CARDS:
                if card['id'] == card_id:
                    self.setProperty('WCS.AIPreview.Title', card['label'])
                    self.setProperty('WCS.AIPreview.Plot', card['description'])
                    self.setProperty('WCS.AIPreview.State', 'visible')
                    break
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Hub preview error: {e}", xbmc.LOGERROR)
    
    def handle_recommendation_click(self):
        """Přetížená metoda - zpracuje klik na kartu nebo výsledek."""
        try:
            list_control = self.getControl(2000)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            # Kontrola zda je to hub karta
            if item.getProperty('is_hub_card') == 'true':
                card_id = item.getProperty('card_id')
                self._handle_card_selection(card_id)
            else:
                # Standardní zpracování výsledku
                super().handle_recommendation_click()
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Hub click error: {e}", xbmc.LOGERROR)
    
    def _handle_card_selection(self, card_id):
        """Zpracuje výběr karty a spustí AI inference - načítá data stejně jako router.py."""
        # Najít kartu
        card = None
        for c in self.RECOMMENDATION_CARDS:
            if c['id'] == card_id:
                card = c
                break
        
        if not card:
            return
        
        # Nastavit media type
        self.media_type = card['media_type']
        self._is_hub_mode = False
        
        # Vytvořit kontext s načtenými daty (stejně jako router.py)
        context_type = card['context_type']
        if context_type:
            self.context = {'type': context_type}
            
            # Načíst data podle typu kontextu
            addon = xbmcaddon.Addon()
            
            if context_type == 'favorite_actors':
                favs = user_data.load_favorite_actors(addon)
                self.context['data'] = favs
            elif context_type == 'favorite_directors':
                favs = user_data.load_favorite_directors(addon)
                self.context['data'] = favs
            elif context_type == 'my_movies':
                data = user_data.load_tmdb_movies(addon)
                self.context['data'] = data
            elif context_type == 'my_series':
                data = user_data.load_tmdb_series(addon)
                self.context['data'] = data
            elif context_type == 'my_mixed_collection':
                movies = user_data.load_tmdb_movies(addon)
                series = user_data.load_tmdb_series(addon)
                self.context['data'] = {'movies': movies, 'series': series}
            elif context_type == 'recently_played':
                data = user_data.load_recently_played(addon)
                self.context['data'] = data
            elif context_type == 'my_collections':
                # Načtení kolekcí uživatele
                tmdb_movies = user_data.load_tmdb_movies(addon)
                colls = {}
                if tmdb_movies:
                    from wcs.metadata.TMDbClient import get_movie_detailed_info
                    for movie in tmdb_movies:
                        mid = movie.get('id')
                        if mid:
                            details = get_movie_detailed_info(mid)
                            if details and details.get('belongs_to_collection'):
                                c_info = details['belongs_to_collection']
                                cid = c_info.get('id')
                                if cid and cid not in colls:
                                    colls[cid] = c_info
                self.context['data'] = list(colls.values())
        else:
            self.context = None
        
        # Inicializovat prompt builder s načtenými daty
        self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
        
        # Vytvořit zprávu podle typu - použít context instruction pokud je dostupná
        context_instruction = self._get_context_instruction() if self.context else None
        
        if context_instruction:
            user_msg = context_instruction
        elif card_id == 'movies':
            user_msg = "Doporuč mi zajímavé filmy k sledování."
        elif card_id == 'series':
            user_msg = "Doporuč mi zajímavé seriály k sledování."
        elif card_id == 'mix':
            user_msg = "Doporuč mi filmy i seriály na základě mé sbírky."
        elif card_id == 'actors':
            user_msg = "Doporuč mi filmy podle mých oblíbených herců."
        elif card_id == 'directors':
            user_msg = "Doporuč mi filmy podle mých oblíbených režisérů."
        elif card_id == 'collections':
            user_msg = "Doporuč mi filmové kolekce a ságy."
        elif card_id == 'history':
            user_msg = "Doporuč mi filmy a seriály na základě mé historie sledování."
        else:
            user_msg = "Doporuč mi něco zajímavého."
        
        # Přidat do historie
        self.history.append({"role": "user", "content": user_msg})
        self.append_chat_message("Ty", user_msg)
        
        # Zobrazit chat sidebar s loading
        self.setProperty('WCS.AIChat.Visible', 'true')
        self._start_button_loading("Analyzuji...")
        
        # Spustit AI konverzaci
        self._start_ai_conversation(is_initial_context=True)

    
    def handle_button_click(self, button_value):
        """Přetížená metoda pro zpracování tlačítek."""
        # Zpracovat hub_select: příkazy
        if button_value.startswith('hub_select:'):
            card_id = button_value.split(':')[1]
            self._handle_card_selection(card_id)
            return
        
        # Standardní zpracování
        super().handle_button_click(button_value)
    
    def onAction(self, action):
        """Přetížená akce pro hub mode.
        
        Hub mode má specifické chování:
        - UP na seznamu zobrazí hub-specifické akce (textový vstup + vybraná karta)
        - LEFT/RIGHT aktualizuje hub preview místo standardního
        - Vše ostatní (včetně nav sidebar) deleguje na parent
        """
        action_id = action.getId()
        
        # V hub mode - šipka nahoru zobrazí akční overlay s textovým vstupem
        if self._is_hub_mode and action_id == xbmcgui.ACTION_MOVE_UP:
            if self.getFocusId() == 2000:
                # Nastavit Hub-specifické akce
                self._setup_hub_actions()
                self.setProperty('WCS.AIPreview.ShowActions', 'true')
                self.setFocusId(6100)
                return
        
        # V hub mode - LEFT/RIGHT aktualizuje hub preview místo standardního
        # DŮLEŽITÉ: Necháme parent zpracovat VŠECHNO včetně nav sidebar logiky
        if self._is_hub_mode and action_id in (xbmcgui.ACTION_MOVE_LEFT, xbmcgui.ACTION_MOVE_RIGHT):
            if self.getFocusId() == 2000:
                # Uložit pozici před delegací na parent
                list_control = self.getControl(2000)
                current_pos = list_control.getSelectedPosition()
                
                # LEFT na první položce - delegovat na parent (může otevřít nav sidebar)
                if action_id == xbmcgui.ACTION_MOVE_LEFT and self._last_list_position == 0:
                    super().onAction(action)
                    return
                
                # Aktualizovat pozici pro příští akci
                self._last_list_position = current_pos
                
                # Aktualizovat hub-specifický preview
                time.sleep(0.05)
                self._update_hub_preview()
                return
        
        # Standardní zpracování (včetně nav sidebar, chat, atd.)
        super().onAction(action)
    
    def _setup_hub_actions(self):
        """Nastaví akční tlačítka pro Hub - textový vstup a vybranou kartu."""
        # Získat vybranou kartu
        try:
            list_control = self.getControl(2000)
            item = list_control.getSelectedItem()
            card_id = item.getProperty('card_id') if item else None
            card_label = item.getLabel() if item else "Dotaz"
        except:
            card_id = None
            card_label = "Dotaz"
        
        # Základní akce: Vlastní dotaz + Vybraná karta
        actions = [
            {"label": "Vlastní", "action": "hub_text_input"},
            {"label": card_label, "action": "hub_select_current"},
        ]
        
        # Přidat kontextové akce pro Filmy a Seriály
        if card_id == 'movies':
            # Přidat "Z mé sbírky" - použije kontext my_movies
            actions.append({"label": "Z mé sbírky", "action": "hub_context:my_movies"})
        elif card_id == 'series':
            # Přidat "Z mé sbírky" - použije kontext my_series
            actions.append({"label": "Z mé sbírky", "action": "hub_context:my_series"})
        
        # Naplnit action list (6100)
        action_list = self.getControl(6100)
        action_list.reset()
        for act in actions:
            item = xbmcgui.ListItem(label=act['label'])
            item.setProperty('action', act['action'])
            action_list.addItem(item)

    
    def _handle_preview_action_click(self):
        """Přetížená metoda pro zpracování akčních tlačítek."""
        if not self._is_hub_mode:
            super()._handle_preview_action_click()
            return
        
        # Hub mode - vlastní zpracování
        try:
            action_list = self.getControl(6100)
            item = action_list.getSelectedItem()
            if not item:
                return
            
            action = item.getProperty('action')
            
            if action == 'hub_text_input':
                # Otevřít klávesnici a odeslat dotaz
                self._handle_hub_text_input()
            elif action == 'hub_select_current':
                # Vybrat aktuální kartu
                self.setFocusId(2000)
                self.handle_recommendation_click()
            elif action.startswith('hub_context:'):
                # Kontextové doporučení (my_movies, my_series)
                context_type = action.split(':')[1]
                self._handle_context_recommendation(context_type)
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Hub action error: {e}", xbmc.LOGERROR)
    
    def _handle_context_recommendation(self, context_type):
        """Spustí AI doporučení s kontextem (my_movies, my_series)."""
        # Skrýt akční overlay
        self.clearProperty('WCS.AIPreview.ShowActions')
        
        # Nastavit media type podle kontextu
        if context_type == 'my_movies':
            self.media_type = 'movie'
        elif context_type == 'my_series':
            self.media_type = 'series'
        else:
            self.media_type = 'movie'
        
        self._is_hub_mode = False
        
        # Vytvořit kontext s načtenými daty
        self.context = {'type': context_type}
        addon = xbmcaddon.Addon()
        
        if context_type == 'my_movies':
            data = user_data.load_tmdb_movies(addon)
            self.context['data'] = data
        elif context_type == 'my_series':
            data = user_data.load_tmdb_series(addon)
            self.context['data'] = data
        
        # Inicializovat prompt builder
        self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
        
        # Získat context instruction
        context_instruction = self._get_context_instruction()
        user_msg = context_instruction or "Doporuč mi něco na základě mé sbírky."
        
        # Přidat do historie a zobrazit
        self.history.append({"role": "user", "content": user_msg})
        self.append_chat_message("Ty", user_msg)
        
        # Zobrazit chat sidebar s loading
        self.setProperty('WCS.AIChat.Visible', 'true')
        self._start_button_loading("Analyzuji...")
        
        # Spustit AI konverzaci
        self._start_ai_conversation(is_initial_context=True)


    
    def _handle_hub_text_input(self):
        """Zpracuje textový vstup od uživatele - používá existující infrastrukturu parent třídy."""
        # Otevřít klávesnici
        keyboard = xbmc.Keyboard('', 'Napište svůj dotaz pro AI')
        keyboard.doModal()
        
        if keyboard.isConfirmed():
            user_text = keyboard.getText().strip()
            if user_text:
                # Skrýt akční overlay
                self.clearProperty('WCS.AIPreview.ShowActions')
                
                # Přepnout z hub mode do normálního režimu
                self._is_hub_mode = False
                
                # Nastavit media_type pokud není nastaven (default movie)
                if not self.media_type:
                    self.media_type = 'movie'
                
                # Inicializovat prompt builder pokud není (pro volný text bez kontextu)
                if not self._prompt_builder:
                    self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
                
                # Použít existující metodu z parent třídy pro zpracování vstupu
                # Tato metoda přidá zprávu do historie, zobrazí v chatu a spustí AI
                self.handle_user_input(user_text)


    
    def _handle_no_recommendations(self):
        """Přetížená metoda - vrátit se zpět do hub mode."""
        self.append_system_message("Model nedoporučil žádné tituly.")
        self._is_hub_mode = True
        self._populate_hub_cards()
        self._setup_hub_buttons()


def show_ai_recommendation_hub_dialog(addon, show_nav_sidebar=False, nav_position=0):
    """Zobrazí dialog AI Doporučení Hub."""
    addon_path = addon.getAddonInfo('path')
    
    dialog = AIRecommendationHubDialog(
        "ai/DialogAIChatRecommendation.xml",
        addon_path,
        "Default",
        "1080i",
        show_nav_sidebar=show_nav_sidebar,
        nav_position=nav_position
    )
    dialog.doModal()
    del dialog
