# wcs/download.py

import os
import io
import re
import unidecode
import xbmcvfs
import xbmc
import time
import requests
import xbmcgui
from wcs import utils, user_data
from wcs.ui import UI as ui
from . import WebshareClient as webshare_api

# Funkce pro stahování souboru
def download_file(ident, addon):
    """Stáhne soubor s daným ident."""
    # Kontrola přihlášení a validace tokenu
    if not user_data.check_providers_and_prompt(addon):
        ui.popinfo("Stahování vyžaduje přihlášení.", addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
        return # Ukončit, pokud není přihlášen
    token = webshare_api.revalidate(addon, ui.popinfo)
    if not token:
        return

    # Získání nastavení a kontrola složky
    where = addon.getSetting('dfolder')
    if not where:
        # Zobrazit dialog pro výběr složky místo otevření nastavení
        from wcs.login.DialogDownloadFolder import show_download_folder_dialog
        if show_download_folder_dialog(addon):
            # Uživatel nastavil složku, pokračovat ve stahování
            where = addon.getSetting('dfolder')
            if not where:
                ui.popinfo('Složka nebyla nastavena, stahování se ruší.', addon.getAddonInfo('name'), sound=True)
                return
        else:
            # Uživatel zrušil nastavování, ukončit stahování
            return
    if not xbmcvfs.exists(where):
        ui.popinfo(f'Nastavená složka pro stahování neexistuje: {where}', addon.getAddonInfo('name'), sound=True)
        addon.openSettings()
        return
        
    local = os.path.exists(where) # Kontrola, zda jde o lokální nebo Kodi VFS cestu
    normalize = addon.getSettingBool('dnormalize')
    notify = addon.getSettingBool('dnotify')
    every_index = addon.getSetting('dnevery')
    
    # Mapování indexu na procenta: 0=1%, 1=5%, 2=10%, 3=20%, 4=25%, 5=50%
    every_percentages = [1, 5, 10, 20, 25, 50]
    try:
        every_index = int(every_index)
        if 0 <= every_index < len(every_percentages):
            every = every_percentages[every_index]
        else:
            every = 10  # Default na 10%
    except ValueError:
        every = 10  # Default na 10%

    # Získání odkazu a informací o souboru
    link = webshare_api.getlink(ident, token, 'file_download', addon)
    info = webshare_api.getinfo(ident, token)

    if not link or not info:
        ui.popinfo('Nepodařilo se získat informace nebo odkaz pro stahování.', addon.getAddonInfo('name'), sound=True)
        return

    name_node = info.find('name')
    if name_node is None or not name_node.text:
        ui.popinfo('Nepodařilo se získat název souboru pro stahování.', addon.getAddonInfo('name'), sound=True)
        return
    name = name_node.text
    
    if normalize:
        name = unidecode.unidecode(name)

    # Vytvoření session pro stahování
    global _current_download_session, _current_download_file
    _current_download_session = requests.Session()
    session = _current_download_session
    bf = None # Inicializace proměnné
    _current_download_file = None
    try:
        filepath = utils.join(where, name)
        bf = io.open(filepath, 'wb') if local else xbmcvfs.File(filepath, 'w')
        _current_download_file = bf
        
        # Notifikace o spuštění stahování - respektuje nastavení dnotify
        if notify:
            ui.popinfo('Spouštím stahování: ' + name, addon.getAddonInfo('name'))
        
        response = session.get(link, stream=True, timeout=30) # Přidán timeout
        response.raise_for_status() # Kontrola HTTP chyb

        total = response.headers.get('content-length')

        if total is None:
            # Neznámá velikost
            if notify:
                ui.popinfo('Neznámá velikost souboru, stahuji: ' + name, addon.getAddonInfo('name'), sound=True)
            # Pro neznámou velikost stáhnout celý obsah najednou
            content = response.content
            bf.write(content)
        elif not notify:
            # Stahování bez notifikací o průběhu
            content = response.content
            bf.write(content)
        else:
            # Stahování s notifikacemi o průběhu
            dl = 0
            total = int(total)
            if total == 0: # Vyhnout se dělení nulou
                 if notify:
                     ui.popinfo('Soubor má nulovou velikost: ' + name, addon.getAddonInfo('name'), sound=True)
                 bf.write(response.content) # Stáhnout prázdný obsah
            else:
                pct = total / 100.0 # Použít float pro přesnější procenta
                lastpop = -1 # Začít na -1, aby se 0% zobrazilo
                start_time = time.time()

                # Stahování s notifikacemi o průběhu
                for data in response.iter_content(chunk_size=8192):
                    dl += len(data)
                    bf.write(data)
                    done = int(dl / pct)
                    if done >= 0 and done <= 100 and done % every == 0 and lastpop != done:
                        # Výpočet rychlosti a zbývajícího času (jednoduchý)
                        elapsed_time = time.time() - start_time
                        speed_bps = dl / elapsed_time if elapsed_time > 0 else 0
                        speed_kbs = speed_bps / 1024
                        remaining_bytes = total - dl
                        eta_sec = remaining_bytes / speed_bps if speed_bps > 0 else 0
                        eta_min = int(eta_sec // 60)
                        eta_s = int(eta_sec % 60)
                        
                        progress_msg = f"{done}% ({utils.format_size(dl)}/{utils.format_size(total)})"
                        speed_msg = f"Rychlost: {speed_kbs:.1f} KB/s"
                        eta_msg = f"ETA: {eta_min}m {eta_s}s"
                        
                        ui.popinfo(f"{progress_msg}", addon.getAddonInfo('name'))
                        lastpop = done
        
        bf.close() # Zavřít soubor po úspěšném stahování
        bf = None # Resetovat proměnnou
        _current_download_file = None
        _current_download_session = None
        
        # Notifikace o dokončení - respektuje nastavení dnotify
        if notify:
            ui.popinfo('Staženo: ' + name, addon.getAddonInfo('name'), sound=True)

    except requests.exceptions.RequestException as e:
        import traceback
        traceback.print_exc()
        # Chybové notifikace se vždy zobrazují (bez ohledu na dnotify)
        ui.popinfo(f'Chyba sítě při stahování: {e} - {name}', addon.getAddonInfo('name'), sound=True)
    except Exception as e:
        import traceback
        traceback.print_exc()
        # Chybové notifikace se vždy zobrazují (bez ohledu na dnotify)
        ui.popinfo(f'Chyba při stahování: {e} - {name}', addon.getAddonInfo('name'), sound=True)
    finally:
        # Vyčistit globální proměnné
        _current_download_file = None
        _current_download_session = None
        
        if bf is not None: # Pokud soubor zůstal otevřený po chybě
            try:
                bf.close()
                # TODO - zvážit smazání nekompletního souboru
                # if xbmcvfs.exists(filepath):
                #    xbmcvfs.delete(filepath)
            except Exception as close_e:
                 import traceback
                 traceback.print_exc()
                 # Chybové notifikace se vždy zobrazují (bez ohledu na dnotify)
                 ui.popinfo(f'Chyba při zavírání souboru po chybě stahování: {close_e}', addon.getAddonInfo('name'), sound=True)


# Logika stahování souborů, práce s adresáři, notifikace průběhu (bude implementováno v dalších krocích)

# Globální proměnná pro aktuální stahování
_current_download_session = None
_current_download_file = None

