# wcs/playback/dialog_scheduler.py

"""Sjednocený plánovač pro zobrazení info/countdown dialogů po startu přehrávání.

Nahrazuje 4 duplicitní threading patterny:
  - utils.py: _show_info_after_play, schedule_episode_info_dialog
  - LibraryManager: _activate_movie_info_only, _activate_episode_info_only
"""

import threading
import time
import xbmc
import xbmcaddon

from wcs.playback.metadata import MediaMetadata

# Konstanty
PLAYBACK_WAIT_ATTEMPTS = 100   # max ~10s čekání na start
PLAYBACK_WAIT_INTERVAL = 0.1   # 100ms mezi pokusy
DIALOG_DELAY = 3               # sjednocený delay po startu přehrávání
SETTINGS_KEY = 'show_episode_info'  # klíč nastavení (sdílený pro film i epizodu)


def schedule_info_dialog(metadata, source='unknown'):
    """Naplánuje zobrazení info dialogu po detekci startu přehrávání.

    Args:
        metadata: MediaMetadata instance s normalizovanými daty
        source: zdroj volání pro logování ('addon', 'library', 'autoplay')
    """
    def _worker():
        try:
            # 1. Čekání na start přehrávání
            for _ in range(PLAYBACK_WAIT_ATTEMPTS):
                if xbmc.Player().isPlaying():
                    break
                time.sleep(PLAYBACK_WAIT_INTERVAL)
            else:
                return

            # 2. Sjednocený delay
            time.sleep(DIALOG_DELAY)

            # 3. Kontrola nastavení
            addon = xbmcaddon.Addon()
            if not addon.getSettingBool(SETTINGS_KEY):
                xbmc.log(
                    f"[plugin.video.milionar] DialogScheduler: Info dialog disabled in settings (source={source})",
                    level=xbmc.LOGINFO,
                )
                return

            # 4. Kontrola že přehrávání stále běží
            if not xbmc.Player().isPlaying():
                return

            # 5. Zobrazení dialogu podle typu média
            if metadata.media_type == 'movie':
                _show_movie_dialog(metadata, source)
            else:
                _show_episode_dialog(metadata, source)

        except Exception as e:
            xbmc.log(
                f"[plugin.video.milionar] DialogScheduler error (source={source}): {e}",
                level=xbmc.LOGWARNING,
            )

    threading.Thread(target=_worker, daemon=True).start()
    xbmc.log(
        f"[plugin.video.milionar] DialogScheduler: Scheduled {metadata.media_type} info dialog (source={source})",
        level=xbmc.LOGINFO,
    )


def _show_movie_dialog(metadata, source):
    """Zobrazí MovieInfoDialog."""
    from wcs.playback.DialogMediaInfoWithGame import get_movie_info_manager

    movie_info_manager = get_movie_info_manager()
    movie_info_manager.show_movie_info(
        movie_title=metadata.title,
        movie_year=metadata.year,
        movie_thumb=metadata.thumb,
        rating=metadata.rating,
        plot=metadata.plot,
        tmdb_id=metadata.tmdb_id,
        fanart=metadata.fanart,
        genre=metadata.genre,
        director=metadata.director,
        runtime=metadata.runtime,
    )
    xbmc.log(
        f"[plugin.video.milionar] DialogScheduler: Movie info shown for {metadata.title} (source={source})",
        level=xbmc.LOGINFO,
    )


def _show_episode_dialog(metadata, source):
    """Zobrazí EpisodeInfoDialog."""
    from wcs.playback.DialogMediaInfoWithGame import get_episode_info_manager

    episode_info_manager = get_episode_info_manager()
    episode_info_manager.show_episode_info(
        series_name=metadata.series_name,
        episode_title=metadata.episode_title or (
            f"Epizoda {metadata.episode_number}" if metadata.episode_number else ""
        ),
        season_number=metadata.season_number,
        episode_number=metadata.episode_number,
        episode_thumb=metadata.thumb,
        rating=metadata.rating,
        year=metadata.year,
        plot=metadata.plot,
        tmdb_id=metadata.tmdb_id,
        fanart=metadata.fanart,
    )
    xbmc.log(
        f"[plugin.video.milionar] DialogScheduler: Episode info shown for {metadata.series_name} (source={source})",
        level=xbmc.LOGINFO,
    )


def schedule_library_info_dialog(metadata_fetcher, fallback_metadata, source='library'):
    """Naplánuje info dialog pro přehrávání z Kodi knihovny.

    Na rozdíl od schedule_info_dialog metadata nezná předem -- získá je
    z knihovny až po startu přehrávání (Kodi JSON-RPC).

    Args:
        metadata_fetcher: callable() -> dict|None, vrátí raw metadata z knihovny
        fallback_metadata: MediaMetadata instance pro použití pokud fetcher selže
        source: zdroj pro logování
    """
    def _worker():
        try:
            # 1. Čekání na start přehrávání
            for _ in range(PLAYBACK_WAIT_ATTEMPTS):
                if xbmc.Player().isPlaying():
                    break
                time.sleep(PLAYBACK_WAIT_INTERVAL)
            else:
                return

            # 2. Sjednocený delay
            time.sleep(DIALOG_DELAY)

            # 3. Kontrola nastavení
            addon = xbmcaddon.Addon()
            if not addon.getSettingBool(SETTINGS_KEY):
                xbmc.log(
                    f"[plugin.video.milionar] DialogScheduler: Info dialog disabled in settings (source={source})",
                    level=xbmc.LOGINFO,
                )
                return

            # 4. Kontrola přehrávání
            if not xbmc.Player().isPlaying():
                return

            # 5. Získání metadat z knihovny
            raw_data = None
            try:
                raw_data = metadata_fetcher()
            except Exception as e:
                xbmc.log(
                    f"[plugin.video.milionar] DialogScheduler: Metadata fetch failed (source={source}): {e}",
                    level=xbmc.LOGWARNING,
                )

            # 6. Konverze na MediaMetadata (nebo fallback)
            if raw_data:
                if fallback_metadata.media_type == 'movie':
                    metadata = MediaMetadata.from_kodi_library_movie(raw_data)
                else:
                    metadata = MediaMetadata.from_kodi_library_episode(raw_data)
            else:
                metadata = fallback_metadata

            # 7. Zobrazení dialogu
            if metadata.media_type == 'movie':
                _show_movie_dialog(metadata, source)
            else:
                _show_episode_dialog(metadata, source)

        except Exception as e:
            xbmc.log(
                f"[plugin.video.milionar] DialogScheduler error (source={source}): {e}",
                level=xbmc.LOGWARNING,
            )

    threading.Thread(target=_worker, daemon=True).start()
    xbmc.log(
        f"[plugin.video.milionar] DialogScheduler: Scheduled library info dialog (source={source})",
        level=xbmc.LOGINFO,
    )
