# wcs/details_window.py
import sys
import os
import xbmc
import xbmcgui
import xbmcplugin
import requests
from urllib.parse import quote_plus
import xbmcaddon
import xbmcvfs
import json
import re
from datetime import datetime

# Import TMDB API Key a případných dalších utilit
# TODO: Lépe spravovat API klíč
from wcs.metadata.TMDbClient import get_tmdb_api_key
from wcs.ui.dialogs.DialogWcsSearch import SearchDialog

# Základní URL pro obrázky TMDB (mělo by se načítat z /configuration)
TMDB_IMAGE_BASE_URL = "https://image.tmdb.org/t/p/"

from wcs import utils # Přidáno pro utils.show_info_window_by_id
from wcs.metadata import TMDbClient as tmdb_handler 

from wcs.ui.dialogs import DialogWcsSearch as search_dialog

class DetailsWindow(xbmcgui.WindowXMLDialog):
    """
    Vlastní okno pro zobrazení detailních informací o filmu/seriálu
    načtené z details_extras/DialogWcsDetails.xml.
    """
    def __init__(self, *args, **kwargs):
        super().__init__() 
        # Uložíme si naše kwargs
        self.media_type = kwargs.get('media_type')
        self.tmdb_id = kwargs.get('tmdb_id')
        self.addon = xbmcaddon.Addon()
        # Trailer URL budeme ukládat zde, až ho získáme
        self.trailer_plugin_url = None

    def onInit(self):
        """
        Volá se při inicializaci okna. Zde načteme data a naplníme kontrolky.
        """
        xbmc.log(f"[plugin.video.milionar] Initializing DetailsWindow for {self.media_type} ID: {self.tmdb_id}", level=xbmc.LOGINFO)
        if not self.media_type or not self.tmdb_id:
            xbmc.log("[plugin.video.milionar] Error: Missing media_type or tmdb_id for DetailsWindow.", level=xbmc.LOGERROR)
            self.close()
            return

        # Zobrazíme indikátor načítání
        xbmc.executebuiltin("ActivateWindow(busydialognocancel)")

        # Získáme data z TMDB
        details_data = self._get_tmdb_details()

        # Skryjeme indikátor načítání
        xbmc.executebuiltin("Dialog.Close(busydialognocancel)")

        if not details_data:
            # Zobrazit chybu a zavřít okno, pokud se data nepodařilo načíst
            xbmcgui.Dialog().ok("Chyba", "Nepodařilo se načíst detaily z TMDb.")
            self.close()
            return

        # Naplníme kontrolky okna daty
        self._populate_controls(details_data)

    def _get_tmdb_details(self):
        """Načte detailní informace z TMDB API."""
        try:
            api_params = {
                'api_key': get_tmdb_api_key(),
                'language': 'cs-CZ',
                'append_to_response': 'credits,videos' # Přidáme credits a videos
            }
            url = f'https://api.themoviedb.org/3/{self.media_type}/{self.tmdb_id}'
            response = requests.get(url, params=api_params, timeout=10) # Timeout pro request
            response.raise_for_status()
            return response.json()
        except ValueError as e:
            # TMDb API klíč není nastaven - zobrazí dialog a ukončí okno
            from wcs.metadata.TMDbClient import handle_tmdb_api_error
            handle_tmdb_api_error(e, self.addon)
            self.close()
            return None
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error fetching TMDB details: {e}", level=xbmc.LOGERROR)
            return None

    def _populate_controls(self, data):
        """Naplní vlastnosti okna a seznam herců získanými daty."""
        try:
            # --- Získání dat (rozšířeno pro seriály) ---
            is_movie = self.media_type == 'movie'
            title = data.get('title') if is_movie else data.get('name', '')
            # original_title = data.get('original_title') if is_movie else data.get('original_name', '') # Zatím nepoužíváme v XML
            tagline = data.get('tagline', '')
            plot = data.get('overview', '')
            year = (data.get('release_date', '')[:4] if is_movie else data.get('first_air_date', '')[:4])
            rating = data.get('vote_average', 0.0)
            
            # Runtime/Episodes info podle typu média
            if is_movie:
                runtime_minutes = data.get('runtime', 0)
                runtime_info = f"{runtime_minutes} min" if runtime_minutes else ""
                status_info = ""
            else:
                # Pro seriály zobrazíme počet sezón a epizod
                num_seasons = data.get('number_of_seasons', 0)
                num_episodes = data.get('number_of_episodes', 0)
                if num_seasons and num_episodes:
                    runtime_info = f"{num_seasons} sezón, {num_episodes} epizod"
                elif num_seasons:
                    runtime_info = f"{num_seasons} sezón"
                elif num_episodes:
                    runtime_info = f"{num_episodes} epizod"
                else:
                    runtime_info = ""
                
                # Status seriálu
                status_info = data.get('status', '')
                if status_info:
                    # Přeložíme některé standardní statusy
                    status_translations = {
                        'Ended': 'Ukončen',
                        'Returning Series': 'Pokračuje',
                        'In Production': 'Ve výrobě',
                        'Canceled': 'Zrušen',
                        'Pilot': 'Pilot'
                    }
                    status_info = status_translations.get(status_info, status_info)
            
            genres = ", ".join([g['name'] for g in data.get('genres', [])])
            poster_path = data.get('poster_path', '')
            fanart_path = data.get('backdrop_path', '')
            poster_url = f"{TMDB_IMAGE_BASE_URL}w500{poster_path}" if poster_path else ''
            fanart_url = f"{TMDB_IMAGE_BASE_URL}original{fanart_path}" if fanart_path else ''
            credits = data.get('credits', {})
            crew = credits.get('crew', [])
            
            # Rozdílné získávání crew informací pro filmy vs seriály
            if is_movie:
                directors = [c['name'] for c in crew if c.get('job') == 'Director']
                # Získání ID prvního režiséra (pokud existuje)
                director_id = ""
                for c in crew:
                    if c.get('job') == 'Director':
                        director_id = str(c.get('id', ''))
                        break
                directors_info = f"Režie: {', '.join(directors)}" if directors else ""
                creators_info = ""
            else:
                # Pro seriály používáme created_by místo directors
                directors = []
                director_id = ""
                directors_info = ""
                creators = [c['name'] for c in data.get('created_by', [])]
                creators_info = f"Tvůrci: {', '.join(creators)}" if creators else ""
            
            writers = [c['name'] for c in crew if c.get('department') == 'Writing' and c.get('job') in ['Writer', 'Screenplay', 'Story']]
            writers = list(dict.fromkeys(writers))
            videos = data.get('videos', {}).get('results', [])
            
            # Získání ID a jména prvního scénáristy
            writer_id = ""
            writer_name = ""
            for c in crew:
                if c.get('department') == 'Writing' and c.get('job') in ['Writer', 'Screenplay', 'Story']:
                    writer_id = str(c.get('id', ''))
                    writer_name = c.get('name', '')
                    break # Bereme prvního nalezeného
                    
            trailer_key = None
            for video in videos:
                if video.get('site') == 'YouTube' and video.get('type') == 'Trailer':
                     if video.get('official') and video.get('iso_639_1') in ['cs', 'en']:
                        trailer_key = video.get('key')
                        break 
            if not trailer_key:
                 for video in videos:
                    if video.get('site') == 'YouTube' and video.get('type') == 'Trailer':
                        trailer_key = video.get('key')
                        break
            self.trailer_plugin_url = f"plugin://plugin.video.youtube/?action=play_video&videoid={trailer_key}" if trailer_key else None
            
            # --- Nastavení VLASTNOSTÍ okna --- 
            self.setProperty('WCS.Details.MediaType', self.media_type)  # Přidáno MediaType
            self.setProperty('WCS.Details.Fanart', fanart_url or 'common/black.png')
            self.setProperty('WCS.Details.Poster', poster_url or 'common/black.png')
            self.setProperty('WCS.Details.Title', title)
            self.setProperty('WCS.Details.Tagline', tagline)
            self.setProperty('WCS.Details.Year', year)
            self.setProperty('WCS.Details.Rating', f"{rating:.1f}/10" if rating else "")
            self.setProperty('WCS.Details.Runtime', runtime_info)
            self.setProperty('WCS.Details.Status', status_info)  # Přidáno pro seriály
            self.setProperty('WCS.Details.Genre', genres)
            self.setProperty('WCS.Details.Plot', plot)
            self.setProperty('WCS.Details.Director', directors_info)
            self.setProperty('WCS.Details.DirectorID', director_id)  # Přidáno ID režiséra
            self.setProperty('WCS.Details.Creators', creators_info)  # Přidáno pro seriály
            self.setProperty('WCS.Details.Writer', f"Scénář: {', '.join(writers)}" if writers else "")
            self.setProperty('WCS.Details.WriterID', writer_id) # Přidáno ID prvního scénáristy
            self.setProperty('WCS.Details.WriterName', writer_name) # Přidáno jméno prvního scénáristy
            self.setProperty('WCS.Details.HasTrailer', str(bool(self.trailer_plugin_url))) # Pro visible podmínku tlačítka
            self.setProperty('WCS.Details.ID', str(self.tmdb_id))  # Přidávám TMDb ID pro onclick akce

            # --- Zpracování kolekce (pouze pro filmy) ---
            if is_movie:
                collection_info = data.get('belongs_to_collection')
                if collection_info and isinstance(collection_info, dict):
                    collection_id = str(collection_info.get('id', ''))
                    collection_name = collection_info.get('name', '')
                    if collection_id:
                        self.setProperty('WCS.Details.CollectionID', collection_id)
                        self.setProperty('WCS.Details.CollectionName', collection_name)
                        xbmc.log(f"[plugin.video.milionar] Movie belongs to collection: ID={collection_id}, Name={collection_name}", level=xbmc.LOGDEBUG)
                    else:
                        # Pokud ID kolekce chybí, vymažeme vlastnosti
                        self.clearProperty('WCS.Details.CollectionID')
                        self.clearProperty('WCS.Details.CollectionName')
                else:
                    # Pokud informace o kolekci chybí nebo není slovník, vymažeme vlastnosti
                    self.clearProperty('WCS.Details.CollectionID')
                    self.clearProperty('WCS.Details.CollectionName')
            else:
                # Pro seriály vymažeme collection vlastnosti
                self.clearProperty('WCS.Details.CollectionID')
                self.clearProperty('WCS.Details.CollectionName')

            # --- Naplnění seznamu herců (ID 50) - Opraveno ID --- 
            try:
                cast_list_control = self.getControl(50) # OPRAVENO ID z 400 na 50
                cast_list_control.reset() 
                cast_data = credits.get('cast', [])
                for actor in cast_data[:15]: 
                    actor_name = actor.get('name', '')
                    character_name = actor.get('character', '')
                    profile_path = actor.get('profile_path', '')
                    actor_id = actor.get('id', '')  # Získání ID herce
                    actor_photo_url = f"{TMDB_IMAGE_BASE_URL}w185{profile_path}" if profile_path else 'DefaultActor.png'
                    list_item = xbmcgui.ListItem(label=actor_name)
                    list_item.setLabel2(character_name)
                    list_item.setArt({'icon': actor_photo_url, 'thumb': actor_photo_url})
                    list_item.setProperty('tmdb_id', str(actor_id))  # Přidání ID herce jako vlastnosti
                    cast_list_control.addItem(list_item)
                
                # Nastavíme focus na seznam herců, pokud tam nějací jsou
                if cast_data:
                    self.setFocusId(50) # OPRAVENO ID z 400 na 50
                else:
                    self.setFocusId(5000) # Jinak focus na první tlačítko (ID listu tlačítek)
                    
            except Exception as list_error:
                 xbmc.log(f"[plugin.video.milionar] Error populating cast list (Control 50): {list_error}", level=xbmc.LOGERROR) # OPRAVENO ID V LOGU
                 # Pokud selže naplnění seznamu, nastavíme focus na tlačítka
                 self.setFocusId(5000)

        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error setting properties or populating controls: {e}", level=xbmc.LOGERROR)
            import traceback
            traceback.print_exc()
            xbmcgui.Dialog().ok("Chyba", f"Nastala chyba při přípravě detailů:\n{e}")
            self.close()

    def onClick(self, controlId):
        """Obsluha kliknutí na kontrolky."""
        if controlId == 9001: # ID tlačítka Trailer
            if self.trailer_plugin_url:
                xbmc.log(f"[plugin.video.milionar] Playing trailer: {self.trailer_plugin_url}", level=xbmc.LOGINFO)
                xbmc.executebuiltin(f"PlayMedia({self.trailer_plugin_url})")
            else:
                xbmc.log("[plugin.video.milionar] Trailer button clicked, but no trailer URL found.", level=xbmc.LOGWARNING)

        elif controlId == 5001: # ID tlačítka Přehrát (v seznamu 5000)
            from wcs.utils import play_movie_from_addon
            meta = {
                'title': self.getProperty('WCS.Details.Title'),
                'year': self.getProperty('WCS.Details.Year'),
                'plot': self.getProperty('WCS.Details.Plot'),
                'poster': self.getProperty('WCS.Details.Poster'),
                'fanart': self.getProperty('WCS.Details.Fanart'),
                'rating': self.getProperty('WCS.Details.Rating'),
                'genre': self.getProperty('WCS.Details.Genre'),
                'director': self.getProperty('WCS.Details.Director'),
                'runtime': self.getProperty('WCS.Details.Runtime'),
                'tmdb_id': self.tmdb_id,  # Pro uložení do historie
            }
            play_movie_from_addon(meta, self.addon)
            self.close()

        elif controlId == 5005: # ID tlačítka Close/OK (v seznamu 5000)
            self.close()
            
        elif controlId == 50: # Seznam herců
            # Získání vybraného herce
            cast_list_control = self.getControl(50)
            selected_item = cast_list_control.getSelectedItem()
            if selected_item:
                actor_id = selected_item.getProperty('tmdb_id')
                actor_name = selected_item.getLabel()
                
                if not actor_id or not actor_name:
                    xbmc.log(f"[plugin.video.milionar] ERROR: Missing actor ID or name on click! actor_id={actor_id}, actor_name={actor_name}", level=xbmc.LOGERROR)
                    xbmcgui.Dialog().notification("Chyba: Chybí ID nebo jméno herce!", str(actor_name), xbmcgui.NOTIFICATION_ERROR)
                    return

                # Výběr mezi filmy a seriály
                dialog = xbmcgui.Dialog()
                options = ["Filmy s hercem", "Seriály s hercem", "Hrát hru s hercem", "Přidat herce do oblíbených"]
                choice = dialog.select(f"Zobrazit pro: {actor_name}", options)
                if choice == 0:
                    # Filmy
                    url = f"plugin://{self.addon.getAddonInfo('id')}?action=display_actor_dialog&actor_id={actor_id}&actor_name={quote_plus(actor_name)}&media_type=movie"
                    xbmc.log(f"[plugin.video.milionar] Otevírám filmy s hercem: {url}", level=xbmc.LOGINFO)
                    xbmc.executebuiltin(f"RunPlugin({url})")
                elif choice == 1:
                    # Seriály
                    url = f"plugin://{self.addon.getAddonInfo('id')}?action=display_actor_dialog&actor_id={actor_id}&actor_name={quote_plus(actor_name)}&media_type=tv"
                    xbmc.log(f"[plugin.video.milionar] Otevírám seriály s hercem: {url}", level=xbmc.LOGINFO)
                    xbmc.executebuiltin(f"RunPlugin({url})")
                elif choice == 2:
                    # Spustit hru s hercem (Milionář)
                    from wcs.games.DialogWcsTextGame import TextGameWindow
                    actor_data = tmdb_handler.get_actor_details(actor_id)
                    if actor_data:
                        profile_path = actor_data.get('profile_path')
                        fanart_url = f"https://image.tmdb.org/t/p/w780{profile_path}" if profile_path else ''
                        window = TextGameWindow(
                            "games/DialogWcsTextGame.xml",
                            self.addon.getAddonInfo('path'),
                            "Default",
                            title=actor_data.get('name', ''),
                            fanart=fanart_url,
                            plot=actor_data.get('biography', '')[:500],
                            genre='',
                            media_type='actor',
                            actor_data=actor_data,
                            banner_fanart=fanart_url,
                            episode_title="",
                            season_number=None,
                            episode_number=None
                        )
                        window.doModal()
                        del window
                elif choice == 3:
                    # Přidat herce do oblíbených
                    url = f"plugin://{self.addon.getAddonInfo('id')}?action=tmdb_add_actor_to_favorites&actor_id={actor_id}&actor_name={quote_plus(actor_name)}"
                    xbmc.log(f"[plugin.video.milionar] Přidávám herce do oblíbených: {url}", level=xbmc.LOGINFO)
                    xbmc.executebuiltin(f"RunPlugin({url})")
                # Pokud choice < 0, nic se neděje

    def onAction(self, action):
        """Obsluha akcí (např. tlačítko Zpět)."""
        if action.getId() in (xbmcgui.ACTION_PREVIOUS_MENU, xbmcgui.ACTION_NAV_BACK):
            self.close()


# --- Kód pro spuštění okna (příklad - bude voláno z routeru) ---
# if __name__ == "__main__":
#     # Příklad volání:
#     # Potřebujeme TMDB ID filmu nebo seriálu
#     example_tmdb_id = 603 # Matrix
#     example_media_type = 'movie'
#     # example_tmdb_id = 1396 # Breaking Bad
#     # example_media_type = 'tv'
#
#     window = DetailsWindow("details_extras/DialogWcsDetails.xml", "special://home/addons/plugin.video.milionar/wcs", # Cesty k XML
#                           media_type=example_media_type, tmdb_id=example_tmdb_id)
#     window.doModal()
#     del window 