# wcs/games/text_formatter.py
"""
Formatter for Millionaire game text content.

Uses Kodi label formatting tags: [COLOR], [B], [CR], [UPPERCASE].
Provides consistent visual hierarchy across all game screens.
"""

# Millionaire color palette (AARRGGBB)
GOLD = "FFFFD700"
BLUE = "FF2997FF"
GREEN = "FF34C759"
RED = "FFFF3B30"
DIM = "80FFFFFF"
WHITE = "FFFFFFFF"
SILVER = "FFCCCCCC"
ORANGE = "FFFF9500"


def title(text):
    """Golden bold title for screen headers."""
    return "[COLOR {}][B]{}[/B][/COLOR]".format(GOLD, text)


def heading(text):
    """White bold section heading."""
    return "[B]{}[/B]".format(text)


def money(amount):
    """Gold-highlighted money amount."""
    return "[COLOR {}][B]{}[/B][/COLOR]".format(GOLD, amount)


def highlight(text, color=BLUE):
    """Colored highlight for emphasis."""
    return "[COLOR {}]{}[/COLOR]".format(color, text)


def bold_highlight(text, color=BLUE):
    """Bold colored highlight."""
    return "[COLOR {}][B]{}[/B][/COLOR]".format(color, text)


def success(text):
    """Green text for correct answers."""
    return "[COLOR {}][B]{}[/B][/COLOR]".format(GREEN, text)


def error(text):
    """Red text for wrong answers / errors."""
    return "[COLOR {}][B]{}[/B][/COLOR]".format(RED, text)


def dim(text):
    """Dimmed text for metadata."""
    return "[COLOR {}]{}[/COLOR]".format(DIM, text)


def silver(text):
    """Silver text for descriptions."""
    return "[COLOR {}]{}[/COLOR]".format(SILVER, text)


def separator():
    """Visual section separator."""
    return dim("─" * 40)


def bullet(text, color=None):
    """Bullet point item, optionally colored."""
    dot = "•"
    if color:
        return "[COLOR {}]{}[/COLOR] {}".format(color, dot, text)
    return "{} {}".format(dot, text)


def labeled_bullet(label, desc, label_color=BLUE):
    """Bullet with colored label and description."""
    return "• [COLOR {}]{}[/COLOR] — {}".format(label_color, label, desc)
