from PIL import Image, ImageDraw, ImageOps
import os

# Configuration
TARGET_SIZE = (440, 660) # 2:3 Aspect ratio, 2x scale of 220x330
RADIUS = 40 # Rounded corner radius
BORDER_WIDTH = 12

BASE_PATH = "/Users/chudy/Library/Application Support/Kodi/addons/plugin.video.milionar/resources/media"
CARDS = [
    "ai_card_movies.png",
    "ai_card_series.png",
    "ai_card_mix.png",
    "ai_card_actors.png",
    "ai_card_directors.png",
    "ai_card_collections.png",
    "ai_card_history.png"
]

def create_mask(size, radius):
    # High-res mask for antialiasing
    scale = 4
    w, h = size[0] * scale, size[1] * scale
    r = radius * scale
    
    mask = Image.new('L', (w, h), 0)
    draw = ImageDraw.Draw(mask)
    draw.rounded_rectangle((0, 0, w, h), radius=r, fill=255)
    
    # Resize down
    mask = mask.resize(size, Image.Resampling.LANCZOS)
    return mask

def create_border(size, radius, thickness):
    scale = 4
    w, h = size[0] * scale, size[1] * scale
    r = radius * scale
    t = thickness * scale
    
    img = Image.new('RGBA', (w, h), (0, 0, 0, 0))
    draw = ImageDraw.Draw(img)
    
    # Draw stroke
    # Inset slightly to ensure it sits ON the edge, not outside
    # Actually, for a focus border, we usually want it slightly outside or exactly on the edge.
    # Let's align it exactly with the mask edge.
    draw.rounded_rectangle((t/2, t/2, w - t/2, h - t/2), radius=r, outline=(255, 255, 255, 255), width=int(t))
    
    img = img.resize(size, Image.Resampling.LANCZOS)
    return img

def process_card(filename, mask):
    filepath = os.path.join(BASE_PATH, filename)
    if not os.path.exists(filepath):
        print(f"File not found: {filepath}")
        return

    try:
        img = Image.open(filepath).convert('RGBA')
        
        # Resize and Crop to fill TARGET_SIZE
        img = ImageOps.fit(img, TARGET_SIZE, method=Image.Resampling.LANCZOS, centering=(0.5, 0.5))
        
        # Apply mask
        img.putalpha(mask)
        
        # Save
        img.save(filepath)
        print(f"Processed and saved: {filename}")
        
    except Exception as e:
        print(f"Error processing {filename}: {e}")

def main():
    print(f"Processing cards in {BASE_PATH}")
    
    # 1. Create shared mask
    mask = create_mask(TARGET_SIZE, RADIUS)
    
    # 2. Process all cards
    for card_name in CARDS:
        process_card(card_name, mask)
        
    # 3. Generate Border (Standard)
    border = create_border(TARGET_SIZE, RADIUS, BORDER_WIDTH)
    border.save(os.path.join(BASE_PATH, "card_border.png"))
    print("Generated card_border.png")

    # 4. Generate Focus Border (Larger)
    # We want it to sit outside the card.
    # Let's say 8px gap (visual) -> 16px (script scale)
    gap = 16 
    # The border stroke itself has thickness.
    focus_size = (TARGET_SIZE[0] + gap*2, TARGET_SIZE[1] + gap*2)
    # Radius must increase by gap to be concentric
    focus_radius = RADIUS + gap
    
    focus_border = create_border(focus_size, focus_radius, BORDER_WIDTH)
    focus_border.save(os.path.join(BASE_PATH, "card_border_focus.png"))
    print("Generated card_border_focus.png")
    
    # 5. Generate pure mask file (just in case we still need it for diffuse anywhere)
    mask_rgba = Image.new('RGBA', TARGET_SIZE, (255, 255, 255, 0))
    mask_rgba.putalpha(mask)
    # Actually, for diffuse, we want visible pixel = white, invisible = transparency?
    # Or just white on black? Kodi diffuse usually uses alpha channel. 
    # Let's verify: Kodi diffuse uses the alpha of the diffuse texture.
    # So a white rounded rect on transparent background is correct.
    mask_img = Image.new('RGBA', TARGET_SIZE, (0,0,0,0))
    draw = ImageDraw.Draw(mask_img)
    draw.rounded_rectangle((0,0, TARGET_SIZE[0], TARGET_SIZE[1]), radius=RADIUS, fill=(255,255,255,255))
    mask_img.save(os.path.join(BASE_PATH, "card_mask.png"))
    print("Generated card_mask.png")

    # 6. Generate Large Poster Assets (Detail View)
    # Visual Size: 460x690 -> Script Size: 920x1380
    LARGE_SIZE = (920, 1380)
    LARGE_RADIUS = 60 # Visual 30px
    
    # Large Mask
    large_mask = create_mask(LARGE_SIZE, LARGE_RADIUS)
    large_mask_img = Image.new('RGBA', LARGE_SIZE, (0,0,0,0))
    draw_large = ImageDraw.Draw(large_mask_img)
    draw_large.rounded_rectangle((0,0, LARGE_SIZE[0], LARGE_SIZE[1]), radius=LARGE_RADIUS, fill=(255,255,255,255))
    large_mask_img.save(os.path.join(BASE_PATH, "large_poster_mask.png"))
    print("Generated large_poster_mask.png")
    
    # Large Border (Focus/Static)
    # Gap 16px (Visual 8px)
    large_border_size = (LARGE_SIZE[0] + gap*2, LARGE_SIZE[1] + gap*2)
    large_border_radius = LARGE_RADIUS + gap
    large_border = create_border(large_border_size, large_border_radius, BORDER_WIDTH)
    large_border.save(os.path.join(BASE_PATH, "large_poster_border.png"))
    print("Generated large_poster_border.png")

if __name__ == "__main__":
    main()
