# wcs/tmdb_handler.py

import xbmc
import xbmcaddon
import xbmcgui
import xbmcplugin
import requests
from urllib.parse import quote_plus
import datetime # Přidán import
import os      # <-- PŘIDÁN IMPORT
import shutil  # <-- PŘIDÁN IMPORT

# Importy našich modulů (budeme přidávat dle potřeby)
from wcs.ui import UI as ui
from wcs import user_data, utils # Odstraněn import api
from wcs.ui.ContextMenu import create_movie_context_menu, create_series_context_menu # <-- PŘIDÁN IMPORT Z UTILS

# Konstanty pro práci s TMDb API
# --- API klíč pro The Movie Database je nyní v nastavení doplňku ---

def get_tmdb_api_key(addon=None):
    """
    Získá TMDB API klíč z nastavení doplňku.
    Vyhodí ValueError pokud klíč není nastaven.
    """
    if addon is None:
        addon = xbmcaddon.Addon()
    
    api_key = addon.getSetting('tmdb_api_key')
    
    # Pokud není klíč nastaven nebo je prázdný, vyhodíme chybu
    if not api_key:
        raise ValueError("TMDB API klíč není nastaven v nastavení doplňku")
    
    return api_key

def handle_tmdb_api_error(e, addon, handle=None):
    """
    Jednotné zpracování všech TMDb API chyb s dialogem a přímým odkazem na nastavení.
    """
    import xbmcgui
    
    error_title = "TMDb API - Chyba"
    
    if isinstance(e, ValueError) and "TMDB API klíč není nastaven" in str(e):
        # Klíč není nastaven
        _show_tmdb_settings_dialog(
            addon,
            error_title,
            "TMDb API klíč není nastaven.\\n\\nPro pokračování je potřeba nastavit platný TMDb API klíč v nastavení doplňku."
        )
    elif isinstance(e, requests.exceptions.HTTPError):
        if e.response.status_code == 401:
            # Neplatný API klíč
            _show_tmdb_settings_dialog(
                addon,
                error_title,
                "TMDb API klíč je neplatný nebo vypršel.\\n\\nPro pokračování je potřeba nastavit platný TMDb API klíč v nastavení doplňku."
            )
        else:
            # Jiné HTTP chyby
            _show_tmdb_settings_dialog(
                addon,
                error_title,
                f"Chyba při komunikaci s TMDb API (HTTP {e.response.status_code}).\\n\\nZkontrolujte TMDb API klíč v nastavení doplňku."
            )
    elif isinstance(e, requests.exceptions.RequestException):
        # Síťové chyby
        ui.popinfo(f"Chyba síťového připojení k TMDb API: {e}", addon.getAddonInfo('name'))
    else:
        # Ostatní chyby
        ui.popinfo(f"Neočekávaná chyba při zpracování TMDb dat: {e}", addon.getAddonInfo('name'))
    
    if handle:
        xbmcplugin.endOfDirectory(handle)

def _show_tmdb_settings_dialog(addon, title, message):
    """
    Zobrazí dialog s chybou TMDb API a nabídkou přejít do nastavení.
    Podobný systém jako u webshare credentials.
    """
    import xbmcgui
    
    dialog = xbmcgui.Dialog()
    if dialog.yesno(
        title,
        message,
        yeslabel="Otevřít Nastavení",
        nolabel="Zrušit"
    ):
        addon.openSettings()

def check_tmdb_api_key_validity(addon, api_key=None):
    """
    Ověří platnost TMDb API klíče testem volání.
    - Pokud je předán api_key, ověří přímo tento klíč (bez zásahu do settings)
    - Jinak ověří klíč uložený v nastavení addonu
    Vrací True pokud je klíč platný, jinak False.
    """
    try:
        key_to_check = api_key if api_key else get_tmdb_api_key(addon)
        test_url = f'https://api.themoviedb.org/3/configuration?api_key={key_to_check}'
        response = requests.get(test_url, timeout=10)
        response.raise_for_status()
        return True
    except requests.exceptions.HTTPError as e:
        try:
            status = e.response.status_code if e.response is not None else None
        except Exception:
            status = None
        if status == 401:
            return False
        return True  # Jiné HTTP chyby neznamenají špatný klíč
    except Exception:
        return True  # Síťové chyby neznamenají špatný klíč

def validate_tmdb_api_key_and_prompt(addon):
    """
    Zkontroluje platnost TMDb API klíče a pokud je neplatný, zobrazí dialog.
    Vrací True pokud je klíč platný nebo uživatel dialog zavřel, jinak False.
    """
    if not check_tmdb_api_key_validity(addon):
        _show_tmdb_settings_dialog(
            addon,
            "TMDb API - Neplatný klíč",
            "TMDb API klíč je neplatný nebo vypršel.\\n\\nPro pokračování je potřeba nastavit platný TMDb API klíč v nastavení doplňku."
        )
        return False
    return True

# Zde budou další funkce pro tmdb_... akce

# --- Funkce pro procházení TMDb ---

def browse_movie_genres(params, addon, handle):
    """Získá žánry filmů z TMDb a předá je do UI vrstvy."""
    try:
        api_params = {'api_key': get_tmdb_api_key(addon), 'language': 'cs-CZ'}
        response = requests.get('https://api.themoviedb.org/3/genre/movie/list', params=api_params)
        response.raise_for_status()
        data = response.json()
        genres = data.get('genres', [])
        ui.display_movie_genres(addon, handle, genres)
    except Exception as e:
        handle_tmdb_api_error(e, addon, handle)
        
# --- Funkce pro zobrazení seznamů filmů/seriálů ---

def list_movies(params, addon, handle):
    """Získá seznam filmů z TMDb a předá je do UI vrstvy."""
    from wcs.metadata.TMDbClient import fetch_tmdb_items
    from wcs.ui.UI import display_movie_list
    import xbmc
    api_params = {
        'page': params.get('page', '1')
    }
    category = params.get('category')
    genre_id = params.get('genre_id')
    country_code = params.get('country_code')
    year = params.get('year')
    endpoint = 'discover/movie'
    category_name = "Filmy"
    # --- OPRAVA: vždy přidat filtry žánru a země, pokud jsou v params ---
    if 'primary_release_date.lte' in params and 'primary_release_date.gte' in params:
        # Položka "Vše" – filtrujeme podle žánru a země, ale ne podle konkrétního roku
        if genre_id:
            api_params['with_genres'] = genre_id
        if country_code:
            api_params['with_origin_country'] = country_code
        api_params['primary_release_date.lte'] = params['primary_release_date.lte']
        api_params['primary_release_date.gte'] = params['primary_release_date.gte']
        api_params['sort_by'] = 'primary_release_date.desc'
        genre_name = params.get('genre_name', f'Žánr {genre_id}')
        country_name = params.get('country_name', country_code)
        category_name = f"Filmy / Výsledky: {genre_name} / {country_name} / Vše"
    elif genre_id and country_code and year:
        genre_name = params.get('genre_name', f'Žánr {genre_id}')
        country_name = params.get('country_name', country_code)
        category_name = f"Filmy / Výsledky: {genre_name} / {country_name} / {year}"
        endpoint = 'discover/movie'
        api_params['with_genres'] = genre_id
        api_params['with_origin_country'] = country_code
        api_params['primary_release_year'] = year
        api_params['sort_by'] = 'primary_release_date.desc'
    elif category:
        categories_map = {'popular', 'top_rated', 'now_playing', 'upcoming'}
        if category in categories_map:
            category_name = f"Filmy / {addon.getLocalizedString({'popular': 30110, 'top_rated': 30111, 'now_playing': 30112, 'upcoming': 30113}[category])}"
            endpoint = f'movie/{category}'
            # NEPŘIDÁVÁME sort_by!
        else:
            category_name = f"Filmy / {category.capitalize()}"
            endpoint = 'discover/movie'
            api_params['sort_by'] = 'primary_release_date.desc'
    elif genre_id:
        genre_name = params.get('genre_name', f'Žánr {genre_id}')
        category_name = f"Filmy / Žánry / {genre_name}"
        endpoint = 'discover/movie'
        api_params['with_genres'] = genre_id
        api_params['sort_by'] = 'primary_release_date.desc'
    elif country_code:
        country_name = params.get('country_name', country_code)
        category_name = f"Filmy / Země původu / {country_name}"
        endpoint = 'discover/movie'
        api_params['with_origin_country'] = country_code
        api_params['sort_by'] = 'primary_release_date.desc'
    else:
        category_name = f"Filmy / Populární"
        endpoint = 'movie/popular'
    results = []
    current_page = 1
    total_pages = 1
    try:
        import requests
        url = f'https://api.themoviedb.org/3/{endpoint}'
        api_params_full = dict(api_params)
        api_params_full['api_key'] = get_tmdb_api_key(addon)
        api_params_full['language'] = 'cs-CZ'
        response = requests.get(url, params=api_params_full, timeout=10)
        response.raise_for_status()
        data = response.json()
        if 'results' in data:
            results = data['results']
        elif 'parts' in data:
            results = data['parts']
        else:
            results = []
        current_page = data.get('page', 1)
        total_pages = data.get('total_pages', 1)
        # --- DODATEČNÁ FILTRACE: nikdy nezobrazuj budoucí filmy ani ve stejném roce ---
        if 'primary_release_date.lte' in params and 'primary_release_date.gte' in params:
            import datetime
            today = datetime.datetime.now().strftime('%Y-%m-%d')
            current_year = datetime.datetime.now().year
            filtered = []
            for m in results:
                rel_date = m.get('release_date', '')
                year = rel_date[:4]
                try:
                    year_int = int(year)
                except Exception:
                    year_int = 0
                if rel_date and rel_date <= today and 1950 <= year_int <= current_year:
                    filtered.append(m)
            results = filtered
    except Exception as e:
        import traceback
        xbmc.log(f"[WCS] Chyba v list_movies: {e}\n{traceback.format_exc()}", xbmc.LOGERROR)
        handle_tmdb_api_error(e, addon, handle)
    display_movie_list(addon, handle, results, category_name, params, current_page, total_pages)

def list_series(params, addon, handle):
    """Získá seznam seriálů z TMDb a předá je do UI vrstvy."""
    from wcs.metadata.TMDbClient import fetch_tmdb_items
    from wcs.ui.UI import display_series_list, popinfo
    import traceback
    import xbmc
    api_params = {
        'page': params.get('page', '1')
    }
    category = params.get('category')
    genre_id = params.get('genre_id')
    country_code = params.get('country_code')
    year = params.get('year')
    endpoint = 'discover/tv'
    category_name = "Seriály"
    # --- OPRAVA: vždy přidat filtry žánru a země, pokud jsou v params ---
    if 'first_air_date.lte' in params and 'first_air_date.gte' in params:
        # Položka "Vše" – filtrujeme podle žánru a země, ale ne podle konkrétního roku
        if genre_id:
            api_params['with_genres'] = genre_id
        if country_code:
            api_params['with_origin_country'] = country_code
        api_params['first_air_date.lte'] = params['first_air_date.lte']
        api_params['first_air_date.gte'] = params['first_air_date.gte']
        api_params['sort_by'] = 'first_air_date.desc'
        genre_name = params.get('genre_name', f'Žánr {genre_id}')
        country_name = params.get('country_name', country_code)
        category_name = f"Seriály / Výsledky: {genre_name} / {country_name} / Vše"
    elif genre_id and country_code and year:
        genre_name = params.get('genre_name', f'Žánr {genre_id}')
        country_name = params.get('country_name', country_code)
        category_name = f"Seriály / Výsledky: {genre_name} / {country_name} / {year}"
        endpoint = 'discover/tv'
        api_params['with_genres'] = genre_id
        api_params['with_origin_country'] = country_code
        api_params['first_air_date_year'] = year
        api_params['sort_by'] = 'first_air_date.desc'
    elif category:
        categories_map = {'popular', 'top_rated', 'on_the_air', 'airing_today'}
        if category in categories_map:
            category_name = f"Seriály / {addon.getLocalizedString({'popular': 30110, 'top_rated': 30111, 'on_the_air': 30112, 'airing_today': 30113}[category])}"
            endpoint = f'tv/{category}'
            # NEPŘIDÁVÁME sort_by!
        else:
            category_name = f"Seriály / {category.capitalize()}"
            endpoint = 'discover/tv'
            api_params['sort_by'] = 'first_air_date.desc'
    elif genre_id:
        genre_name = params.get('genre_name', f'Žánr {genre_id}')
        category_name = f"Seriály / Žánry / {genre_name}"
        endpoint = 'discover/tv'
        api_params['with_genres'] = genre_id
        api_params['sort_by'] = 'first_air_date.desc'
    elif country_code:
        country_name = params.get('country_name', country_code)
        category_name = f"Seriály / Země původu / {country_name}"
        endpoint = 'discover/tv'
        api_params['with_origin_country'] = country_code
        api_params['sort_by'] = 'first_air_date.desc'
    else:
        category_name = f"Seriály / Populární"
        endpoint = 'tv/popular'
    results = []
    current_page = 1
    total_pages = 1
    try:
        import requests
        url = f'https://api.themoviedb.org/3/{endpoint}'
        api_params_full = dict(api_params)
        # OPRAVA: vždy použijeme platný klíč z kódu
        api_params_full['api_key'] = get_tmdb_api_key(addon)
        api_params_full['language'] = 'cs-CZ'
        xbmc.log(f"[WCS] TMDb list_series: endpoint={endpoint}, params={api_params_full}", xbmc.LOGINFO)
        response = requests.get(url, params=api_params_full, timeout=10)
        response.raise_for_status()
        data = response.json()
        xbmc.log(f"[WCS] TMDb list_series: response={data}", xbmc.LOGINFO)
        if 'results' in data:
            results = data['results']
        elif 'parts' in data:
            results = data['parts']
        else:
            results = []
        current_page = data.get('page', 1)
        total_pages = data.get('total_pages', 1)
        # --- DODATEČNÁ FILTRACE: nikdy nezobrazuj budoucí roky ---
        if 'first_air_date.lte' in params:
            import datetime
            current_year = datetime.datetime.now().year
            filtered = []
            for s in results:
                y = s.get('first_air_date', '')[:4]
                try:
                    y_int = int(y)
                except Exception:
                    y_int = 0
                if y_int <= current_year:
                    filtered.append(s)
            results = filtered
    except Exception as e:
        tb = traceback.format_exc()
        xbmc.log(f"[WCS] Chyba v list_series: {e}\n{tb}", xbmc.LOGERROR)
        handle_tmdb_api_error(e, addon, handle)
    display_series_list(addon, handle, results, category_name, params, current_page, total_pages)

# --- Funkce pro Kombinovaný výběr (Žánr -> Země -> Rok) --- 
# Funkce combo_select_countries a combo_select_years byly přesunuty do ui.py dle architektonických pravidel.

def combo_select_genres(params, addon, handle):
    """Zobrazí první krok kombinovaného výběru: výběr žánru."""
    media_type = params.get('media_type')
    if not media_type:
        ui.popinfo("Chyba: Chybí typ média pro kombinovaný výběr.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
        
    media_label = addon.getLocalizedString(30103) if media_type == 'tv' else addon.getLocalizedString(30102)
    xbmcplugin.setPluginCategory(handle, f'TMDb / {media_label} / {addon.getLocalizedString(30114)} (1/3: {addon.getLocalizedString(30104)})')
    
    genre_url_part = 'tv' if media_type == 'tv' else 'movie'
    url = f'https://api.themoviedb.org/3/genre/{genre_url_part}/list?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
    
    try:
        response = requests.get(url)
        response.raise_for_status()
        data = response.json()
        genres = data.get('genres', [])
        if not genres:
            ui.popinfo(f"Nepodařilo se načíst žánry ({media_label}) z TMDb.", addon.getAddonInfo('name'))
        
        for genre in genres:
            genre_id = genre['id']
            genre_name = genre['name']
            listitem = xbmcgui.ListItem(label=genre_name)
            listitem.setArt({'icon': 'DefaultGenre.png'})
            # Odkaz na další krok: výběr země
            next_url = f"plugin://{addon.getAddonInfo('id')}?action=tmdb_combo_countries&media_type={media_type}&genre_id={genre_id}&genre_name={quote_plus(genre_name)}"
            xbmcplugin.addDirectoryItem(handle, next_url, listitem, True)
            
    except Exception as e:
        handle_tmdb_api_error(e, addon, handle)
        
    xbmcplugin.endOfDirectory(handle)

def combo_select_years(params, addon, handle):
    """Zobrazí třetí krok kombinovaného výběru: výběr roku."""
    media_type = params.get('media_type')
    genre_id = params.get('genre_id')
    genre_name = params.get('genre_name')
    country_code = params.get('country_code')
    country_name = params.get('country_name')
    
    if not all([media_type, genre_id, genre_name, country_code, country_name]):
        ui.popinfo("Chyba: Chybějící parametry pro výběr roku.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
        
    media_label = addon.getLocalizedString(30103) if media_type == 'tv' else addon.getLocalizedString(30102)
    xbmcplugin.setPluginCategory(handle, f'TMDb / {media_label} / {addon.getLocalizedString(30114)} (3/3: {addon.getLocalizedString(30106)}) - {genre_name} / {country_name}')
    
    try:
        current_year = datetime.datetime.now().year
        # Zobrazíme roky od aktuálního do 1950
        for y in range(current_year, 1949, -1):
            listitem = xbmcgui.ListItem(label=str(y))
            listitem.setArt({'icon': 'DefaultYear.png'})
            # Odkaz na finální zobrazení výsledků (akce tmdb_combo_results v router.py)
            # Tato akce už deleguje na list_movies/list_series
            next_url = f"plugin://{addon.getAddonInfo('id')}?action=tmdb_combo_results&media_type={media_type}&genre_id={genre_id}&country_code={country_code}&year={y}&genre_name={quote_plus(genre_name)}&country_name={quote_plus(country_name)}"
            xbmcplugin.addDirectoryItem(handle, next_url, listitem, True)
            
    except Exception as e:
        ui.popinfo(f"Chyba při generování seznamu roků: {e}", addon.getAddonInfo('name'))
        import traceback
        traceback.print_exc()
            
    xbmcplugin.endOfDirectory(handle)

# --- Funkce pro zobrazení detailů ---

def show_item_details(params, addon):
    """Získá a zobrazí detailní informace o filmu/seriálu z TMDb."""
    media_type = params.get('media_type')
    tmdb_id = params.get('tmdb_id')

    if not media_type or not tmdb_id:
        ui.popinfo("Chyba: Chybí parametry pro zobrazení detailů (media_type, tmdb_id).", addon.getAddonInfo('name'))
        return

    try:
        tmdb_id = int(tmdb_id) # Ujistíme se, že ID je číslo
    except ValueError:
        ui.popinfo(f"Chyba: Neplatné TMDb ID: {tmdb_id}", addon.getAddonInfo('name'))
        return
        
    if media_type not in ['movie', 'tv']:
        ui.popinfo(f"Chyba: Neplatný typ média: {media_type}", addon.getAddonInfo('name'))
        return

    details_text = ""
    dialog_title = "Zobrazit detaily"

    try:
        # Přidáme credits a keywords pro více informací
        api_params = {
            'api_key': get_tmdb_api_key(addon),
            'language': 'cs-CZ',
            'append_to_response': 'credits,keywords' 
        }
        url = f'https://api.themoviedb.org/3/{media_type}/{tmdb_id}'
        response = requests.get(url, params=api_params)
        response.raise_for_status() # Kontrola HTTP chyb
        data = response.json()

        # Získání lokalizovaných popisků
        L = addon.getLocalizedString
        label_title = L(30130) # Název
        label_orig_title = L(30131) # Původní název
        label_year = L(30106) # Rok
        label_first_air = L(30132) # První vysílání
        label_last_air = L(30133) # Poslední vysílání
        label_status = L(30134) # Stav
        label_seasons = L(30135) # Počet sérií
        label_episodes = L(30136) # Počet epizod
        label_runtime = L(30137) # Délka
        label_genres = L(30104) # Žánry
        label_plot = L(30138) # Popis
        label_director = L(30139) # Režie
        label_created_by = L(30140) # Tvůrci
        label_cast = L(30141) # Hrají
        label_rating = L(30142) # Hodnocení
        label_tagline = L(30143) # Motto
        label_keywords = L(30144) # Klíčová slova
        label_minutes = L(30145) # min.

        if media_type == 'movie':
            dialog_title = data.get('title', L(30102))
            details_text += f"[B]{label_title}:[/B] {data.get('title', '?')}\n"
            if data.get('original_title') and data.get('original_title') != data.get('title'):
                 details_text += f"[I]{label_orig_title}:[/I] {data.get('original_title')}\n"
            year_val = data.get('release_date', '')[:4]
            if year_val:
                details_text += f"[B]{label_year}:[/B] {year_val}\n"
            runtime = data.get('runtime')
            if runtime:
                 details_text += f"[B]{label_runtime}:[/B] {runtime} {label_minutes}\n"
            genres = ", ".join([g['name'] for g in data.get('genres', [])])
            if genres:
                 details_text += f"[B]{label_genres}:[/B] {genres}\n"
            tagline = data.get('tagline')
            if tagline:
                details_text += f"[I]{label_tagline}:[/I] {tagline}\n"
            rating = data.get('vote_average')
            if rating:
                 details_text += f"[B]{label_rating}:[/B] {rating:.1f}/10\n"
            
            # Režie (z credits)
            directors = [c['name'] for c in data.get('credits', {}).get('crew', []) if c['job'] == 'Director']
            if directors:
                details_text += f"[B]{label_director}:[/B] {', '.join(directors)}\n"
            
            # Herci (prvních 5 z credits)
            cast = [c['name'] for c in data.get('credits', {}).get('cast', [])[:5]]
            if cast:
                details_text += f"[B]{label_cast}:[/B] {', '.join(cast)}, ...\n"
                
            details_text += f"\n[B]{label_plot}:[/B]\n{data.get('overview', '')}\n"
            
            # Klíčová slova
            keywords = ", ".join([k['name'] for k in data.get('keywords', {}).get('keywords', [])])
            if keywords:
                 details_text += f"\n[B]{label_keywords}:[/B] {keywords}\n"

        elif media_type == 'tv':
            dialog_title = data.get('name', L(30103))
            details_text += f"[B]{label_title}:[/B] {data.get('name', '?')}\n"
            if data.get('original_name') and data.get('original_name') != data.get('name'):
                 details_text += f"[I]{label_orig_title}:[/I] {data.get('original_name')}\n"
            first_air = data.get('first_air_date', '')[:4]
            last_air = data.get('last_air_date', '')[:4]
            if first_air:
                 air_info = f"{first_air} - {last_air}" if last_air and last_air != first_air else first_air
                 details_text += f"[B]{label_first_air}:[/B] {air_info}\n"
            status = data.get('status')
            if status:
                 details_text += f"[B]{label_status}:[/B] {status}\n"
            num_seasons = data.get('number_of_seasons')
            num_episodes = data.get('number_of_episodes')
            if num_seasons is not None:
                 details_text += f"[B]{label_seasons}:[/B] {num_seasons}\n"
            if num_episodes is not None:
                 details_text += f"[B]{label_episodes}:[/B] {num_episodes}\n"
            genres = ", ".join([g['name'] for g in data.get('genres', [])])
            if genres:
                 details_text += f"[B]{label_genres}:[/B] {genres}\n"
            tagline = data.get('tagline')
            if tagline:
                details_text += f"[I]{label_tagline}:[/I] {tagline}\n"
            rating = data.get('vote_average')
            if rating:
                 details_text += f"[B]{label_rating}:[/B] {rating:.1f}/10\n"
                 
            # Tvůrci
            creators = [c['name'] for c in data.get('created_by', [])]
            if creators:
                 details_text += f"[B]{label_created_by}:[/B] {', '.join(creators)}\n"
                 
            # Herci (prvních 5 z credits)
            cast = [c['name'] for c in data.get('credits', {}).get('cast', [])[:5]]
            if cast:
                details_text += f"[B]{label_cast}:[/B] {', '.join(cast)}, ...\n"
                
            details_text += f"\n[B]{label_plot}:[/B]\n{data.get('overview', '')}\n"
            
            # Klíčová slova
            keywords_data = data.get('keywords', {}).get('results', []) # U seriálů je to 'results'
            keywords = ", ".join([k['name'] for k in keywords_data])
            if keywords:
                 details_text += f"\n[B]{label_keywords}:[/B] {keywords}\n"

    except Exception as e:
        handle_tmdb_api_error(e, addon)
        return

    # Zobrazení dialogu
    if details_text:
        xbmcgui.Dialog().textviewer(dialog_title, details_text)
    else:
        ui.popinfo("Nepodařilo se získat žádné detaily.", addon.getAddonInfo('name'))

# --- Funkce pro detaily (budou přidány později) ---

# --- Funkce pro přidávání (budou přidány později) ---

# --- Funkce pro vyhledávání (budou přidány později) --- 

# --- NOVÉ FUNKCE PRO ZOBRAZENÍ SEZÓN A EPIZOD ---

def display_series_seasons(params, addon, handle):
    """Zobrazí seznam sezón pro daný seriál."""
    series_id = params.get('series_id')
    series_name = params.get('name', 'Seriál') # Fallback jméno
    if not series_id:
        ui.popinfo("Chyba: Chybí ID seriálu pro zobrazení sezón.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return

    xbmcplugin.setPluginCategory(handle, f'TMDb / {series_name} / Sezóny')
    
    try:
        # Použijeme stejné API volání jako pro detail seriálu, abychom získali info o sezónách
        url = f'https://api.themoviedb.org/3/tv/{series_id}?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
        response = requests.get(url)
        response.raise_for_status()
        data = response.json()
        seasons = data.get('seasons', [])
        
        if not seasons:
            ui.popinfo("Nebyly nalezeny žádné sezóny.", addon.getAddonInfo('name'))
            
        # Bezpečné načtení nastavení 'show_specials'
        show_specials_setting = addon.getSetting('show_specials') # Načíst jako string
        show_specials = show_specials_setting == 'true' # Bezpečné porovnání

        for season in seasons:
            # Přeskočíme "Specials" (season_number 0), pokud nemají epizody
            # nebo pokud to explicitně nenastavíme v addonu
            # if season['season_number'] == 0 and not addon.getSettingBool('show_specials'): # Původní problematický řádek
            if season['season_number'] == 0 and not show_specials: # Použití bezpečně načtené hodnoty
                continue
            
            season_number = season['season_number']
            label = season.get('name') or f"{addon.getLocalizedString(30135)} {season_number}" # Použijeme lokalizovaný string
            listitem = xbmcgui.ListItem(label=label)
            if season.get('poster_path'):
                listitem.setArt({'thumb': f"https://image.tmdb.org/t/p/w500{season.get('poster_path')}"})
            utils.set_video_info_tag(listitem, {'title': label, 'plot': season.get('overview', '')})
            # Odkaz na zobrazení epizod dané sezóny
            next_url = f"plugin://{addon.getAddonInfo('id')}?action=tmdb_season_detail&series_id={series_id}&season_number={season_number}&series_name={quote_plus(series_name)}"
            xbmcplugin.addDirectoryItem(handle, next_url, listitem, True)
            
    except Exception as e:
        handle_tmdb_api_error(e, addon, handle)
        
    xbmcplugin.endOfDirectory(handle)


def get_next_episode_from_tmdb(tmdb_id, current_season, current_episode, addon=None):
    """
    Získá skutečný další díl z TMDb včetně přechodu mezi sezónami.
    
    Args:
        tmdb_id: TMDb ID seriálu
        current_season: Aktuální sezóna
        current_episode: Aktuální epizoda
        addon: Instance addonu
        
    Returns:
        dict: Metadata další epizody nebo None pokud neexistuje
            {
                'season': int,
                'episode': int,
                'episode_title': str,
                'plot': str,
                'poster': str,
                'episode_thumb': str,
                'series_name': str,
                'year': str,
                'rating': str,
                'genre': str,
                'fanart': str,
                'runtime': str
            }
    """
    if addon is None:
        addon = xbmcaddon.Addon()
    
    current_season = int(current_season)
    current_episode = int(current_episode)
    
    # [FIX] Pokud začínáme od nuly (nový seriál), rovnou hledáme v Sezóně 1 (přeskočíme Specials/S0)
    if current_season == 0 and current_episode == 0:
        current_season = 1
    
    try:
        api_key = get_tmdb_api_key(addon)
        
        # 1) Získáme data aktuální sezóny
        season_url = f'https://api.themoviedb.org/3/tv/{tmdb_id}/season/{current_season}?api_key={api_key}&language=cs-CZ'
        response = requests.get(season_url)
        response.raise_for_status()
        season_data = response.json()
        episodes = season_data.get('episodes', [])
        
        next_season = current_season
        next_episode = current_episode + 1
        
        # 2) Zjistíme jestli další epizoda existuje v aktuální sezóně
        episode_exists = any(ep['episode_number'] == next_episode for ep in episodes)
        
        if not episode_exists:
            # Zkusíme první epizodu další sezóny
            next_season = current_season + 1
            next_episode = 1
            
            try:
                next_season_url = f'https://api.themoviedb.org/3/tv/{tmdb_id}/season/{next_season}?api_key={api_key}&language=cs-CZ'
                next_response = requests.get(next_season_url)
                next_response.raise_for_status()
                next_season_data = next_response.json()
                episodes = next_season_data.get('episodes', [])
                
                # Ověříme že existuje E01 v další sezóně
                episode_exists = any(ep['episode_number'] == 1 for ep in episodes)
                
                if not episode_exists:
                    xbmc.log(f"[plugin.video.milionar] get_next_episode: No next episode found for S{current_season:02d}E{current_episode:02d}", level=xbmc.LOGINFO)
                    return None
                    
            except requests.exceptions.HTTPError as e:
                if e.response.status_code == 404:
                    # Další sezóna neexistuje
                    xbmc.log(f"[plugin.video.milionar] get_next_episode: Season {next_season} doesn't exist", level=xbmc.LOGINFO)
                    return None
                raise
        
        # 3) Najdeme metadata epizody
        episode_data = None
        for ep in episodes:
            if ep['episode_number'] == next_episode:
                episode_data = ep
                break
        
        if not episode_data:
            return None
        
        # 4) Získáme metadata seriálu
        series_url = f'https://api.themoviedb.org/3/tv/{tmdb_id}?api_key={api_key}&language=cs-CZ'
        series_response = requests.get(series_url)
        series_response.raise_for_status()
        series_data = series_response.json()
        
        series_name = series_data.get('name', '')
        year = str(series_data.get('first_air_date', '')[:4]) if series_data.get('first_air_date') else ''
        rating = f"{series_data.get('vote_average', 0):.1f}"
        genre = ', '.join([g['name'] for g in series_data.get('genres', [])[:2]])
        
        fanart = ''
        if series_data.get('backdrop_path'):
            fanart = f"https://image.tmdb.org/t/p/original{series_data['backdrop_path']}"
        
        poster = ''
        if series_data.get('poster_path'):
            poster = f"https://image.tmdb.org/t/p/w500{series_data['poster_path']}"
        
        episode_thumb = ''
        if episode_data.get('still_path'):
            episode_thumb = f"https://image.tmdb.org/t/p/w500{episode_data['still_path']}"
        
        runtime_minutes = episode_data.get('runtime', 0)
        runtime = f"{runtime_minutes} min" if runtime_minutes and runtime_minutes > 0 else ""
        
        ep_name = episode_data.get('name') or f"Epizoda {next_episode}"
        
        # 5) Ziskame logo serialu z /images endpointu
        series_logo = ''
        try:
            images_url = f'https://api.themoviedb.org/3/tv/{tmdb_id}/images?api_key={api_key}'
            images_response = requests.get(images_url, timeout=5)
            if images_response.status_code == 200:
                images_data = images_response.json()
                logos = images_data.get('logos', [])
                # Preferujeme anglicke logo nebo logo bez jazyka, serazene podle vote_average
                # Kodi neumi SVG -- preskocime SVG soubory
                best_logo = None
                best_score = -1
                for logo in logos:
                    file_path = logo.get('file_path', '')
                    if file_path.lower().endswith('.svg'):
                        continue  # Kodi cannot render SVG
                    lang = logo.get('iso_639_1')
                    # Preferujeme: en > null > ostatni
                    if lang == 'en':
                        lang_score = 2
                    elif lang is None or lang == '':
                        lang_score = 1
                    else:
                        lang_score = 0
                    vote = logo.get('vote_average', 0)
                    score = lang_score * 100 + vote
                    if score > best_score:
                        best_score = score
                        best_logo = logo
                if best_logo and best_logo.get('file_path'):
                    series_logo = f"https://image.tmdb.org/t/p/w300{best_logo['file_path']}"
        except Exception as logo_err:
            xbmc.log(f"[plugin.video.milionar] get_next_episode: Logo fetch error: {logo_err}", level=xbmc.LOGDEBUG)
        
        result = {
            'season': next_season,
            'episode': next_episode,
            'episode_title': ep_name,
            'plot': episode_data.get('overview', ''),
            'poster': poster,
            'episode_thumb': episode_thumb,
            'series_name': series_name,
            'series_logo': series_logo,
            'year': year,
            'rating': rating,
            'genre': genre,
            'fanart': fanart,
            'runtime': runtime,
            'tmdb_id': tmdb_id
        }
        
        xbmc.log(f"[plugin.video.milionar] get_next_episode: Found S{next_season:02d}E{next_episode:02d} '{ep_name}'", level=xbmc.LOGINFO)
        return result
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] get_next_episode error: {e}", level=xbmc.LOGERROR)
        return None

def display_season_episodes(params, addon, handle):
    """Zobrazí seznam epizod pro danou sezónu seriálu."""
    series_id = params.get('series_id')
    season_number = params.get('season_number')
    series_name = params.get('series_name', 'Seriál')
    
    if not series_id or season_number is None: # Kontrola i na season_number
        ui.popinfo("Chyba: Chybí ID seriálu nebo číslo sezóny pro zobrazení epizod.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
    
    # Získání názvu sezóny (pro titulek)
    season_name = f"{addon.getLocalizedString(30135)} {season_number}" # Výchozí
    try:
        # Rychlé volání pro detail sezóny kvůli názvu
        season_url = f'https://api.themoviedb.org/3/tv/{series_id}/season/{season_number}?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
        season_resp = requests.get(season_url)
        season_resp.raise_for_status()
        season_data = season_resp.json()
        season_name = season_data.get('name', season_name)
    except: pass # Nevadí, když se nepodaří, použije se výchozí název
    
    xbmcplugin.setPluginCategory(handle, f'TMDb / {series_name} / {season_name}')

    try:
        # API volání pro získání epizod (stejné jako bylo v routeru)
        url = f'https://api.themoviedb.org/3/tv/{series_id}/season/{season_number}?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
        response = requests.get(url)
        response.raise_for_status()
        data = response.json()
        episodes = data.get('episodes', [])

        if not episodes:
            ui.popinfo("Nebyly nalezeny žádné epizody.", addon.getAddonInfo('name'))

        for ep in episodes:
            ep_num = ep['episode_number']
            ep_name = ep.get('name') or f"{addon.getLocalizedString(30146)} {ep_num}" # Použijeme lokalizovaný string (ID 30146 pro "Epizoda")
            label = f"S{int(season_number):02d}E{int(ep_num):02d} {ep_name}"
            plot = ep.get('overview', '')
            # Použijeme obrázek epizody pro zobrazení v menu, ale poster seriálu pro search dialog
            episode_thumb = f"https://image.tmdb.org/t/p/w500{ep['still_path']}" if ep.get('still_path') else ''
            listitem = xbmcgui.ListItem(label=label)
            if episode_thumb:
                listitem.setArt({'thumb': episode_thumb, 'icon': episode_thumb})
            utils.set_video_info_tag(listitem, {
                'title': label,
                'plot': plot,
                'season': int(season_number),
                'episode': int(ep_num)
                # Můžeme přidat i další info jako 'aired', 'rating', atd.
            })
            listitem.setProperty('IsPlayable', 'true') # Epizoda je přehratelná (spustí vyhledávání)

            # Kontextové menu pro epizody
            from wcs.ui.ContextMenu import create_episode_context_menu
            commands = create_episode_context_menu(
                addon=addon,
                series_tmdb_id=series_id,
                series_name=series_name,
                episode_title=ep_name,
                season_number=season_number,
                episode_number=ep_num,
                plot=plot,
                poster=episode_thumb
            )
            listitem.addContextMenuItems(commands)

            # Kliknutí na epizodu spustí vyhledávání na WS
            # Předáme relevantní metadata pro vyhledávání a pro ListItem v Kodi
            # Získáme fanart z backdrop_path nebo poster_path
            fanart = ""
            rating = ""
            genre = ""
            year = ""
            runtime = ""
            
            try:
                # Načteme data seriálu pro získání fanart a dalších metadat
                series_url = f'https://api.themoviedb.org/3/tv/{series_id}?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
                series_resp = requests.get(series_url)
                series_resp.raise_for_status()
                series_data = series_resp.json()
                
                if series_data.get('backdrop_path'):
                    fanart = f"https://image.tmdb.org/t/p/original{series_data['backdrop_path']}"
                elif series_data.get('poster_path'):
                    fanart = f"https://image.tmdb.org/t/p/original{series_data['poster_path']}"
                
                # Načteme rok seriálu
                year = str(series_data.get('first_air_date', '')[:4]) if series_data.get('first_air_date') else ''
                
                # Načteme žánr seriálu
                genre = ', '.join([g['name'] for g in series_data.get('genres', [])[:2]])
                
                # Načteme hodnocení seriálu
                rating = f"{series_data.get('vote_average', 0):.1f}"
                
                # Načteme délku epizody
                runtime_minutes = ep.get('runtime', 0)
                runtime = f"{runtime_minutes} min" if runtime_minutes > 0 else ""
                
            except:
                # Pokud se nepodaří načíst data seriálu, použijeme prázdné hodnoty
                pass
            
            # Sestavíme správný název ve formátu "Název seriálu – Název epizody"
            display_title = f"{series_name} – {ep_name}" if ep_name and ep_name != f"Epizoda {ep_num}" else series_name
            
            # Pro search dialog použijeme poster seriálu místo obrázku epizody
            poster = ""
            if series_data.get('poster_path'):
                poster = f"https://image.tmdb.org/t/p/w500{series_data['poster_path']}"
            
            # Do search dialogu posíláme poster seriálu jako tmdb_thumb (UI poster)
            # a still epizody pošleme zvlášť jako tmdb_episode_thumb pro úvodní dialog
            episode_thumb_enc = quote_plus(episode_thumb) if episode_thumb else ''
            tmdb_params = (f"&tmdb_title={quote_plus(display_title)}"
                           f"&tmdb_plot={quote_plus(plot)}"
                           f"&tmdb_thumb={quote_plus(poster)}"
                           f"&tmdb_episode_thumb={episode_thumb_enc}"
                           f"&tmdb_fanart={quote_plus(fanart)}"
                           f"&tmdb_rating={quote_plus(rating)}"
                           f"&tmdb_genre={quote_plus(genre)}"
                           f"&tmdb_year={quote_plus(year)}"
                           f"&tmdb_runtime={quote_plus(runtime)}"
                           f"&tmdb_season={season_number}"
                           f"&tmdb_episode={ep_num}"
                           f"&tmdb_series_name={quote_plus(series_name)}"
                           f"&tmdb_id={series_id}")
            search_url = f"plugin://{addon.getAddonInfo('id')}?action=tmdb_episode_search&series_name={quote_plus(series_name)}&season={season_number}&episode={ep_num}&ep_name={quote_plus(ep_name)}{tmdb_params}"
            xbmcplugin.addDirectoryItem(handle, search_url, listitem, False)
            
    except Exception as e:
        handle_tmdb_api_error(e, addon, handle)
        
    xbmcplugin.endOfDirectory(handle)

# Původní akce tmdb_movie_detail z router.py
def display_movie_text_details(params, addon, handle):
    """Zobrazí jednoduché textové detaily filmu (zastaralé?)."""
    # Tuto funkci možná později nahradíme voláním show_item_details
    idx = params.get('idx') # Tato akce byla volána s 'idx' zřejmě z tmdb_added_movies
    movie_id = params.get('movie_id') # Nebo možná s movie_id?
    
    movie_data = None
    if idx is not None:
        try:
            tmdb_movies = user_data.load_tmdb_movies(addon)
            movie_data = tmdb_movies[int(idx)]
        except: pass # Chybu ignorujeme, zkusíme movie_id
    
    if not movie_data and movie_id:
        # Pokud máme ID, mohli bychom načíst data z API, ale show_item_details to dělá lépe
        # Prozatím jen zobrazíme chybu, pokud nemáme data z idx
        ui.popinfo("Nelze zobrazit detaily filmu jen s ID v této funkci.", addon.getAddonInfo('name'))
        # xbmcplugin.endOfDirectory(handle) # Tato akce by neměla končit adresář, je volána z menu
        return
    
    if movie_data:
        title = movie_data.get('title', 'Film')
        text = f"Název: {title}\n"
        text += f"Rok: {movie_data.get('year', '?')}\n"
        text += f"Popis: {movie_data.get('overview', 'Není k dispozici.')}\n"
        xbmcgui.Dialog().textviewer(title, text)
    else:
        ui.popinfo("Nebylo možné načíst data filmu pro zobrazení detailů.", addon.getAddonInfo('name'))
    
def add_movie_to_list(params, addon):
    """Přidá film do seznamu Moje Filmy (voláno z kontextového menu nebo detailního okna)."""
    tmdb_id = params.get('tmdb_id')
    title = params.get('title') # Název pro případnou notifikaci "už existuje"
    add_to_my_movies = params.get('add_to_my_movies')
    if not tmdb_id:
        ui.popinfo("Chyba: Chybí TMDb ID pro přidání filmu.", "Moje filmy")
        return
    try:
        tmdb_id = int(tmdb_id)
    except ValueError:
        ui.popinfo(f"Chyba: Neplatné TMDb ID: {tmdb_id}", "Moje filmy")
        return
    # Zkontrolujeme, zda film již není v seznamu
    tmdb_movies = user_data.load_tmdb_movies(addon)
    if any(m['id'] == tmdb_id for m in tmdb_movies):
        title_disp = title or f"ID {tmdb_id}"
        ui.popinfo(f"Film '{title_disp}' už v seznamu existuje.", "Moje filmy")
        return
    # Získáme detaily z TMDb API
    api_params = {'api_key': get_tmdb_api_key(addon), 'language': 'cs-CZ'}
    url = f'https://api.themoviedb.org/3/movie/{tmdb_id}'
    try:
        response = requests.get(url, params=api_params)
        response.raise_for_status()
        selected_item = response.json()
    except Exception as e:
        ui.popinfo(f"Chyba při získávání detailů filmu: {e}", "Moje filmy")
        return
    # Přidáme film do seznamu na začátek
    tmdb_movies.insert(0, {
        'id': selected_item['id'],
        'title': selected_item['title'],
        'year': selected_item.get('release_date', '')[:4],
        'overview': selected_item.get('overview', ''),
        'poster_path': selected_item.get('poster_path', '')
    })
    user_data.store_tmdb_movies(addon, tmdb_movies)
    ui.popinfo(f"Film '{selected_item['title']}' byl přidán.", "Moje filmy")
    import xbmc
    
    # Refresh pouze pokud NEJSME v hledacích seznamech
    container_path = xbmc.getInfoLabel('Container.FolderPath')
    if not any(search_action in container_path for search_action in ['tmdb_add_movie', 'tmdb_add_series', 'tmdb_universal_search']):
        xbmc.executebuiltin('Container.Refresh')

# --- Funkce pro správu oblíbených herců ---

def search_actor(addon, handle):
    """Vyhledá herce podle zadaného textu."""
    # Dialog pro zadání jména herce
    keyboard = xbmc.Keyboard('', 'Zadejte jméno herce')
    keyboard.doModal()
    
    if not keyboard.isConfirmed():
        return
    
    search_text = keyboard.getText()
    if not search_text:
        return
    
    # Vyhledání herce
    api_params = {
        'api_key': get_tmdb_api_key(addon),
        'language': 'cs-CZ',
        'query': search_text,
        'page': 1,
        'include_adult': 'false'
    }
    
    try:
        response = requests.get('https://api.themoviedb.org/3/search/person', params=api_params)
        response.raise_for_status()
        data = response.json()
        
        if not data.get('results'):
            ui.popinfo(f"Nenalezen žádný herec s názvem '{search_text}'", addon.getAddonInfo('name'))
            return
        
        # Zobrazení výsledků
        results = data.get('results', [])
        
        # Dialog pro výběr herce
        actors = []
        for actor in results:
            name = actor.get('name', '')
            known_for = []
            
            # Získání známých filmů
            for item in actor.get('known_for', []):
                if item.get('media_type') == 'movie':
                    known_for.append(item.get('title', ''))
                elif item.get('media_type') == 'tv':
                    known_for.append(item.get('name', ''))
            
            known_for_text = ', '.join(known_for[:3])
            display_text = f"{name}"
            if known_for_text:
                display_text += f" ({known_for_text})"
            
            actors.append(display_text)
        
        if not actors:
            ui.popinfo(f"Nenalezen žádný herec s názvem '{search_text}'", addon.getAddonInfo('name'))
            return
        
        # Zobrazení dialogu pro výběr
        dialog = xbmcgui.Dialog()
        selected = dialog.select('Vyberte herce', actors)
        
        if selected < 0:
            return
        
        # Získání detailních informací o herci
        selected_actor = results[selected]
        add_actor_to_favorites(addon, selected_actor.get('id'), selected_actor.get('name'))
        
    except Exception as e:
        handle_tmdb_api_error(e, addon)

def add_actor_to_favorites(addon, actor_id, actor_name=None):
    """Přidá herce do seznamu oblíbených."""
    if not actor_id:
        ui.popinfo("Chybí ID herce pro přidání do oblíbených.", "Oblíbení herci", xbmcgui.NOTIFICATION_WARNING)
        return False
    favorite_actors = user_data.load_favorite_actors(addon)
    if user_data.check_actor_exists(actor_id, favorite_actors):
        ui.popinfo(f"Herec '{actor_name}' je již v seznamu oblíbených.", "Oblíbení herci", xbmcgui.NOTIFICATION_INFO)
        return False
    try:
        response = requests.get(
            f'https://api.themoviedb.org/3/person/{actor_id}',
            params={'api_key': get_tmdb_api_key(addon), 'language': 'cs-CZ'}
        )
        response.raise_for_status()
        actor_data = response.json()
        actor = {
            'id': actor_id,
            'name': actor_data.get('name', actor_name if actor_name else 'Neznámý herec'),
            'profile_path': actor_data.get('profile_path', ''),
            'biography': actor_data.get('biography', ''),
            'known_for_department': actor_data.get('known_for_department', '')
        }
        favorite_actors.append(actor)
        user_data.store_favorite_actors(addon, favorite_actors)
        ui.popinfo(f"Herec '{actor['name']}' byl přidán do oblíbených.", "Oblíbení herci", xbmcgui.NOTIFICATION_INFO)
        xbmc.executebuiltin('Container.Refresh')
        return True
    except Exception as e:
        handle_tmdb_api_error(e, addon)
        return False

def get_movie_detailed_info(movie_id):
    """Získá detailní informace o filmu z TMDb, včetně credits."""
    if not movie_id:
        xbmc.log(f"[WCS TMDB Handler] get_movie_detailed_info: Chybí movie_id.", xbmc.LOGWARNING)
        return None
    
    try:
        api_params = {
            'api_key': get_tmdb_api_key(), 
            'language': 'cs-CZ', 
            'append_to_response': 'credits,videos' # Přidáme credits a také videa (pro případný trailer)
        }
        # xbmc.log(f"[WCS TMDB Handler] Volání TMDB API: https://api.themoviedb.org/3/movie/{movie_id} s parametry: {api_params}", xbmc.LOGDEBUG)
        response = requests.get(f'https://api.themoviedb.org/3/movie/{movie_id}', params=api_params, timeout=10)
        response.raise_for_status() 
        data = response.json()
        
        director_name = ''
        writer_names_list = []
        
        credits_data = data.get('credits')
        if credits_data and 'crew' in credits_data:
            directors = [person['name'] for person in credits_data['crew'] if person.get('job') == 'Director']
            if directors:
                director_name = ", ".join(directors) 
            
            # Scenáristé - různé role
            writers_crew = [person['name'] for person in credits_data['crew'] if person.get('job') in ['Screenplay', 'Writer', 'Story', 'Teleplay']]
            # Odstranění duplicit při zachování pořadí
            seen_writers = set()
            for writer in writers_crew:
                if writer not in seen_writers:
                    writer_names_list.append(writer)
                    seen_writers.add(writer)
        
        # Formátování scenáristů
        writer_display_name = ''
        if writer_names_list:
            # Omezíme počet zobrazených scenáristů, např. na 2-3, aby to nebylo příliš dlouhé
            max_writers_to_show = 2 
            if len(writer_names_list) > max_writers_to_show:
                writer_display_name = ", ".join(writer_names_list[:max_writers_to_show]) + ", ..."
            else:
                writer_display_name = ", ".join(writer_names_list)

        # Přidáme extrahované informace do hlavního slovníku dat
        # Použijeme prefix, abychom nekolidovali s možnými existujícími klíči, 
        # nebo přímo klíče, které očekává ListItem.setInfo
        data['director'] = director_name 
        data['writer'] = writer_display_name 
        # data['_director_name'] = director_name # Alternativa s prefixem
        # data['_writer_names'] = writer_display_name

        # xbmc.log(f"[WCS TMDB Handler] get_movie_detailed_info pro ID {movie_id} vrátil: Režie='{director_name}', Scénář='{writer_display_name}'", xbmc.LOGDEBUG)
        return data
        
    except Exception as ex:
        import traceback
        xbmc.log(f"[WCS TMDB Handler] Chyba v get_movie_detailed_info pro ID {movie_id}: {ex}\n{traceback.format_exc()}", xbmc.LOGERROR)
        return None

def search_actor_movies(params, addon, handle):
    """Vyhledá filmy s vybraným hercem."""
    actor_id = params.get('actor_id')
    actor_name = params.get('actor_name', 'Neznámý herec')
    
    if not actor_id:
        ui.popinfo("Chybí ID herce pro vyhledání filmů.", addon.getAddonInfo('name'), xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle)
        return
    
    # Nastavení kategorie
    xbmcplugin.setPluginCategory(handle, f'Filmy s hercem: {actor_name}')
    xbmcplugin.setContent(handle, 'movies')
    
    try:
        # Získání čísla stránky z parametrů
        page = params.get('page', '1')
        try:
            page = int(page)
        except ValueError:
            page = 1
        
        # Získání filmů s hercem
        api_params = {
            'api_key': get_tmdb_api_key(),
            'language': 'cs-CZ',
            'append_to_response': 'credits' # Zajistí data o 'cast' a 'crew'
        }
        
        # Získání filmových kreditů pro herce
        response = requests.get(
            f'https://api.themoviedb.org/3/person/{actor_id}/movie_credits',
            params=api_params,
            timeout=10
        )
        response.raise_for_status()
        credits_data = response.json()
        
        # Získáme seznam filmů, ve kterých herec hrál ('cast')
        movies = credits_data.get('cast', [])
        
        # --- Přidáno ŘAZENÍ podle roku ---
        # Seřadíme filmy podle roku vydání sestupně (nejnovější první)
        # Použijeme 'release_date' z každého filmu v seznamu 'cast'
        movies.sort(key=utils.get_year_movie, reverse=True)
        
        # --- Manuální stránkování ---
        items_per_page = 20 # Můžeme nastavit podle potřeby nebo preference
        total_items = len(movies)
        total_pages = (total_items + items_per_page - 1) // items_per_page
        start_index = (page - 1) * items_per_page
        end_index = start_index + items_per_page
        paginated_movies = movies[start_index:end_index]

        context_params = {
            'context_type': 'person', 
            'context_value': actor_name, 
            'person_type': 'actor',
            'media_type': 'movie'
        }
        ui.add_ai_recommend_button(handle, addon, 'ai_chat_recommendation', label_text="Poradit se s AI", context_params=context_params)

        if not paginated_movies and page == 1:
            ui.popinfo(f"Nebyly nalezeny žádné filmy pro herce {actor_name}.", addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
            return

        # Zpracování stránkovaných výsledků
        for movie in paginated_movies:
            movie_id = movie.get('id')
            if not movie_id: continue # Přeskočíme, pokud chybí ID

            title = movie.get('title', 'Neznámý titul')
            year_str = movie.get('release_date', '')[:4]
            year = int(year_str) if year_str.isdigit() else 0
            overview = movie.get('overview', '')
            poster_path = movie.get('poster_path', '')
            backdrop_path = movie.get('backdrop_path', '') # Získáme i backdrop

            # Zkusíme získat detailnější data, pokud chybí (např. backdrop z hlavní role)
            # Můžeme optimalizovat voláním detailů jen v případě potřeby
            # backdrop_path = movie.get('backdrop_path') # Už máme
            
            poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
            fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}" if backdrop_path else poster # Fanart z backdrop, fallback na poster

            # Vytvoření položky
            label = f"{title} ({year})" if year else title
            listitem = xbmcgui.ListItem(label=label)
            
            # Nastavení obrázků
            art = {'thumb': poster, 'poster': poster, 'icon': poster} # Přidáme i poster a icon
            if fanart:
                art['fanart'] = fanart
            listitem.setArt(art)
            
            utils.set_video_info_tag(listitem, {
                'title': title,
                'year': year,
                'plot': overview,
                'mediatype': 'movie' # Důležité pro Kodi
            })
            
            # Kontextové menu
            commands = create_movie_context_menu(addon, movie_id, title, year, overview, poster)
            listitem.addContextMenuItems(commands)
            
            
            # Odkaz na detailní okno
            # Podle nastavení určíme, jakou akci provést při kliknutí na film
            default_action = addon.getSetting('default_movie_action')
            
            if default_action == '1':  # 40004 - Přehrát ihned
                play_url = f"plugin://{addon.getAddonInfo('id')}?action=play_movie_from_addon&title={quote_plus(title)}&year={quote_plus(str(year))}&plot={quote_plus(overview)}&poster={quote_plus(poster)}&tmdb_id={movie_id}"
                xbmcplugin.addDirectoryItem(handle, play_url, listitem, False)
            else:
                details_url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={movie_id}"
                xbmcplugin.addDirectoryItem(handle, details_url, listitem, False)
        
        # Přidání položky "Další stránka" - bude zobrazena jako poslední položka
        if page < total_pages:
            next_page_params = params.copy()
            next_page_params['page'] = str(page + 1)
            # Sestavení URL pro další stránku se všemi relevantními parametry
            next_url = f"plugin://{addon.getAddonInfo('id')}?action=tmdb_search_actor_movies"
            for key, value in next_page_params.items():
                if key != 'action': # Akci už máme
                    next_url += f"&{key}={quote_plus(str(value))}" 
            
            listitem = xbmcgui.ListItem(label=f"Další stránka ({page + 1}/{total_pages})")
            # Upravený způsob nastavení ikony, který by měl být konzistentní s ostatními funkcemi
            icon_path = 'DefaultFolderNext.png'
            listitem.setArt({'thumb': icon_path, 'icon': icon_path})
            xbmcplugin.addDirectoryItem(handle, next_url, listitem, True)
        
        xbmcplugin.endOfDirectory(handle)
        
    except Exception as e:
        handle_tmdb_api_error(e, addon, handle)

def search_director(addon, handle):
    """Vyhledá režiséry podle zadaného textu."""
    # Dialog pro zadání jména režiséra
    keyboard = xbmc.Keyboard('', 'Zadejte jméno režiséra')
    keyboard.doModal()
    
    if not keyboard.isConfirmed():
        return
    
    search_text = keyboard.getText()
    if not search_text:
        return
    
    # Vyhledání režiséra
    api_params = {
        'api_key': get_tmdb_api_key(),
        'language': 'cs-CZ',
        'query': search_text,
        'page': 1,
        'include_adult': 'false'
    }
    
    try:
        response = requests.get('https://api.themoviedb.org/3/search/person', params=api_params)
        response.raise_for_status()
        data = response.json()
        
        if not data.get('results'):
            ui.popinfo(f"Nenalezen žádný režisér s názvem '{search_text}'", addon.getAddonInfo('name'))
            return
        
        # Filtrování výsledků - jen lidé, kteří režírovali alespoň jeden film
        results = []
        for person in data.get('results', []):
            if person.get('known_for_department') == 'Directing':
                results.append(person)
            else:
                # Zkusíme zjistit, jestli osoba režírovala nějaké filmy
                try:
                    credits_response = requests.get(
                        f'https://api.themoviedb.org/3/person/{person["id"]}/movie_credits',
                        params={'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
                    )
                    credits_response.raise_for_status()
                    credits_data = credits_response.json()
                    
                    # Ověříme, zda osoba režírovala nějaké filmy
                    for movie in credits_data.get('crew', []):
                        if movie.get('job') == 'Director':
                            results.append(person)
                            break
                except:
                    pass
        
        if not results:
            ui.popinfo(f"Nenalezen žádný režisér s názvem '{search_text}'", addon.getAddonInfo('name'))
            return
        
        # Zobrazení výsledků
        # Dialog pro výběr režiséra
        directors = []
        for director in results:
            name = director.get('name', '')
            known_for = []
            
            # Získání známých filmů
            for item in director.get('known_for', []):
                if item.get('media_type') == 'movie':
                    known_for.append(item.get('title', ''))
                elif item.get('media_type') == 'tv':
                    known_for.append(item.get('name', ''))
            
            known_for_text = ', '.join(known_for[:3])
            display_text = f"{name}"
            if known_for_text:
                display_text += f" ({known_for_text})"
            
            directors.append(display_text)
        
        # Zobrazení dialogu pro výběr
        dialog = xbmcgui.Dialog()
        selected = dialog.select('Vyberte režiséra', directors)
        
        if selected < 0:
            return
        
        # Získání detailních informací o režiséru
        selected_director = results[selected]
        add_director_to_favorites(addon, selected_director.get('id'), selected_director.get('name'))
        
    except Exception as e:
        handle_tmdb_api_error(e, addon)

def add_director_to_favorites(addon, director_id, director_name=None):
    """Přidá režiséra do seznamu oblíbených."""
    if not director_id:
        ui.popinfo("Chybí ID režiséra pro přidání do oblíbených.", "Oblíbení režiséři", xbmcgui.NOTIFICATION_WARNING)
        return False
    favorite_directors = user_data.load_favorite_directors(addon)
    if user_data.check_director_exists(director_id, favorite_directors):
        ui.popinfo(f"Režisér '{director_name}' je již v seznamu oblíbených.", "Oblíbení režiséři", xbmcgui.NOTIFICATION_INFO)
        return False
    try:
        response = requests.get(
            f'https://api.themoviedb.org/3/person/{director_id}',
            params={'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        )
        response.raise_for_status()
        director_data = response.json()
        director = {
            'id': director_id,
            'name': director_data.get('name', director_name if director_name else 'Neznámý režisér'),
            'profile_path': director_data.get('profile_path', ''),
            'biography': director_data.get('biography', ''),
            'known_for_department': director_data.get('known_for_department', '')
        }
        favorite_directors.append(director)
        user_data.store_favorite_directors(addon, favorite_directors)
        ui.popinfo(f"Režisér '{director['name']}' byl přidán do oblíbených.", "Oblíbení režiséři", xbmcgui.NOTIFICATION_INFO)
        xbmc.executebuiltin('Container.Refresh')
        return True
    except Exception as e:
        handle_tmdb_api_error(e, addon)
        return False

def search_director_movies(params, addon, handle):
    """Vyhledá filmy s vybraným režisérem."""
    director_id = params.get('director_id')
    director_name = params.get('director_name', 'Neznámý režisér')
    
    if not director_id:
        ui.popinfo("Chybí ID režiséra pro vyhledání filmů.", addon.getAddonInfo('name'), xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle)
        return
    
    # Nastavení kategorie
    xbmcplugin.setPluginCategory(handle, f'Filmy s režisérem: {director_name}')
    xbmcplugin.setContent(handle, 'movies')
    
    try:
        # Získání čísla stránky z parametrů
        page = params.get('page', '1')
        try:
            page = int(page)
        except ValueError:
            page = 1
        
        # Získání všech filmů, na kterých režisér spolupracoval (použití nové funkce)
        all_movies = get_creator_movies(director_id)
        
        # Seřazení podle názvu (abecedně) - ZMĚNA NA ŘAZENÍ PODLE ROKU
        # all_movies.sort(key=lambda x: x.get('title', '').lower())
        all_movies.sort(key=utils.get_year_movie, reverse=True) # Řadíme sestupně podle roku
        
        if not all_movies:
            ui.popinfo(f"Nenalezeny žádné filmy s režisérem {director_name}.", addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
            return
        
        # Implementace stránkování (20 položek na stránku)
        items_per_page = 20
        total_items = len(all_movies)
        total_pages = (total_items + items_per_page - 1) // items_per_page  # Zaokrouhlení nahoru
        
        # Získání filmů pro aktuální stránku
        start_index = (page - 1) * items_per_page
        end_index = min(start_index + items_per_page, total_items)
        current_page_movies = all_movies[start_index:end_index]

        context_params = {
            'context_type': 'person', 
            'context_value': director_name, 
            'person_type': 'director',
            'media_type': 'movie'
        }
        ui.add_ai_recommend_button(handle, addon, 'ai_chat_recommendation', label_text="Poradit se s AI", context_params=context_params)
        
        for movie in current_page_movies:
            movie_title = movie.get('title', 'Neznámý film')
            movie_id = movie.get('id', '')
            poster_path = movie.get('poster_path', '')
            release_date = movie.get('release_date', '')
            vote_average = movie.get('vote_average', 0)
            overview = movie.get('overview', '') # Přidáno pro kontextové menu
            
            # Získání roku z release_date
            year = release_date[:4] if release_date else ""
            
            # Vytvoření ListItem - PŘIDÁNÍ ROKU DO LABELU
            label = f"{movie_title} ({year})" if year else movie_title
            listitem = xbmcgui.ListItem(label=label)
            
            # Nastavení obrázku
            if poster_path:
                poster_url = f"https://image.tmdb.org/t/p/w500{poster_path}"
            else:
                poster_url = "DefaultVideo.png"
            
            listitem.setArt({'thumb': poster_url, 'icon': poster_url, 'poster': poster_url})
            
            # Nastavení dalších informací
            utils.set_video_info_tag(listitem, {
                'title': movie_title,
                'year': year,
                'rating': vote_average,
                'director': director_name,
                'plot': overview, # Přidáno pro úplnost
                'mediatype': 'movie' # Přidáno pro konzistenci
            })
            
            # Přidání kontextového menu
            commands = create_movie_context_menu(addon, movie_id, movie_title, year, overview, poster_url)
            listitem.addContextMenuItems(commands)
            
            # Vyhledávání filmu na Webshare po kliknutí
            # Podle nastavení určíme, jakou akci provést při kliknutí na film
            default_action = addon.getSetting('default_movie_action')
            
            if default_action == '1':  # 40004 - Přehrát ihned
                play_url = f"plugin://{addon.getAddonInfo('id')}?action=play_movie_from_addon&title={quote_plus(movie_title)}&year={quote_plus(str(year))}&plot={quote_plus(overview)}&poster={quote_plus(poster_url)}&tmdb_id={movie_id}"
                xbmcplugin.addDirectoryItem(handle, play_url, listitem, False)
            else:
                details_url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={movie_id}"
                xbmcplugin.addDirectoryItem(handle, details_url, listitem, False)
        
        # Přidání položky "Další stránka" - bude zobrazena jako poslední položka
        if page < total_pages:
            next_page_params = params.copy()
            next_page_params['page'] = str(page + 1)
            # Sestavení URL pro další stránku se všemi relevantními parametry
            next_url = f"plugin://{addon.getAddonInfo('id')}?action=tmdb_search_director_movies"
            for key, value in next_page_params.items():
                if key != 'action': # Akci už máme
                    next_url += f"&{key}={quote_plus(str(value))}"
            listitem = xbmcgui.ListItem(label=f"Další stránka ({page + 1}/{total_pages})")
            # Upravený způsob nastavení ikony
            icon_path = 'DefaultFolderNext.png'
            listitem.setArt({'thumb': icon_path, 'icon': icon_path})
            xbmcplugin.addDirectoryItem(handle, next_url, listitem, True)
        
        xbmcplugin.endOfDirectory(handle)
        
        
    except Exception as e:
        handle_tmdb_api_error(e, addon, handle)



# --- Funkce pro práci s oblíbenými scénáristy --- 

def search_writer(addon, handle):
    """Vyhledá scénáristy podle zadaného textu."""
    keyboard = xbmc.Keyboard('', 'Zadejte jméno scénáristy')
    keyboard.doModal()
    
    if not keyboard.isConfirmed():
        return
    
    search_text = keyboard.getText()
    if not search_text:
        return
    
    api_params = {
        'api_key': get_tmdb_api_key(),
        'language': 'cs-CZ',
        'query': search_text,
        'page': 1,
        'include_adult': 'false'
    }
    
    try:
        response = requests.get('https://api.themoviedb.org/3/search/person', params=api_params)
        response.raise_for_status()
        data = response.json()
        
        if not data.get('results'):
            ui.popinfo(f"Nenalezen žádný scénárista s názvem '{search_text}'", addon.getAddonInfo('name'))
            return
        
        # Filtrování výsledků - jen lidé, kteří psali scénář
        results = []
        for person in data.get('results', []):
            # Primární kontrola podle oddělení
            if person.get('known_for_department') == 'Writing':
                results.append(person)
                continue
            # Sekundární kontrola kreditů (může být pomalejší)
            try:
                credits_response = requests.get(
                    f'https://api.themoviedb.org/3/person/{person["id"]}/movie_credits',
                    params={'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'},
                    timeout=2 # Krátký timeout
                )
                credits_response.raise_for_status()
                credits_data = credits_response.json()
                # Ověříme, zda osoba psala scénář
                for movie in credits_data.get('crew', []):
                    if movie.get('department') == 'Writing' and movie.get('job') in ['Writer', 'Screenplay', 'Story']:
                        if person not in results: # Přidáme jen pokud tam ještě není
                             results.append(person)
                        break # Stačí najít jeden film
            except: # Ignorujeme chyby při dotahování kreditů pro jednotlivé osoby
                pass
        
        if not results:
            ui.popinfo(f"Nenalezen žádný scénárista s názvem '{search_text}'", addon.getAddonInfo('name'))
            return
        
        # Zobrazení výsledků
        writers_display = []
        for writer in results:
            name = writer.get('name', '')
            known_for_text = ", ".join([item.get('title') or item.get('name') for item in writer.get('known_for', []) if item.get('media_type') in ['movie', 'tv']])
            display_text = f"{name}{f' ({known_for_text})' if known_for_text else ''}"
            writers_display.append(display_text)
        
        dialog = xbmcgui.Dialog()
        selected = dialog.select('Vyberte scénáristu', writers_display)
        
        if selected < 0:
            return
        
        selected_writer = results[selected]
        add_writer_to_favorites(addon, selected_writer.get('id'), selected_writer.get('name'))
        
    except Exception as e:
        handle_tmdb_api_error(e, addon)

def add_writer_to_favorites(addon, writer_id, writer_name=None):
    """Přidá scénáristu do seznamu oblíbených."""
    if not writer_id:
        ui.popinfo("Chybí ID scénáristy pro přidání do oblíbených.", "Oblíbení scénáristé", xbmcgui.NOTIFICATION_WARNING)
        return False
    favorite_writers = user_data.load_favorite_writers(addon)
    if user_data.check_writer_exists(writer_id, favorite_writers):
        ui.popinfo(f"Scenárista '{writer_name}' je již v seznamu oblíbených.", "Oblíbení scénáristé", xbmcgui.NOTIFICATION_INFO)
        return False
    try:
        response = requests.get(
            f'https://api.themoviedb.org/3/person/{writer_id}',
            params={'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        )
        response.raise_for_status()
        writer_data = response.json()
        writer = {
            'id': writer_id,
            'name': writer_data.get('name', writer_name if writer_name else 'Neznámý scénárista'),
            'profile_path': writer_data.get('profile_path', ''),
            'biography': writer_data.get('biography', ''),
            'known_for_department': writer_data.get('known_for_department', '')
        }
        favorite_writers.append(writer)
        user_data.store_favorite_writers(addon, favorite_writers)
        ui.popinfo(f"Scenárista '{writer['name']}' byl přidán do oblíbených.", "Oblíbení scénáristé", xbmcgui.NOTIFICATION_INFO)
        xbmc.executebuiltin('Container.Refresh')
        return True
    except Exception as e:
        handle_tmdb_api_error(e, addon)
        return False

def search_writer_movies(params, addon, handle):
    """Vyhledá filmy, na kterých se podílel vybraný scénárista."""
    writer_id = params.get('writer_id')
    writer_name = params.get('writer_name', 'Neznámý scénárista')
    
    if not writer_id:
        ui.popinfo("Chybí ID scénáristy pro vyhledání filmů.", addon.getAddonInfo('name'), xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(handle)
        return
    
    xbmcplugin.setPluginCategory(handle, f'Filmy scénáristy: {writer_name}')
    xbmcplugin.setContent(handle, 'movies')
    
    try:
        page = int(params.get('page', '1'))
        api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        
        response = requests.get(f'https://api.themoviedb.org/3/person/{writer_id}/movie_credits', params=api_params, timeout=10)
        response.raise_for_status()
        data = response.json()
        
        # Získání filmů, kde osoba byla ve 'crew' a job == 'Writer'/'Screenplay'/'Story'
        # Použijeme slovník pro odstranění duplicit podle movie_id
        movies_dict = {}
        crew = data.get('crew', [])
        for movie in crew:
            if movie.get('department') == 'Writing' and movie.get('job') in ['Writer', 'Screenplay', 'Story']:
                movie_id = movie.get('id')
                if movie_id and movie_id not in movies_dict:
                    movies_dict[movie_id] = movie # Uložíme celý objekt filmu
        
        all_movies = list(movies_dict.values())
        
        # Řazení podle roku vydání sestupně
        all_movies.sort(key=utils.get_year_movie, reverse=True)
        
        # Stránkování
        items_per_page = 20
        total_items = len(all_movies)
        total_pages = (total_items + items_per_page - 1) // items_per_page
        start_index = (page - 1) * items_per_page
        end_index = start_index + items_per_page
        paginated_movies = all_movies[start_index:end_index]

        # Stránkování
        items_per_page = 20
        total_items = len(all_movies)
        total_pages = (total_items + items_per_page - 1) // items_per_page
        start_index = (page - 1) * items_per_page
        end_index = start_index + items_per_page
        paginated_movies = all_movies[start_index:end_index]

        context_params = {
            'context_type': 'person', 
            'context_value': writer_name, 
            'person_type': 'writer',
            'media_type': 'movie'
        }
        ui.add_ai_recommend_button(handle, addon, 'ai_chat_recommendation', label_text="Poradit se s AI", context_params=context_params)

        if not paginated_movies and page == 1:
            ui.popinfo(f"Nebyly nalezeny žádné filmy pro scénáristu {writer_name}.", addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
            return

        for movie in paginated_movies:
            movie_id = movie.get('id')
            if not movie_id: continue

            title = movie.get('title', 'Neznámý titul')
            year_str = movie.get('release_date', '')[:4]
            year = int(year_str) if year_str.isdigit() else 0
            overview = movie.get('overview', '')
            poster_path = movie.get('poster_path', '')
            backdrop_path = movie.get('backdrop_path', '')
            
            poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
            fanart = f"https://image.tmdb.org/t/p/original{backdrop_path}" if backdrop_path else poster

            label = f"{title} ({year})" if year else title
            listitem = xbmcgui.ListItem(label=label)
            
            art = {'thumb': poster, 'poster': poster, 'icon': poster}
            if fanart:
                art['fanart'] = fanart
            listitem.setArt(art)
            
            utils.set_video_info_tag(listitem, {
                'title': title,
                'year': year,
                'plot': overview,
                'writer': writer_name, # Přidáme jméno scénáristy
                'mediatype': 'movie'
            })
            
            commands = create_movie_context_menu(addon, movie_id, title, year, overview, poster)
            listitem.addContextMenuItems(commands)

            # Podle nastavení určíme, jakou akci provést při kliknutí na film
            default_action = addon.getSetting('default_movie_action')
            
            if default_action == '1':  # 40004 - Přehrát ihned
                play_url = f"plugin://{addon.getAddonInfo('id')}?action=play_movie_from_addon&title={quote_plus(title)}&year={quote_plus(str(year))}&plot={quote_plus(overview)}&poster={quote_plus(poster)}&tmdb_id={movie_id}"
                xbmcplugin.addDirectoryItem(handle, play_url, listitem, False)
            else:
                details_url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={movie_id}"
                xbmcplugin.addDirectoryItem(handle, details_url, listitem, False)
        
        if page < total_pages:
            next_page_params = params.copy()
            next_page_params['page'] = str(page + 1)
            next_url = f"plugin://{addon.getAddonInfo('id')}?action=tmdb_search_writer_movies"
            for key, value in next_page_params.items():
                 if key != 'action':
                      next_url += f"&{key}={quote_plus(str(value))}"
            listitem = xbmcgui.ListItem(label=f"Další stránka ({page + 1}/{total_pages})")
            icon_path = 'DefaultFolderNext.png'
            listitem.setArt({'thumb': icon_path, 'icon': icon_path})
            xbmcplugin.addDirectoryItem(handle, next_url, listitem, True)
        
        xbmcplugin.endOfDirectory(handle)
        
    except Exception as e:
        handle_tmdb_api_error(e, addon, handle)

def display_writer_dialog(params, addon):
    """Zobrazí dialog s akcemi pro scénáristu a provede vybranou akci."""
    writer_id = params.get('writer_id')
    full_writer_text = params.get('writer_name', '')
    
    if not writer_id or not full_writer_text:
        ui.popinfo("Chyba: Informace o scénáristovi nejsou k dispozici.", addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
        return

    # Extrahujeme čisté jméno scenáristy (předpokládáme, že zde není prefix)
    writer_name = full_writer_text # Může obsahovat více jmen, použijeme celé
    
    dialog = xbmcgui.Dialog()
    # TODO: Použít lokalizované stringy pro dialog
    options = ["Přidat do oblíbených scénáristů", "Zobrazit filmy se scénáristou", "Zrušit"]
    choice = dialog.select(f"Akce pro scénáristu: {writer_name}", options)
    
    if choice == 0:  # Přidat do oblíbených
        xbmc.log(f"[plugin.video.milionar] Dialog choice -> Adding writer to favorites: {writer_name} (ID: {writer_id})", level=xbmc.LOGINFO)
        add_writer_to_favorites(addon, writer_id, writer_name)
    
    elif choice == 1:  # Zobrazit filmy se scénáristou
        xbmc.log(f"[plugin.video.milionar] Dialog choice -> Showing movies with writer: {writer_name} (ID: {writer_id})", level=xbmc.LOGINFO)
        url = f"plugin://{addon.getAddonInfo('id')}?action=tmdb_search_writer_movies&writer_id={writer_id}&writer_name={quote_plus(writer_name)}"
        xbmc.executebuiltin('Dialog.Close(all, true)') 
        xbmc.sleep(100)
        xbmc.executebuiltin(f"Notification(Otevírám..., Filmy se scénáristou: {writer_name})")
        xbmc.executebuiltin(f"Container.Update({url})")


# --- Ostatní funkce (např. detail seriálu, kombinovaný výběr atd.) ---

# --- Nové funkce pro práci s kolekcemi --- 

def get_collection_details(collection_id):
    """Získá detaily kolekce z TMDb API."""
    try:
        response = requests.get(
            f'https://api.themoviedb.org/3/collection/{collection_id}',
            params={
                'api_key': get_tmdb_api_key(),
                'language': 'cs-CZ'
            },
            timeout=10
        )
        response.raise_for_status()
        return response.json()
    except Exception as e:
        xbmc.log(f"Chyba při načítání detailů kolekce {collection_id}: {e}", xbmc.LOGERROR)
        return None

def display_movie_collection(tmdb_id, addon, handle):
    """Zobrazí seznam filmů patřících do stejné kolekce jako daný film."""
    movie_details = get_movie_detailed_info(tmdb_id)
    if not movie_details or 'belongs_to_collection' not in movie_details or not movie_details['belongs_to_collection']:
        ui.popinfo("Film nepatří do žádné kolekce nebo se nepodařilo načíst detaily.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return

    collection_info = movie_details['belongs_to_collection']
    collection_id = collection_info.get('id')
    collection_name = collection_info.get('name', 'Neznámá kolekce')

    if not collection_id:
        ui.popinfo("Nepodařilo se získat ID kolekce.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
        
    xbmcplugin.setPluginCategory(handle, f"Kolekce: {collection_name}")
    xbmcplugin.setContent(handle, 'movies')

    collection_details = get_collection_details(collection_id)
    if not collection_details or 'parts' not in collection_details:
        ui.popinfo(f"Nepodařilo se načíst filmy v kolekci '{collection_name}'.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
        
    movies_in_collection = collection_details['parts']
    # Seřadíme filmy v kolekci podle data vydání (pokud je dostupné)
    movies_in_collection.sort(key=utils.get_year_movie, reverse=False) # Od nejstaršího

    for movie in movies_in_collection:
        movie_id_coll = movie['id']
        title = movie['title']
        year_val = movie.get('release_date', '')[:4]
        plot = movie.get('overview', '')
        poster_path = movie.get('poster_path', '')
        poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
        label = f"{title} ({year_val})" if year_val else title
        
        try: info_year = int(year_val) 
        except: info_year = 0

        listitem = xbmcgui.ListItem(label=label)
        if poster:
             listitem.setArt({'thumb': poster, 'poster': poster, 'icon': poster})
        utils.set_video_info_tag(listitem, {'title': label, 'plot': plot, 'year': info_year})
            
        commands = create_movie_context_menu(addon, movie_id_coll, title, info_year, plot, poster)
        listitem.addContextMenuItems(commands)
            
        # Podle nastavení určíme, jakou akci provést při kliknutí na film
        default_action = addon.getSetting('default_movie_action')
        
        if default_action == '1':  # 40004 - Přehrát ihned
            play_url = f"plugin://{addon.getAddonInfo('id')}?action=play_movie_from_addon&title={quote_plus(title)}&year={quote_plus(year_val)}&plot={quote_plus(plot)}&poster={quote_plus(poster)}&tmdb_id={movie_id_coll}"
            xbmcplugin.addDirectoryItem(handle, play_url, listitem, False)
        else:
            details_url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={movie_id_coll}"
            xbmcplugin.addDirectoryItem(handle, details_url, listitem, False)

    xbmcplugin.endOfDirectory(handle)

# --- Funkce pro práci s kolekcemi ---

def display_collection_dialog(params, addon):
    """Zobrazí dialog s možnostmi pro filmovou kolekci."""
    collection_id = params.get('collection_id')
    collection_name = params.get('collection_name', 'Tato kolekce')

    if not collection_id:
        ui.popinfo("Chybí ID kolekce.", addon.getAddonInfo('name'))
        return

    dialog = xbmcgui.Dialog()
    options = [
        f"Přidat kolekci '{collection_name}' do Moje filmy",
        f"Zobrazit filmy v kolekci '{collection_name}'",
        "Zrušit"
    ]
    choice = dialog.select(f"Možnosti pro kolekci", options)

    if choice == 0: # Přidat kolekci do Moje filmy
        _add_collection_movies_to_list(collection_id, addon)
    elif choice == 1: # Zobrazit filmy v kolekci
        # Použijeme Container.Update pro zobrazení seznamu filmů v aktuálním okně
        url = f"plugin://{addon.getAddonInfo('id')}?action=list_collection_movies&collection_id={collection_id}&collection_name={quote_plus(collection_name)}"
        # Zavřeme dialog PŘED aktualizací kontejneru
        xbmc.executebuiltin('Dialog.Close(all, true)') 
        xbmc.sleep(100) # Krátká pauza pro jistotu
        xbmc.executebuiltin(f"Container.Update({url})")
    # choice == 2 nebo -1 (Zrušit) nedělá nic

def _add_collection_movies_to_list(collection_id, addon):
    """Načte detaily kolekce a přidá všechny její filmy do 'Moje filmy'."""
    xbmc.executebuiltin("ActivateWindow(busydialognocancel)")
    added_count = 0
    skipped_count = 0
    try:
        collection_details = get_collection_details(collection_id)
        if collection_details and 'parts' in collection_details:
            movie_ids_to_add = [part['id'] for part in collection_details['parts'] if 'id' in part]
            
            if not movie_ids_to_add:
                 ui.popinfo("Kolekce neobsahuje žádné filmy.", addon.getAddonInfo('name'))
                 xbmc.executebuiltin("Dialog.Close(busydialognocancel)")
                 return

            # Načteme existující filmy, abychom předešli duplicitám
            existing_movie_ids = {movie['id'] for movie in user_data.load_tmdb_movies(addon)}

            for movie_id in movie_ids_to_add:
                if movie_id not in existing_movie_ids:
                    # Použijeme interní funkci pro přidání, která již obsahuje logiku
                    if add_movie_item_to_list(addon, movie_id):
                        added_count += 1
                    else:
                        # Chyba při přidávání konkrétního filmu (např. chyba API)
                        xbmc.log(f"[plugin.video.milionar] Chyba při přidávání filmu ID {movie_id} z kolekce.", level=xbmc.LOGWARNING)
                else:
                    skipped_count += 1
            
            if added_count > 0:
                 ui.popinfo(f"Úspěšně přidáno {added_count} filmů z kolekce. Přeskočeno {skipped_count} duplicitních.", addon.getAddonInfo('name'))
                 # Obnovíme seznam "Moje filmy", pokud jsme v něm
                 if "action=tmdb_added_movies" in xbmc.getInfoLabel('Container.FolderPath'):
                     xbmc.executebuiltin("Container.Refresh")
            elif skipped_count > 0:
                 ui.popinfo(f"Žádné nové filmy nebyly přidány. Všechny filmy z kolekce již jsou v seznamu ({skipped_count}).", addon.getAddonInfo('name'))
            else:
                 ui.popinfo("Nepodařilo se přidat žádné filmy z kolekce.", addon.getAddonInfo('name'), icon=xbmcgui.NOTIFICATION_WARNING)
        else:
            ui.popinfo("Nepodařilo se načíst detaily kolekce.", addon.getAddonInfo('name'))
    except Exception as e:
        ui.popinfo(f"Chyba při přidávání kolekce: {e}", addon.getAddonInfo('name'))
        xbmc.log(f"[plugin.video.milionar] Error adding collection {collection_id}: {e}", level=xbmc.LOGERROR)
        import traceback
        traceback.print_exc()
    finally:
        xbmc.executebuiltin("Dialog.Close(busydialognocancel)")


def list_collection_movies(params, addon, handle):
    """Získá detaily kolekce a předá je do UI vrstvy pro zobrazení filmů v kolekci."""
    collection_id = params.get('collection_id')
    collection_name = params.get('collection_name', 'Filmy v kolekci')
    if not collection_id:
        ui.popinfo("Chybí ID kolekce pro zobrazení filmů.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return
    try:
        collection_details = get_collection_details(collection_id)
        if collection_details and 'parts' in collection_details:
            movies = collection_details['parts']
            ui.display_collection_movies(addon, handle, collection_name, movies)
        else:
            ui.popinfo(f"Nepodařilo se načíst detaily kolekce.", addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
    except Exception as e:
        ui.popinfo(f"Nastala chyba při zobrazování kolekce: {e}", addon.getAddonInfo('name'))
        import traceback
        traceback.print_exc()
        xbmcplugin.endOfDirectory(handle)

def add_collection_movies_to_list(collection_id, addon):
    """Přidá všechny filmy z kolekce do seznamu 'Moje filmy'."""
    try:
        if not collection_id:
            ui.popinfo("Chybí ID kolekce.", addon.getAddonInfo('name'))
            return

        collection_details = get_collection_details(collection_id)
        if not collection_details or 'parts' not in collection_details:
             ui.popinfo("Nepodařilo se načíst filmy z kolekce.", addon.getAddonInfo('name'))
             return

        parts = collection_details['parts']
        count = 0
        
        # Progress dialog
        p_dialog = xbmcgui.DialogProgress()
        p_dialog.create(addon.getAddonInfo('name'), "Přidávám filmy do Moje filmy...")

        # Načteme existující seznam pro kontrolu duplicit a přidání
        from wcs import user_data
        current_movies = user_data.load_tmdb_movies(addon)
        existing_ids = set(str(m.get('id')) for m in current_movies)
        added_count = 0

        for i, movie in enumerate(parts):
            if p_dialog.iscanceled():
                break
            
            mid = str(movie.get('id'))
            title = movie.get('title', 'Neznámý film')
            p_dialog.update(int((i / len(parts)) * 100), f"Zpracovávám: {title}")
            
            if mid and mid not in existing_ids:
                poster_path = movie.get('poster_path')
                backdrop_path = movie.get('backdrop_path')
                release_date = movie.get('release_date', '')
                year = release_date[:4] if release_date else ''
                overview = movie.get('overview', '')
                
                movie_data = {
                    'id': mid,
                    'title': title, # Sjednoceno s UI.py (očekává 'title')
                    'name': title,  # Pro jistotu zachováme i 'name'
                    'year': year,
                    'overview': overview,
                    'poster_path': poster_path,
                    'backdrop_path': backdrop_path,
                    'added_at': datetime.datetime.now().isoformat()
                }
                current_movies.append(movie_data)
                existing_ids.add(mid)
                added_count += 1
                count += 1
                
        # Uložíme aktualizovaný seznam najednou
        if added_count > 0:
            user_data.store_tmdb_movies(addon, current_movies)
                
        p_dialog.close()
        ui.popinfo(f"Přidáno {count} filmů.", addon.getAddonInfo('name'))
        
    except Exception as e:
        xbmc.log(f"Error adding collection to My Movies: {e}", xbmc.LOGERROR)
        ui.popinfo(f"Chyba při přidávání kolekce: {e}", addon.getAddonInfo('name'))

# Stávající funkce pro získání detailů kolekce (pokud již existuje, jinak ji přidej)
# Předpokládáme, že funkce get_collection_details již existuje a vrací slovník s detaily
# Pokud neexistuje, je třeba ji implementovat - např.:
# def get_collection_details(collection_id):
#     try:
#         api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
#         response = requests.get(f'https://api.themoviedb.org/3/collection/{collection_id}', params=api_params, timeout=10)
#         response.raise_for_status()
#         return response.json()
#     except requests.exceptions.RequestException as e:
#         xbmc.log(f"[plugin.video.milionar] Error fetching collection details for ID {collection_id}: {e}", level=xbmc.LOGERROR)
#         return None
#     except Exception as e:
#         xbmc.log(f"[plugin.video.milionar] General error fetching collection details for ID {collection_id}: {e}", level=xbmc.LOGERROR)
#         import traceback
#         traceback.print_exc()
#         return None

# --- NOVÁ FUNKCE PRO ZOBRAZENÍ KOLEKCÍ UŽIVATELE ---
def display_my_movie_collections(addon, handle):
    """Předá kolekce do UI vrstvy pro zobrazení seznamu kolekcí uživatele."""
    tmdb_movies = user_data.load_tmdb_movies(addon)
    if not tmdb_movies:
        ui.display_my_movie_collections(addon, handle, {})
        return
    collections = {}
    try:
        for movie in tmdb_movies:
            movie_id = movie.get('id')
            if not movie_id:
                continue
            movie_details = get_movie_detailed_info(movie_id)
            if movie_details and movie_details.get('belongs_to_collection'):
                collection_info = movie_details['belongs_to_collection']
                if collection_info and collection_info.get('id'):
                    coll_id = collection_info['id']
                    if coll_id not in collections:
                        collections[coll_id] = {
                            'name': collection_info.get('name', 'Neznámá kolekce'),
                            'poster_path': collection_info.get('poster_path', ''),
                            'backdrop_path': collection_info.get('backdrop_path', '')
                        }
        ui.display_my_movie_collections(addon, handle, collections)
    except Exception as e:
        ui.popinfo(f"Nastala chyba při načítání kolekcí: {e}", addon.getAddonInfo('name'))
        import traceback
        traceback.print_exc()
        xbmcplugin.endOfDirectory(handle)

# --- Funkce pro zobrazení podobných filmů ---
def display_similar_movies(params, addon, handle):
    """Zobrazí seznam filmů podobných danému filmu z TMDb."""
    tmdb_id = params.get('tmdb_id')
    original_title = params.get('original_title', 'Vybraný film') # Výchozí, pokud název není předán

    if not tmdb_id:
        ui.popinfo("Chybí ID filmu pro zobrazení podobných.", addon.getAddonInfo('name'))
        xbmcplugin.endOfDirectory(handle)
        return

    category_name = addon.getLocalizedString(30183).replace('%s', str(original_title)) # <-- ZMĚNA NA NOVÉ ID
    xbmcplugin.setPluginCategory(handle, category_name)
    xbmcplugin.setContent(handle, 'movies')

    api_params = {
        'api_key': get_tmdb_api_key(),
        'language': 'cs-CZ',
        'page': params.get('page', '1') # Pro případné budoucí stránkování podobných filmů
    }
    
    try:
        url = f'https://api.themoviedb.org/3/movie/{tmdb_id}/similar'
        response = requests.get(url, params=api_params)
        response.raise_for_status()
        data = response.json()
        results = data.get('results', [])
        current_page = data.get('page', 1)
        total_pages = data.get('total_pages', 1) # TMDB může stránkovat i similar

        if not results and current_page == 1:
            ui.popinfo(f"Pro film '{original_title}' nebyly nalezeny žádné podobné filmy.", addon.getAddonInfo('name'))
            xbmcplugin.endOfDirectory(handle)
            return
            
        results.sort(key=utils.get_year_movie, reverse=True) # Řazení filmů sestupně podle roku

        for movie in results:
            movie_id_similar = movie['id']
            title_similar = movie['title']
            year_val_similar = movie.get('release_date', '')[:4]
            plot_similar = movie.get('overview', '')
            poster_path_similar = movie.get('poster_path', '')
            poster_similar = f"https://image.tmdb.org/t/p/w500{poster_path_similar}" if poster_path_similar else ''
            backdrop_path_similar = movie.get('backdrop_path', '')
            fanart_similar = f"https://image.tmdb.org/t/p/original{backdrop_path_similar}" if backdrop_path_similar else poster_similar
            
            label_similar = f"{title_similar} ({year_val_similar})" if year_val_similar else title_similar
            
            try: info_year_similar = int(year_val_similar)
            except: info_year_similar = 0
            
            listitem = xbmcgui.ListItem(label=label_similar)
            if poster_similar:
                listitem.setArt({'thumb': poster_similar, 'poster': poster_similar, 'icon': poster_similar, 'fanart': fanart_similar})
            utils.set_video_info_tag(listitem, {'title': label_similar, 'plot': plot_similar, 'year': info_year_similar, 'mediatype': 'movie'})
            
            # Kontextové menu pro podobný film - předáváme jeho vlastní údaje
            commands = create_movie_context_menu(
                addon=addon,
                tmdb_id=movie_id_similar,
                title=title_similar,
                year=year_val_similar,
                plot=plot_similar,
                poster=poster_similar,
                original_title=title_similar # Pro případné další zanoření "Podobné filmy"
            )
            listitem.addContextMenuItems(commands)

            # Kliknutí na podobný film otevře jeho WCS Podrobnosti
            # Podle nastavení určíme, jakou akci provést při kliknutí na film
            default_action = addon.getSetting('default_movie_action')
            
            if default_action == '1':  # 40004 - Přehrát ihned
                play_url = f"plugin://{addon.getAddonInfo('id')}?action=play_movie_from_addon&title={quote_plus(title_similar)}&year={quote_plus(year_val_similar)}&plot={quote_plus(plot_similar)}&poster={quote_plus(poster_similar)}&tmdb_id={movie_id_similar}"
                xbmcplugin.addDirectoryItem(handle, play_url, listitem, False)
            else:
                details_url = f"plugin://{addon.getAddonInfo('id')}?action=show_wcs_details&media_type=movie&tmdb_id={movie_id_similar}"
                xbmcplugin.addDirectoryItem(handle, details_url, listitem, False)
        
        # Přidání položky "Další stránka" pro podobné filmy, pokud TMDB vrátí více stránek
        if current_page < total_pages:
            next_page_params = params.copy()
            next_page_params['page'] = str(current_page + 1)
            # Ujistíme se, že tmdb_id a original_title jsou stále v params pro další stránku
            next_page_params['tmdb_id'] = tmdb_id 
            next_page_params['original_title'] = original_title

            next_url = f"plugin://{addon.getAddonInfo('id')}?action=show_similar_movies"
            for key, value in next_page_params.items():
                if key != 'action':
                    next_url += f"&{key}={quote_plus(str(value))}"
            
            listitem_next = xbmcgui.ListItem(label=f"Další stránka ({current_page + 1}/{total_pages})")
            icon_path = 'DefaultFolderNext.png' 
            listitem_next.setArt({'thumb': icon_path, 'icon': icon_path})
            xbmcplugin.addDirectoryItem(handle, next_url, listitem_next, True)

    except Exception as e:
        handle_tmdb_api_error(e, addon, handle)
        
    xbmcplugin.endOfDirectory(handle)

# --- Funkce pro pokročilý výběr (Žánr > Země > Rok) ---

def get_director_details(director_id):
    """Vrátí detailní informace o režisérovi z TMDb."""
    if not director_id:
        return None
    try:
        api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        response = requests.get(f'https://api.themoviedb.org/3/person/{director_id}', params=api_params, timeout=10)
        response.raise_for_status()
        return response.json()
    except Exception as e:
        xbmc.log(f"[WCS] Chyba při načítání detailů režiséra {director_id}: {e}", xbmc.LOGERROR)
        return None

def get_director_movies(director_id):
    """Vrátí seznam filmů, které režíroval daný režisér (pouze job=Director, seřazeno sestupně podle roku)."""
    if not director_id:
        return []
    try:
        api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        response = requests.get(f'https://api.themoviedb.org/3/person/{director_id}/movie_credits', params=api_params, timeout=10)
        response.raise_for_status()
        data = response.json()
        crew = data.get('crew', [])
        movies = [m for m in crew if m.get('job') == 'Director']
        # Seřadit podle roku sestupně
        def get_year(m):
            date = m.get('release_date', '')
            try:
                return int(date[:4]) if date else 0
            except Exception:
                return 0
        movies.sort(key=get_year, reverse=True)
        return movies
    except Exception as e:
        xbmc.log(f"[WCS] Chyba při načítání filmů režiséra {director_id}: {e}", xbmc.LOGERROR)
        return []

def get_movie_details_for_my_movies(tmdb_id, addon):
    """Vrátí slovník s daty o filmu pro přidání do Moje filmy (id, title, year, overview, poster_path)."""
    api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
    url = f'https://api.themoviedb.org/3/movie/{tmdb_id}'
    import requests
    try:
        response = requests.get(url, params=api_params)
        response.raise_for_status()
        selected_item = response.json()
        return {
            'id': selected_item['id'],
            'title': selected_item['title'],
            'year': selected_item.get('release_date', '')[:4],
            'overview': selected_item.get('overview', ''),
            'poster_path': selected_item.get('poster_path', '')
        }
    except Exception as e:
        return {'error': str(e)}

def get_genres(media_type, addon):
    """Vrací žánry pro filmy nebo seriály podle media_type ('movie' nebo 'tv')."""
    if media_type not in ['movie', 'tv']:
        raise ValueError('media_type musí být "movie" nebo "tv"')
    api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
    url = f'https://api.themoviedb.org/3/genre/{media_type}/list'
    response = requests.get(url, params=api_params)
    response.raise_for_status()
    data = response.json()
    return data.get('genres', [])

def fetch_tmdb_items(media_type: str, endpoint: str, params: dict, addon) -> list:
    """
    Univerzální funkce pro získání seznamu položek z TMDb.
    - media_type: 'movie', 'tv', 'collection', ...
    - endpoint: např. 'discover/movie', 'movie/popular', 'tv/top_rated', 'collection/{id}', atd.
    - params: parametry pro API (page, genre_id, country_code, year, atd.)
    - addon: instance addonu pro lokalizaci, atd.
    Vrací: list slovníků s daty položek (přímo z TMDb API).
    """
    # Nechť ValueError z get_tmdb_api_key() se propaguje
    api_key = get_tmdb_api_key(addon)
    base_url = 'https://api.themoviedb.org/3/'
    url = f'{base_url}{endpoint}'
    params = dict(params)  # kopie, abychom neměnili původní
    params['api_key'] = api_key
    if 'language' not in params:
        params['language'] = 'cs-CZ'
    try:
        response = requests.get(url, params=params, timeout=10)
        response.raise_for_status()
        data = response.json()
        # Pokud je to kolekce, položky jsou v 'parts', jinak v 'results'
        if 'results' in data:
            return data['results']
        elif 'parts' in data:
            return data['parts']
        else:
            return []
    except requests.exceptions.RequestException as e:
        # Logování chyby, pokud je addon k dispozici
        if addon:
            try:
                addon.log(f"fetch_tmdb_items error: {e}")
            except Exception:
                pass
        return []

def fetch_tmdb_item_detail(media_type: str, tmdb_id, params: dict, addon) -> dict:
    """
    Univerzální funkce pro získání detailních informací o položce z TMDb.
    - media_type: 'movie', 'tv', 'person', 'collection', ...
    - tmdb_id: ID položky
    - params: další parametry pro API (např. append_to_response, language, atd.)
    - addon: instance addonu
    Vrací: slovník s daty položky nebo prázdný slovník při chybě.
    """
    # Nechť ValueError z get_tmdb_api_key() se propaguje
    api_key = get_tmdb_api_key(addon)
    base_url = 'https://api.themoviedb.org/3/'
    url = f'{base_url}{media_type}/{tmdb_id}'
    params = dict(params) if params else {}
    params['api_key'] = api_key
    if 'language' not in params:
        params['language'] = 'cs-CZ'
    try:
        import requests
        response = requests.get(url, params=params, timeout=10)
        response.raise_for_status()
        return response.json()
    except requests.exceptions.RequestException as e:
        if addon:
            try:
                addon.log(f"fetch_tmdb_item_detail error: {e}")
            except Exception:
                pass
        return {}

def fetch_tmdb_list(endpoint: str, params: dict, addon) -> list:
    """
    Univerzální funkce pro získání libovolného seznamu položek z TMDb.
    - endpoint: např. 'genre/movie/list', 'person/popular', 'collection/{id}', atd.
    - params: parametry pro API (page, language, atd.)
    - addon: instance addonu
    Vrací: list slovníků (výsledek z 'results', 'genres', 'parts', ... podle typu dotazu).
    """
    # Nechť ValueError z get_tmdb_api_key() se propaguje
    api_key = get_tmdb_api_key(addon)
    base_url = 'https://api.themoviedb.org/3/'
    url = f'{base_url}{endpoint}'
    params = dict(params) if params else {}
    params['api_key'] = api_key
    if 'language' not in params:
        params['language'] = 'cs-CZ'
    try:
        import requests
        response = requests.get(url, params=params, timeout=10)
        response.raise_for_status()
        data = response.json()
        # Vracíme první nalezený seznam (results, genres, parts, cast, crew, ...)
        for key in ['results', 'genres', 'parts', 'cast', 'crew', 'keywords', 'seasons', 'episodes']:
            if key in data:
                return data[key]
        # Pokud není žádný známý klíč, vrátíme prázdný list
        return []
    except requests.exceptions.RequestException as e:
        if addon:
            try:
                addon.log(f"fetch_tmdb_list error: {e}")
            except Exception:
                pass
        return []

# --- Funkce pro načtení detailů herce ---
def get_actor_details(actor_id):
    """Vrátí detailní informace o herci z TMDb včetně known_for dat."""
    if not actor_id:
        return None
    try:
        # Nejdříve získáme základní detaily
        api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        response = requests.get(f'https://api.themoviedb.org/3/person/{actor_id}', params=api_params, timeout=10)
        response.raise_for_status()
        actor_data = response.json()
        
        # Získáme jméno herce pro search
        actor_name = actor_data.get('name', '')
        if not actor_name:
            return actor_data
        
        # Nyní získáme known_for data z search endpointu
        search_params = {
            'api_key': get_tmdb_api_key(),
            'language': 'cs-CZ',
            'query': actor_name,
            'page': 1,
            'include_adult': 'false'
        }
        
        search_response = requests.get('https://api.themoviedb.org/3/search/person', params=search_params, timeout=10)
        search_response.raise_for_status()
        search_data = search_response.json()
        
        # Najdeme herce v search výsledcích podle ID
        for person in search_data.get('results', []):
            if person.get('id') == int(actor_id):
                # Přidáme known_for data do actor_data
                actor_data['known_for'] = person.get('known_for', [])
                break
        
        return actor_data
    except Exception as e:
        xbmc.log(f"[WCS] Chyba při načítání detailů herce {actor_id}: {e}", xbmc.LOGERROR)
        return None

# --- Funkce pro načtení filmů, ve kterých herec hrál ---
def get_actor_movies(actor_id):
    """Vrátí seznam filmů, ve kterých herec hrál (cast) seřazený sestupně podle roku."""
    if not actor_id:
        return []
    try:
        api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        response = requests.get(f'https://api.themoviedb.org/3/person/{actor_id}/movie_credits', params=api_params, timeout=10)
        response.raise_for_status()
        data = response.json()
        cast = data.get('cast', [])
        # Seřadit podle roku sestupně
        def get_year(m):
            date = m.get('release_date', '')
            try:
                return int(date[:4]) if date else 0
            except Exception:
                return 0
        cast.sort(key=get_year, reverse=True)
        return cast
    except Exception as e:
        xbmc.log(f"[WCS] Chyba při načítání filmů herce {actor_id}: {e}", xbmc.LOGERROR)
        return []

def get_creator_movies(creator_id):
    """Vrátí seznam filmů, na kterých tvůrce pracoval (crew) - jako režisér, producent, scenárista apod."""
    if not creator_id:
        return []
    try:
        api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        response = requests.get(f'https://api.themoviedb.org/3/person/{creator_id}/movie_credits', params=api_params, timeout=10)
        response.raise_for_status()
        data = response.json()
        
        # Kombinujeme cast i crew pro úplný přehled
        cast = data.get('cast', [])
        crew = data.get('crew', [])
        
        # Pro crew filtrujeme relevantní role
        relevant_crew = []
        for item in crew:
            job = item.get('job', '').lower()
            department = item.get('department', '').lower()
            # Relevantní role pro tvůrce
            if (job in ['director', 'executive producer', 'producer', 'writer', 'screenplay', 'story', 'creator'] or
                department in ['writing', 'production', 'directing']):
                relevant_crew.append(item)
        
        # Sloučíme cast a crew, odstraníme duplikáty podle ID
        all_movies = cast + relevant_crew
        unique_movies = {}
        for movie in all_movies:
            movie_id = movie.get('id')
            if movie_id and movie_id not in unique_movies:
                unique_movies[movie_id] = movie
        
        # Seřadit podle roku sestupně
        def get_year(m):
            date = m.get('release_date', '')
            try:
                return int(date[:4]) if date else 0
            except Exception:
                return 0
        
        movies_list = list(unique_movies.values())
        movies_list.sort(key=get_year, reverse=True)
        return movies_list
    except Exception as e:
        xbmc.log(f"[WCS] Chyba při načítání filmů tvůrce {creator_id}: {e}", xbmc.LOGERROR)
        return []

def get_actor_tv(actor_id):
    """Vrátí seznam seriálů, ve kterých herec hrál (cast) seřazený sestupně podle roku."""
    if not actor_id:
        return []
    try:
        api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        import requests
        response = requests.get(f'https://api.themoviedb.org/3/person/{actor_id}/tv_credits', params=api_params, timeout=10)
        response.raise_for_status()
        data = response.json()
        cast = data.get('cast', [])
        # Seřadit podle roku sestupně
        def get_year(m):
            date = m.get('first_air_date', '')
            try:
                return int(date[:4]) if date else 0
            except Exception:
                return 0
        cast.sort(key=get_year, reverse=True)
        return cast
    except Exception as e:
        xbmc.log(f"[WCS] Chyba při načítání seriálů herce {actor_id}: {e}", xbmc.LOGERROR)
        return []

def get_creator_tv(creator_id):
    """Vrátí seznam seriálů, které vytvořil daný tvůrce (pouze job=Creator, seřazeno sestupně podle roku)."""
    if not creator_id:
        return []
    try:
        api_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
        response = requests.get(f'https://api.themoviedb.org/3/person/{creator_id}/tv_credits', params=api_params, timeout=10)
        response.raise_for_status()
        data = response.json()
        crew = data.get('crew', [])
        tv_shows = [m for m in crew if m.get('job') == 'Creator']
        # Seřadit podle roku sestupně
        def get_year(m):
            date = m.get('first_air_date', '')
            try:
                return int(date[:4]) if date else 0
            except Exception:
                return 0
        tv_shows.sort(key=get_year, reverse=True)
        return tv_shows
    except Exception as e:
        xbmc.log(f"[WCS] Chyba při načítání seriálů tvůrce {creator_id}: {e}", xbmc.LOGERROR)
        return []

def get_person_details(person_id, person_type='actor'):
    """
    Univerzální funkce pro načtení detailů osoby (herce, režiséra, scénáristy) z TMDb.
    
    Args:
        person_id: TMDb ID osoby
        person_type: typ osoby ('actor', 'director', 'writer')
    
    Returns:
        dict: Kompletní data osoby včetně known_for, nebo None při chybě
    """
    if not person_id:
        return None
    
    try:
        # Všechny osoby (herci, režiséři, scénáristé) jsou v TMDb API pod endpointem 'person'
        # Použijeme get_actor_details, který už má logiku pro načtení known_for dat
        person_data = get_actor_details(person_id)
        
        if person_data:
            # Přidáme informaci o typu osoby
            person_data['person_type'] = person_type
            
            # Pro režiséry a scénáristy můžeme přidat specifické informace
            if person_type in ['director', 'writer']:
                # Získáme filmové kredity pro režiséry/scénáristy
                try:
                    movie_credits_params = {'api_key': get_tmdb_api_key(), 'language': 'cs-CZ'}
                    movie_response = requests.get(f'https://api.themoviedb.org/3/person/{person_id}/movie_credits', 
                                                params=movie_credits_params, timeout=10)
                    movie_response.raise_for_status()
                    movie_data = movie_response.json()
                    
                    if person_type == 'director':
                        # Pro režiséry: filmy kde byli režiséry
                        crew = movie_data.get('crew', [])
                        directed_movies = [m for m in crew if m.get('job') == 'Director']
                        person_data['directed_movies'] = directed_movies
                    elif person_type == 'writer':
                        # Pro scénáristy: filmy kde byli scénáristy
                        crew = movie_data.get('crew', [])
                        written_movies = [m for m in crew if m.get('job') in ['Writer', 'Screenplay', 'Story']]
                        person_data['written_movies'] = written_movies
                        
                except Exception as e:
                    xbmc.log(f"[WCS] Chyba při načítání filmových kreditů pro {person_type} {person_id}: {e}", xbmc.LOGERROR)
        
        return person_data
        
    except Exception as e:
        xbmc.log(f"[WCS] Chyba při načítání detailů {person_type} {person_id}: {e}", xbmc.LOGERROR)
        return None