#!/usr/bin/env python3
"""Generate dashed card border texture for program reorder mode focus indicator.
Wider landscape card for episode stills + text below.
"""
from PIL import Image, ImageDraw
import math

# Final output dimensions (covers still 290x163 + text + slide 25px + margin)
WIDTH, HEIGHT = 306, 258
BORDER = 3
RADIUS = 15
DASH_LEN = 12
GAP_LEN = 8
SCALE = 4  # Antialiasing

w, h = WIDTH * SCALE, HEIGHT * SCALE
r = RADIUS * SCALE
b = BORDER * SCALE
dash = DASH_LEN * SCALE
gap = GAP_LEN * SCALE

img = Image.new('RGBA', (w, h), (0, 0, 0, 0))

# Step 1: Create full solid border
border_img = Image.new('RGBA', (w, h), (0, 0, 0, 0))
border_draw = ImageDraw.Draw(border_img)
border_draw.rounded_rectangle([(0, 0), (w - 1, h - 1)], radius=r, fill=(255, 255, 255, 255))
border_draw.rounded_rectangle([(b, b), (w - 1 - b, h - 1 - b)], radius=r - b, fill=(0, 0, 0, 0))

# Step 2: Create dash mask along the perimeter
mask = Image.new('L', (w, h), 0)
mask_draw = ImageDraw.Draw(mask)

def rounded_rect_points(W, H, R):
    """Generate points along rounded rectangle perimeter."""
    points = []
    segments = [
        ('line', R, 0, W - R, 0),
        ('arc', W - R, R, R, 270, 360),
        ('line', W, R, W, H - R),
        ('arc', W - R, H - R, R, 0, 90),
        ('line', W - R, H, R, H),
        ('arc', R, H - R, R, 90, 180),
        ('line', 0, H - R, 0, R),
        ('arc', R, R, R, 180, 270),
    ]
    for seg in segments:
        if seg[0] == 'line':
            _, x1, y1, x2, y2 = seg
            length = math.sqrt((x2-x1)**2 + (y2-y1)**2)
            n = max(int(length / 2), 2)
            for i in range(n):
                t = i / n
                points.append((x1 + t*(x2-x1), y1 + t*(y2-y1)))
        else:
            _, cx, cy, radius, start_deg, end_deg = seg
            n = max(int(radius * abs(end_deg - start_deg) / 360 * 0.5), 8)
            for i in range(n):
                t = i / n
                angle = math.radians(start_deg + t * (end_deg - start_deg))
                points.append((cx + radius * math.cos(angle), cy + radius * math.sin(angle)))
    return points

pts = rounded_rect_points(w - 1, h - 1, r)

dists = [0]
for i in range(1, len(pts)):
    d = math.sqrt((pts[i][0]-pts[i-1][0])**2 + (pts[i][1]-pts[i-1][1])**2)
    dists.append(dists[-1] + d)

total_len = dists[-1]
cycle = dash + gap
pos = 0
while pos < total_len:
    dash_end = min(pos + dash, total_len)
    dash_pts = []
    for i, d in enumerate(dists):
        if pos <= d <= dash_end:
            dash_pts.append(pts[i])
    if len(dash_pts) >= 2:
        for i in range(len(dash_pts) - 1):
            x1, y1 = dash_pts[i]
            x2, y2 = dash_pts[i+1]
            mask_draw.line([(x1, y1), (x2, y2)], fill=255, width=b + 2)
    pos += cycle

# Step 3: Apply mask to border
result = Image.new('RGBA', (w, h), (0, 0, 0, 0))
result.paste(border_img, mask=mask)

result = result.resize((WIDTH, HEIGHT), Image.LANCZOS)
result.save('card_border_program.png', 'PNG')
print(f"Generated card_border_program.png ({WIDTH}x{HEIGHT}, dashed border={BORDER}px, radius={RADIUS}px)")
