# wcs/games/MillionaireMyTVAdapter.py
"""
Adapter between MillionaireGameController and MyTV inline dialog.

Maps controller callbacks to window property setProperty() calls,
ensuring all UI updates go through Kodi's property system (thread-safe).
"""
import xbmc
import xbmcgui


# Default answer button colors
COLOR_ANSWER_DEFAULT = "40FFFFFF"     # nofocus normal
COLOR_ANSWER_FOCUS = "CC2997FF"       # focus normal (blue)
COLOR_ANSWER_CORRECT = "FF34C759"     # correct (green)
COLOR_ANSWER_WRONG = "FFFF3B30"       # wrong (red)
COLOR_ANSWER_DISABLED = "15FFFFFF"    # 50:50 removed

# Ladder row colors (matching existing fullscreen dialog)
COLOR_LADDER_DEFAULT = "FF232323"
COLOR_LADDER_CURRENT = "FF335C8F"
COLOR_LADDER_COMPLETED = "FF3CB371"
COLOR_LADDER_FAILED = "FFB04A4A"
COLOR_LADDER_TEXT_DEFAULT = "AAFFFFFF"
COLOR_LADDER_TEXT_CURRENT = "FFFFFFFF"
COLOR_LADDER_TEXT_COMPLETED = "FFFFFFFF"
COLOR_LADDER_TEXT_DIM = "60FFFFFF"


class MillionaireMyTVAdapter:
    """Adapter between MillionaireGameController and MyTV dialog."""

    def __init__(self, dialog, series_id, title, plot, poster, fanart, logo):
        """
        Args:
            dialog: MyTVDialog instance (has setProperty/clearProperty)
            series_id: TMDb series ID
            title: Series name
            plot: Series plot
            poster: Poster URL
            fanart: Fanart URL
            logo: Logo URL
        """
        from wcs.games.MillionaireGame import MillionaireGameController

        self.dialog = dialog
        self.series_id = series_id
        self.title = title
        self.controller = MillionaireGameController(
            title=title, plot=plot, media_type='series'
        )

        # Set UI callbacks to our adapter methods
        self.controller.set_ui_callbacks(
            update_display=self._on_display_update,
            update_buttons=self._on_buttons_update,
            update_sidebar=self._on_sidebar_update,
            show_error=self._on_error,
            set_focus=self._on_set_focus,
            start_countdown=self._on_countdown,
            close_dialog=self._on_close,
            update_millionaire_ladder=self._on_ladder_update,
            set_answer_button_colors=self._on_answer_colors
        )

        # Return-to-menu callback
        self.controller.set_return_to_menu_callback(self._on_return_to_menu)

        # Set static properties
        dialog.setProperty('WCS.Game.Poster', poster)
        dialog.setProperty('WCS.Game.Logo', logo)
        dialog.setProperty('WCS.Game.Fanart', fanart)
        dialog.setProperty('WCS.Game.SeriesName', title)

        # Initialize answer colors to default
        self._reset_answer_colors()
        self._update_ladder_defaults()

    def start(self):
        """Start the game (show welcome screen)."""
        self.controller.start_specific_game()

    def handle_answer(self, index):
        """Handle answer button press (0-3)."""
        self.controller.answer_question(index)

    def handle_choice(self, choice_index):
        """Handle menu choice (welcome/feedback buttons)."""
        self.controller.handle_game_choice(choice_index)

    def use_lifeline_5050(self):
        """Use 50:50 lifeline."""
        self.controller.use_5050_lifeline()

    def use_lifeline_tip(self):
        """Use tip lifeline."""
        self.controller.use_tip_lifeline()

    def use_lifeline_audience(self):
        """Use audience poll lifeline."""
        self.controller.use_audience_poll_lifeline()

    @property
    def game_state(self):
        return self.controller.game_state

    def is_playing(self):
        """Return True if in active question state."""
        return self.controller.game_state in (
            'playing', 'answer_pending'
        )

    def is_answer_pending(self):
        """Return True if waiting for answer evaluation."""
        return self.controller.game_state == 'answer_pending'

    # ---- Callback handlers ----

    def _on_display_update(self, text):
        """Update title label and main content text area.

        Extracts the first non-empty line as the title (stripping Kodi
        formatting tags) and puts the remaining text into the textbox.
        """
        if not text:
            self.dialog.setProperty('WCS.Game.Title', '')
            self.dialog.setProperty('WCS.Game.Text', '')
            return

        lines = text.split('\n')
        # First non-empty line is the title
        title_line = lines[0] if lines else ''
        # Strip Kodi formatting tags for clean label display
        import re
        clean_title = re.sub(r'\[/?(?:COLOR|B|I|UPPERCASE)[^\]]*\]', '', title_line).strip()

        # Remaining lines (skip first line and optional empty second line)
        rest_start = 1
        if rest_start < len(lines) and lines[rest_start].strip() == '':
            rest_start = 2
        remaining = '\n'.join(lines[rest_start:])

        self.dialog.setProperty('WCS.Game.Title', clean_title)
        self.dialog.setProperty('WCS.Game.Text', remaining)

    def _on_buttons_update(self, *labels):
        """Update answer button labels and show/hide footer."""
        letter_map = ['A', 'B', 'C', 'D']

        # Determine if this is an answer state (show 2x2 grid) or menu state
        answer_states = ('playing', 'answer_pending')
        is_answer_state = getattr(self.controller, 'game_state', '') in answer_states

        if is_answer_state:
            # Show answer buttons (A, B, C, D grid)
            for i, label in enumerate(labels):
                if i < 4:
                    letter = letter_map[i]
                    if label:
                        self.dialog.setProperty(f'WCS.Game.Answer{letter}', f'{letter}: {label}')
                    else:
                        self.dialog.setProperty(f'WCS.Game.Answer{letter}', '')

            self.dialog.setProperty('WCS.Game.ShowAnswers', 'true')
            # Also update the question text
            question = getattr(self.controller, 'current_question', {})
            question_text = question.get('question', '') if question else ''
            self.dialog.setProperty('WCS.Game.Question', question_text)
            # Reset colors for new question
            self._reset_answer_colors()
            # Focus on first answer button
            if hasattr(self.dialog, '_delayed_focus'):
                self.dialog._delayed_focus(9500, delay=0.15)
        else:
            # Menu state - map labels to menu buttons
            self.dialog.clearProperty('WCS.Game.ShowAnswers')
            # First label = MenuBtn1, second = MenuBtn2
            # Strip A:, B:, C: prefixes if present
            menu_labels = []
            for label in labels:
                if label:
                    # Remove "A: ", "B: " etc. prefix
                    if len(label) > 2 and label[1] == ':':
                        menu_labels.append(label[3:].strip())
                    else:
                        menu_labels.append(label)
            
            self.dialog.setProperty('WCS.Game.MenuBtn1', menu_labels[0] if len(menu_labels) > 0 else '')
            self.dialog.setProperty('WCS.Game.MenuBtn2', menu_labels[1] if len(menu_labels) > 1 else '')
            # Focus on first menu button
            if hasattr(self.dialog, '_delayed_focus'):
                self.dialog._delayed_focus(9514, delay=0.15)

    def _on_sidebar_update(self):
        """Update sidebar (we use ladder + lifeline state)."""
        self._update_lifeline_state()

    def _on_error(self, message):
        """Show error via toast."""
        xbmc.log(f"[MyTV Game] Error: {message}", xbmc.LOGERROR)
        if hasattr(self.dialog, '_show_toast'):
            self.dialog._show_toast(message, 'error')

    def _on_set_focus(self, control_id):
        """Map old control IDs to new MyTV IDs."""
        # In the fullscreen dialog, focus targets are different
        # For MyTV, we focus on first answer button or menu button
        if self.is_playing():
            try:
                self.dialog.setFocusId(9500)
            except Exception:
                pass
        else:
            try:
                self.dialog.setFocusId(9514)
            except Exception:
                pass

    def _on_countdown(self, seconds, callback):
        """Start countdown for answer evaluation."""
        self.controller.start_countdown_with_dynamic_buttons(seconds, callback)

    def _on_close(self):
        """Close game dialog."""
        if hasattr(self.dialog, '_close_game_dialog'):
            self.dialog._close_game_dialog()

    def _on_return_to_menu(self):
        """Return to welcome menu."""
        self.dialog.clearProperty('WCS.Game.ShowAnswers')
        self.dialog.clearProperty('WCS.Game.Question')
        self._reset_answer_colors()
        # Set menu buttons for welcome
        self.dialog.setProperty('WCS.Game.MenuBtn1', 'Hrát hru')
        self.dialog.setProperty('WCS.Game.MenuBtn2', 'Pravidla')
        try:
            self.dialog.setFocusId(9514)
        except Exception:
            pass

    def _on_ladder_update(self):
        """Update ladder from controller state."""
        ladder_states = self.controller.get_millionaire_ladder_states()
        for _ctrl_id, state in ladder_states.items():
            q_num = state['question_num']
            self.dialog.setProperty(f'WCS.Game.Ladder.{q_num}', state['bgcolor'])

            # Text color
            if state['state'] == 'current':
                text_color = COLOR_LADDER_TEXT_CURRENT
            elif state['state'] == 'completed':
                text_color = COLOR_LADDER_TEXT_COMPLETED
            elif state['state'] in ('empty', 'generating'):
                text_color = COLOR_LADDER_TEXT_DEFAULT
            else:
                text_color = COLOR_LADDER_TEXT_DIM
            self.dialog.setProperty(f'WCS.Game.Ladder.{q_num}.Color', text_color)

    # ---- Helper methods ----

    def _reset_answer_colors(self):
        """Reset all answer button colors to default."""
        for letter in ['A', 'B', 'C', 'D']:
            self.dialog.setProperty(f'WCS.Game.Answer{letter}.Color', COLOR_ANSWER_DEFAULT)
            self.dialog.setProperty(f'WCS.Game.Answer{letter}.FocusColor', COLOR_ANSWER_FOCUS)

    def set_answer_color(self, index, is_correct):
        """Set answer button color after evaluation."""
        letters = ['A', 'B', 'C', 'D']
        if 0 <= index < 4:
            letter = letters[index]
            color = COLOR_ANSWER_CORRECT if is_correct else COLOR_ANSWER_WRONG
            self.dialog.setProperty(f'WCS.Game.Answer{letter}.Color', color)
            self.dialog.setProperty(f'WCS.Game.Answer{letter}.FocusColor', color)

    def _on_answer_colors(self, selected_index, correct_index=None):
        """Callback pro nastavení barev odpovědí na tlačítkách.
        
        Fáze 1 (countdown): correct_index=None -> vybraná = modrá focus barva
        Fáze 2 (reveal): correct_index set -> správná = zelená, špatná = červená
        Reset: selected_index=-1 -> vše default
        """
        letters = ['A', 'B', 'C', 'D']
        
        if selected_index == -1:
            # Reset všech barev
            self._reset_answer_colors()
            return
        
        if correct_index is not None:
            # Fáze 2: Reveal
            for i in range(4):
                letter = letters[i]
                if i == correct_index:
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.Color', COLOR_ANSWER_CORRECT)
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.FocusColor', COLOR_ANSWER_CORRECT)
                elif i == selected_index and selected_index != correct_index:
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.Color', COLOR_ANSWER_WRONG)
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.FocusColor', COLOR_ANSWER_WRONG)
                else:
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.Color', COLOR_ANSWER_DEFAULT)
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.FocusColor', COLOR_ANSWER_FOCUS)
        else:
            # Fáze 1: Countdown - modrý highlight na vybrané
            for i in range(4):
                letter = letters[i]
                if i == selected_index:
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.Color', COLOR_ANSWER_FOCUS)
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.FocusColor', COLOR_ANSWER_FOCUS)
                else:
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.Color', COLOR_ANSWER_DEFAULT)
                    self.dialog.setProperty(f'WCS.Game.Answer{letter}.FocusColor', COLOR_ANSWER_FOCUS)

    def _update_lifeline_state(self):
        """Update lifeline button states."""
        if self.controller.lifeline_5050_used:
            self.dialog.setProperty('WCS.Game.Lifeline5050', 'used')
        if self.controller.lifeline_tip_used:
            self.dialog.setProperty('WCS.Game.LifelineTip', 'used')
        if self.controller.lifeline_audience_used:
            self.dialog.setProperty('WCS.Game.LifelineAudience', 'used')

    def _update_ladder_defaults(self):
        """Set default ladder colors (all rows default state)."""
        for i in range(1, 10):
            self.dialog.setProperty(f'WCS.Game.Ladder.{i}', COLOR_LADDER_DEFAULT)
            self.dialog.setProperty(f'WCS.Game.Ladder.{i}.Color', COLOR_LADDER_TEXT_DEFAULT)

    def cleanup(self):
        """Clear all game properties."""
        props = [
            'WCS.MyTV.Game.Active', 'WCS.Game.State', 'WCS.Game.Text',
            'WCS.Game.Title',
            'WCS.Game.Question', 'WCS.Game.ShowAnswers',
            'WCS.Game.Poster', 'WCS.Game.Logo', 'WCS.Game.Fanart',
            'WCS.Game.SeriesName',
            'WCS.Game.MenuBtn1', 'WCS.Game.MenuBtn2',
            'WCS.Game.Lifeline5050', 'WCS.Game.LifelineTip',
            'WCS.Game.LifelineAudience',
        ]
        for p in props:
            self.dialog.clearProperty(p)

        for letter in ['A', 'B', 'C', 'D']:
            self.dialog.clearProperty(f'WCS.Game.Answer{letter}')
            self.dialog.clearProperty(f'WCS.Game.Answer{letter}.Color')
            self.dialog.clearProperty(f'WCS.Game.Answer{letter}.FocusColor')

        for i in range(1, 10):
            self.dialog.clearProperty(f'WCS.Game.Ladder.{i}')
            self.dialog.clearProperty(f'WCS.Game.Ladder.{i}.Color')
