from PIL import Image, ImageDraw

def create_rounded_mask(width, height, radius, output_path):
    # Create high-res image for anti-aliasing
    scale = 4
    w, h, r = width * scale, height * scale, radius * scale
    
    # Create image with transparent background
    # Mode 'L' is 8-bit pixels, black and white. For diffuse in Kodi, 
    # the alpha channel of the diffuse texture is used.
    # So we want an image where the shape is white (or any color) and opaque, 
    # and the outside is transparent.
    # Actually for "diffuse", Kodi uses the alpha channel. 
    # So we need RGBA.
    
    img = Image.new('RGBA', (w, h), (0, 0, 0, 0))
    draw = ImageDraw.Draw(img)
    
    # Draw white rounded rectangle (fully opaque)
    draw.rounded_rectangle((0, 0, w, h), radius=r, fill=(255, 255, 255, 255))
    
    # Downsample
    img = img.resize((width, height), Image.Resampling.LANCZOS)
    img.save(output_path)
    print(f"Saved mask to {output_path}")

def create_rounded_border(width, height, radius, thickness, output_path):
    scale = 4
    w, h, r, t = width * scale, height * scale, radius * scale, thickness * scale
    
    img = Image.new('RGBA', (w, h), (0, 0, 0, 0))
    draw = ImageDraw.Draw(img)
    
    # Draw rounded rectangle outline
    # PIL doesn't support 'width' in rounded_rectangle nicely for strokes inside/outside sometimes
    # so we draw a filled one and cut out the middle.
    
    # Outer
    draw.rounded_rectangle((0, 0, w, h), radius=r, fill=(255, 255, 255, 255))
    
    # Inner (Cutout)
    # Inset by thickness
    draw.rounded_rectangle((t, t, w - t, h - t), radius=r-t, fill=(0, 0, 0, 0), outline=None)
    
    # Since we can't easily "clear" pixels with draw in one go on RGBA without composite...
    # Better way: Draw Stroke.
    # clean slate
    img = Image.new('RGBA', (w, h), (0, 0, 0, 0))
    draw = ImageDraw.Draw(img)
    draw.rounded_rectangle((t/2, t/2, w - t/2, h - t/2), radius=r, outline=(255, 255, 255, 255), width=int(t))
    
    img = img.resize((width, height), Image.Resampling.LANCZOS)
    img.save(output_path)
    print(f"Saved border to {output_path}")


# Dimensions
# XML says: width=220, height=330.
# We'll make it 440x660 for 2x crispness if needed, or just 220x330.
# Let's go with 440x660 (2x) to be safe for 1080p scaling.
W, H = 440, 660
CornerRadius = 40 # Appropriate for this size
BorderWidth = 12

base_path = "/Users/chudy/Library/Application Support/Kodi/addons/plugin.video.milionar/resources/media"

create_rounded_mask(W, H, CornerRadius, f"{base_path}/card_mask.png")
# We don't strictly need a border texture if we use code to draw it, but having one helps
create_rounded_border(W, H, CornerRadius, BorderWidth, f"{base_path}/card_border.png")
