# wcs/user_data.py

import os
import io
import json
import traceback
import xbmcgui
import xbmc
import xbmcaddon
import xbmcvfs
import datetime

# Získáme profilovou cestu a přeložíme ji na reálnou filesystém cestu
_profile_special = xbmcaddon.Addon().getAddonInfo('profile')
_profile = xbmcvfs.translatePath(_profile_special)

# Správa uživatelských dat, nastavení, historie a oblíbených seriálů (bude implementováno v dalších krocích) 

# Historie hledani

_SEARCH_HISTORY_FILE = os.path.join(_profile, 'search_history.json')
_SEARCH_HISTORY_DEFAULT = {"universal": [], "tmdb": [], "tv": []}

def _migrate_search_history_from_settings(addon):
    """Migruje historii hledani ze 3 oddelenyc settings do jednoho JSON."""
    migrated = {"universal": [], "tmdb": [], "tv": []}
    needs_save = False
    
    for key in ["universal", "tmdb", "tv"]:
        setting_key = f"search_history_{key}"
        data = addon.getSetting(setting_key)
        if data:
            try:
                items = json.loads(data)
                if items:
                    migrated[key] = items
                    needs_save = True
                    addon.setSetting(setting_key, '')
            except (ValueError, json.JSONDecodeError):
                pass
    
    if needs_save:
        _store_json_data(_SEARCH_HISTORY_FILE, migrated)
        xbmc.log('[plugin.video.milionar] Migrated search history from settings to search_history.json', level=xbmc.LOGINFO)
        return migrated
    return None

def _load_search_history_all(addon):
    """Nacte celou strukturu historie hledani z JSON souboru."""
    # Zkusit migraci ze settings (pouze pokud JSON neexistuje)
    if not os.path.exists(_SEARCH_HISTORY_FILE):
        migrated = _migrate_search_history_from_settings(addon)
        if migrated:
            return migrated
    
    # Pouzit genericke nacteni
    return _load_json_data(_SEARCH_HISTORY_FILE, None, addon, default_value=_SEARCH_HISTORY_DEFAULT.copy())

def _store_search_history_all(data):
    """Ulozi celou strukturu historie hledani do JSON souboru."""
    _store_json_data(_SEARCH_HISTORY_FILE, data)

def _get_history_key(search_history_file):
    """Vrací klíč historie podle názvu souboru (pro kompatibilitu)."""
    if 'universal' in search_history_file:
        return 'universal'
    elif 'tmdb' in search_history_file:
        return 'tmdb'
    return 'tv'

def load_search_history_setting(addon, key):
    """Načte historii pro daný typ (universal/tmdb/tv)."""
    # Normalizace klíče (odstraní prefix search_history_)
    if key.startswith('search_history_'):
        key = key.replace('search_history_', '')
    data = _load_search_history_all(addon)
    return data.get(key, [])

def store_search_history_setting(addon, key, what, max_size):
    """Přidá položku do historie daného typu."""
    if not what:
        return
    if key.startswith('search_history_'):
        key = key.replace('search_history_', '')
    
    data = _load_search_history_all(addon)
    history = data.get(key, [])
    
    if what in history:
        history.remove(what)
    history = [what] + history
    if len(history) > max_size:
        history = history[:max_size]
    
    data[key] = history
    _store_search_history_all(data)

def remove_search_history_setting(addon, key, what):
    """Odstraní položku z historie daného typu."""
    if not what:
        return
    if key.startswith('search_history_'):
        key = key.replace('search_history_', '')
    
    data = _load_search_history_all(addon)
    history = data.get(key, [])
    
    if what in history:
        history.remove(what)
        data[key] = history
        _store_search_history_all(data)

def clear_search_history_setting(addon, key):
    """Vymaže historii daného typu."""
    if key.startswith('search_history_'):
        key = key.replace('search_history_', '')
    
    data = _load_search_history_all(addon)
    data[key] = []
    _store_search_history_all(data)

# Původní funkce pro kompatibilitu s UI
def load_search_history(profile_path, search_history_file):
    addon = xbmcaddon.Addon()
    key = _get_history_key(search_history_file)
    return load_search_history_setting(addon, key)

def store_search_history(what, profile_path, search_history_file, max_size):
    addon = xbmcaddon.Addon()
    key = _get_history_key(search_history_file)
    store_search_history_setting(addon, key, what, max_size)

def remove_search_history(what, profile_path, search_history_file):
    addon = xbmcaddon.Addon()
    key = _get_history_key(search_history_file)
    remove_search_history_setting(addon, key, what)

def clear_search_history(profile_path, search_history_file):
    addon = xbmcaddon.Addon()
    key = _get_history_key(search_history_file)
    clear_search_history_setting(addon, key)

# Správa TMDb seriálů (Watchlist)

_WATCHLIST_SERIES_FILE = os.path.join(_profile, 'watchlist_series.json')

def load_tmdb_series(addon):
    """Načte seznam seriálů ve watchlistu z JSON souboru."""
    return _load_json_data(_WATCHLIST_SERIES_FILE, 'tmdb_series', addon)

def store_tmdb_series(addon, series_list):
    """Uloží seznam seriálů ve watchlistu do JSON souboru."""
    _store_json_data(_WATCHLIST_SERIES_FILE, series_list)

def remove_tmdb_series_by_index(addon, idx):
    """Odstraní seriál podle indexu ze seznamu 'Moje seriály'. Vrací True při úspěchu, jinak False."""
    series = load_tmdb_series(addon)
    if not (0 <= idx < len(series)):
        return False
    series.pop(idx)
    store_tmdb_series(addon, series)
    return True

# Správa TMDb filmů (Watchlist)

_WATCHLIST_MOVIES_FILE = os.path.join(_profile, 'watchlist_movies.json')

def load_tmdb_movies(addon):
    """Načte seznam filmů ve watchlistu z JSON souboru."""
    return _load_json_data(_WATCHLIST_MOVIES_FILE, 'tmdb_movies', addon)

def store_tmdb_movies(addon, movies_list):
    """Uloží seznam filmů ve watchlistu do JSON souboru."""
    _store_json_data(_WATCHLIST_MOVIES_FILE, movies_list)

def remove_tmdb_movie_by_index(addon, idx):
    """Odstraní film podle indexu ze seznamu 'Moje filmy'. Vrací True při úspěchu, jinak False."""
    movies = load_tmdb_movies(addon)
    if not (0 <= idx < len(movies)):
        return False
    movies.pop(idx)
    store_tmdb_movies(addon, movies)
    return True

# Správa oblíbených herců

_FAVORITE_ACTORS_FILE = os.path.join(_profile, 'favorite_actors.json')

def load_favorite_actors(addon):
    """Načte seznam oblíbených herců z JSON souboru."""
    return _load_json_data(_FAVORITE_ACTORS_FILE, 'favorite_actors', addon)

def store_favorite_actors(addon, actors_list):
    """Uloží seznam oblíbených herců do JSON souboru."""
    _store_json_data(_FAVORITE_ACTORS_FILE, actors_list)

def check_actor_exists(actor_id, actors_list):
    """Kontroluje, zda herec s daným ID již existuje v seznamu."""
    for actor in actors_list:
        if actor.get('id') == actor_id:
            return True
    return False

def remove_favorite_actor_by_index(addon, idx):
    """Odstraní herce podle indexu ze seznamu oblíbených. Vrací True při úspěchu, jinak False."""
    actors = load_favorite_actors(addon)
    if not (0 <= idx < len(actors)):
        return False
    actors.pop(idx)
    store_favorite_actors(addon, actors)
    return True

# Správa oblíbených režisérů

_FAVORITE_DIRECTORS_FILE = os.path.join(_profile, 'favorite_directors.json')

def load_favorite_directors(addon):
    """Načte seznam oblíbených režisérů z JSON souboru."""
    return _load_json_data(_FAVORITE_DIRECTORS_FILE, 'favorite_directors', addon)

def store_favorite_directors(addon, directors_list):
    """Uloží seznam oblíbených režisérů do JSON souboru."""
    _store_json_data(_FAVORITE_DIRECTORS_FILE, directors_list)

def check_director_exists(director_id, directors_list):
    """Kontroluje, zda režisér s daným ID již existuje v seznamu."""
    for director in directors_list:
        if director.get('id') == director_id:
            return True
    return False

def remove_favorite_director_by_index(addon, idx):
    """Odstraní režiséra podle indexu ze seznamu oblíbených. Vrací True při úspěchu, jinak False."""
    directors = load_favorite_directors(addon)
    if not (0 <= idx < len(directors)):
        return False
    directors.pop(idx)
    store_favorite_directors(addon, directors)
    return True

def _check_credentials_and_prompt(addon, on_success_callback=None):
    """
    Zkontroluje přihlašovací údaje. Pokud chybí, zobrazí dialog.
    Vždy zavolá `on_success_callback` po úspěšném přihlášení nebo pokud je uživatel již přihlášen.
    Vrací `True`, pokud je uživatel přihlášen (nebo se právě přihlásil), jinak `False`.
    """
    ws_user = addon.getSetting('wsuser')
    ws_pass = addon.getSetting('wspass')

    if ws_user and ws_pass:
        if on_success_callback:
            on_success_callback()
        return True
    
    from wcs.login.DialogWebshareLogin import show_webshare_login_dialog
    
    # Předáme callback přímo do dialogu.
    # Dialog se postará o jeho zavolání po úspěšném přihlášení.
    show_webshare_login_dialog(addon, on_success_callback)
    
    # Funkce nyní nevrací stav přihlášení, o to se postará callback.
    # Vracíme False, protože v momentě volání není jisté, zda se uživatel přihlásí.
    # Logika pokračování je plně v rukou callbacku.
    return False


def check_providers_and_prompt(addon, on_success_callback=None):
    """
    Zkontroluje oba zapnuté providery a zobrazí login dialog pro nepřihlášené.
    Zobrazuje dialogy postupně pro každého nepřihlášeného providera.
    
    Vrací True pokud je alespoň jeden provider přihlášen, jinak False.
    """
    from wcs.login.DialogWebshareLogin import show_webshare_login_dialog
    from wcs.login.DialogKraskaLogin import show_kraska_login_dialog
    
    webshare_enabled = addon.getSetting('webshare_enabled') == 'true'
    kraska_enabled = addon.getSetting('kraska_enabled') == 'true'
    
    # Kontrola přihlášení
    webshare_logged_in = bool(addon.getSetting('wsuser') and addon.getSetting('wspass'))
    kraska_logged_in = bool(addon.getSetting('kraska_token'))
    
    xbmc.log(f"[plugin.video.milionar] check_providers_and_prompt: webshare_enabled={webshare_enabled}, logged_in={webshare_logged_in}", level=xbmc.LOGINFO)
    xbmc.log(f"[plugin.video.milionar] check_providers_and_prompt: kraska_enabled={kraska_enabled}, logged_in={kraska_logged_in}", level=xbmc.LOGINFO)
    
    # Webshare zapnutý ale nepřihlášený
    if webshare_enabled and not webshare_logged_in:
        show_webshare_login_dialog(addon)
    
    # KRA.sk zapnutý ale nepřihlášený
    if kraska_enabled and not kraska_logged_in:
        show_kraska_login_dialog(addon)
    
    # Znovu zkontrolovat stav po dialozích
    webshare_logged_in = bool(addon.getSetting('wsuser') and addon.getSetting('wspass'))
    kraska_logged_in = bool(addon.getSetting('kraska_token'))
    
    # Máme alespoň jednoho funkčního providera?
    has_valid_provider = (webshare_enabled and webshare_logged_in) or \
                         (kraska_enabled and kraska_logged_in)
    
    if has_valid_provider and on_success_callback:
        on_success_callback()
    
    return has_valid_provider

# --- Oblíbení Scenáristé ---

FAVORITE_WRITERS_FILE = os.path.join(_profile, 'favorite_writers.json')

def load_favorite_writers(addon):
    """Načte seznam oblíbených scénáristů z JSON souboru."""
    try:
        if os.path.exists(FAVORITE_WRITERS_FILE):
            with open(FAVORITE_WRITERS_FILE, 'r', encoding='utf-8') as f:
                return json.load(f)
        else:
            return []
    except (IOError, ValueError, json.JSONDecodeError) as e:
        xbmc.log(f'Error loading favorite writers: {e}', level=xbmc.LOGERROR)
        return []

def store_favorite_writers(addon, writers):
    """Uloží seznam oblíbených scénáristů do JSON souboru."""
    try:
        if not os.path.exists(_profile):
            os.makedirs(_profile)
        with open(FAVORITE_WRITERS_FILE, 'w', encoding='utf-8') as f:
            json.dump(writers, f, ensure_ascii=False, indent=4)
    except (IOError, TypeError) as e:
        xbmc.log(f'Error storing favorite writers: {e}', level=xbmc.LOGERROR)

def check_writer_exists(writer_id, writers_list):
    """Zkontroluje, zda scénárista s daným ID již existuje v seznamu."""
    return any(writer['id'] == writer_id for writer in writers_list)

def remove_favorite_writer_by_index(addon, idx):
    """Odstraní scénáristu podle indexu ze seznamu oblíbených. Vrací True při úspěchu, jinak False."""
    writers = load_favorite_writers(addon)
    if not (0 <= idx < len(writers)):
        return False
    writers.pop(idx)
    store_favorite_writers(addon, writers)
    return True

# --- Generické helper funkce pro JSON data ---

def _load_json_data(file_path, setting_key, addon, old_file_path=None, default_value=None):
    """
    Genericka funkce pro nacteni dat s automatickou migraci.
    
    Args:
        file_path: Cesta k novemu JSON souboru
        setting_key: Klic v Kodi settings pro migraci (muze byt None)
        addon: Kodi addon objekt
        old_file_path: Volitelna cesta ke staremu JSON souboru pro migraci
        default_value: Vychozi hodnota pokud data neexistuji (default: [])
    
    Returns:
        Nactena data nebo default_value
    """
    if default_value is None:
        default_value = []
    
    # 1. Zkusit nacist z JSON souboru
    if os.path.exists(file_path):
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                return json.load(f)
        except (IOError, ValueError, json.JSONDecodeError) as e:
            xbmc.log(f'[plugin.video.milionar] Error loading {os.path.basename(file_path)}: {e}', level=xbmc.LOGERROR)
            return default_value
    
    # 2. Migrace z Kodi settings
    if setting_key:
        data = addon.getSetting(setting_key)
        if data:
            try:
                items = json.loads(data)
                if items:
                    _store_json_data(file_path, items)
                    addon.setSetting(setting_key, '')
                    xbmc.log(f'[plugin.video.milionar] Migrated {setting_key} from settings to {os.path.basename(file_path)}', level=xbmc.LOGINFO)
                    return items
            except (ValueError, json.JSONDecodeError) as e:
                xbmc.log(f'[plugin.video.milionar] Error parsing {setting_key} from settings: {e}', level=xbmc.LOGERROR)
    
    # 3. Migrace ze stareho JSON souboru (pokud existuje)
    if old_file_path and os.path.exists(old_file_path):
        try:
            with open(old_file_path, 'r', encoding='utf-8') as f:
                old_data = json.load(f)
            if old_data:
                _store_json_data(file_path, old_data)
                try:
                    os.remove(old_file_path)
                    xbmc.log(f'[plugin.video.milionar] Migrated {os.path.basename(old_file_path)} to {os.path.basename(file_path)}', level=xbmc.LOGINFO)
                except:
                    pass
                return old_data
        except (IOError, ValueError, json.JSONDecodeError) as e:
            xbmc.log(f'[plugin.video.milionar] Error loading old file {os.path.basename(old_file_path)}: {e}', level=xbmc.LOGERROR)
    
    return default_value

def _store_json_data(file_path, data):
    """
    Generická funkce pro uložení dat do JSON souboru.
    
    Args:
        file_path: Cesta k JSON souboru
        data: Data k uložení (list nebo dict)
    """
    try:
        if not os.path.exists(_profile):
            os.makedirs(_profile)
        with open(file_path, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=2)
    except (IOError, TypeError) as e:
        xbmc.log(f'[plugin.video.milionar] Error storing {os.path.basename(file_path)}: {e}', level=xbmc.LOGERROR)


# --- MyTV Kanály ---

_MYTV_CHANNELS_FILE = os.path.join(_profile, 'mytv_channels.json')

def load_mytv_channels(addon):
    """Načte seznam MyTV kanálů z JSON souboru."""
    return _load_json_data(_MYTV_CHANNELS_FILE, 'my_tv_channels', addon)

def store_mytv_channels(addon, channels):
    """Uloží seznam MyTV kanálů do JSON souboru."""
    _store_json_data(_MYTV_CHANNELS_FILE, channels)


# --- MyCinema Kanály (Kina) ---

_MYCINEMA_CHANNELS_FILE = os.path.join(_profile, 'mycinema_channels.json')

def load_mycinema_channels(addon):
    """Načte seznam MyCinema kanálů (kin) z JSON souboru."""
    return _load_json_data(_MYCINEMA_CHANNELS_FILE, None, addon)

def store_mycinema_channels(addon, channels):
    """Uloží seznam MyCinema kanálů (kin) do JSON souboru."""
    _store_json_data(_MYCINEMA_CHANNELS_FILE, channels)


# --- Naposledy přehrané ---

_GLOBAL_WATCH_HISTORY_FILE = os.path.join(_profile, 'global_watch_history.json')
_RECENTLY_PLAYED_FILE_OLD = os.path.join(_profile, 'recently_played.json')

def load_recently_played(addon):
    """Načte seznam naposledy přehraných položek z JSON souboru."""
    return _load_json_data(_GLOBAL_WATCH_HISTORY_FILE, 'recently_played', addon, _RECENTLY_PLAYED_FILE_OLD)

def store_recently_played(addon, items):
    """Uloží seznam naposledy přehraných položek do JSON souboru."""
    _store_json_data(_GLOBAL_WATCH_HISTORY_FILE, items)


def add_recently_played_item(addon, item):
    """
    Přidá položku na začátek seznamu naposledy přehraných.
    Omezuje velikost seznamu na 50 položek.
    Duplicitní položky (stejné ID/titul) přesune na začátek.
    """
    items = load_recently_played(addon)
    
    # Identifikace položky (pro duplicitu) - stačí porovnat tmdb_id a typ, nebo titul pokud ID chybí
    tmdb_id = item.get('tmdb_id')
    title = item.get('title')
    media_type = item.get('media_type', 'movie')
    
    # Odstranění existující stejné položky
    new_items = []
    for existing in items:
        ex_id = existing.get('tmdb_id')
        ex_title = existing.get('title')
        ex_type = existing.get('media_type', 'movie')
        
        # Shoda ID a typu
        if tmdb_id and ex_id and str(tmdb_id) == str(ex_id) and media_type == ex_type:
            continue
        # Shoda titulu (fallback pro položky bez ID, např. manuální hledání)
        if (not tmdb_id or not ex_id) and title == ex_title:
            continue
            
        new_items.append(existing)
    
    # Přidání nové na začátek (včetně timestampu)
    item['timestamp'] = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    new_items.insert(0, item)
    
    # Limit na 50 (odpovídá max nastavení pro AI kontext)
    if len(new_items) > 50:
        new_items = new_items[:50]
        
    store_recently_played(addon, new_items)

# Starý AI usage systém byl odstraněn - používáme nyní ai_usage_tracker.py 