import xbmc
import xbmcaddon
from wcs.ai import AIHandler as ai_handler
from wcs import user_data
from wcs.ai.AIProvider import ai_provider


class GameController:
    """
    Hlavní herní kontroler, který deleguje na specializované kontrolery podle typu hry.
    Zachovává plnou kompatibilitu s existujícím kódem.
    """
    
    def __init__(self, title="", plot="", genre="", media_type="movie", episode_title="", season_number=None, episode_number=None, actor_data=None):
        self.addon = xbmcaddon.Addon()
        
        # Informace o médiu
        self.title = title
        self.plot = plot
        self.genre = genre
        self.media_type = media_type  # "movie" nebo "series"
        self.episode_title = episode_title  # Pro seriály
        self.season_number = season_number  # Pro seriály
        self.episode_number = episode_number  # Pro seriály
        
        # --- Nové: data o herci (pouze pro media_type 'actor') ---
        self.actor_data = actor_data or {}
        
        # Základní herní stav
        self.game_state = "intro"
        self.game_type = None
        
        # UI callbacks
        self.update_display_callback = None
        self.update_buttons_callback = None
        self.update_sidebar_callback = None
        self.show_error_callback = None
        self.set_focus_callback = None
        self.start_countdown_callback = None
        self.close_dialog_callback = None
        
        # Současný aktivní controller (None na začátku)
        self.active_controller = None
        
        # Uložení parametrů pro předání potomkům
        self._constructor_params = {
            'title': title,
            'plot': plot,
            'genre': genre,
            'media_type': media_type,
            'episode_title': episode_title,
            'season_number': season_number,
            'episode_number': episode_number,
            'actor_data': self.actor_data
        }
    
    def set_ui_callbacks(self, update_display, update_buttons, update_sidebar, show_error, set_focus=None, start_countdown=None, close_dialog=None, update_millionaire_ladder=None, set_answer_button_colors=None):
        """Nastaví callback funkce pro komunikaci s UI."""
        self.update_display_callback = update_display
        self.update_buttons_callback = update_buttons
        self.update_sidebar_callback = update_sidebar
        self.show_error_callback = show_error
        self.set_focus_callback = set_focus
        self.start_countdown_callback = start_countdown
        self.close_dialog_callback = close_dialog
        self.update_millionaire_ladder_callback = update_millionaire_ladder
        self.set_answer_button_colors_callback = set_answer_button_colors
    
    def _add_ai_info_header(self, text: str) -> str:
        """Přidá informaci o AI modelu na začátek textu velmi malým písmem."""
        # AI info odstraněno na požádání uživatele
        return text
        
    def _track_ai_usage(self, usage, usage_type):
        """Společná metoda pro sledování AI usage."""
        if usage:
            from wcs.ai.UsageTracker import track_ai_usage
            provider = ai_provider.get_current_provider()
            model = ai_provider.get_current_model()
            track_ai_usage(provider, model, usage, usage_type)
    
    def start_game_selection(self):
        """Spustí výběr typu hry."""
        xbmc.log("[plugin.video.milionar] GameController.start_game_selection called", level=xbmc.LOGINFO)
        self.game_state = "game_selection"
        self.active_controller = None  # Reset aktivního kontroleru
        
        # Rozlišení typu média pro popisy her
        if self.media_type == "series":
            media_reference = "seriálem"
            media_about = "seriálu"
            game_title = "SERIÁLOVÝ MILIONÁŘ"
        elif self.media_type == "actor":
            media_reference = "hercem"
            media_about = "herci"
            game_title = "MILIONÁŘ S HERCEM"
        elif self.media_type == "director":
            media_reference = "režisérem"
            media_about = "režiséru"
            game_title = "MILIONÁŘ S REŽISÉREM"
        elif self.media_type == "writer":
            media_reference = "scénáristou"
            media_about = "scénáristovi"
            game_title = "MILIONÁŘ SE SCÉNÁRISTOU"
        elif self.media_type == "creator":
            media_reference = "tvůrcem"
            media_about = "tvůrci"
            game_title = "MILIONÁŘ S TVŮRCEM"
        else:
            media_reference = "filmem"
            media_about = "filmu"
            game_title = "FILMOVÝ MILIONÁŘ"
        
        intro_text = (
            f"ADVENTURE (beta):\n"
            f"• Interaktivní příběh inspirovaný {media_reference}\n"
            "• Vaše volby ovlivní průběh a konec\n"
            f"• Hrajete jako jedna z postav {media_about}\n\n"
            "KVÍZOVÁ HRA:\n"
            f"• 5 otázek o {media_about} z různých oblastí\n"
            f"• Testování znalostí o {media_about}\n"
            "• Získejte co nejvyšší skóre!\n\n"
            f"{game_title}:\n"
            "• Hra inspirovaná Chcete být milionářem\n"
            "• Otázky s postupně narůstající obtížností\n"
            "• Pomocníci a šance na velkou výhru!\n\n"
        )
        
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(intro_text))
        if self.update_buttons_callback:
            # Všechny hry dostupné vždy
            millionaire_button = "Milionář"
            self.update_buttons_callback(millionaire_button, "Kvízová hra", "Adventure (beta)", "Zavřít")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        # Nastavit fokus na první tlačítko (Milionář)
        if self.set_focus_callback:
            self.set_focus_callback(0)
    
    def return_to_main_menu(self):
        """Jednoduchý návrat na hlavní menu - resetuje stav a spustí výběr hry."""
        xbmc.log("[plugin.video.milionar] GameController.return_to_main_menu called", level=xbmc.LOGINFO)
        self.active_controller = None
        self.game_state = "game_selection"
        self.start_game_selection()
    
    def end_game(self):
        """Ukončí hru - zavře dialog."""
        if self.close_dialog_callback:
            self.close_dialog_callback()
    
    def get_sidebar_items(self):
        """Vrací položky pro sidebar podle stavu hry."""
        if self.active_controller:
            return self.active_controller.get_sidebar_items()
        elif self.game_state == "game_selection":
            return ["Výběr typu hry", f"Médium: {self.title[:20]}", "Vyberte hru"]
        else:
            return ["Neznámý stav", "", ""]
    
    def _start_adventure_game(self):
        """Vytvoří a přepne na AdventureGameController."""
        from .AdventureGame import AdventureGameController
        
        self.active_controller = AdventureGameController(**self._constructor_params)
        
        # Přenos UI callbacks
        self.active_controller.set_ui_callbacks(
            self.update_display_callback,
            self.update_buttons_callback,
            self.update_sidebar_callback,
            self.show_error_callback,
            self.set_focus_callback,
            self.start_countdown_callback,
            self.close_dialog_callback
        )
        
        # Nastavení return to menu callback
        self.active_controller.set_return_to_menu_callback(self.return_to_main_menu)
        
        # Spuštění adventure hry
        self.active_controller.start_specific_game()
    
    def _start_quiz_game(self):
        """Vytvoří a přepne na QuizGameController."""
        from .QuizGame import QuizGameController
        
        self.active_controller = QuizGameController(**self._constructor_params)
        
        # Přenos UI callbacks
        self.active_controller.set_ui_callbacks(
            self.update_display_callback,
            self.update_buttons_callback,
            self.update_sidebar_callback,
            self.show_error_callback,
            self.set_focus_callback,
            self.start_countdown_callback,
            self.close_dialog_callback
        )
        
        # Nastavení return to menu callback
        self.active_controller.set_return_to_menu_callback(self.return_to_main_menu)
        
        # Spuštění quiz hry
        self.active_controller.start_specific_game()
    
    def _start_millionaire_game(self):
        """Vytvoří a přepne na MillionaireGameController."""
        # Vybereme správný kontroler podle typu média
        if self.media_type == 'actor':
            from .MillionaireGame import MillionaireGameController
            controller_cls = MillionaireGameController
        else:
            from .MillionaireGame import MillionaireGameController
            controller_cls = MillionaireGameController
        
        params = self._constructor_params.copy()
        self.active_controller = controller_cls(**params)
        
        # Přenos UI callbacks (včetně speciálních pro millionaire)
        self.active_controller.set_ui_callbacks(
            self.update_display_callback,
            self.update_buttons_callback,
            self.update_sidebar_callback,
            self.show_error_callback,
            self.set_focus_callback,
            self.start_countdown_callback,
            self.close_dialog_callback,
            self.update_millionaire_ladder_callback,
            getattr(self, 'set_answer_button_colors_callback', None)
        )
        
        # Nastavení return to menu callback  
        self.active_controller.set_return_to_menu_callback(self.return_to_main_menu)
        
        # Spuštění millionaire hry
        self.active_controller.start_specific_game()
    
    @property
    def current_game_state(self):
        """Vrací aktuální stav hry - buď z aktivního controlleru nebo vlastní."""
        if self.active_controller and hasattr(self.active_controller, 'game_state'):
            return self.active_controller.game_state
        return self.game_state
    
    def make_choice(self, choice_index):
        """Deleguje volby na aktivní controller nebo řeší game_selection."""
        xbmc.log(f"[plugin.video.milionar] GameController.make_choice: choice_index={choice_index}, game_state={self.current_game_state}, active_controller={type(self.active_controller).__name__ if self.active_controller else 'None'}", level=xbmc.LOGINFO)
        
        # OPRAVA: Pokud existuje aktivní controller, VŽDY deleguj na něj
        if self.active_controller:
            # Delegace na aktivní controller
            xbmc.log(f"[plugin.video.milionar] Delegating to {type(self.active_controller).__name__}", level=xbmc.LOGINFO)
            self.active_controller.make_choice(choice_index)
        elif self.game_state == "game_selection":
            # Standardní režim (včetně actor) - všechny tři hry
            if choice_index == 0:  # Milionář (první)
                xbmc.log("[plugin.video.milionar] Starting Millionaire game", level=xbmc.LOGINFO)
                self._start_millionaire_game()
            elif choice_index == 1:  # Kvíz (druhé)
                xbmc.log("[plugin.video.milionar] Starting Quiz game", level=xbmc.LOGINFO)
                self._start_quiz_game()
            elif choice_index == 2:  # Adventure (třetí)
                xbmc.log("[plugin.video.milionar] Starting Adventure game", level=xbmc.LOGINFO)
                self._start_adventure_game()
            elif choice_index == 3:  # Zavřít
                xbmc.log("[plugin.video.milionar] Ending game", level=xbmc.LOGINFO)
                self.end_game()
    
    # Kompatibilní metody pro zachování zpětné kompatibility
    def start_adventure_game(self):
        """Kompatibilní metoda pro přímé spuštění adventure."""
        self._start_adventure_game()
    
    def start_quiz_game(self):
        """Kompatibilní metoda pro přímé spuštění kvízu.""" 
        self._start_quiz_game()
    
    def answer_quiz_question(self, answer_index):
        """Kompatibilní metoda pro quiz odpovědi."""
        if self.active_controller and hasattr(self.active_controller, 'answer_quiz_question'):
            self.active_controller.answer_quiz_question(answer_index)
    
    def make_adventure_choice(self, choice_index):
        """Kompatibilní metoda pro adventure volby."""
        if self.active_controller and hasattr(self.active_controller, 'make_adventure_choice'):
            self.active_controller.make_adventure_choice(choice_index)
    
    def finish_answer_evaluation(self):
        """Kompatibilní metoda pro dokončení quiz vyhodnocení."""
        if self.active_controller and hasattr(self.active_controller, 'finish_answer_evaluation'):
            self.active_controller.finish_answer_evaluation()
