#!/usr/bin/env python3
"""Generate a cinematic bottom gradient for fullscreen play selection overlay.

Creates a vertical gradient texture:
- Top: fully transparent (still image visible)
- Middle (~40%): starts gradually fading to black
- Bottom: 100% opaque black

Uses an eased curve (cubic) for smooth, cinematic transition
typical of Netflix/Apple TV detail screens.
"""
from PIL import Image
import math

WIDTH = 16  # Narrow - Kodi stretches horizontally
HEIGHT = 1080  # Full screen height

img = Image.new('RGBA', (WIDTH, HEIGHT), (0, 0, 0, 0))

# Gradient starts at ~35% from top, fully black at ~85%
FADE_START = 0.35  # Where gradient begins (transparent above)
FADE_END = 0.85    # Where it reaches full black

for y in range(HEIGHT):
    ratio = y / (HEIGHT - 1)
    
    if ratio <= FADE_START:
        alpha = 0
    elif ratio >= FADE_END:
        alpha = 255
    else:
        # Normalized progress within fade zone [0..1]
        t = (ratio - FADE_START) / (FADE_END - FADE_START)
        # Smooth cubic ease-in for cinematic feel
        t = t * t * (3 - 2 * t)  # smoothstep
        alpha = int(255 * t)
    
    for x in range(WIDTH):
        img.putpixel((x, y), (0, 0, 0, alpha))

output = '/Users/chudy/Library/Application Support/Kodi/addons/plugin.video.milionar/resources/media/gradient_cinematic_bottom.png'
img.save(output, 'PNG')
print(f"Created: {output} ({WIDTH}x{HEIGHT})")
