# wcs/library.py

import os
import re
import xbmc
import xbmcaddon
import xbmcgui
import requests  # Přidáno pro _create_series_strms
from urllib.parse import quote_plus
from wcs.ui import UI as ui
from wcs import user_data, utils
from wcs.webshare import WebshareClient as webshare_api # Odstraněn import search
import difflib # Přidán import
import xml.etree.ElementTree as ET # Přidán import
import xbmcplugin # Přidán import
import datetime
import threading
import json  # Přidáno pro JSON-RPC
from wcs.ui.dialogs.DialogWcsSearch import show_search_dialog
from wcs.search.MovieSearch import sort_files_by_priority
from wcs.utils import calculate_advanced_score
from wcs.metadata.TMDbClient import get_tmdb_api_key

# TMDB API klíč je nyní centralizovaně spravován v tmdb_handler.py 

def _create_movie_strm(movie_data, addon):
    """Vytvoří .strm soubor pro film, pokud je nastavena složka knihovny."""
    movies_folder = addon.getSetting('library_movies_folder')
    if not movies_folder or not os.path.isdir(movies_folder):
        if movies_folder:
             ui.popinfo(f"Nastavená složka pro filmy neexistuje: {movies_folder}", addon.getAddonInfo('name'))
        else:
             ui.popinfo("Složka pro filmy není nastavena v integraci knihovny.", addon.getAddonInfo('name'))
        return False

    try:
        movie_title = movie_data['title']
        # Rok získáme buď přímo z 'year' nebo z 'release_date'
        movie_year = movie_data.get('year') or movie_data.get('release_date', '')[:4]
        if not movie_year:
             ui.popinfo(f"Nelze vytvořit .strm pro '{movie_title}', chybí rok vydání.", addon.getAddonInfo('name'))
             return False

        # Použijeme sanitize_filename z modulu utils
        strm_filename = utils.sanitize_filename(f"{movie_title} ({movie_year}).strm")
        strm_filepath = os.path.join(movies_folder, strm_filename)
        
        # OPRAVA: Předání handle do play_movie_from_library není možné zde
        # Obsah .strm musí být pouze URL, které plugin později zpracuje
        strm_content = f"plugin://{addon.getAddonInfo('id')}?action=play_movie_from_library&title={quote_plus(movie_title)}&year={movie_year}"
        
        # Vytvoření souboru
        os.makedirs(os.path.dirname(strm_filepath), exist_ok=True)
        with open(strm_filepath, 'w', encoding='utf-8') as f:
            f.write(strm_content)
        # Formátovaný string s názvem filmu
        popup_message = addon.getLocalizedString(30131).replace('%s', str(movie_title)) 
        ui.popinfo(popup_message, addon.getAddonInfo('name'))
        # --- Automatická aktualizace knihovny ---
        if addon.getSettingBool('library_auto_update'):
            def delayed_refresh():
                xbmc.sleep(500)
                if not xbmc.getCondVisibility('Library.IsScanningVideo'):
                    xbmc.executebuiltin(f'UpdateLibrary(video, "{movies_folder}")')
            threading.Thread(target=delayed_refresh).start()
        # -------------------------------------
    except KeyError as e:
        ui.popinfo(f"Chyba při vytváření .strm: Chybějící data filmu ({e})", addon.getAddonInfo('name'))
        return False
    except Exception as e:
        ui.popinfo(f"Chyba při vytváření .strm souboru: {e}", addon.getAddonInfo('name'))
        return False

def _create_series_strms(series_data, addon):
    """Vytvoří strukturu složek a .strm souborů pro seriál, pokud je nastavena složka knihovny."""
    series_folder = addon.getSetting('library_series_folder')
    if not series_folder or not os.path.isdir(series_folder):
        if series_folder:
            ui.popinfo(f"Nastavená složka pro seriály neexistuje: {series_folder}", addon.getAddonInfo('name'))
        else:
            ui.popinfo("Složka pro seriály není nastavena v integraci knihovny.", addon.getAddonInfo('name'))
        return False

    try:
        # import requests # Již importováno na začátku souboru
        # API klíč získáváme z centrální funkce
        series_id = series_data['id']
        series_name = series_data['name']
        # Použijeme sanitize_filename z modulu utils
        sanitized_series_name = utils.sanitize_filename(series_name)
        series_path = os.path.join(series_folder, sanitized_series_name)

        # Získání detailů seriálu včetně sezón (znovu, pro jistotu aktuálních dat)
        series_details_url = f'https://api.themoviedb.org/3/tv/{series_id}?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
        series_response = requests.get(series_details_url)
        series_data_full = series_response.json()
        seasons = series_data_full.get('seasons', [])

        os.makedirs(series_path, exist_ok=True)


        episode_count = 0
        for season in seasons:
            season_number = season.get('season_number')
            if season_number is None or season_number == 0: continue

            season_path = os.path.join(series_path, f"Season {season_number:02d}")
            os.makedirs(season_path, exist_ok=True)

            season_details_url = f'https://api.themoviedb.org/3/tv/{series_id}/season/{season_number}?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
            season_response = requests.get(season_details_url)
            season_data = season_response.json()
            episodes = season_data.get('episodes', [])

            for ep in episodes:
                ep_num = ep.get('episode_number')
                ep_name = ep.get('name', f'Epizoda {ep_num}')
                if ep_num is None: continue

                # Použijeme sanitize_filename z modulu utils
                sanitized_ep_name = utils.sanitize_filename(ep_name)
                strm_filename = utils.sanitize_filename(f"{sanitized_series_name} S{season_number:02d}E{ep_num:02d} {sanitized_ep_name}.strm")
                strm_filepath = os.path.join(season_path, strm_filename)
                
                # OPRAVA: Předání handle do play_episode_from_library není možné zde
                strm_content = f"plugin://{addon.getAddonInfo('id')}?action=play_episode_from_library&series_name={quote_plus(series_name)}&season={season_number}&episode={ep_num}&tmdb_id={series_id}"

                with open(strm_filepath, 'w', encoding='utf-8') as f:
                    f.write(strm_content)
                episode_count += 1
        
        # Formátovaný string s názvem seriálu
        popup_message = addon.getLocalizedString(30132).replace('%s', str(series_name))
        ui.popinfo(popup_message, addon.getAddonInfo('name'))
        # --- Automatická aktualizace knihovny ---
        if addon.getSettingBool('library_auto_update'):
            def delayed_refresh():
                xbmc.sleep(500)
                if not xbmc.getCondVisibility('Library.IsScanningVideo'):
                    xbmc.executebuiltin(f'UpdateLibrary(video, "{series_folder}")')
            threading.Thread(target=delayed_refresh).start()
        # -------------------------------------

    except KeyError as e:
        ui.popinfo(f"Chyba při vytváření .strm pro seriál: Chybějící data ({e})", addon.getAddonInfo('name'))
        return False
    except Exception as e:
        ui.popinfo(f"Chyba při vytváření .strm pro seriál: {e}", addon.getAddonInfo('name'))
        return False

# --- Funkce pro obsluhu akcí ---

def check_settings_and_add_item(params, addon):
    """
    Zkontroluje nastavení knihovny a pokud je v pořádku,
    zavolá akci pro přidání filmu/seriálu. Jinak zobrazí dialog pro výběr složky.
    """
    import xbmcgui
    media_type = params.get('media_type')
    if not media_type:
        ui.popinfo("Chyba: Chybí typ média pro přidání do knihovny.", addon.getAddonInfo('name'))
        return
    folder = ""
    folder_setting_name = ""
    item_id = ""
    id_param_name = ""
    folder_label = ""
    if media_type == 'movie':
        folder_setting_name = 'library_movies_folder'
        id_param_name = 'movie_id'
        folder = addon.getSetting(folder_setting_name)
        item_id = params.get(id_param_name)
        folder_label = "filmů"
    elif media_type == 'series':
        folder_setting_name = 'library_series_folder'
        id_param_name = 'series_id'
        folder = addon.getSetting(folder_setting_name)
        item_id = params.get(id_param_name)
        folder_label = "seriálů"
    else:
        ui.popinfo(f"Neznámý typ média: {media_type}", addon.getAddonInfo('name'))
        return
    if not item_id:
        ui.popinfo(f"Chyba: Chybí ID pro přidání do knihovny ({id_param_name}).", addon.getAddonInfo('name'))
        return
    # Kontroly nastavení
    if not folder or not os.path.isdir(folder):
        from wcs.login.DialogLibraryFolder import show_library_folder_dialog
        # show_library_folder_dialog vrací True pokud uživatel úspěšně vybral složku
        if show_library_folder_dialog(addon, media_type):
            # Po úspěšném výběru složky pokračuj v akci
            check_settings_and_add_item(params, addon)
        return
    # Vše je v pořádku, spusť akci přidání
    if media_type == 'movie':
        add_movie_to_library(params, addon)
    elif media_type == 'series':
        add_series_to_library(params, addon)
    
def add_movie_to_library(params, addon):
    """Zpracuje akci 'library_add_movie'. Získá data z TMDb a vytvoří .strm."""
    movie_id = params.get('movie_id')
    if not movie_id:
        ui.popinfo("Chybí ID filmu pro přidání do knihovny.", addon.getAddonInfo('name'))
        return

    try:
        # Získání detailů filmu z TMDb API
        # Předpokládáme, že TMDb API klíč je dostupný (buď globálně nebo přes addon settings)
        movie_details_url = f'https://api.themoviedb.org/3/movie/{movie_id}?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
        response = requests.get(movie_details_url)
        movie_data = response.json() # Získaná data z TMDb
        
        # Přejmenování klíčů, aby odpovídaly očekávání _create_movie_strm
        # 'title' je již přítomno
        movie_data['year'] = movie_data.get('release_date', '')[:4] 
        
        # Volání funkce pro vytvoření .strm souboru
        _create_movie_strm(movie_data, addon)
        
    except Exception as e:
        from wcs.metadata.TMDbClient import handle_tmdb_api_error
        handle_tmdb_api_error(e, addon)


def add_series_to_library(params, addon):
    """Zpracuje akci 'library_add_series'. Získá data z TMDb a vytvoří .strm soubory."""
    series_id = params.get('series_id')
    if not series_id:
        ui.popinfo("Chybí ID seriálu pro přidání do knihovny.", addon.getAddonInfo('name'))
        return

    try:
        # Získání detailů seriálu z TMDb API (stačí základní data, _create_series_strms si načte zbytek)
        series_details_url = f'https://api.themoviedb.org/3/tv/{series_id}?api_key={get_tmdb_api_key(addon)}&language=cs-CZ'
        response = requests.get(series_details_url)
        series_data = response.json() # Získaná data z TMDb

        # Volání funkce pro vytvoření .strm souborů
        # series_data zde musí obsahovat 'id' a 'name'
        if 'id' in series_data and 'name' in series_data:
             _create_series_strms(series_data, addon)
        else:
             ui.popinfo(f"Nepodařilo se získat potřebná data pro seriál ID: {series_id}", addon.getAddonInfo('name'))

    except Exception as e:
        from wcs.metadata.TMDbClient import handle_tmdb_api_error
        handle_tmdb_api_error(e, addon)

def play_episode_from_library(params, addon, handle):
    """Zpracuje akci 'play_episode_from_library'. Deleguje na resolve_episode_from_library."""
    # Tato funkce nyní pouze deleguje na funkci, která řeší URL
    resolve_episode_from_library(params, addon, handle)

# --- Funkce pro přehrávání z knihovny ---

# Funkce pro nalezení a vyřešení filmu z knihovny
def play_movie_from_library(params, addon, handle):
    """
    Zpracuje přehrání filmu z knihovny.
    Pokud uživatel není přihlášen, zobrazí dialog a skončí.
    Jinak se pokusí film přehrát.
    """
    def login_success_callback():
        # Po úspěšném přihlášení pouze zobrazíme notifikaci.
        # Uživatel musí přehrávání spustit znovu.
        from wcs.ui import UI as ui

    if not user_data.check_providers_and_prompt(addon, on_success_callback=login_success_callback):
        # Pokud se zobrazil dialog, skončíme. Callback se postará o notifikaci.
        xbmcplugin.setResolvedUrl(handle, False, xbmcgui.ListItem())
        return

    # Pokud je uživatel přihlášen, pokračujeme v původní logice přehrávání
    title = params.get('title')
    year = params.get('year')

    if not title:
        ui.popinfo("Chybí název filmu pro přehrání z knihovny.", addon.getAddonInfo('name'))
        xbmcplugin.setResolvedUrl(handle, False, xbmcgui.ListItem())
        return

    _activate_movie_info_only(title, year)

    from wcs.utils import resolve_movie_for_library
    resolve_movie_for_library(title, year, addon, handle)

def resolve_episode_from_library(params, addon, handle):
    """
    Zpracuje přehrání epizody z knihovny.
    Pokud uživatel není přihlášen, zobrazí dialog a skončí.
    Jinak se pokusí epizodu přehrát.
    """
    def login_success_callback():
        # Po úspěšném přihlášení pouze zobrazíme notifikaci.
        # Uživatel musí přehrávání spustit znovu.
        from wcs.ui import UI as ui

    if not user_data.check_providers_and_prompt(addon, on_success_callback=login_success_callback):
        # Pokud se zobrazil dialog, skončíme. Callback se postará o notifikaci.
        xbmcplugin.setResolvedUrl(handle, False, xbmcgui.ListItem())
        return

    # Pokud je uživatel přihlášen, pokračujeme v původní logice přehrávání
    series_name = params.get('series_name')
    season_num_str = params.get('season')
    episode_num_str = params.get('episode')
    autoplay_request = params.get('autoplay_request') == 'true'

    if not all([series_name, season_num_str, episode_num_str]):
        ui.popinfo("Chybí údaje o seriálu/epizodě pro přehrání z knihovny.", addon.getAddonInfo('name'))
        xbmcplugin.setResolvedUrl(handle, False, xbmcgui.ListItem())
        return
    
    try:
        season = int(season_num_str)
        episode = int(episode_num_str)
    except ValueError:
        ui.popinfo("Neplatné číslo sezóny nebo epizody.", addon.getAddonInfo('name'))
        xbmcplugin.setResolvedUrl(handle, False, xbmcgui.ListItem())
        return

    # Získání metadat epizody z knihovny
    from wcs.library.LibraryManager import get_episode_metadata_from_library
    episode_metadata = get_episode_metadata_from_library(series_name, season, episode) or {}
    # Sestav název pro dialog: Název seriálu – Název epizody (pokud existuje)
    if episode_metadata.get('episode_title'):
        movie_title = f"{episode_metadata.get('series_title', series_name)} – {episode_metadata['episode_title']}"
    else:
        movie_title = episode_metadata.get('series_title', series_name)

    search_str = f"{series_name} S{season:02d}E{episode:02d}"

    if autoplay_request:
        xbmc.log(f"[plugin.video.milionar] Library Autoplay: AUTOPLAY REQUEST DETECTED", level=xbmc.LOGINFO)

    from wcs.ui.dialogs.DialogWcsSearch import show_search_dialog

    selected_file = show_search_dialog(
        search_query=search_str,
        movie_title=movie_title,
        movie_year=episode_metadata.get('episode_year', ''),
        poster_url=episode_metadata.get('series_poster', ''),
        fanart_url=episode_metadata.get('series_fanart', ''),
        genre=episode_metadata.get('series_genre', ''),
        rating=episode_metadata.get('episode_rating', ''),
        runtime=episode_metadata.get('episode_runtime', ''),
        plot=episode_metadata.get('episode_plot', '') or episode_metadata.get('series_plot', ''),
        search_results=[],
        is_episode=True,
        series_name=series_name,
        season_number=season,
        episode_number=episode,
        episode_title=episode_metadata.get('episode_title', ''),
        tmdb_id=params.get('tmdb_id'),
        autoplay_request=autoplay_request,
        from_library=True
    )
    
    if selected_file and selected_file.get('ident'):
        ident = selected_file['ident']
        
        if autoplay_request:
            # Nastavení informací o epizodě pro autoplay
            from wcs.playback.DialogMediaInfoWithGame import get_autoplay_manager
            autoplay_manager = get_autoplay_manager()
            autoplay_manager.set_autoplay_enabled(True)
            
            from wcs.utils import extract_quality
            selected_quality = extract_quality(selected_file.get('name', ''))
            
            autoplay_manager.set_current_episode(
                series_name, season, episode, params.get('tmdb_id'), selected_quality
            )
            
            _activate_singleton_monitoring("Library Autoplay")
            selected_filename = selected_file.get('name', '')
            xbmc.log(f"[plugin.video.milionar] Library Autoplay: Selected file: {selected_filename}", level=xbmc.LOGINFO)
        else:
            from wcs.playback.DialogMediaInfoWithGame import get_autoplay_manager
            autoplay_manager = get_autoplay_manager()
            
            global_autoplay_enabled = addon.getSettingBool('autoplay_enabled_global')
            
            if global_autoplay_enabled:
                autoplay_manager.set_autoplay_enabled(True)
                
                from wcs.utils import extract_quality
                selected_quality = extract_quality(selected_file.get('name', ''))
                
                autoplay_manager.set_current_episode(
                    series_name, season, episode, params.get('tmdb_id'), selected_quality
                )
                try:
                    addon.setSetting('temp_autoplay_active', 'true')
                    xbmc.log(f"[plugin.video.milionar] Library: Activated autoplay for {series_name} S{season:02d}E{episode:02d}, quality: {selected_quality}", level=xbmc.LOGINFO)
                except Exception as e:
                    xbmc.log(f"[plugin.video.milionar] Library: Error setting temp_autoplay_active: {e}", level=xbmc.LOGWARNING)
                    
                _activate_singleton_monitoring("Library Manual")
            else:
                _activate_episode_info_only(series_name, season, episode)
        
        token = webshare_api.revalidate(addon, ui.popinfo)
        if token:
            # Před přehráním z knihovny: zkontroluj VIP expiraci s varováním
            try:
                webshare_api.refresh_vip_info(addon, token_override=token, show_notification=False, log_context="library_episode", show_warning=True)
            except Exception:
                pass
            link = webshare_api.getlink(ident, token, dtype='video_stream', addon=addon)
            if link:
                # Přidání do historie naposledy přehraných - řeší volající (utils.search_and_play_episode)
                # nebo pokud voláme přímo play_episode_from_library z kontextu kde není utils
                # V našem novém designu MyTV volá utils, takže zde to není třeba dublovat.
                # Ponecháme jen logování nebo nic.

                listitem = xbmcgui.ListItem(path=link)
                listitem.setProperty('IsPlayable', 'true')
                xbmcplugin.setResolvedUrl(handle, True, listitem)
                return
        
    xbmcplugin.setResolvedUrl(handle, False, xbmcgui.ListItem()) 

def get_episode_metadata_from_library(series_name, season_number, episode_number):
    """
    Získá metadata epizody z Kodi knihovny pomocí JSON-RPC API.
    
    Args:
        series_name (str): Název seriálu
        season_number (int): Číslo sezóny
        episode_number (int): Číslo epizody
        
    Returns:
        dict: Metadata epizody nebo None pokud epizoda nebyla nalezena
    """
    import xbmc
    xbmc.log(f"[plugin.video.milionar] Library metadata: series='{series_name}', season={season_number}, episode={episode_number}", level=xbmc.LOGINFO)
    try:
        # Nejprve najdeme TV show podle názvu
        shows_query = {
            "jsonrpc": "2.0",
            "method": "VideoLibrary.GetTVShows",
            "params": {
                "properties": ["title", "year", "rating", "genre", "plot"],
                "filter": {
                    "field": "title",
                    "operator": "contains",
                    "value": series_name
                }
            },
            "id": 1
        }
        
        shows_response = xbmc.executeJSONRPC(json.dumps(shows_query))
        shows_data = json.loads(shows_response)
        
        if 'result' not in shows_data or 'tvshows' not in shows_data['result']:
            xbmc.log(f"[plugin.video.milionar] Library: TV show '{series_name}' not found in library (no results)", level=xbmc.LOGINFO)
            return None
            
        shows_list = shows_data['result']['tvshows']
        xbmc.log(f"[plugin.video.milionar] Library: Found {len(shows_list)} potential candidates for '{series_name}'", level=xbmc.LOGINFO)

        # Najdeme nejlepší shodu podle názvu
        best_match = None
        best_score = 0
        
        for show in shows_list:
            # Jednoduchá podobnost podle názvu
            title = show.get('title', '').lower()
            search_name = series_name.lower()
            
            xbmc.log(f"[plugin.video.milionar] Library: Checking candidate '{title}' against '{search_name}'", level=xbmc.LOGDEBUG)
            
            if search_name in title or title in search_name:
                score = len(search_name) if search_name in title else len(title)
                # Prefer exact match
                if title == search_name:
                    score += 100
                    
                if score > best_score:
                    best_score = score
                    best_match = show
        
        if not best_match:
            xbmc.log(f"[plugin.video.milionar] Library: No matching TV show found for '{series_name}'. Candidates were: {[s.get('title') for s in shows_list]}", level=xbmc.LOGINFO)
            return None
        tv_show_id = best_match['tvshowid']
        xbmc.log(f"[plugin.video.milionar] Library: best_match for show: {json.dumps(best_match)}", level=xbmc.LOGINFO)
        
        # Nyní získáme epizody pro danou sezónu
        episodes_query = {
            "jsonrpc": "2.0",
            "method": "VideoLibrary.GetEpisodes",
            "params": {
                "tvshowid": tv_show_id,
                "season": season_number,
                "properties": ["title", "season", "episode", "rating", "firstaired", "plot", "art", "playcount", "streamdetails"],
            },
            "id": 2
        }
        
        episodes_response = xbmc.executeJSONRPC(json.dumps(episodes_query))
        episodes_data = json.loads(episodes_response)
        
        if 'result' not in episodes_data or 'episodes' not in episodes_data['result']:
            xbmc.log(f"[plugin.video.milionar] DEBUG: No episodes found for {series_name} season {season_number}", level=xbmc.LOGINFO)
            return None
            
        # Najdeme požadovanou epizodu
        for episode in episodes_data['result']['episodes']:
            if episode.get('episode') == episode_number:
                # Výpočet délky epizody v minutách
                duration_sec = episode.get('streamdetails', {}).get('video', [{}])[0].get('duration', 0)
                runtime_minutes = int(duration_sec) // 60 if duration_sec else 66
                episode_runtime = f"{runtime_minutes} min"
                art = episode.get('art', {})
                series_poster = art.get('poster', '') or art.get('tvshow.poster', '') or art.get('thumb', '')
                series_fanart = art.get('fanart', '') or art.get('tvshow.fanart', '')
                episode_thumb = art.get('thumb', '') or art.get('tvshow.poster', '') or art.get('poster', '')
                metadata = {
                    'series_title': best_match.get('title', series_name),
                    'series_poster': series_poster,
                    'series_fanart': series_fanart,
                    'series_genre': ', '.join(best_match.get('genre', [])) if best_match.get('genre') else '',
                    'series_plot': best_match.get('plot', ''),
                    'episode_title': episode.get('title', f'Epizoda {episode_number}'),
                    'episode_plot': episode.get('plot', ''),
                    'episode_year': episode.get('firstaired', '')[:4] if episode.get('firstaired') else '',
                    'episode_runtime': episode_runtime,
                    'episode_rating': episode.get('rating', 0),
                    'episode_thumb': episode_thumb,
                    'season_number': season_number,
                    'episode_number': episode_number,
                }
                xbmc.log(f"[plugin.video.milionar] Library: Found episode metadata for {series_name} S{season_number:02d}E{episode_number:02d}", level=xbmc.LOGINFO)
                return metadata
                
        xbmc.log(f"[plugin.video.milionar] Library: Episode {episode_number} not found in season {season_number} of {series_name}", level=xbmc.LOGINFO)
        return None
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Library: Error getting episode metadata: {e}", level=xbmc.LOGERROR)
        import traceback
        traceback.print_exc()
        return None

def get_next_episode_metadata_from_library(series_name, current_season, current_episode):
    """
    Získá metadata následující epizody z Kodi knihovny.
    
    Args:
        series_name (str): Název seriálu
        current_season (int): Aktuální sezóna
        current_episode (int): Aktuální epizoda
        
    Returns:
        dict: Metadata následující epizody nebo None pokud neexistuje
    """
    # Nejprve zkusíme další epizodu ve stejné sezóně
    next_episode_metadata = get_episode_metadata_from_library(series_name, current_season, current_episode + 1)
    
    if next_episode_metadata:
        return next_episode_metadata
        
    # Pokud není další epizoda ve stejné sezóně, zkusíme první epizodu další sezóny
    next_season_metadata = get_episode_metadata_from_library(series_name, current_season + 1, 1)
    
    return next_season_metadata 

def _activate_singleton_monitoring(source):
    """
    Aktivuje singleton monitoring pro daný zdroj.
    
    Args:
        source (str): Zdroj aktivace (např. "Library Auto" nebo "Library Manual")
    """
    def activate_monitoring():
        import time
        from wcs.playback.PlayerMonitor import get_player_monitor
        
        # Počkat na začátek přehrávání
        for i in range(100):  # 10s timeout
            if xbmc.Player().isPlaying():
                xbmc.log(f"[plugin.video.milionar] {source}: Playback detected, activating singleton monitoring", level=xbmc.LOGINFO)
                break
            time.sleep(0.1)
        else:
            xbmc.log(f"[plugin.video.milionar] {source}: Timeout waiting for playbook", level=xbmc.LOGWARNING)
            return
        
        # Získat singleton player monitor
        monitor = get_player_monitor()
        
        # Nastavit speciální flag aby se zabránilo duplicitnímu spuštění přes onPlayBackStarted
        monitor._manual_activation = True
        
        # Simulovat callback onPlayBackStarted s nastaveným temp_autoplay_active
        # temp_autoplay_active už je nastaveno v library.py před voláním této funkce
        time.sleep(1)  # Zkráceno na 1 sekundu pro rychlejší zobrazení
        
        if xbmc.Player().isPlaying():
            # Přímo zavolat _handle_test_playback_start pro aktivaci monitoring
            monitor._handle_test_playback_start()
        
    # Spustit aktivaci monitoring v threadu
    import threading
    monitoring_thread = threading.Thread(target=activate_monitoring, daemon=True)
    monitoring_thread.start()
    xbmc.log(f"[plugin.video.milionar] {source}: Started monitoring activation thread", level=xbmc.LOGINFO)

def _activate_episode_info_only(series_name, season, episode):
    """Aktivuje zobrazení informací o epizodě bez aktivace autoplay."""
    from wcs.playback.metadata import MediaMetadata
    from wcs.playback.dialog_scheduler import schedule_library_info_dialog

    fallback = MediaMetadata(
        media_type='series',
        title=series_name,
        series_name=series_name,
        episode_title=f"Epizoda {episode}",
        season_number=season,
        episode_number=episode,
    )
    schedule_library_info_dialog(
        metadata_fetcher=lambda: get_episode_metadata_from_library(series_name, season, episode),
        fallback_metadata=fallback,
        source='library_episode',
    )

def _activate_movie_info_only(movie_title, movie_year):
    """Aktivuje zobrazení informací o filmu bez dalších funkcí."""
    from wcs.playback.metadata import MediaMetadata
    from wcs.playback.dialog_scheduler import schedule_library_info_dialog

    fallback = MediaMetadata(
        media_type='movie',
        title=movie_title,
        year=movie_year,
    )
    schedule_library_info_dialog(
        metadata_fetcher=lambda: get_movie_metadata_from_library(movie_title, movie_year),
        fallback_metadata=fallback,
        source='library_movie',
    )

def get_movie_metadata_from_library(movie_title, movie_year):
    """
    Získá metadata filmu z Kodi knihovny pomocí JSON-RPC API.
    
    Args:
        movie_title (str): Název filmu
        movie_year (str): Rok filmu
        
    Returns:
        dict: Metadata filmu nebo None pokud film nebyl nalezen
    """
    try:
        # Najdeme film podle názvu a roku
        movies_query = {
            "jsonrpc": "2.0",
            "method": "VideoLibrary.GetMovies",
            "params": {
                "properties": ["title", "year", "rating", "genre", "plot", "director", "runtime", "art", "playcount"],
                "filter": {
                    "field": "title",
                    "operator": "contains",
                    "value": movie_title
                }
            },
            "id": 1
        }
        
        movies_response = xbmc.executeJSONRPC(json.dumps(movies_query))
        movies_data = json.loads(movies_response)
        
        if 'result' not in movies_data or 'movies' not in movies_data['result']:
            xbmc.log(f"[plugin.video.milionar] Library: Movie '{movie_title}' not found in library", level=xbmc.LOGINFO)
            return None
            
        # Najdeme nejlepší shodu podle názvu a roku
        best_match = None
        best_score = 0
        
        for movie in movies_data['result']['movies']:
            # Podobnost podle názvu
            title = movie.get('title', '').lower()
            search_title = movie_title.lower()
            
            title_match = search_title in title or title in search_title
            
            # Bonus za shodu roku
            year_match = str(movie.get('year', '')) == movie_year if movie_year else True
            
            if title_match:
                score = len(search_title) if search_title in title else len(title)
                if year_match:
                    score += 1000  # Vysoký bonus za shodu roku
                    
                if score > best_score:
                    best_score = score
                    best_match = movie
        
        if not best_match:
            xbmc.log(f"[plugin.video.milionar] Library: No matching movie found for '{movie_title}' ({movie_year})", level=xbmc.LOGINFO)
            return None
            
        # Sestavíme metadata
        runtime_seconds = best_match.get('runtime', 0)
        runtime_minutes = int(runtime_seconds / 60) if runtime_seconds > 0 else 0  # Konverze ze sekund na minuty
        
        metadata = {
            'title': best_match.get('title', movie_title),
            'year': str(best_match.get('year', movie_year)),
            'rating': best_match.get('rating', 0),
            'plot': best_match.get('plot', ''),
            # Poster: preferuj 'poster', fallback na 'thumb'
            'poster': best_match.get('art', {}).get('poster', '') or best_match.get('art', {}).get('thumb', ''),
            'thumb': best_match.get('art', {}).get('thumb', ''),
            'fanart': best_match.get('art', {}).get('fanart', ''),
            'genre': ', '.join(best_match.get('genre', [])) if best_match.get('genre') else '',
            'director': ', '.join(best_match.get('director', [])) if best_match.get('director') else '',
            'runtime': f"{runtime_minutes} min" if runtime_minutes > 0 else '',
            'tmdb_id': best_match.get('imdbnumber', '').replace('tmdb/', '') if best_match.get('imdbnumber', '').startswith('tmdb/') else ''
        }
        
        xbmc.log(f"[plugin.video.milionar] Library: Found movie metadata for {movie_title} ({movie_year})", level=xbmc.LOGINFO)
        return metadata
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Library: Error getting movie metadata: {e}", level=xbmc.LOGERROR)
        import traceback
        traceback.print_exc()
        return None 