# wcs/ai/DialogMyMovies.py
"""
Dialog "Moje filmy" - Fullscreen dialog pro uložené filmy uživatele.
Dědí z AIChatRecommendationDialog a sdílí jeho XML grafiku.
"""

import xbmc
import xbmcgui
import xbmcaddon
import threading
from urllib.parse import quote_plus

from wcs.ai.DialogAIChatRecommendation import AIChatRecommendationDialog, AIRecommendationPromptBuilder
from wcs import user_data
from wcs import utils as wcs_utils


class MyMoviesDialog(AIChatRecommendationDialog):
    """
    Dialog pro uložené filmy uživatele.
    Dědí všechny funkce z AI Chat dialogu, ale přetěžuje:
    - onInit() - načítá z user_data místo AI inference
    - _populate_preview_actions() - upravené akční tlačítka (Přehrát, Detaily, Podobné, Smazat)
    """
    
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        # Parent zpracovává show_nav_sidebar, nav_position, context
        super(MyMoviesDialog, self).__init__(xml_filename, addon_path, *args, **kwargs)
        self._movie_items = []  # Cache pro položky filmů
        self._current_item_index = 0  # Index aktuální položky pro akce
        # Identifikátor sekce pro navigační sidebar
        self._nav_section_id = 'my_movies'


    
    def onInit(self):
        """Přetížená inicializace - načte položky z uložených filmů."""
        # Resetovat stav (využívá parent metodu)
        self._reset_state()
        self.media_type = 'movie'
        self._is_showing_collection = True  # Flag pro rozlišení mezi sbírkou a AI doporučeními
        
        # Načíst filmy
        self._movie_items = user_data.load_tmdb_movies(self.addon)
        
        # Nastavit titulek
        self.setProperty('WCS.AIChat.Title', 'Moje filmy')
        self.setProperty('WCS.AIChat.Status', 'Připraven')
        
        if not self._movie_items:
            # Prázdná sbírka - zobrazit sidebar s možností přidat
            self.append_chat_message("AI", "Vaše sbírka filmů je prázdná. Přidejte první film pomocí tlačítka níže.")
            self._setup_empty_state_buttons()
            # Zobrazit chat sidebar
            self.setProperty('WCS.AIChat.Visible', 'true')
            self.setFocusId(4000)
            return
        
        # Zobrazit uvítací zprávu v chatu
        self.append_chat_message("AI", f"Ve vaší sbírce je {len(self._movie_items)} filmů. Vyberte film nebo si nechte doporučit podobné.")
        
        # Nastavit AI tlačítka
        self._setup_initial_buttons()
        
        # Naplnit poster flow z filmů
        self._populate_from_items()
        
        # Pokud byl dialog spuštěn z navigačního sidebaru, zobrazit ho bez animace
        if self._show_nav_on_init:
            self._show_nav_sidebar(animate=False)
        else:
            # Skrýt chat sidebar, fokus na postery s krátkým zpožděním
            self.clearProperty('WCS.AIChat.Visible')
            self._delayed_focus(2000, 0.5)

    
    def _setup_empty_state_buttons(self):
        """Nastaví tlačítka pro prázdný stav."""
        buttons = [
            {"label": "Přidat film", "value": "add_item"},
            {"label": "Doporučit nový film", "value": "recommend_new"}
        ]
        self.update_buttons(buttons)
    
    def _setup_initial_buttons(self):
        """Nastaví počáteční AI tlačítka."""
        buttons = [
            {"label": "Přidat film", "value": "add_item"},
            {"label": "Doporučit na základě sbírky", "value": "recommend_from_collection"},
            {"label": "Doporučit nový film", "value": "recommend_new"}
        ]
        self.update_buttons(buttons)
    
    def _populate_from_items(self):
        """Naplní poster flow položkami z filmů."""
        list_control = self.getControl(2000)
        list_control.reset()
        
        for item in self._movie_items:
            title = item.get('title') or item.get('name', '')
            year = item.get('year', '')
            poster_path = item.get('poster_path', '')
            poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
            tmdb_id = item.get('id', '')
            
            display_title = f"{title} ({year})" if year else title
            
            list_item = xbmcgui.ListItem(label=display_title)
            
            if poster:
                list_item.setArt({'poster': poster})
            
            # Uložit metadata pro akce
            list_item.setProperty('tmdb_id', str(tmdb_id) if tmdb_id else '')
            list_item.setProperty('media_type', 'movie')
            list_item.setProperty('title', title)
            list_item.setProperty('year', str(year) if year else '')
            
            list_control.addItem(list_item)
        
        # Pokud máme položky, nastavit preview pro první
        if list_control.size() > 0:
            self._update_preview_for_current_item()
    
    def _populate_preview_actions(self, media_type, tmdb_id, title, year):
        """Přetíženo - upravené akční tlačítka pro Moje filmy (Unified)."""
        try:
            list_control = self.getControl(6100)
            list_control.reset()
            
            # Získat aktuální index
            poster_list = self.getControl(2000)
            current_index = poster_list.getSelectedPosition()
            
            # Rozlišit mezi položkami ze sbírky a AI doporučeními
            is_collection_item = getattr(self, '_is_showing_collection', False) and current_index >= 0 and current_index < len(self._movie_items)
            
            # 1. URČIT KONTEXT
            if is_collection_item:
                context = 'col_movie'
            else:
                context = 'ai_movie' # Standardni AI doporuceni
            
            # 2. ZISKAT TLACITKA Z REGISTRU
            items = self.build_action_items(context)
            
            # 3. DOPLNIT DATA
            for item in items:
                item.setProperty('tmdb_id', str(tmdb_id))
                item.setProperty('title', title)
                item.setProperty('year', year)
                item.setProperty('media_type', media_type)
                item.setProperty('collection_index', str(current_index))
                
                list_control.addItem(item)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] MyMovies preview actions error: {e}", xbmc.LOGERROR)
    
    def _handle_preview_action_click(self):
        """Přetíženo - zpracování akcí specifických pro Moje filmy."""
        try:
            # Reset automatického zobrazení tlačítek (jakákoliv akce ukončí delay)
            self.clearProperty('WCS.AIPreview.ShowActions')
            
            list_control = self.getControl(6100)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            action = item.getProperty('action')
            tmdb_id = item.getProperty('tmdb_id')
            title = item.getProperty('title')
            year = item.getProperty('year')
            media_type = item.getProperty('media_type')
            collection_index = item.getProperty('collection_index')
            
            if action == 'remove_from_list':
                self._handle_remove_from_list(collection_index)
            elif action == 'play':
                self._handle_play_movie(collection_index)
            elif action == 'similar':
                self._handle_similar(title, year)
            else:
                # Delegovat na parent pro standardní akce (details, add_list, atd.)
                super(MyMoviesDialog, self)._handle_preview_action_click()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] MyMovies action error: {e}", xbmc.LOGERROR)
    
    def _handle_similar(self, title, year):
        """Spustí AI doporučení podobných filmů."""
        try:
            self.media_type = 'movie'
            inspiration_text = f"Film '{title}' ({year}) mě zaujal. Chci zažít něco podobného. Nemusí to být stejný žánr, ale ať to má podobnou atmosféru."
            
            # Vymazat starý kontext (jde o prostý dotaz bez kontextu)
            self.context = None
            
            # Reinicializovat prompt builder s novým media_type a bez kontextu
            self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
            
            # Skrýt overlay a fokus na chat
            self._clear_preview_properties()
            self.setFocusId(4000)
            
            # Spustit načítací animaci
            self._start_button_loading("Hledám podobné...")
            self.handle_user_input(inspiration_text)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Similar action error: {e}", xbmc.LOGERROR)
    
    def fetch_and_display_recommendations(self, recommendations):
        """Přetíženo - nastaví flag že už nezobrazujeme sbírku."""
        # Po AI doporučení už nezobrazujeme sbírku
        self._is_showing_collection = False
        # Volat parent implementaci
        super(MyMoviesDialog, self).fetch_and_display_recommendations(recommendations)
    
    def _handle_no_recommendations(self):
        """Přetíženo - obnoví původní tlačítka když AI nevrátí nic."""
        self.append_system_message("Model nedoporučil žádné tituly.")
        # Obnovit původní tlačítka
        self._setup_initial_buttons()

    
    def _handle_play_movie(self, collection_index):
        """Přehraje film ze sbírky."""
        try:
            idx = int(collection_index) if collection_index else 0
            if idx < 0 or idx >= len(self._movie_items):
                return
            
            movie_item = self._movie_items[idx]
            
            from wcs.utils import play_movie_from_addon
            
            poster_path = movie_item.get('poster_path', '')
            poster = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
            
            # Sestavit meta slovník
            meta = {
                'title': movie_item.get('title') or movie_item.get('name', ''),
                'year': movie_item.get('year', ''),
                'plot': movie_item.get('overview', ''),
                'poster': poster,
                'fanart': '',
                'rating': '',
                'genre': '',
                'director': '',
                'runtime': '',
                'tmdb_id': movie_item.get('id'),
            }
            
            # Zavřít dialog před přehráváním
            self.close()
            
            # Spustit klasické přehrávání
            play_movie_from_addon(meta, self.addon)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Play movie error: {e}", xbmc.LOGERROR)
    
    def _handle_remove_from_list(self, collection_index):
        """Smaže položku ze sbírky."""
        try:
            idx = int(collection_index) if collection_index else 0
            if idx < 0 or idx >= len(self._movie_items):
                return
            
            removed_item = self._movie_items[idx]
            item_title = removed_item.get('title') or removed_item.get('name', 'Neznámý')
            
            # Potvrzovací dialog
            dialog = xbmcgui.Dialog()
            if not dialog.yesno("Smazat ze sbírky", f"Opravdu chcete smazat '{item_title}' ze sbírky?"):
                return
            
            # Smazat položku
            self._movie_items.pop(idx)
            user_data.store_tmdb_movies(self.addon, self._movie_items)
            
            # Notifikace
            xbmcgui.Dialog().notification(
                self.addon.getAddonInfo('name'),
                f"'{item_title}' smazáno ze sbírky",
                xbmcgui.NOTIFICATION_INFO
            )
            
            # Obnovit poster flow
            if self._movie_items:
                self._populate_from_items()
            else:
                # Sbírka je prázdná - přepnout na prázdný stav
                self.getControl(2000).reset()
                self._setup_empty_state_buttons()
                self.setProperty('WCS.AIChat.Visible', 'true')
                self.setFocusId(4000)
                self.append_chat_message("AI", "Vaše sbírka filmů je prázdná. Přidejte první film pomocí tlačítka níže.")
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Remove from list error: {e}", xbmc.LOGERROR)
    
    def _handle_add_movie(self):
        """Otevře dialog pro přidání nového filmu - BEZ zavření našeho dialogu."""
        try:
            import requests
            from wcs.metadata.TMDbClient import get_tmdb_api_key
            
            # 1. Zobrazit klávesnici pro vyhledávání
            search_query = xbmcgui.Dialog().input(
                self.addon.getLocalizedString(30152),  # "Hledat film"
                type=xbmcgui.INPUT_ALPHANUM
            )
            
            if not search_query:
                return  # Uživatel zrušil
            
            # 2. Vyhledat na TMDb
            try:
                url = f'https://api.themoviedb.org/3/search/movie?api_key={get_tmdb_api_key()}&language=cs-CZ&query={quote_plus(search_query)}'
                response = requests.get(url, timeout=10)
                response.raise_for_status()
                results = response.json().get('results', [])
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] TMDb search error: {e}", xbmc.LOGERROR)
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    "Chyba při vyhledávání",
                    xbmcgui.NOTIFICATION_ERROR
                )
                return
            
            if not results:
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    f"Nenalezen žádný film: {search_query}",
                    xbmcgui.NOTIFICATION_INFO
                )
                return
            
            # 3. Zobrazit výběrový dialog
            choices = [f"{item['title']} ({item.get('release_date', '')[:4]})" for item in results]
            selected_index = xbmcgui.Dialog().select(
                self.addon.getLocalizedString(30154),  # "Vyberte film"
                choices
            )
            
            if selected_index < 0:
                return  # Uživatel zrušil
            
            selected_item = results[selected_index]
            
            # 4. Zkontrolovat, zda film už není ve sbírce
            if any(m.get('id') == selected_item['id'] for m in self._movie_items):
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    f"'{selected_item['title']}' už je ve sbírce",
                    xbmcgui.NOTIFICATION_INFO
                )
                return
            
            # 5. Přidat film do sbírky
            new_movie = {
                'id': selected_item['id'],
                'title': selected_item['title'],
                'year': selected_item.get('release_date', '')[:4],
                'overview': selected_item.get('overview', ''),
                'poster_path': selected_item.get('poster_path', ''),
                'backdrop_path': selected_item.get('backdrop_path', '')
            }
            self._movie_items.insert(0, new_movie)
            user_data.store_tmdb_movies(self.addon, self._movie_items)
            
            # 6. Notifikace o úspěchu
            xbmcgui.Dialog().notification(
                self.addon.getAddonInfo('name'),
                f"'{selected_item['title']}' přidán do sbírky",
                xbmcgui.NOTIFICATION_INFO
            )
            
            # 7. Obnovit poster flow a nastavit flag
            self._is_showing_collection = True
            self._populate_from_items()
            
            # Skrýt sidebar a fokusovat na postery
            self.clearProperty('WCS.AIChat.Visible')
            self.setFocusId(2000)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Add movie error: {e}", xbmc.LOGERROR)

    
    def onClick(self, controlId):
        """Přetíženo pro handling speciálních tlačítek."""
        # AI tlačítka
        if controlId == 4000:
            list_control = self.getControl(4000)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            val = item.getProperty('value')
            label = item.getLabel()
            
            if val == 'add_item':
                self._handle_add_movie()
            elif val == 'recommend_from_collection':
                self._start_ai_from_collection()
            elif val == 'recommend_new':
                self._start_ai_without_context()
            else:
                # Delegovat na parent
                super(MyMoviesDialog, self).onClick(controlId)
        elif controlId == 2000:
            # Poster flow klik - přesunout focus na preview akce
            # (parent handle_recommendation_click zobrazuje akční dialog, to nechceme)
            self.setFocusId(6100)
        else:
            # Delegovat na parent pro ostatní controls
            super(MyMoviesDialog, self).onClick(controlId)
    
    def _start_ai_from_collection(self):
        """Spustí AI doporučení na základě sbírky filmů."""
        if not self._movie_items:
            self.append_system_message("Vaše sbírka filmů je prázdná. Nejdříve přidejte nějaké filmy.")
            return
            
        # Spustit loading animaci
        self._start_button_loading("Analyzuji sbírku...")
        
        user_msg = "Doporuč mi filmy na základě mé sbírky."
        
        # Unified Start
        self.start_ai_with_context('my_movies', data=self._movie_items, user_text=user_msg)

    def _start_ai_without_context(self):
        """Spustí AI doporučení bez kontextu (nové filmy)."""
        # Spustit loading animaci
        self._start_button_loading("Hledám filmy...")
        
        user_msg = "Doporuč mi nové zajímavé filmy."
        
        # Unified Start
        self.start_ai_with_context('default_movie', data=None, user_text=user_msg, is_initial=False)


def show_my_movies_dialog(addon, show_nav_sidebar=False, nav_position=0):
    """Zobrazí dialog Moje filmy."""
    addon_path = addon.getAddonInfo('path')
    
    # Použijeme SDÍLENÝ XML z AI Chat dialogu
    dialog = MyMoviesDialog(
        "ai/DialogAIChatRecommendation.xml",
        addon_path,
        "Default",
        "1080i",
        show_nav_sidebar=show_nav_sidebar,
        nav_position=nav_position
    )
    dialog.doModal()
    del dialog


