# wcs/ai/DialogContinueWatching.py
"""
Dialog "Pokračovat ve sledování" - Fullscreen dialog pro naposledy přehrané položky.
Dědí z AIChatRecommendationDialog a sdílí jeho XML grafiku.
"""

import xbmc
import xbmcgui
import xbmcaddon
import threading
from urllib.parse import quote_plus

from wcs.ai.DialogAIChatRecommendation import AIChatRecommendationDialog
from wcs import user_data
from wcs import utils as wcs_utils


class ContinueWatchingDialog(AIChatRecommendationDialog):
    """
    Dialog pro naposledy přehrané položky.
    Dědí všechny funkce z AI Chat dialogu, ale přetěžuje:
    - onInit() - načítá z historie místo AI inference
    - _populate_preview_actions() - upravené akční tlačítka (Další díl, Smazat)
    """
    
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        # Parent zpracovává show_nav_sidebar, nav_position, context
        super(ContinueWatchingDialog, self).__init__(xml_filename, addon_path, *args, **kwargs)
        self._history_items = []  # Cache pro položky historie
        self._current_item_index = 0  # Index aktuální položky pro akce
        # Identifikátor sekce pro navigační sidebar
        self._nav_section_id = 'continue_watching'


    
    def onInit(self):
        """Přetížená inicializace - načte položky z historie místo AI inference."""
        # Resetovat stav (využívá parent metodu)
        self._reset_state()
        self.media_type = 'movie'  # Default
        self._is_showing_history = True  # Flag pro rozlišení mezi historií a AI doporučeními
        
        # Načíst historii
        self._history_items = user_data.load_recently_played(self.addon)
        
        if not self._history_items:
            # Prázdná historie - zobrazit hlášku (podobně jako v Naposledy přehrané)
            xbmcgui.Dialog().ok(
                "Pokračovat ve sledování",
                "Historie přehrávání je prázdná.\n\nPřehrajte nějaký film nebo seriál z menu a položka se zde zobrazí."
            )
            self.close()
            return
        
        # Nastavit titulek
        self.setProperty('WCS.AIChat.Title', 'Pokračovat ve sledování')
        self.setProperty('WCS.AIChat.Status', 'Připraven')
        
        # Zobrazit uvítací zprávu v chatu
        self.append_chat_message("AI", "Vyberte titul z historie nebo si nechte doporučit podobné.")
        
        # Nastavit AI tlačítka
        self._setup_initial_buttons()
        
        # Naplnit poster flow z historie
        self._populate_from_history()
        
        # Pokud byl dialog spuštěn z navigačního sidebaru, zobrazit ho bez animace
        if self._show_nav_on_init:
            self._show_nav_sidebar(animate=False)
        else:
            # Skrýt chat sidebar, fokus na postery s krátkým zpožděním
            self.clearProperty('WCS.AIChat.Visible')
            self._delayed_focus(2000, 0.5)

    
    def onClick(self, controlId):
        """Přetíženo - klik na poster zobrazí preview akce místo akčního dialogu."""
        if controlId == 2000:
            # Poster flow klik - přesunout focus na preview akce
            # (parent handle_recommendation_click zobrazuje akční dialog, to nechceme)
            self.setFocusId(6100)
        else:
            # Delegovat na parent pro ostatní controls
            super(ContinueWatchingDialog, self).onClick(controlId)
    
    def _setup_initial_buttons(self):
        """Nastaví počáteční AI tlačítka."""
        buttons = [
            {"label": "Doporuč mi na základě historie", "value": "recommend_from_history"},
            {"label": "Doporuč podobné filmy", "value": "recommend_similar_movies"},
            {"label": "Doporuč podobné seriály", "value": "recommend_similar_series"}
        ]
        self.update_buttons(buttons)
    
    def _populate_from_history(self):
        """Naplní poster flow položkami z historie."""
        list_control = self.getControl(2000)
        list_control.reset()
        
        for item in self._history_items:
            title = item.get('title', '')
            year = item.get('year', '')
            poster = item.get('poster', '')
            media_type = item.get('media_type', 'movie')
            tmdb_id = item.get('tmdb_id', '')
            
            # Pro seriály přidáme info o epizodě do labelu
            if media_type == 'series':
                season = item.get('season')
                episode = item.get('episode')
                episode_title = item.get('episode_title', '')
                
                display_title = title
                if season is not None and episode is not None:
                    display_title = f"{title} S{int(season):02d}E{int(episode):02d}"
                    if episode_title and not episode_title.startswith('S') and episode_title != f"Epizoda {episode}":
                        display_title += f" {episode_title}"
            else:
                display_title = f"{title} ({year})" if year else title
            
            list_item = xbmcgui.ListItem(label=display_title)
            
            if poster:
                list_item.setArt({'poster': poster})
            
            # Uložit metadata pro akce
            list_item.setProperty('tmdb_id', str(tmdb_id) if tmdb_id else '')
            list_item.setProperty('media_type', media_type)
            list_item.setProperty('title', title)
            list_item.setProperty('year', str(year) if year else '')
            
            # Pro seriály uložíme i info o epizodě
            if media_type == 'series':
                list_item.setProperty('season', str(item.get('season', '')))
                list_item.setProperty('episode', str(item.get('episode', '')))
                list_item.setProperty('episode_title', item.get('episode_title', ''))
            
            list_control.addItem(list_item)
        
        # Pokud máme položky, nastavit preview pro první
        if list_control.size() > 0:
            self._update_preview_for_current_item()
    
    def _populate_preview_actions(self, media_type, tmdb_id, title, year):
        """Přetíženo - upravené akční tlačítka pro Continue Watching (Unified)."""
        try:
            list_control = self.getControl(6100)
            list_control.reset()
            
            # Získat aktuální index
            poster_list = self.getControl(2000)
            current_index = poster_list.getSelectedPosition()
            
            # Rozlišit mezi položkami z historie a AI doporučeními
            is_history_item = getattr(self, '_is_showing_history', False) and current_index >= 0 and current_index < len(self._history_items)
            
            # 1. URČIT KONTEXT
            if is_history_item:
                if media_type == 'movie':
                    context = 'hist_movie'
                elif media_type in ['series', 'tv']:
                    context = 'hist_series'
                else:
                    context = 'hist_other'
            else:
                # AI doporučení - stejné jako v base class
                if media_type == 'movie':
                    context = 'ai_movie'
                elif media_type in ['series', 'tv']:
                    context = 'ai_series'
                else:
                    context = 'ai_col'
            
            # 2. SESTAVIT TLAČÍTKA Z REGISTRU
            items = self.build_action_items(context)
            
            # 3. DOPLNIT DATA (společná + specifická pro tento dialog)
            for item in items:
                item.setProperty('tmdb_id', str(tmdb_id))
                item.setProperty('title', title)
                item.setProperty('year', year)
                item.setProperty('media_type', media_type)
                item.setProperty('history_index', str(current_index)) # Specifické pro CW
                
                list_control.addItem(item)
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] ContinueWatching preview actions error: {e}", xbmc.LOGERROR)
    
    def _handle_preview_action_click(self):
        """Přetíženo - zpracování akcí specifických pro Continue Watching."""
        try:
            # Reset automatického zobrazení tlačítek (jakákoliv akce ukončí delay)
            self.clearProperty('WCS.AIPreview.ShowActions')
            
            list_control = self.getControl(6100)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            action = item.getProperty('action')
            tmdb_id = item.getProperty('tmdb_id')
            title = item.getProperty('title')
            year = item.getProperty('year')
            media_type = item.getProperty('media_type')
            history_index = item.getProperty('history_index')
            
            if action == 'next_episode':
                self._handle_next_episode(title, tmdb_id, history_index)
            elif action == 'play_episode':
                self._handle_play_episode(history_index)
            elif action == 'remove_from_history':
                self._handle_remove_from_history(history_index)
            elif action == 'play':
                # Přehrát film pomocí klasického search dialogu
                self._handle_play_movie(history_index)
            elif action == 'similar':
                # Podobné - s rozlišením pro filmy a seriály
                self._handle_similar(title, year, media_type)
            else:
                # Delegovat na parent pro standardní akce (details, add_list, atd.)
                super(ContinueWatchingDialog, self)._handle_preview_action_click()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] ContinueWatching action error: {e}", xbmc.LOGERROR)

    def _handle_next_episode(self, title, tmdb_id, history_index):
        """Spustí přehrávání další epizody (Next Episode)."""
        try:
            # Zavřít dialog
            self.close()
            
            # Získat item z historie
            item = self._history_items[int(history_index)]
            
            # Parametry pro utils
            params = {
                'tmdb_id': tmdb_id,
                'series_name': title,
                'current_season': item.get('season'),
                'current_episode': item.get('episode')
            }
            
            from wcs import utils
            utils.play_next_episode_with_metadata(params, self.addon)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Handler next episode error: {e}", xbmc.LOGERROR)

    
    def _handle_similar(self, title, year, media_type):
        """Spustí AI doporučení podobných filmů nebo seriálů (Unified)."""
        try:
            # 1. Připravit text inspirace
            if media_type in ['series', 'tv']:
                context_key = 'similar_ser'
                inspiration_text = f"Seriál '{title}' ({year}) mě nadchl. Hledám podobný seriál - může mít jinou atmosféru, ale ať mě to vtáhne stejně."
            else:
                context_key = 'similar_mov'
                inspiration_text = f"Film '{title}' ({year}) mě zaujal. Chci zažít něco podobného. Nemusí to být stejný žánr, ale ať to má podobnou atmosféru."
            
            # 2. Skrýt overlay a fokus na chat (UI logika zustava zde)
            self._clear_preview_properties()
            self.setFocusId(4000)
            
            # 3. Spustit načítací animaci
            self._start_button_loading("Hledám podobné...")
            
            # 4. Spustit AI pomocí Unified metody
            # Data jsou None (nejde o kontext dat, ale o volnou inspiraci)
            self.start_ai_with_context(context_key, data=None, user_text=inspiration_text)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Similar action error: {e}", xbmc.LOGERROR)
    
    def _start_ai_from_history(self, media_filter='mixed'):
        """Spustí AI doporučení na základě historie (Unified)."""
        # Filtrovat položky podle typu
        if media_filter == 'movie':
            filtered = [i for i in self._history_items if i.get('media_type') == 'movie']
            context_key = 'hist_movie'
            user_msg = "Doporuč mi filmy na základě mé historie sledování."
        elif media_filter == 'series':
            filtered = [i for i in self._history_items if i.get('media_type') == 'series']
            context_key = 'hist_series'
            user_msg = "Doporuč mi seriály na základě mé historie sledování."
        else:
            filtered = self._history_items
            context_key = 'hist_mixed'
            user_msg = "Doporuč mi filmy a seriály na základě mé historie sledování."
        
        if not filtered:
            self.append_system_message(f"V historii nejsou žádné {'filmy' if media_filter == 'movie' else 'seriály' if media_filter == 'series' else 'položky'}.")
            return
            
        # Spustit loading animaci
        self._start_button_loading("Analyzuji historii...")
        
        # Spustit AI pomocí Unified metody
        self.start_ai_with_context(context_key, data=filtered, user_text=user_msg)


def show_continue_watching_dialog(addon, show_nav_sidebar=False, nav_position=0):
    """Zobrazí dialog Pokračovat ve sledování."""
    # Nejdřív zkontrolovat, jestli je historie neprázdná
    items = user_data.load_recently_played(addon)
    if not items:
        xbmcgui.Dialog().notification(
            addon.getAddonInfo('name'),
            "Historie je prázdná",
            xbmcgui.NOTIFICATION_INFO
        )
        return
    
    addon_path = addon.getAddonInfo('path')
    
    # Použijeme SDÍLENÝ XML z AI Chat dialogu
    dialog = ContinueWatchingDialog(
        "ai/DialogAIChatRecommendation.xml",
        addon_path,
        "Default",
        "1080i",
        show_nav_sidebar=show_nav_sidebar,
        nav_position=nav_position
    )
    dialog.doModal()
    del dialog


