"""
Anthropic Claude API handler
Kompletní implementace pro Anthropic Claude API podle oficiální dokumentace
"""

import os
import requests
import xbmcaddon
import xbmc
import json
from wcs.ai.DialogAIDebugRequest import show_api_request_debug, show_api_response_debug, show_api_error_debug

# Načtení nastavení addonu
addon = xbmcaddon.Addon()
API_BASE_URL = "https://api.anthropic.com"
API_VERSION = "2023-06-01"

def get_current_model():
    """Vrací aktuálně vybraný Anthropic model z nastavení."""
    addon = xbmcaddon.Addon()
    selected_model = addon.getSetting("anthropic_model")
    if not selected_model:
        selected_model = "claude-3-5-sonnet-20241022"  # Výchozí model
    return selected_model

def send_prompt(prompt: str, model: str = None) -> tuple:
    """
    Odešle prompt na Anthropic Claude API.
    
    Args:
        prompt (str): Text promptu
        model (str, optional): Specifický model (pokud None, použije se z nastavení)
        
    Returns:
        tuple: (response_text, usage_data)
    """
    addon = xbmcaddon.Addon()
    API_KEY = addon.getSetting("anthropic_api_key")
    
    if not API_KEY:
        raise ValueError("Anthropic API klíč není nastaven.")
    
    # Pokud není model specifikován, použij model z nastavení
    if model is None:
        model = get_current_model()
    
    url = f"{API_BASE_URL}/v1/messages"
    
    headers = {
        "x-api-key": API_KEY,
        "anthropic-version": API_VERSION,
        "Content-Type": "application/json",
        "Accept": "application/json"
    }
    
    # Načtení parametrů z nastavení
    temperature = float(addon.getSetting("anthropic_temperature") or "1.0")
    top_p = float(addon.getSetting("anthropic_top_p") or "1.0")
    top_k = int(addon.getSetting("anthropic_top_k") or "5")
    max_tokens = int(addon.getSetting("anthropic_max_tokens") or "4096")

    payload = {
        "model": model,
        "max_tokens": max_tokens,
        "temperature": temperature,
        "top_p": top_p,
        "top_k": top_k,
        "messages": [
            {
                "role": "user",
                "content": prompt
            }
        ]
    }

    try:
        xbmc.log(f"[Anthropic API] Odesílám prompt na model {model}", xbmc.LOGINFO)
        
        # Debug: Zobrazí raw request před odesláním
        if not show_api_request_debug("Anthropic Claude", model, payload):
            raise Exception("Odeslání API requestu bylo zrušeno uživatelem.")
        
        response = requests.post(url, headers=headers, json=payload, timeout=120)
        response.raise_for_status()
        
        data = response.json()
        
        # Debug: Zobrazí raw response po obdržení
        
        # Extrakce odpovědi z Anthropic API formátu
        if 'content' in data and len(data['content']) > 0:
            response_text = data['content'][0]['text']
        else:
            response_text = "Chyba: Neplatná odpověď z Anthropic API"
        
        # Debug: Zobrazí response s extrahovaným textem
        show_api_response_debug("Anthropic Claude", model, data, response_text)
        
        # Extrakce usage informací (pokud jsou dostupné)
        usage_data = {}
        if 'usage' in data:
            usage_data = {
                'input_tokens': data['usage'].get('input_tokens', 0),
                'output_tokens': data['usage'].get('output_tokens', 0),
                'total_tokens': data['usage'].get('input_tokens', 0) + data['usage'].get('output_tokens', 0)
            }
        
        xbmc.log(f"[Anthropic API] Úspěšně získána odpověď: {len(response_text)} znaků", xbmc.LOGINFO)
        
        return response_text, usage_data
        
    except requests.exceptions.RequestException as e:
        error_msg = f"Chyba při komunikaci s Anthropic API: {str(e)}"
        xbmc.log(f"[Anthropic API] {error_msg}", xbmc.LOGERROR)
        # Debug: Zobrazí chybu
        show_api_error_debug("Anthropic Claude", model, error_msg, {"exception": str(e)})
        raise Exception(error_msg)
    except Exception as e:
        error_msg = f"Neočekávaná chyba v Anthropic API: {str(e)}"
        xbmc.log(f"[Anthropic API] {error_msg}", xbmc.LOGERROR)
        # Debug: Zobrazí chybu
        show_api_error_debug("Anthropic Claude", model, error_msg, {"exception": str(e)})
        raise Exception(error_msg)

def get_available_models():
    """
    Získá seznam dostupných modelů z Anthropic API.
    
    Returns:
        list: Seznam dostupných modelů
    """
    addon = xbmcaddon.Addon()
    API_KEY = addon.getSetting("anthropic_api_key")
    
    if not API_KEY:
        return []
    
    # Anthropic API nemá endpoint pro seznam modelů, vracíme známé modely
    return [
        "claude-3-5-sonnet-20241022",
        "claude-3-5-haiku-20241022", 
        "claude-3-opus-20240229",
        "claude-3-sonnet-20240229",
        "claude-3-haiku-20240307",
        "claude-2.1",
        "claude-2.0",
        "claude-instant-1.2"
    ]

def send_multimodal_prompt(text_prompt: str, image_data: str = None, model: str = None) -> tuple:
    """
    Odešle multimodální prompt (text + obrázek) na Anthropic Claude API.
    
    Args:
        text_prompt (str): Textový prompt
        image_data (str, optional): Base64 kódovaný obrázek
        model (str, optional): Specifický model (doporučeno claude-3-5-sonnet-20241022)
        
    Returns:
        tuple: (response_text, usage_data)
    """
    addon = xbmcaddon.Addon()
    API_KEY = addon.getSetting("anthropic_api_key")
    
    if not API_KEY:
        raise ValueError("Anthropic API klíč není nastaven.")
    
    # Pro multimodální úlohy použij vhodný model
    if model is None:
        model = "claude-3-5-sonnet-20241022"
    
    url = f"{API_BASE_URL}/v1/messages"
    
    headers = {
        "x-api-key": API_KEY,
        "anthropic-version": API_VERSION,
        "Content-Type": "application/json",
        "Accept": "application/json"
    }
    
    # Sestavení zprávy s obrázkem
    content = [{"type": "text", "text": text_prompt}]
    
    if image_data:
        content.append({
            "type": "image",
            "source": {
                "type": "base64",
                "media_type": "image/jpeg",
                "data": image_data
            }
        })
    
    # Načtení parametrů z nastavení
    temperature = float(addon.getSetting("anthropic_temperature") or "1.0")
    top_p = float(addon.getSetting("anthropic_top_p") or "1.0")
    top_k = int(addon.getSetting("anthropic_top_k") or "5")
    max_tokens = int(addon.getSetting("anthropic_max_tokens") or "4096")

    payload = {
        "model": model,
        "max_tokens": max_tokens,
        "temperature": temperature,
        "top_p": top_p,
        "top_k": top_k,
        "messages": [
            {
                "role": "user",
                "content": content
            }
        ]
    }
    
    try:
        xbmc.log(f"[Anthropic API] Odesílám multimodální prompt na model {model}", xbmc.LOGINFO)
        
        # Debug: Zobrazí raw multimodal request před odesláním
        show_api_request_debug("Anthropic Claude (Multimodal)", model, payload)
        
        response = requests.post(url, headers=headers, json=payload, timeout=120)
        response.raise_for_status()
        
        data = response.json()
        
        if 'content' in data and len(data['content']) > 0:
            response_text = data['content'][0]['text']
        else:
            response_text = "Chyba: Neplatná odpověď z Anthropic API"
        
        # Debug: Zobrazí multimodal response
        show_api_response_debug("Anthropic Claude (Multimodal)", model, data, response_text)
        
        usage_data = {}
        if 'usage' in data:
            usage_data = data['usage']
        
        return response_text, usage_data
        
    except Exception as e:
        error_msg = f"Chyba při multimodálním zpracování: {str(e)}"
        xbmc.log(f"[Anthropic API] {error_msg}", xbmc.LOGERROR)
        # Debug: Zobrazí multimodal chybu
        show_api_error_debug("Anthropic Claude (Multimodal)", model, error_msg, {"exception": str(e)})
        raise Exception(error_msg)

def validate_api_key():
    """
    Ověří platnost API klíče.
    
    Returns:
        bool: True pokud je klíč platný, False jinak
    """
    addon = xbmcaddon.Addon()
    API_KEY = addon.getSetting("anthropic_api_key")
    
    if not API_KEY:
        return False
    
    try:
        # Skutečné API volání pro ověření klíče - jednoduchý test prompt
        url = f"{API_BASE_URL}/v1/messages"
        headers = {
            "x-api-key": API_KEY,
            "anthropic-version": API_VERSION,
            "Content-Type": "application/json",
            "Accept": "application/json"
        }
        payload = {
            "model": "claude-3-5-sonnet-20241022",
            "max_tokens": 1,
            "messages": [
                {
                    "role": "user",
                    "content": "test"
                }
            ]
        }
        response = requests.post(url, headers=headers, json=payload, timeout=10)
        response.raise_for_status()
        return True
    except requests.exceptions.HTTPError as e:
        if e.response.status_code == 401:
            xbmc.log("[wcs][anthropic_api] validate_api_key: Neplatný API klíč (401)", xbmc.LOGWARNING)
            return False
        else:
            xbmc.log(f"[wcs][anthropic_api] validate_api_key: HTTP chyba {e.response.status_code}", xbmc.LOGWARNING)
            return False
    except requests.exceptions.RequestException as e:
        xbmc.log(f"[wcs][anthropic_api] validate_api_key: Síťová chyba - {e}", xbmc.LOGWARNING)
        return False
    except Exception as e:
        xbmc.log(f"[wcs][anthropic_api] validate_api_key: Neočekávaná chyba - {e}", xbmc.LOGWARNING)
        return False

def test_api_connection():
    """
    Testuje připojení k Anthropic Claude API s jednoduchým promptem.
    
    Returns:
        tuple: (success: bool, message: str)
    """
    addon = xbmcaddon.Addon()
    API_KEY = addon.getSetting("anthropic_api_key")
    
    if not API_KEY:
        return False, "Anthropic API klíč není nastaven."
    
    url = f"{API_BASE_URL}/v1/messages"
    
    headers = {
        "x-api-key": API_KEY,
        "anthropic-version": API_VERSION,
        "Content-Type": "application/json",
        "Accept": "application/json"
    }
    
    payload = {
        "model": "claude-3-5-sonnet-20241022",
        "max_tokens": 50,
        "messages": [
            {
                "role": "user",
                "content": "Odpověz pouze 'Anthropic Claude API funguje' a nic víc."
            }
        ]
    }
    
    try:
        xbmc.log("[Anthropic API] Testování připojení", xbmc.LOGINFO)
        
        response = requests.post(url, headers=headers, json=payload, timeout=120)
        response.raise_for_status()
        
        data = response.json()
        
        if 'content' in data and len(data['content']) > 0:
            xbmc.log("[Anthropic API] Test připojení úspěšný", xbmc.LOGINFO)
            return True, "Anthropic Claude API je funkční a připojení proběhlo úspěšně."
        else:
            xbmc.log("[Anthropic API] Test připojení neúspěšný - neplatná odpověď", xbmc.LOGWARNING)
            return False, "Test Anthropic Claude API selhal - neplatná odpověď."
            
    except requests.exceptions.HTTPError as e:
        if e.response.status_code == 401:
            xbmc.log("[Anthropic API] Test připojení neúspěšný - neplatný API klíč", xbmc.LOGWARNING)
            return False, "Test Anthropic Claude API selhal - neplatný API klíč."
        else:
            xbmc.log(f"[Anthropic API] Test připojení neúspěšný - HTTP {e.response.status_code}", xbmc.LOGWARNING)
            return False, f"Test Anthropic Claude API selhal - HTTP {e.response.status_code}."
    except Exception as e:
        xbmc.log(f"[Anthropic API] Test připojení neúspěšný: {str(e)}", xbmc.LOGWARNING)
        return False, f"Test Anthropic Claude API selhal: {str(e)}" 