"""
AI Provider abstraction layer pro různé poskytovatele AI.
Podporuje OpenAI, Anthropic Claude a Google AI.
"""

import xbmcaddon
import xbmc
from wcs.ai.OpenAIClient import send_prompt as openai_send_prompt


def check_ai_api_key_and_handle_error(provider=None):
    """
    Kontroluje AI API klíč a v případě chyby zobrazí dialog.
    Vrací True pokud je klíč v pořádku, False pokud nastala chyba.
    
    Args:
        provider: Volitelný parametr pro specifikaci poskytovatele
    """
    try:
        addon = xbmcaddon.Addon()
        # Použij poskytovatele z parametru nebo z nastavení jako fallback
        current_provider = provider or addon.getSetting("ai_provider") or "mistral"
        
        # Normalizace názvu poskytovatele
        if "mistral" in current_provider.lower():
            setting_key = "mistral_api_key"
            provider_name = "Mistral AI"
        elif "openai" in current_provider.lower():
            setting_key = "openai_api_key"
            provider_name = "OpenAI"
        elif "anthropic" in current_provider.lower():
            setting_key = "anthropic_api_key"
            provider_name = "Anthropic"
        elif "google" in current_provider.lower():
            setting_key = "google_api_key"
            provider_name = "Google AI"
        else:
            setting_key = "mistral_api_key"
            provider_name = "AI"
        
        # Kontrola zda je klíč nastaven
        api_key = addon.getSetting(setting_key)
        if not api_key:
            # Klíč není nastaven - zobrazit dialog s poskytovatelem
            from wcs.login.DialogAILogin import show_ai_login_dialog
            return show_ai_login_dialog(provider=current_provider)
        
        # Ověření platnosti klíče
        try:
            if "mistral" in current_provider.lower():
                from wcs.ai.MistralClient import validate_api_key
                is_valid = validate_api_key()
            elif "openai" in current_provider.lower():
                from wcs.ai.OpenAIClient import validate_api_key
                is_valid = validate_api_key()
            elif "anthropic" in current_provider.lower():
                from wcs.ai.AnthropicClient import validate_api_key
                is_valid = validate_api_key()
            elif "google" in current_provider.lower():
                from wcs.ai.GoogleClient import validate_api_key
                is_valid = validate_api_key()
            else:
                is_valid = False
            
            if not is_valid:
                # Klíč je neplatný - zobrazit dialog pro nový klíč
                xbmc.log(f"[plugin.video.milionar] {provider_name} API klíč je neplatný", level=xbmc.LOGWARNING)
                from wcs.login.DialogAILogin import show_ai_login_dialog
                return show_ai_login_dialog(provider=current_provider)
            
            return True
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Chyba při ověřování {provider_name} API klíče: {e}", level=xbmc.LOGWARNING)
            # Při chybě ověření zobrazit dialog pro nový klíč
            from wcs.login.DialogAILogin import show_ai_login_dialog
            return show_ai_login_dialog(provider=current_provider)
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error checking AI API key: {e}", level=xbmc.LOGERROR)
        return False


class AIProvider:
    """Základní třída pro AI poskytovatele."""
    
    def __init__(self, provider=None):
        self.addon = xbmcaddon.Addon()
        # Použij poskytovatele z parametru nebo z nastavení jako fallback
        self.current_provider = provider or self.addon.getSetting("ai_provider") or "mistral"
        # Normalizace názvu poskytovatele
        if "mistral" in self.current_provider.lower():
            self.current_provider = "mistral"
    
    def send_prompt(self, prompt: str, model: str = None, use_tools: bool = False) -> tuple:
        """
        Odešle prompt podle vybraného poskytovatele.
        Automaticky kontroluje API klíč před odesláním.
        
        Args:
            prompt (str): Text promptu
            model (str, optional): Specifický model (pokud None, použije se z nastavení)
            use_tools (bool): Použít tools jako web search (pouze pro OpenAI)
            
        Returns:
            tuple: (response_text, usage_data)
        """
        # Kontrola API klíče před odesláním
        if not check_ai_api_key_and_handle_error(provider=self.current_provider):
            raise ValueError("AI API klíč není nastaven nebo dialog byl zrušen.")
        
        if self.current_provider == "openai":
            return self._send_openai_prompt(prompt, model, use_tools)
        elif self.current_provider == "anthropic":
            return self._send_anthropic_prompt(prompt, model)
        elif self.current_provider == "google":
            return self._send_google_prompt(prompt, model)
        elif self.current_provider == "mistral":
            return self._send_mistral_prompt(prompt, model)
        else:
            raise ValueError(f"Nepodporovaný AI poskytovatel: {self.current_provider}")
    
    def _send_openai_prompt(self, prompt: str, model: str = None, use_tools: bool = False) -> tuple:
        """Odešle prompt přes OpenAI API."""
        return openai_send_prompt(prompt, model, use_tools)
    
    def _send_anthropic_prompt(self, prompt: str, model: str = None) -> tuple:
        """Odešle prompt přes Anthropic Claude API."""
        from wcs.ai.AnthropicClient import send_prompt as anthropic_send_prompt
        return anthropic_send_prompt(prompt, model)
    
    def _send_google_prompt(self, prompt: str, model: str = None) -> tuple:
        """Odešle prompt přes Google AI API."""
        from wcs.ai.GoogleClient import send_prompt as google_send_prompt
        return google_send_prompt(prompt, model)
    
    def _send_mistral_prompt(self, prompt: str, model: str = None) -> tuple:
        """Odešle prompt přes Mistral AI API."""
        from wcs.ai.MistralClient import send_prompt as mistral_send_prompt
        return mistral_send_prompt(prompt, model)
    
    def get_current_provider(self) -> str:
        """Vrací aktuálně vybraný poskytovatel."""
        return self.current_provider
    
    def get_available_providers(self) -> list:
        """Vrací seznam dostupných poskytovatelů."""
        return ["openai", "anthropic", "google", "mistral"]
    
    def get_current_model(self) -> str:
        """Vrací aktuálně vybraný model podle poskytovatele."""
        if self.current_provider == "openai":
            from wcs.ai.OpenAIClient import get_current_model
            return get_current_model()
        elif self.current_provider == "anthropic":
            from wcs.ai.AnthropicClient import get_current_model
            return get_current_model()
        elif self.current_provider == "google":
            from wcs.ai.GoogleClient import get_current_model
            return get_current_model()
        elif self.current_provider == "mistral":
            from wcs.ai.MistralClient import get_current_model
            return get_current_model()
        else:
            return "neznámý"
    
    def get_model_info_string(self) -> str:
        """Vrací formátovaný řetězec s informací o aktuálním modelu a poskytovateli."""
        provider_names = {
            "openai": "OpenAI",
            "anthropic": "Anthropic", 
            "google": "Google AI",
            "mistral": "Mistral AI"
        }
        
        provider_display = provider_names.get(self.current_provider, self.current_provider)
        model = self.get_current_model()
        
        return f"AI: {provider_display} - {model}"

# Globální instance pro snadné použití
ai_provider = AIProvider()

def send_prompt(prompt: str, model: str = None, use_tools: bool = False, provider: str = None) -> tuple:
    """
    Wrapper funkce pro posílání promptů.
    Automaticky používá aktuálně vybraný AI poskytovatel.
    Automaticky kontroluje API klíč před odesláním.
    
    Args:
        prompt (str): Text promptu
        model (str, optional): Specifický model
        use_tools (bool): Použít tools jako web search (pouze pro OpenAI)
        provider (str, optional): Specifický poskytovatel
        
    Returns:
        tuple: (response_text, usage_data)
    """
    # Pokud je specifikován provider, vytvoř novou instanci
    if provider:
        temp_provider = AIProvider(provider=provider)
        return temp_provider.send_prompt(prompt, model, use_tools)
    else:
        return ai_provider.send_prompt(prompt, model, use_tools) 