# Multi-Provider Systém

**Status:** COMPLETED  
**Vytvořeno:** 2026-02-06  
**Updated:** 2026-02-22 23:14

---

## Související dokumenty

- [001-kraska-provider.md](001-kraska-provider.md) - KRA.sk provider
- [002-webshare-provider.md](002-webshare-provider.md) - Webshare provider

---

## Účel

Multi-provider systém umožňuje používat **Webshare a KRA.sk současně nebo samostatně**. Výsledky z obou providerů se sloučí a jednotně seřadí.

---

## Nastavení

### Umístění

**Kodi → Doplňky → plugin.video.milionar → Nastavení → Providery**

### Možnosti

| Nastavení | Klíč | Default | Popis |
|-----------|------|---------|-------|
| Webshare | `webshare_enabled` | true | Zapnout Webshare provider |
| KRA.sk | `kraska_enabled` | false | Zapnout KRA.sk provider |

### Kombinace a chování

| Webshare | KRA.sk | Chování |
|----------|--------|---------|
| ❌ | ❌ | Warning dialog, nabídka otevřít nastavení |
| ✅ | ❌ | Výsledky pouze z Webshare |
| ❌ | ✅ | Výsledky pouze z KRA.sk |
| ✅ | ✅ | Sloučení výsledků, jednotné řazení |

---

## Jak to funguje v doplňku

### Detekce aktivních providerů

```python
# wcs/search/SeriesSearch.py
def _get_enabled_providers():
    providers = []
    addon = xbmcaddon.Addon()
    
    if addon.getSettingBool('webshare_enabled'):
        providers.append('webshare')
    if addon.getSettingBool('kraska_enabled'):
        providers.append('kraska')
    
    return providers
```

### Paralelní vyhledávání

```
┌─────────────────┐
│  Uživatel       │
│  spustí film    │
└────────┬────────┘
         │
         ▼
┌─────────────────────────────────────┐
│  Kontrola aktivních providerů       │
│  ['webshare', 'kraska']             │
└─────────────────┬───────────────────┘
         │
         ├──────────────────────┐
         │                      │
         ▼                      ▼
┌─────────────────┐    ┌─────────────────┐
│   Webshare      │    │    KRA.sk       │
│   search API    │    │   SC API        │
└────────┬────────┘    └────────┬────────┘
         │                      │
         │  file_items[]        │  file_items[]
         │                      │
         └──────────┬───────────┘
                    │
                    ▼
         ┌──────────────────────┐
         │  Sloučení výsledků   │
         │  all_items = ws + kr │
         └──────────┬───────────┘
                    │
                    ▼
         ┌──────────────────────┐
         │  Jednotné řazení     │
         │  sort_files_by_...() │
         └──────────┬───────────┘
                    │
                    ▼
         ┌──────────────────────┐
         │  Zobrazení v UI      │
         │  nebo autoplay       │
         └──────────────────────┘
```

---

## Sloučení Výsledků

### Filmy

```python
# wcs/ui/dialogs/DialogWcsSearch.py
def _perform_search(self):
    file_items = []
    
    if self.webshare_enabled:
        ws_results = self._perform_search_webshare()
        file_items.extend(ws_results)
    
    if self.kraska_enabled:
        kr_results = self._perform_search_kraska()
        file_items.extend(kr_results)
    
    # Jednotné řazení všech výsledků
    file_items = sort_files_by_priority(file_items, self.search_query)
    return file_items
```

### Seriály

```python
# wcs/search/SeriesSearch.py
def search_files_for_episode():
    all_files = []
    providers = _get_enabled_providers()
    
    if 'webshare' in providers:
        ws_files = _search_via_webshare(...)
        all_files.extend(ws_files)
    
    if 'kraska' in providers:
        kr_files = _search_via_kraska(...)
        all_files.extend(kr_files)
    
    # Jednotné řazení
    sorted_files = sort_series_files_by_priority(all_files, query)
    return sorted_files
```

---

## Jednotné Řazení

Obě varianty (filmy i seriály) používají `calculate_advanced_score()` z `wcs/utils.py`:

### Priorita

| Pořadí | Kritérium | Max body |
|--------|-----------|----------|
| 1 | Kvalita videa | 50000 |
| 2 | Velikost souboru | 3000 |
| 3 | CZ jazyk | 500 |
| 4 | Bonusy (HDR, HEVC) | 200 |

### Jak to funguje pro oba providery

| Provider | Zdroj kvality | Zdroj jazyka |
|----------|---------------|--------------|
| Webshare | Extrakce z názvu souboru | Extrakce z názvu souboru |
| KRA.sk | Z SC API metadata + syntetický název | Z SC API metadata + syntetický název |

Díky **syntetickému názvu** u KRA.sk (např. `Film.1080p.CZ.mkv`) fungují stejné funkce `extract_quality()` a `extract_language()` pro oba providery.

---

## Zobrazení v UI

### Barevné rozlišení providerů

```
Webshare: [COLOR FF4488FF]Film.1080p.CZ.mkv[/COLOR]   (modrá)
KRA.sk:   [COLOR FF00CC66]Film.1080p.CZ.mkv[/COLOR]   (zelená)
```

### Formát položky

```
{provider}: {filename}
• {quality} - {language} - {format} - {size} - {votes}
```

---

## Klíčové Soubory

| Soubor | Účel |
|--------|------|
| `wcs/search/SeriesSearch.py` | Multi-provider vyhledávání seriálů |
| `wcs/ui/dialogs/DialogWcsSearch.py` | Multi-provider vyhledávání filmů |
| `wcs/utils.py` | Jednotné řazení (`calculate_advanced_score`) |
| `wcs/router.py` | Provider-aware playback |

---

## Validace nastavení

Pokud jsou oba providery vypnuté:

```python
# wcs/ui/dialogs/DialogWcsSearch.py
if not self.webshare_enabled and not self.kraska_enabled:
    dialog = xbmcgui.Dialog()
    result = dialog.yesno(
        "Žádný provider",
        "Není zapnutý žádný provider. Otevřít nastavení?"
    )
    if result:
        addon.openSettings()
    return []
```

---

## Provider-Aware Playback

Při přehrání se automaticky detekuje provider podle položky:

```python
# wcs/router.py
def play_file(file_item):
    provider = file_item.get('provider', 'webshare')
    
    if provider == 'kraska':
        # KRA.sk flow: SC API → ident → resolve
        url = resolve_kraska_stream(file_item)
    else:
        # Webshare flow: file_link API
        url = webshare_api.getlink(file_item['ident'], token)
    
    play_url(url)
```
