#!/usr/bin/env python3
"""
Standalone test harness for Continuous Chat Millionaire.

Runs production prompt code (millionaire_prompts.py) with real API calls
outside of Kodi. Simulates entire game flow: welcome -> 9 questions -> feedbacks.

Usage:
    python3 test_millionaire_game.py --scenario all-correct
    python3 test_millionaire_game.py --scenario fail-at-5
    python3 test_millionaire_game.py --scenario fail-at-1 --use-lifeline 5050:3,tip:5
"""

import sys
import os
import json
import time
import argparse
import subprocess
import urllib.request
import urllib.error
import xml.etree.ElementTree as ET
from datetime import datetime

# ---------------------------------------------------------------------------
# Kodi module stubs (minimal mocks for millionaire_prompts.py import)
# ---------------------------------------------------------------------------

ADDON_DIR = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
SETTINGS_PATH = os.path.expanduser(
    "~/Library/Application Support/Kodi/userdata/addon_data/"
    "plugin.video.milionar/settings.xml"
)


class _FakeAddon:
    """Minimal xbmcaddon.Addon stub that reads real Kodi settings XML."""

    def __init__(self, _id=None):
        self._settings = {}
        self._load_settings()

    def _load_settings(self):
        if not os.path.exists(SETTINGS_PATH):
            return
        tree = ET.parse(SETTINGS_PATH)
        root = tree.getroot()
        for setting in root.findall("setting"):
            sid = setting.get("id")
            value = setting.text or ""
            if sid:
                self._settings[sid] = value

    def getSetting(self, key):
        return self._settings.get(key, "")

    def getSettingBool(self, key):
        return self._settings.get(key, "false").lower() == "true"

    def getSettingInt(self, key):
        try:
            return int(self._settings.get(key, "0"))
        except ValueError:
            return 0

    def getAddonInfo(self, key):
        return ""


# Inject fake Kodi modules BEFORE importing prompts
import types

xbmcaddon_mod = types.ModuleType("xbmcaddon")
xbmcaddon_mod.Addon = _FakeAddon
sys.modules["xbmcaddon"] = xbmcaddon_mod

xbmc_mod = types.ModuleType("xbmc")
xbmc_mod.LOGINFO = 1
xbmc_mod.LOGWARNING = 2
xbmc_mod.LOGERROR = 3
xbmc_mod.LOGDEBUG = 0
xbmc_mod.log = lambda msg, level=1: None  # silent
sys.modules["xbmc"] = xbmc_mod

xbmcgui_mod = types.ModuleType("xbmcgui")
sys.modules["xbmcgui"] = xbmcgui_mod

xbmcvfs_mod = types.ModuleType("xbmcvfs")
xbmcvfs_mod.translatePath = lambda p: p.replace("special://temp/", "/tmp/").replace("special://home/", os.path.expanduser("~/"))
xbmcvfs_mod.File = None  # not used in test
sys.modules["xbmcvfs"] = xbmcvfs_mod

# Now import production prompts
sys.path.insert(0, ADDON_DIR)
from wcs.games.millionaire_prompts import (
    build_system_prompt,
    build_welcome_message,
    build_question_message,
    build_feedback_message,
    MONEY_AMOUNTS,
)
from wcs.games.millionaire_tts import (
    compose_welcome_tts,
    compose_question_tts,
    compose_feedback_tts,
)
from wcs.games.moderator_styles import get_tts_instructions


# ---------------------------------------------------------------------------
# Settings loader
# ---------------------------------------------------------------------------

def load_settings():
    """Load AI settings from Kodi addon settings XML."""
    addon = _FakeAddon()
    provider = addon.getSetting("ai_provider") or "mistral"
    settings = {"provider": provider}

    if provider == "mistral":
        settings["api_key"] = addon.getSetting("mistral_api_key")
        settings["model"] = addon.getSetting("mistral_model") or "mistral-large-latest"
        settings["temperature"] = float(addon.getSetting("mistral_temperature") or "0.7")
        settings["top_p"] = float(addon.getSetting("mistral_top_p") or "1.0")
        settings["max_tokens"] = int(addon.getSetting("mistral_max_tokens") or "4096")
        settings["api_url"] = "https://api.mistral.ai/v1/chat/completions"
    elif provider == "openai":
        settings["api_key"] = addon.getSetting("openai_api_key")
        settings["model"] = addon.getSetting("openai_model") or "gpt-4.1-mini"
        settings["temperature"] = float(addon.getSetting("openai_temperature") or "1.0")
        settings["top_p"] = float(addon.getSetting("openai_top_p") or "1.0")
        settings["max_tokens"] = int(addon.getSetting("openai_max_output_tokens") or "4096")
        settings["api_url"] = "https://api.openai.com/v1/chat/completions"
    else:
        raise ValueError(f"Unsupported provider for test: {provider}")

    settings["difficulty"] = addon.getSetting("millionaire_difficulty") or "1"

    # TTS settings (same keys as production _load_tts_settings)
    settings["tts"] = {
        "tts_welcome": addon.getSetting("milionar_tts_welcome") == "true",
        "tts_intro": addon.getSetting("milionar_tts_intro") == "true",
        "tts_question": addon.getSetting("milionar_tts_question") == "true",
        "tts_options": addon.getSetting("milionar_tts_options") == "true",
        "tts_result": addon.getSetting("milionar_tts_result") == "true",
        "tts_transition": addon.getSetting("milionar_tts_transition") == "true",
        "tts_explanation": addon.getSetting("milionar_tts_explanation") == "true",
        "voice": addon.getSetting("milionar_tts_voice") or "alloy",
    }
    settings["tts_model"] = addon.getSetting("milionar_tts_model") or "gpt-4o-mini-tts-2025-03-20"
    settings["tts_instructions"] = get_tts_instructions()
    settings["openai_api_key"] = addon.getSetting("openai_api_key")
    return settings


# ---------------------------------------------------------------------------
# TTS caller (pure urllib -- standalone version of TTSClient)
# ---------------------------------------------------------------------------

TTS_OUTPUT_PATH = "/tmp/tts_test_milionar.wav"


def call_tts_api(settings, text):
    """Call OpenAI TTS API. Returns (audio_path, latency)."""
    api_key = settings.get("openai_api_key", "")
    if not api_key:
        print(c("red", "  TTS: Chybi OpenAI API klic!"))
        return None, 0.0

    payload = {
        "model": settings.get("tts_model", "gpt-4o-mini-tts-2025-03-20"),
        "input": text,
        "voice": settings["tts"]["voice"],
        "instructions": settings.get("tts_instructions", ""),
        "response_format": "wav",
    }
    data_bytes = json.dumps(payload).encode("utf-8")
    req = urllib.request.Request(
        "https://api.openai.com/v1/audio/speech",
        data=data_bytes,
        headers={
            "Authorization": f"Bearer {api_key}",
            "Content-Type": "application/json",
        },
        method="POST",
    )

    start = time.time()
    try:
        with urllib.request.urlopen(req, timeout=30) as resp:
            audio_data = resp.read()
        latency = time.time() - start
        with open(TTS_OUTPUT_PATH, "wb") as f:
            f.write(audio_data)
        return TTS_OUTPUT_PATH, latency
    except Exception as e:
        latency = time.time() - start
        print(c("red", f"  TTS error: {e}"))
        return None, latency


def play_tts_sync(settings, text):
    """Generate TTS and play synchronously via afplay. Blocks until done."""
    if not text or not text.strip():
        return
    print(f"  {c('blue', 'TTS:')} {text[:80]}{'...' if len(text) > 80 else ''}")
    audio_path, tts_latency = call_tts_api(settings, text.strip())
    if audio_path:
        print(f"  {c('dim', f'TTS generovano za {tts_latency:.1f}s, prehrava se...')}")
        try:
            subprocess.run(["afplay", audio_path], check=True)
        except Exception as e:
            print(c("red", f"  afplay error: {e}"))
    else:
        print(c("yellow", "  TTS: audio nebylo vygenerovano"))


# ---------------------------------------------------------------------------
# API caller (pure urllib -- same format as production MistralClient/OpenAIClient)
# ---------------------------------------------------------------------------

def call_chat_api(settings, system_prompt, messages, json_mode=True):
    """Call chat completion API. Returns (parsed_data, usage, latency)."""
    api_messages = [{"role": "system", "content": system_prompt}]
    for msg in messages:
        api_messages.append({"role": msg["role"], "content": msg["content"]})

    payload = {
        "model": settings["model"],
        "messages": api_messages,
        "temperature": settings["temperature"],
        "top_p": settings["top_p"],
        "max_tokens": settings["max_tokens"],
    }
    if json_mode:
        payload["response_format"] = {"type": "json_object"}

    data_bytes = json.dumps(payload).encode("utf-8")
    req = urllib.request.Request(
        settings["api_url"],
        data=data_bytes,
        headers={
            "Authorization": f"Bearer {settings['api_key']}",
            "Content-Type": "application/json",
            "Accept": "application/json",
        },
        method="POST",
    )

    max_retries = 3
    retry_delays = [5, 15, 30]
    last_error = None

    for attempt in range(max_retries + 1):
        start = time.time()
        try:
            with urllib.request.urlopen(req, timeout=120) as resp:
                body = json.loads(resp.read().decode("utf-8"))
            last_error = None
            break
        except urllib.error.HTTPError as e:
            error_body = e.read().decode("utf-8") if e.fp else ""
            if e.code == 429 and attempt < max_retries:
                delay = retry_delays[attempt]
                print(f"  {c('yellow', f'Rate limit 429 -- retry za {delay}s (pokus {attempt + 2}/{max_retries + 1})')}")
                time.sleep(delay)
                # Rebuild request (urlopen consumes it)
                req = urllib.request.Request(
                    settings["api_url"],
                    data=data_bytes,
                    headers={
                        "Authorization": f"Bearer {settings['api_key']}",
                        "Content-Type": "application/json",
                        "Accept": "application/json",
                    },
                    method="POST",
                )
                last_error = RuntimeError(f"API HTTP {e.code}: {error_body}")
                continue
            raise RuntimeError(f"API HTTP {e.code}: {error_body}")

    if last_error:
        raise last_error
    latency = time.time() - start

    # Extract content
    content = ""
    choices = body.get("choices", [])
    if choices:
        content = choices[0].get("message", {}).get("content", "")

    usage = body.get("usage", {})

    # Parse JSON
    clean = content.strip()
    if clean.startswith("```"):
        # Strip markdown code block
        lines = clean.split("\n")
        clean = "\n".join(lines[1:-1]) if len(lines) > 2 else clean
    parsed = json.loads(clean)

    return parsed, usage, latency


# ---------------------------------------------------------------------------
# Formatting helpers
# ---------------------------------------------------------------------------

COLORS = {
    "reset": "\033[0m",
    "bold": "\033[1m",
    "green": "\033[32m",
    "red": "\033[31m",
    "yellow": "\033[33m",
    "blue": "\033[34m",
    "cyan": "\033[36m",
    "dim": "\033[2m",
    "magenta": "\033[35m",
}


def c(color, text):
    return f"{COLORS.get(color, '')}{text}{COLORS['reset']}"


def separator():
    print(c("dim", "─" * 70))


def header(text):
    print()
    print(c("bold", f"{'═' * 70}"))
    print(c("bold", f"  {text}"))
    print(c("bold", f"{'═' * 70}"))


# ---------------------------------------------------------------------------
# Game simulator
# ---------------------------------------------------------------------------

class MillionaireTestRunner:
    """Runs a complete Millionaire game with real API calls."""

    def __init__(self, settings, media_title, media_plot, scenario,
                 lifeline_schedule, media_type="series", genre="Fantasy, Drama", tts_enabled=False):
        self.settings = settings
        self.media_title = media_title
        self.media_plot = media_plot
        self.media_type = media_type
        self.genre = genre
        self.scenario = scenario  # "all-correct" | "fail-at-N"
        self.fail_at = self._parse_fail_at(scenario)
        self.lifeline_schedule = lifeline_schedule  # {round: [lifeline_type]}
        self.tts_enabled = tts_enabled

        # State
        self.chat_history = []
        self.system_prompt = ""
        self.current_round = 0
        self.score = 0
        self.total_prompt_tokens = 0
        self.total_completion_tokens = 0
        self.total_latency = 0.0
        self.inference_count = 0
        self.report_lines = []
        self.errors = []

    def _parse_fail_at(self, scenario):
        if scenario.startswith("fail-at-"):
            try:
                return int(scenario.split("-")[-1])
            except ValueError:
                return None
        return None

    def _chat_inference(self, user_msg):
        """Send a chat inference and accumulate history."""
        self.chat_history.append({"role": "user", "content": user_msg})
        self.inference_count += 1

        parsed, usage, latency = call_chat_api(
            self.settings, self.system_prompt, self.chat_history
        )

        # Add assistant response to history
        self.chat_history.append({"role": "assistant", "content": json.dumps(parsed, ensure_ascii=False)})

        # Track usage
        p_tokens = usage.get("prompt_tokens", 0)
        c_tokens = usage.get("completion_tokens", 0)
        self.total_prompt_tokens += p_tokens
        self.total_completion_tokens += c_tokens
        self.total_latency += latency

        # Log
        print(f"  {c('dim', f'[inference #{self.inference_count}]')} "
              f"{c('cyan', f'{p_tokens}+{c_tokens} tokens')} | "
              f"{c('yellow', f'{latency:.1f}s')}")

        return parsed, usage, latency

    def _select_answer(self, question_data, round_num):
        """Agent selects an answer. Returns (index, letter)."""
        correct_letter = question_data.get("correct", "A")
        correct_index = {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct_letter.upper(), 0)

        if self.fail_at and round_num == self.fail_at:
            # Pick wrong answer
            wrong_index = (correct_index + 1) % 4
            wrong_letter = ["A", "B", "C", "D"][wrong_index]
            return wrong_index, wrong_letter
        return correct_index, correct_letter

    def run(self):
        """Execute full game flow."""
        header(f"MILIONAR TEST -- {self.media_title}")
        print(f"  Provider: {c('cyan', self.settings['provider'])}")
        print(f"  Model: {c('cyan', self.settings['model'])}")
        print(f"  Scenario: {c('yellow', self.scenario)}")
        print(f"  Media: {c('green', self.media_title)}")
        separator()

        # --- Build system prompt ---
        self.system_prompt = build_system_prompt(
            media_type=self.media_type,
            title=self.media_title,
            plot=self.media_plot,
            genre=self.genre,
        )
        self._report(f"## System Prompt\n\n```\n{self.system_prompt[:500]}...\n```\n")
        print(f"\n{c('bold', 'System prompt built')} ({len(self.system_prompt)} chars)")

        # --- Welcome inference ---
        header("INFERENCE 1: WELCOME")
        welcome_msg = build_welcome_message()
        print(f"  User: {c('dim', welcome_msg)}")

        welcome_data, welcome_usage, welcome_lat = self._chat_inference(welcome_msg)

        moderator_welcome = welcome_data.get("moderator_welcome", "")
        rules_summary = welcome_data.get("rules_summary", "")
        print(f"\n  {c('green', 'moderator_welcome:')} {moderator_welcome}")
        print(f"  {c('green', 'rules_summary:')} {rules_summary}")
        self._report(f"## Welcome (inference #1)\n\n**Moderator:** {moderator_welcome}\n\n**Rules:** {rules_summary}\n")

        # TTS: welcome
        if self.tts_enabled:
            tts_text = compose_welcome_tts(
                self.settings["tts"], moderator_welcome, rules_summary
            )
            play_tts_sync(self.settings, tts_text)

        # --- Game rounds ---
        for round_num in range(1, 10):
            self.current_round = round_num
            money = MONEY_AMOUNTS.get(round_num, "?")

            # --- Question inference ---
            header(f"KOLO {round_num}/9 -- {money}")
            question_msg = build_question_message(round_num)
            print(f"  User: {c('dim', question_msg)}")

            question_data, q_usage, q_lat = self._chat_inference(question_msg)

            moderator_intro = question_data.get("moderator_intro", "")
            question_text = question_data.get("question", "")
            options = question_data.get("options", {})
            correct = question_data.get("correct", "A")
            explanation = question_data.get("explanation", "")

            print(f"\n  {c('magenta', 'Moderator:')} {moderator_intro}")
            print(f"  {c('bold', 'Otazka:')} {question_text}")
            for letter in ["A", "B", "C", "D"]:
                opt = options.get(letter, "") if isinstance(options, dict) else ""
                marker = c('green', ' <--') if letter == correct else ""
                print(f"    {letter}: {opt}{marker}")
            print(f"  {c('dim', f'Vysvetleni: {explanation}')}")

            # TTS: question
            if self.tts_enabled:
                opt_list_tts = [options.get(l, "") for l in ["A", "B", "C", "D"]] if isinstance(options, dict) else []
                tts_text = compose_question_tts(
                    self.settings["tts"], moderator_intro, question_text, opt_list_tts
                )
                play_tts_sync(self.settings, tts_text)

            # --- Lifeline simulation ---
            if round_num in self.lifeline_schedule:
                for ll_type in self.lifeline_schedule[round_num]:
                    print(f"  {c('yellow', f'>> NAPOVEDA: {ll_type.upper()}')}")
                    self._report(f"**Napoveda v kole {round_num}:** {ll_type}\n")

            # --- Agent selects answer ---
            answer_idx, answer_letter = self._select_answer(question_data, round_num)
            opt_list = [options.get(l, "") for l in ["A", "B", "C", "D"]] if isinstance(options, dict) else []
            answer_text = opt_list[answer_idx] if answer_idx < len(opt_list) else ""
            correct_idx = {"A": 0, "B": 1, "C": 2, "D": 3}.get(correct.upper(), 0)
            correct_text = opt_list[correct_idx] if correct_idx < len(opt_list) else ""
            is_correct = (answer_idx == correct_idx)

            status = c('green', 'SPRAVNE') if is_correct else c('red', 'SPATNE')
            print(f"\n  >> Agent vybral: {c('bold', f'{answer_letter}: {answer_text}')} -- {status}")

            # --- Feedback inference ---
            is_last = (round_num >= 9)
            feedback_msg = build_feedback_message(
                round_num=round_num,
                player_answer_letter=answer_letter,
                player_answer_text=answer_text,
                correct_answer_letter=correct,
                correct_answer_text=correct_text,
                is_correct=is_correct,
                is_last_round=is_last,
            )
            print(f"  User: {c('dim', feedback_msg[:120])}")

            feedback_data, f_usage, f_lat = self._chat_inference(feedback_msg)

            result_text = feedback_data.get("result_announcement", "")
            transition_text = feedback_data.get("transition_text", "")
            print(f"  {c('magenta', 'Vysledek:')} {result_text}")
            print(f"  {c('magenta', 'Prechod:')} {transition_text}")

            # TTS: feedback
            if self.tts_enabled:
                tts_text = compose_feedback_tts(
                    self.settings["tts"], result_text, transition_text, explanation
                )
                play_tts_sync(self.settings, tts_text)

            # Report
            self._report(
                f"### Kolo {round_num}/9 -- {money}\n\n"
                f"**Moderator intro:** {moderator_intro}\n\n"
                f"**Otazka:** {question_text}\n\n"
                f"| | Odpoved |\n|---|---|\n"
                f"| A | {options.get('A', '')} |\n"
                f"| B | {options.get('B', '')} |\n"
                f"| C | {options.get('C', '')} |\n"
                f"| D | {options.get('D', '')} |\n\n"
                f"**Spravna:** {correct} | "
                f"**Agent vybral:** {answer_letter} | "
                f"**Vysledek:** {'SPRAVNE' if is_correct else 'SPATNE'}\n\n"
                f"**Vysvetleni:** {explanation}\n\n"
                f"**Feedback:** {result_text}\n\n"
                f"**Prechod:** {transition_text}\n\n"
            )

            if is_correct:
                self.score += 1
            else:
                print(f"\n  {c('red', 'KONEC HRY -- spatna odpoved v kole ' + str(round_num))}")
                break

        # --- Summary ---
        self._print_summary()
        return self._generate_report()

    def _print_summary(self):
        """Print final summary to console."""
        header("SOUHRN")
        total_tokens = self.total_prompt_tokens + self.total_completion_tokens
        # Mistral Large pricing: $2/$6 per 1M tokens (input/output)
        input_cost = (self.total_prompt_tokens / 1_000_000) * 2.0
        output_cost = (self.total_completion_tokens / 1_000_000) * 6.0
        total_cost = input_cost + output_cost

        print(f"  Score: {c('bold', f'{self.score}/9')}")
        print(f"  Inferencí: {c('cyan', str(self.inference_count))}")
        print(f"  Prompt tokens: {c('cyan', str(self.total_prompt_tokens))}")
        print(f"  Completion tokens: {c('cyan', str(self.total_completion_tokens))}")
        print(f"  Celkem tokens: {c('bold', str(total_tokens))}")
        print(f"  Celkova latence: {c('yellow', f'{self.total_latency:.1f}s')}")
        print(f"  Prumerna latence: {c('yellow', f'{self.total_latency / max(self.inference_count, 1):.1f}s')}")
        print(f"  Odhadovana cena: {c('green', f'${total_cost:.4f}')}")
        print(f"    Input: ${input_cost:.4f} ({self.total_prompt_tokens} tokens x $2/1M)")
        print(f"    Output: ${output_cost:.4f} ({self.total_completion_tokens} tokens x $6/1M)")
        separator()

    def _report(self, text):
        self.report_lines.append(text)

    def _generate_report(self):
        """Generate final markdown report."""
        total_tokens = self.total_prompt_tokens + self.total_completion_tokens
        input_cost = (self.total_prompt_tokens / 1_000_000) * 2.0
        output_cost = (self.total_completion_tokens / 1_000_000) * 6.0
        total_cost = input_cost + output_cost
        avg_latency = self.total_latency / max(self.inference_count, 1)

        report = f"""# Milionar Test Report

**Datum:** {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}
**Provider:** {self.settings['provider']}
**Model:** {self.settings['model']}
**Scenario:** {self.scenario}
**Media:** {self.media_title} (series)
**Difficulty:** {self.settings['difficulty']} (0=easy, 1=medium, 2=hard)

---

## Vysledky

| Metrika | Hodnota |
|---------|---------|
| Score | {self.score}/9 |
| Pocet inferencí | {self.inference_count} |
| Prompt tokens | {self.total_prompt_tokens:,} |
| Completion tokens | {self.total_completion_tokens:,} |
| Celkem tokens | {total_tokens:,} |
| Celkova latence | {self.total_latency:.1f}s |
| Prumerna latence | {avg_latency:.1f}s |
| Cena input | ${input_cost:.4f} |
| Cena output | ${output_cost:.4f} |
| **Celkova cena** | **${total_cost:.4f}** |

---

## Prubeh hry

"""
        report += "\n".join(self.report_lines)

        report += f"""
---

## Token Usage Detail

| Inference | Popis | Prompt Tokens | Completion Tokens |
|-----------|-------|---------------|-------------------|
"""
        # We don't have per-inference data here, but total is useful

        report += f"""
**Celkem:** {self.total_prompt_tokens:,} prompt + {self.total_completion_tokens:,} completion = {total_tokens:,} tokens

### Cenova analyza (Mistral Large: $2/$6 per 1M)

- Jedna kompletni hra (9 kol): **~${total_cost:.4f}**
- Odhad na 100 her: **~${total_cost * 100:.2f}**
- Odhad na 1000 her: **~${total_cost * 1000:.2f}**

---

## Hodnoceni a doporuceni

### Silne stranky
- (bude doplneno agentem po behu testu)

### Slabe stranky
- (bude doplneno agentem po behu testu)

### Doporuceni
- (bude doplneno agentem po behu testu)
"""
        return report


# ---------------------------------------------------------------------------
# CLI
# ---------------------------------------------------------------------------

def parse_lifeline_schedule(spec):
    """Parse lifeline spec like '5050:3,tip:5' -> {3: ['5050'], 5: ['tip']}"""
    schedule = {}
    if not spec:
        return schedule
    for part in spec.split(","):
        parts = part.strip().split(":")
        if len(parts) == 2:
            ll_type = parts[0].strip()
            round_num = int(parts[1].strip())
            schedule.setdefault(round_num, []).append(ll_type)
    return schedule


def main():
    parser = argparse.ArgumentParser(description="Millionaire Game Test Harness")
    parser.add_argument(
        "--scenario", default="all-correct",
        help="Test scenario: all-correct, fail-at-N (e.g. fail-at-5), random"
    )
    parser.add_argument(
        "--use-lifeline", default="",
        help="Lifeline schedule: 5050:3,tip:5 (type:round)"
    )
    parser.add_argument(
        "--media", default="Zaklínač",
        help="Media title for the game"
    )
    parser.add_argument(
        "--type", default="series",
        help="Media type for the game (movie or series)"
    )
    parser.add_argument(
        "--genre", default="Fantasy, Drama",
        help="Media genre"
    )
    parser.add_argument(
        "--plot", default="Geralt z Rivie, zaklínač, lovec monster, putuje Kontinentem a hledá svůj osud. Postupně se setkává s čarodějkou Yennefer a s princeznou Ciri, která je jeho osudem spojená.",
        help="Plot description"
    )
    parser.add_argument(
        "--report-dir", default="",
        help="Directory for report output (default: .agent/tests/reports/)"
    )
    parser.add_argument(
        "--tts", action="store_true", default=False,
        help="Enable TTS playback (generates audio via OpenAI API and plays via afplay)"
    )
    args = parser.parse_args()

    # Load settings
    settings = load_settings()
    if not settings.get("api_key"):
        print(c("red", "ERROR: API klic neni nastaven v Kodi settings!"))
        sys.exit(1)

    # Parse lifelines
    lifeline_schedule = parse_lifeline_schedule(args.use_lifeline)

    # Run test
    runner = MillionaireTestRunner(
        settings=settings,
        media_title=args.media,
        media_plot=args.plot,
        scenario=args.scenario,
        lifeline_schedule=lifeline_schedule,
        media_type=args.type,
        genre=args.genre,
        tts_enabled=args.tts,
    )

    report = runner.run()

    # Save report
    report_dir = args.report_dir or os.path.join(
        ADDON_DIR, ".agent", "tests", "reports"
    )
    os.makedirs(report_dir, exist_ok=True)
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    report_path = os.path.join(report_dir, f"millionaire_test_{timestamp}.md")
    with open(report_path, "w", encoding="utf-8") as f:
        f.write(report)
    print(f"\n{c('green', 'Report ulozen:')} {report_path}")


if __name__ == "__main__":
    main()
