---
name: openrouter-api-dev
description: >
  Use this skill when building applications with OpenRouter unified LLM proxy,
  accessing 500+ models from 60+ providers through a single API.
  OpenAI-compatible, supports streaming, tool calling, and model routing.
---

# OpenRouter API Development Skill

You are an expert at integrating OpenRouter unified LLM API into Python applications.

## Overview

OpenRouter provides a single API gateway to 500+ models from 60+ providers:
- **Unified API** -- one endpoint, one key, all providers
- **OpenAI-compatible** -- drop-in replacement for OpenAI endpoints
- **Streaming** -- real-time SSE output
- **Tool Calling** -- standardized across providers
- **Model Routing** -- automatic fallbacks and cost optimization
- **Provider Preferences** -- choose specific providers per request

## Why Use OpenRouter

- **Single integration** for all providers (OpenAI, Anthropic, Google, Mistral, Meta, etc.)
- **Automatic fallbacks** -- if one provider fails, routes to another
- **Cost optimization** -- choose cheapest provider for each model
- **No vendor lock-in** -- switch models without code changes
- **Free tier** -- some models available for free

## API Base

```
https://openrouter.ai/api/v1/
```

Authentication: `Authorization: Bearer $OPENROUTER_API_KEY`

## Model Naming

Models use `provider/model-name` format:
```
openai/gpt-4.1
anthropic/claude-sonnet-4.5
google/gemini-3-flash-preview
mistralai/mistral-large-2512
meta-llama/llama-3.3-70b-instruct
```

## Quick Start (Python)

```python
import json
import urllib.request

payload = json.dumps({
    'model': 'openai/gpt-4.1-mini',
    'messages': [
        {'role': 'system', 'content': 'You are helpful.'},
        {'role': 'user', 'content': user_message}
    ]
}).encode('utf-8')

req = urllib.request.Request(
    'https://openrouter.ai/api/v1/chat/completions',
    data=payload,
    headers={
        'Authorization': f'Bearer {api_key}',
        'Content-Type': 'application/json'
    }
)

with urllib.request.urlopen(req) as resp:
    data = json.loads(resp.read().decode('utf-8'))
    answer = data['choices'][0]['message']['content']
```

## Provider Routing

```python
payload['provider'] = {
    'order': ['Anthropic', 'OpenAI'],  # prefer Anthropic, fallback OpenAI
    'allow_fallbacks': True
}
```

## Cost Control

```python
# Use auto-routing for cheapest option
payload['model'] = 'openrouter/auto'

# Or sort by price
payload['provider'] = {
    'sort': 'price'  # cheapest first
}
```

## API Docs (source of truth)

**llms.txt URL**: `https://openrouter.ai/docs/llms.txt`

Key documentation pages:

- [Quick Start](https://openrouter.ai/docs/quickstart)
- [Models](https://openrouter.ai/docs/models)
- [Tool Calling](https://openrouter.ai/docs/tool-calling)
- [Streaming](https://openrouter.ai/docs/streaming)
- [Provider Routing](https://openrouter.ai/docs/provider-routing)

## Related Skills
- `ai-api` -- general AI integration patterns (Python, streaming, agent loops)
- `system-prompt-master` -- prompt engineering
