---
name: kodi-texture-generator
description: Programmatic generation of masks and textures using PIL/Pillow. Use for creating rounded corners, pill shapes, scrollbars and other geometric UI elements in Python code. DO NOT use for generating photos or complex graphics.
---

# Kodi Texture Generator (PIL)

## When to Use
- Creating **masks** for rounded corners
- Generating **geometric shapes** (pill, rounded rect, scrollbar)
- Programmatic textures with **transparency** (alpha channel)
- Python code using **PIL/Pillow**

## When NOT to Use
- Generating photos, icons, illustrations -> use `kodi-image-generator`
- Complex graphics requiring AI generator

---

## Code Pattern

### Basic Structure
```python
#!/usr/bin/env python3
from PIL import Image, ImageDraw

# Dimensions
WIDTH, HEIGHT = 100, 100
RADIUS = 25

# Transparent background
img = Image.new('RGBA', (WIDTH, HEIGHT), (0, 0, 0, 0))
draw = ImageDraw.Draw(img)

# Draw shape (white fill for colordiffuse)
draw.rounded_rectangle(
    [(0, 0), (WIDTH - 1, HEIGHT - 1)],
    radius=RADIUS,
    fill=(255, 255, 255, 255)
)

# Save
img.save('output.png', 'PNG')
```

### Antialiasing (for smooth edges)
```python
SCALE = 4  # Upscale factor
img = Image.new('RGBA', (WIDTH * SCALE, HEIGHT * SCALE), (0, 0, 0, 0))
# ... draw at high resolution ...
img = img.resize((WIDTH, HEIGHT), Image.LANCZOS)  # Downscale
```

---

## Texture Types

| Type | Description | Radius |
|------|-------------|--------|
| Rounded rect | Rounded rectangle | `radius = 25` (or as needed) |
| Pill | Fully rounded ends | `radius = HEIGHT // 2` |
| Circle | Circle | `radius = WIDTH // 2` |

## Rules

1. **Always white fill** `(255, 255, 255, 255)` - color is set by Kodi via `colordiffuse`
2. **Transparent background** `(0, 0, 0, 0)`
3. **PNG format** - preserves alpha channel
4. **Script location** - `resources/media/generate_*.py`
5. **Output** - `resources/media/*.png`

## Diffuse Mask Rules (CRITICAL)

When generating masks used as `diffuse` attribute with `scalediffuse="false"`:

1. **Mask dimensions MUST match the XML control dimensions exactly (1:1)**
   - Control `<width>1560</width> <height>290</height>` → mask must be 1560x290px
   - NEVER use smaller mask and expect Kodi to scale it correctly
2. **NEVER share one mask across controls of different sizes**
   - Each unique control size needs its own dedicated mask file
   - A 200x200 square mask CANNOT work for a 1560x290 rectangle
3. **Radius is in screen pixels, not proportional**
   - For consistent visual radius across dialog bg and content images, use the SAME radius value (e.g. 35px) in ALL masks
   - The radius must represent actual screen pixels, so masks must be at control resolution
4. **9-slice textures (border) vs diffuse masks are different**
   - 9-slice: corners render at native pixel size, center stretches → radius is preserved
   - Diffuse with scalediffuse="false": mask maps to image texture space → dimensions must match control

## Common Use Cases

- `btn_rounded.png` - button with rounded corners
- `btn_pill.png` - pill-shaped button
- `scrollbar_rounded.png` - scrollbar with rounded ends
- `card_mask.png` - mask for cropping posters
- `still_mask_rounded.png` - mask for images
