---
name: kodi-refactorer
description: >
  Principles and techniques for refactoring Kodi addon code -- Python, XML, and general architecture.
  Use when eliminating duplication (DRY), restructuring modules, splitting files,
  improving code architecture, or cleaning up dead code.
---

# Kodi Refactorer

Unified skill for refactoring Kodi addon code (Python, XML, general architecture). ALWAYS follow `modular-code.md` and `python-syntax.md`. Use `kodi-researcher` for API verification.

## Principles
- DRY: 2+ occurrences -> shared function/module
- Single Responsibility: 1 function = 1 task
- Separation of Concerns: UI / business logic / API in separate files

## Python Refactoring
```python
# DRY -- shared utility
# wcs/utils/formatting.py
def format_duration(seconds):
    """Convert seconds to human readable duration."""
    hours, remainder = divmod(seconds, 3600)
    minutes, secs = divmod(remainder, 60)
    return f"{hours}h {minutes}m"

# Configuration instead of hardcoded values
# wcs/constants.py
AI_PROVIDERS = ['openai', 'anthropic', 'google', 'mistral', 'groq']

# Extract class to separate file
# BAD: everything in one file
# GOOD: wcs/gui/game_window.py, wcs/gui/chat_window.py

# Extract method
def validate_answer(self, answer):
    """Single responsibility -- only validates."""
    return answer == self.correct_answer
```

## XML Refactoring
```xml
<!-- Shared includes instead of duplicated controls -->
<include name="CommonButton">
    <control type="button">
        <width>300</width>
        <height>60</height>
    </control>
</include>
```

## Procedure
1. **Identify** -- find duplicates, determine scope, map dependencies
2. **Plan** -- target structure, order of changes, fallback strategy
3. **Implement** -- create abstraction, replace references gradually
4. **Cleanup** -- remove dead code, update imports, verify references

## Safety Rules
NEVER: change function signatures without checking all callers, remove exports, rename without searching for occurrences, refactor multiple things at once.
ALWAYS: `grep_search` for all occurrences, check imports, verify nothing breaks.

## Checklist
- [ ] Followed `modular-code.md`?
- [ ] Follows project rules in `.agent/rules/`?
- [ ] Change is needed (2+ occurrences)?
- [ ] Backward compatibility preserved?
- [ ] All occurrences found via `grep_search`?
- [ ] Imports updated in all affected files?
- [ ] No dead code left behind?
