---
name: elevenlabs-tts
description: >
  Use this skill when integrating ElevenLabs TTS API for text-to-speech,
  voice cloning, streaming audio, or voice generation.
  Covers REST API, Python urllib integration, models, voices, pricing, cost calculation, and streaming.
---

# ElevenLabs TTS Skill

Expert na integraci ElevenLabs TTS API do Python aplikací.

## Přehled

- **5000+ hlasů** -- library, community, vlastní klony
- **Voice cloning** -- instant a professional
- **Streaming** -- WebSocket a HTTP streaming
- **70+ jazyků** -- multilingvální podpora
- **48kHz sample rate** -- přirozené dechy, mikro-pauzy
- **Audio tags** (v3) -- inline kontrola emocí a tónu

---

## Modely (aktuální, únor 2026)

| Model ID | Latence | Jazyky | Limit/req | Nejlepší pro |
|----------|---------|--------|-----------|--------------|
| `eleven_flash_v2_5` | ~75ms | 32 | 40 000 zn | Real-time konverzace |
| `eleven_turbo_v2_5` | ~250ms | 32 | 40 000 zn | Kvalita + rychlost |
| `eleven_multilingual_v2` | ~300ms | 29 | 40 000 zn | Narrace, konzistence |
| `eleven_v3` | ~300ms | 70+ | **5 000 zn** | Expresivita, drama, audio tags |

> Pro real-time a herní TTS: `eleven_flash_v2_5`
> Pro nejlepší expresivitu a emoce: `eleven_v3`

---

## eleven_v3 -- Nová funkce (GA únor 2026)

Model vyšel z alpha fáze. Nové schopnosti:

### Audio Tags
Inline kontrola emocí a výkonu pomocí tagů v `[hranatých závorkách]`:

**Emoce:**
```
[excited] [nervous] [frustrated] [sorrowful] [calm] [sad] [angry]
[happily] [tired] [curious] [mischievously]
```

**Reakce a neverbální projevy:**
```
[sigh] [sighs] [laughs] [laughing] [gulps] [gasps] [whispers]
[crying] [clears throat] [snorts] [exhales] [panting] [hesitates]
[stammers] [pauses]
```

**Tón a styl:**
```
[cheerfully] [flatly] [deadpan] [playfully] [dramatic] [sarcastically]
[matter-of-fact] [whiny] [whispering] [shouting] [quietly] [loudly] [rushed]
```

**Charakter a přízvuk:**
```
[pirate voice] [evil scientist voice] [childlike tone]
[British accent] [Australian accent] [Southern US accent] [French accent]
```

**Zvukové efekty:**
```
[gunshot] [clapping] [explosion] [door creaks]
```

> Celkem 1806+ tagů ve 15 kategoriích (viz [v3 Tag Library](https://elevenlabs.io/docs/capabilities/text-to-speech/prompting))

**Příklad:**
```python
text = "[excited] Správná odpověď! [laughs] Výborně! [calm] Přejdeme na další otázku."
```

**Doporučení pro audio tags:** Nastavit `stability` na nízku hodnotu (0.3--0.5) pro více expresivity.

### Dialogue Mode (multi-speaker)
Nový API endpoint pro přirozené multi-speaker konverzace:
- Přirozené pauzy a přerušení mezi mluvčími
- Emoční přechody
- Překrývající se řeč

```
POST /v1/text-to-dialogue
```

---

## API -- Základní informace

```
Base URL: https://api.elevenlabs.io/v1/
Auth header: xi-api-key: <API_KEY>
```

### Klíčové endpointy

| Endpoint | Popis |
|----------|-------|
| `POST /v1/text-to-speech/{voice_id}` | Generování audia (binary) |
| `POST /v1/text-to-speech/{voice_id}/stream` | Streaming audio chunks |
| `POST /v1/text-to-dialogue` | Multi-speaker dialogue (v3) |
| `GET /v1/voices` | Seznam hlasů s voice_id |
| `GET /v1/models` | Seznam modelů s vlastnostmi |
| `GET /v1/user/subscription` | Stav předplatného + kredity |

---

## Quick Start (Python urllib)

```python
import json
import urllib.request

ELEVENLABS_API_KEY = 'your_api_key'
voice_id = 'pNInz6obpgDQGcFmaJgB'  # Adam

payload = json.dumps({
    'text': 'Text ke konverzi',
    'model_id': 'eleven_flash_v2_5',
    'voice_settings': {
        'stability': 0.5,
        'similarity_boost': 0.75,
        'style': 0.0,
        'use_speaker_boost': True
    }
}).encode('utf-8')

req = urllib.request.Request(
    f'https://api.elevenlabs.io/v1/text-to-speech/{voice_id}',
    data=payload,
    headers={
        'xi-api-key': ELEVENLABS_API_KEY,
        'Content-Type': 'application/json',
        'Accept': 'audio/mpeg'
    }
)

with urllib.request.urlopen(req) as resp:
    # Přečíst x-character-count z headers
    char_count = resp.headers.get('x-character-count', 0)
    audio_data = resp.read()
    with open('output.mp3', 'wb') as f:
        f.write(audio_data)
```

---

## Response Headers

| Header | Popis |
|--------|-------|
| `x-character-count` | Počet zpracovaných znaků (základ pro výpočet ceny) |
| `x-request-id` | ID requestu (pro `previous_request_ids`) |
| `Content-Type` | `audio/mpeg` nebo dle `output_format` |

> **DŮLEŽITÉ:** API **nevrací cenu v Kč/USD**. Cenu je třeba počítat ručně z `x-character-count`.

---

## Výpočet ceny

```python
# Ceny za 1 000 znaků podle plánu (zahrnuté znaky)
PRICE_PER_1K = {
    'free':     0.00,   # do limitu zdarma
    'starter':  0.22,
    'creator':  0.20,
    'pro':      0.17,
    'scale':    0.12,
}

# Overage ceny (po vyčerpání limitu)
OVERAGE_PER_1K = {
    'starter':  0.30,
    'creator':  0.24,
    'pro':      0.18,
    'scale':    0.12,
}

def calculate_cost(char_count, plan='creator', is_overage=False):
    """Vypočítá cenu TTS requestu v USD."""
    prices = OVERAGE_PER_1K if is_overage else PRICE_PER_1K
    cost = (char_count / 1000.0) * prices.get(plan, 0.20)
    return round(cost, 6)

# Příklad: 44 znaků na Creator plánu
# calculate_cost(44, 'creator') -> 0.00088 USD
```

### Zjištění stavu kreditů

```python
def get_subscription_info(api_key):
    """Vrátí zbývající kredity a limit z ElevenLabs API."""
    req = urllib.request.Request(
        'https://api.elevenlabs.io/v1/user/subscription',
        headers={'xi-api-key': api_key}
    )
    with urllib.request.urlopen(req) as resp:
        data = json.loads(resp.read())
    return {
        'limit': data['character_limit'],
        'used': data['character_count'],
        'remaining': data['character_limit'] - data['character_count'],
        'plan': data.get('tier', 'unknown')
    }
```

---

## Pricing (únor 2026)

### TTS -- cena za 1 000 znaků

| Plán | Cena/měsíc | Znaky/měsíc | Cena/1K (inc.) | Overage/1K |
|------|-----------|-------------|-----------------|------------|
| Free | $0 | 10 000 | -- | -- |
| Starter | $5 | 30 000 | $0.22 | $0.30 |
| Creator | $22 | 100 000 | $0.20 | $0.24 |
| Pro | $99 | 500 000 | $0.17 | $0.18 |
| Scale | $330 | 2 000 000 | $0.12 | $0.12 |
| Business | $1 100+ | 11 000 000 | -- | -- |

> Flash/Turbo modely = **0.5 kreditu/znak** (poloviční cena oproti v2/v3 = 1 kredit/znak) na self-serve plánech.

---

## Zjištění hlasů

### Všechny hlasy (API)

```python
def get_voices(api_key):
    """Vrátí seznam všech dostupných hlasů s voice_id."""
    req = urllib.request.Request(
        'https://api.elevenlabs.io/v1/voices',
        headers={'xi-api-key': api_key}
    )
    with urllib.request.urlopen(req) as resp:
        data = json.loads(resp.read())
    return [
        {
            'voice_id': v['voice_id'],
            'name': v['name'],
            'labels': v.get('labels', {}),
            'preview_url': v.get('preview_url')
        }
        for v in data.get('voices', [])
    ]
```

### Výchozí hlasy (Default Voices)

| Hlas | Voice ID | Pohlaví | Styl |
|------|----------|---------|------|
| Adam | `pNInz6obpgDQGcFmaJgB` | M | Neutral, hluboký |
| Rachel | `21m00Tcm4TlvDq8ikWAM` | F | Calm, narrace |
| Antoni | `ErXwobaYiN019PkySvjV` | M | Warm |
| Josh | `TxGEqnHWrfWFTfGW9XjX` | M | Deep |

> Kompletní seznam s preview: [ElevenLabs Voice Library](https://elevenlabs.io/voice-library)
> API endpoint: `GET /v1/voices`

### Hlasy podporující eleven_v3 a audio tags

Všechny hlasy z `GET /v1/voices` jsou kompatibilní s `eleven_v3`. Doporučené pro audio tags:

```python
# Filtrovat hlasy vhodné pro eleven_v3
def get_v3_compatible_voices(api_key):
    voices = get_voices(api_key)
    # Všechny hlasy jsou kompatibilní, ale v3 lépe funguje s
    # hlasy navrženými pro expresivitu (labels: use_case = 'narration' nebo 'characters')
    return [v for v in voices if v.get('labels', {}).get('use_case') in ('narration', 'characters', None)]
```

### Voice Design (only v3)

```python
# Vytvořit hlas pomocí textového promptu
payload = json.dumps({
    'voice_description': 'An elderly Czech storyteller with a warm, gravelly voice, slow pacing',
    'text': 'Testovací věta pro náhled hlasu.'
}).encode('utf-8')

req = urllib.request.Request(
    'https://api.elevenlabs.io/v1/voice-generation/generate-voice',
    data=payload,
    headers={
        'xi-api-key': api_key,
        'Content-Type': 'application/json'
    }
)
```

---

## Voice Settings

| Parametr | Rozsah | Doporučeno (standard) | Pro audio tags v3 |
|----------|--------|----------------------|-------------------|
| `stability` | 0.0--1.0 (flash: continuous, **v3: enum 0.0/0.5/1.0**) | 0.5 | **0.0** (Creative -- max expresivita) |
| `similarity_boost` | 0.0--1.0 | 0.75 | 0.75 |
| `style` | 0.0--1.0 | 0.0 | 0.3--0.5 |
| `use_speaker_boost` | bool | `true` | `true` |

> **POZOR:** Model `eleven_v3` akceptuje POUZE diskrétní hodnoty stability: `0.0` (Creative), `0.5` (Natural), `1.0` (Robust). Jakákoliv jiná hodnota vrátí HTTP 400.

---

## Výstupní formáty

| Formát | Poznámka |
|--------|----------|
| `mp3_44100_128` | Default, doporučeno |
| `mp3_22050_32` | Menší soubor |
| `mp3_44100_192` | Vyžaduje Creator+ |
| `pcm_44100` | Vyžaduje Pro+ |
| `ulaw_8000` | Twilio |

Nastavení: query parametr `?output_format=mp3_44100_128`

---

## Dokumentace a zdroje

- [Text-to-Speech API](https://elevenlabs.io/docs/api-reference/text-to-speech)
- [eleven_v3 Overview](https://elevenlabs.io/docs/capabilities/text-to-speech/overview)
- [eleven_v3 Prompting Guide (audio tags)](https://elevenlabs.io/docs/capabilities/text-to-speech/prompting)
- [Text to Dialogue API](https://elevenlabs.io/docs/api-reference/text-to-dialogue)
- [Voices API](https://elevenlabs.io/docs/api-reference/voices)
- [Models API](https://elevenlabs.io/docs/api-reference/models)
- [Voice Cloning](https://elevenlabs.io/docs/guides/voice-cloning)
- [Streaming](https://elevenlabs.io/docs/api-reference/streaming)
- [API Pricing](https://elevenlabs.io/pricing/api)
- [Voice Library](https://elevenlabs.io/voice-library)

## Související skills
- `ai-api` -- AI integrace (Python, TTS)
- `tts-voice-instructor` -- voice instruction engineering (OpenAI TTS)