# Debugging Strategies Implementation Playbook

This file contains detailed patterns, checklists, and code samples referenced by the skill.

## Core Principles

### 1. The Scientific Method

**1. Observe**: What's the actual behavior?
**2. Hypothesize**: What could be causing it?
**3. Experiment**: Test your hypothesis
**4. Analyze**: Did it prove/disprove your theory?
**5. Repeat**: Until you find the root cause

### 2. Debugging Mindset

**Don't Assume:**
- "It can't be X" -- Yes it can
- "I didn't change Y" -- Check anyway
- "It works on my machine" -- Find out why

**Do:**
- Reproduce consistently
- Isolate the problem
- Keep detailed notes
- Question everything
- Take breaks when stuck

### 3. Rubber Duck Debugging

Explain your code and problem out loud (to a rubber duck, colleague, or yourself). Often reveals the issue.

## Systematic Debugging Process

### Phase 1: Reproduce

```markdown
## Reproduction Checklist

1. **Can you reproduce it?**
   - Always? Sometimes? Randomly?
   - Specific conditions needed?
   - Specific Kodi version or platform?

2. **Create minimal reproduction**
   - Simplify to smallest example
   - Remove unrelated code
   - Isolate the problem

3. **Document steps**
   - Write down exact steps
   - Note Kodi version, OS version
   - Capture error messages from kodi.log
```

### Phase 2: Gather Information

```markdown
## Information Collection

1. **Error Messages**
   - Full traceback from kodi.log
   - ERROR and WARNING lines
   - Filter for addon name: plugin.video.milionar

2. **Environment**
   - Kodi version (19/20/21+)
   - OS version (macOS, Windows, Linux, Android)
   - Python version (Kodi 19+ = Python 3)
   - Addon version

3. **Recent Changes**
   - Git history
   - Recent file modifications
   - Settings changes

4. **Scope**
   - All users or specific setup?
   - All screens or specific dialog?
   - Always or intermittent?
```

### Phase 3: Form Hypothesis

```markdown
## Hypothesis Formation

Based on gathered info, ask:

1. **What changed?**
   - Recent code changes
   - XML layout modifications
   - Settings or data changes

2. **What's different?**
   - Working vs broken environment
   - Before vs after the change
   - Different Kodi versions

3. **Where could this fail?**
   - Python logic (lifecycle, threading)
   - XML layout (IDs, navigation, includes)
   - File paths (casing, missing resources)
   - External services (network, API)
```

### Phase 4: Test & Verify

```markdown
## Testing Strategies

1. **Binary Search**
   - Comment out half the code
   - Narrow down problematic section
   - Repeat until found

2. **Add Debug Logging**
   - Strategic xbmc.log() with [DEBUG] prefix
   - Track variable values at key points
   - Trace execution flow through methods

3. **Isolate Components**
   - Test Python logic separately from XML
   - Mock Kodi API calls where possible
   - Remove complexity layer by layer

4. **Compare Working vs Broken**
   - Diff XML files
   - Diff Python modules
   - Check git blame
```

## Python/Kodi Debugging Tools

### Kodi Log Analysis

```python
import xbmc

# Strategic debug logging
xbmc.log(f"[DEBUG] Method entry: {method_name}", xbmc.LOGINFO)
xbmc.log(f"[DEBUG] State: {self._state}", xbmc.LOGINFO)
xbmc.log(f"[DEBUG] Control value: {value}", xbmc.LOGINFO)

# Error-level for caught exceptions
try:
    result = risky_operation()
except Exception as e:
    xbmc.log(f"[Milionar] Operation failed: {e}", xbmc.LOGERROR)
    import traceback
    xbmc.log(f"[Milionar] Traceback: {traceback.format_exc()}", xbmc.LOGERROR)
```

### Lifecycle Debugging

```python
# Debug WindowXML lifecycle
class MyDialog(xbmcgui.WindowXMLDialog):
    def __init__(self, *args, **kwargs):
        xbmc.log("[DEBUG] __init__ called", xbmc.LOGINFO)

    def onInit(self):
        xbmc.log("[DEBUG] onInit called", xbmc.LOGINFO)
        # Safe to access controls HERE

    def onClick(self, controlId):
        xbmc.log(f"[DEBUG] onClick: {controlId}", xbmc.LOGINFO)

    def onAction(self, action):
        xbmc.log(f"[DEBUG] onAction: {action.getId()}", xbmc.LOGINFO)

    def onFocus(self, controlId):
        xbmc.log(f"[DEBUG] onFocus: {controlId}", xbmc.LOGINFO)
```

### Threading Debugging

```python
import threading
import xbmc

def background_task():
    xbmc.log(f"[DEBUG] Thread: {threading.current_thread().name}", xbmc.LOGINFO)
    # NEVER access UI controls here!
    # Use callback to main thread instead

# Check thread safety
def check_thread():
    if threading.current_thread().name != 'MainThread':
        xbmc.log("[WARNING] UI access from background thread!", xbmc.LOGWARNING)
```

### Performance Profiling

```python
import time
import xbmc

# Simple timing
start = time.time()
# ... operation ...
elapsed = time.time() - start
xbmc.log(f"[DEBUG] Operation took: {elapsed:.3f}s", xbmc.LOGINFO)

# Context manager for profiling
class Timer:
    def __init__(self, name):
        self.name = name

    def __enter__(self):
        self.start = time.time()
        return self

    def __exit__(self, *args):
        elapsed = time.time() - self.start
        xbmc.log(f"[DEBUG] {self.name}: {elapsed:.3f}s", xbmc.LOGINFO)

with Timer("load_questions"):
    questions = load_questions_from_file()
```

## Debugging Patterns by Issue Type

### Pattern 1: Intermittent Bugs

```markdown
## Strategies for Flaky Bugs

1. **Add extensive logging**
   - Log timing information
   - Log all state transitions
   - Log thread names

2. **Look for race conditions**
   - Background threads touching UI
   - Async operations completing out of order
   - Missing locks on shared state

3. **Check Kodi lifecycle timing**
   - onInit not yet called
   - Controls not ready
   - Window closing during operation
```

### Pattern 2: XML Layout Issues

```markdown
## XML Debugging

1. **Validate XML syntax**
   - Missing closing tags
   - Wrong attribute names
   - Invalid control types

2. **Check control IDs**
   - Unique numeric IDs
   - Correct references in Python
   - Navigation IDs exist

3. **Focus debugging**
   - Default control set in window
   - All focusable controls have navigation
   - Focus chain has no dead ends

4. **Visual debugging**
   - Add temporary colored background to controls
   - Log control positions: getX(), getY(), getWidth(), getHeight()
```

## Best Practices

1. **Reproduce First**: Can't fix what you can't reproduce
2. **Isolate the Problem**: Remove complexity until minimal case
3. **Read Kodi Log**: Full tracebacks are usually there
4. **Check Recent Changes**: Most bugs are recent
5. **Use Version Control**: Git blame, diff, bisect
6. **Take Breaks**: Fresh eyes see better
7. **Document Findings**: Help future you
8. **Fix Root Cause**: Not just symptoms
9. **Clean Up Debug Logs**: ALWAYS remove `[DEBUG]` lines after fix

## Quick Debugging Checklist

```markdown
## When Stuck, Check:

- [ ] Full traceback in kodi.log
- [ ] Spelling errors (typos in variable/control names)
- [ ] Case sensitivity (file paths, XML attributes)
- [ ] None/missing values
- [ ] Thread safety (background thread touching UI)
- [ ] Lifecycle order (accessing controls before onInit)
- [ ] Control IDs match between XML and Python
- [ ] File paths exist (correct casing)
- [ ] __pycache__ stale files (clear and restart)
- [ ] XML syntax valid (matching tags, attributes)
- [ ] Import order (circular dependencies)
```
