---
name: debug
description: >
  Kodi addon debugging system -- adding diagnostic logs, reading/analyzing
  Kodi log output, systematic root cause analysis, and cleanup. Use when
  debugging errors, investigating unexpected behavior, analyzing Kodi log,
  diagnosing threading/lifecycle/focus issues, or running /debug workflow.
---

# Debug

You are an expert at systematic debugging of Kodi addons. You transform debugging from guesswork into methodical problem-solving using log-based diagnostics.

## System Overview

Three linked components form the debug system:

| Component | File | Purpose |
|-----------|------|---------|
| **Skill** | `.agent/skills/debug/SKILL.md` | Authority -- all debug procedures (this file) |
| **Rule** | `.agent/rules/debug.md` | Trigger -- when to activate debug skill |
| **Workflow** | `.agent/workflows/debug.md` | Interface -- `/debug create\|log\|end` commands |

---

## Kodi Log

**Path (macOS):**
```
/Users/chudy/Library/Logs/kodi.log
```

**Alternative paths:**
- Windows: `%APPDATA%\Kodi\kodi.log`
- Linux: `~/.kodi/temp/kodi.log`

---

## Phase 1: Create (`/debug create`)

Add diagnostic `[DEBUG]` logs to Python code.

### Format

```python
import xbmc
xbmc.log(f"[DEBUG] ComponentName.function_name: description {variable}", xbmc.LOGINFO)
```

### Rules

1. **Prefix** -- always `[DEBUG]` for easy filtering
2. **Component + function** -- include both in every log message
3. **Descriptive messages** -- explain WHAT is being measured
4. **Comment marker** -- add `# DEBUG` comment above each debug log line for easy cleanup
5. **Log level** -- use `xbmc.LOGINFO` for debug output

### Placement Strategy

| Issue Type | Where to Add Logs |
|------------|-------------------|
| Python crash | Around the traceback location, function inputs |
| UI not updating | In callback functions, before/after setLabel/setImage |
| Threading | Thread name, before/after thread operations |
| Focus/Navigation | onFocus, onAction, setFocusId calls |
| Lifecycle | onInit, __init__, close, doModal |
| Data flow | API responses, data transformations |
| Performance | Timing with time.time() around operations |
| Intermittent | State transitions, thread names, timing info |

### Example

```python
# DEBUG
xbmc.log(f"[DEBUG] GameDialog.on_answer_selected: answer_index={idx}, correct={is_correct}", xbmc.LOGINFO)
# DEBUG
xbmc.log(f"[DEBUG] GameDialog.on_answer_selected: callback result={result}", xbmc.LOGINFO)
```

---

## Phase 2: Log Analysis (`/debug log`)

Autonomously read and analyze Kodi log to find the root cause.

### Procedure

1. **Primary search** -- grep for `[DEBUG]` in kodi.log
2. **Error search** -- grep for `ERROR`, `WARNING`, `Traceback` related to `plugin.video.milionar`
3. **Iterate** -- if initial grep does not reveal the cause:
   - Try different search terms based on the problem context
   - Narrow or widen the search scope
   - Look for patterns, sequences, or missing expected output
4. **Fallback** -- if grep yields nothing useful, read last 200-300 lines of kodi.log
5. **Analyze** -- correlate debug output with the reported problem
6. **Conclude** -- identify root cause or request more debug logs (back to Phase 1)

### Search Strategy

```
Priority order:
1. grep "[DEBUG]" -- our diagnostic output
2. grep "ERROR" -- Kodi errors
3. grep "plugin.video.milionar" -- all addon output
4. grep "WARNING" -- potential issues
5. grep "Traceback" -- Python stack traces
6. tail -200 -- raw last lines as fallback
```

> [!IMPORTANT]
> The agent decides autonomously what to search for based on context.
> Do NOT stop after one grep -- iterate until the root cause is found or
> it is clear that more debug logs are needed.

---

## Phase 3: Cleanup (`/debug end`)

Remove all temporary debug output from the codebase.

### Procedure

1. **Search** -- grep for `[DEBUG]` across all `.py` files in the addon
2. **Identify** -- find all lines with `# DEBUG` comment marker and their associated `xbmc.log` lines
3. **Remove** -- delete both the comment marker and the log line
4. **Verify** -- grep again to confirm no `[DEBUG]` lines remain
5. **Report** -- list files cleaned and number of debug lines removed

> [!CAUTION]
> Only remove lines that were added as part of the debug session.
> Do NOT remove permanent logging that is part of production code.
> Production logs do NOT have `[DEBUG]` prefix or `# DEBUG` comment.

---

## Kodi-Specific Debug Patterns

### Common Issue Categories

| Category | Symptoms | First Check |
|----------|----------|-------------|
| **Python crash** | Traceback in log, addon stops | Kodi log -- last 200 lines |
| **XML layout** | Controls missing, wrong position | XML syntax, control IDs, includes |
| **Focus/Navigation** | Can't navigate, wrong focus | `onleft`/`onright`/`onup`/`ondown` in XML |
| **Threading** | Random crashes, UI freeze | Background thread touching UI controls |
| **Import error** | `ModuleNotFoundError` | `__pycache__` stale files, wrong path |
| **Lifecycle** | Controls not ready, `NoneType` | Accessing UI before `onInit()` completes |
| **Texture** | Missing images, white squares | Path casing, file existence, Textures13.db |

### Control Existence Check
```python
# DEBUG
try:
    ctrl = self.getControl(CONTROL_ID)
    xbmc.log(f"[DEBUG] Control {CONTROL_ID}: exists, type={type(ctrl).__name__}", xbmc.LOGINFO)
except RuntimeError:
    xbmc.log(f"[DEBUG] Control {CONTROL_ID}: NOT FOUND!", xbmc.LOGERROR)
```

### Thread Safety Check
```python
# DEBUG
import threading
xbmc.log(f"[DEBUG] Current thread: {threading.current_thread().name}", xbmc.LOGINFO)
```

### Lifecycle Debugging
```python
# DEBUG
class MyDialog(xbmcgui.WindowXMLDialog):
    def onInit(self):
        xbmc.log("[DEBUG] onInit called", xbmc.LOGINFO)
    def onClick(self, controlId):
        xbmc.log(f"[DEBUG] onClick: {controlId}", xbmc.LOGINFO)
    def onAction(self, action):
        xbmc.log(f"[DEBUG] onAction: {action.getId()}", xbmc.LOGINFO)
    def onFocus(self, controlId):
        xbmc.log(f"[DEBUG] onFocus: {controlId}", xbmc.LOGINFO)
```

### Performance Timing
```python
# DEBUG
import time
start = time.time()
# ... operation ...
elapsed = time.time() - start
xbmc.log(f"[DEBUG] Operation took: {elapsed:.3f}s", xbmc.LOGINFO)
```

### Stale __pycache__ Fix

```bash
# When getting mysterious ImportError or wrong traceback lines
find "/Users/chudy/Library/Application Support/Kodi/addons/plugin.video.milionar" \
  -type d -name __pycache__ -exec rm -rf {} +
```

> See also rule `pycache-clear.md` for when to apply this fix.

---

## Debugging Methodology

### Scientific Method

1. **Observe** -- What's the actual behavior?
2. **Hypothesize** -- What could be causing it?
3. **Experiment** -- Test your hypothesis with targeted logs
4. **Analyze** -- Did it prove/disprove your theory?
5. **Repeat** -- Until root cause is found

### When Stuck Checklist

- [ ] Full traceback in kodi.log
- [ ] Spelling errors (typos in variable/control names)
- [ ] Case sensitivity (file paths, XML attributes)
- [ ] None/missing values
- [ ] Thread safety (background thread touching UI)
- [ ] Lifecycle order (accessing controls before onInit)
- [ ] Control IDs match between XML and Python
- [ ] File paths exist (correct casing)
- [ ] `__pycache__` stale files (clear and restart)
- [ ] XML syntax valid (matching tags, attributes)
- [ ] Import order (circular dependencies)

---

## Resources

- `resources/implementation-playbook.md` for detailed debugging patterns and checklists
- Rule `debug.md` for activation trigger
- Rule `pycache-clear.md` for mysterious Python import errors
