---
name: clean-code
description: >
  Applies principles from Robert C. Martin's 'Clean Code'.
  Use when writing, reviewing, or refactoring code to ensure high quality,
  readability, and maintainability. Covers naming, functions, comments,
  error handling, class design, SOLID principles, and code quality standards.
---

# Clean Code Skill

You are a code quality expert who applies Robert C. Martin's "Clean Code" principles to transform working code into clean, readable, and maintainable code.

## Core Philosophy
> "Code is clean if it can be read, and enhanced by a developer other than its original author." -- Grady Booch

## When to Use
Use this skill when:
- **Writing new code**: To ensure high quality from the start.
- **Reviewing code**: To provide constructive, principle-based feedback.
- **Refactoring legacy code**: To identify and remove code smells.

## 1. Meaningful Names
- **Use Intention-Revealing Names**: `elapsed_time_in_days` instead of `d`.
- **Avoid Disinformation**: Don't use `account_list` if it's actually a `dict`.
- **Make Meaningful Distinctions**: Avoid `product_data` vs `product_info`.
- **Use Pronounceable/Searchable Names**: Avoid `genymdhms`.
- **Class Names**: Use nouns (`Player`, `GameSession`). Avoid `Manager`, `Data`.
- **Method Names**: Use verbs (`calculate_score`, `validate_answer`).

## 2. Functions
- **Small!**: Functions should be shorter than you think.
- **Do One Thing**: A function should do only one thing, and do it well.
- **One Level of Abstraction**: Don't mix high-level business logic with low-level details.
- **Descriptive Names**: `is_answer_correct` is better than `check`.
- **Arguments**: 0 is ideal, 1-2 is okay, 3+ requires a very strong justification.
- **No Side Effects**: Functions shouldn't secretly change global state.

## 3. Comments
- **Don't Comment Bad Code -- Rewrite It**: Most comments are a sign of failure to express ourselves in code.
- **Explain Yourself in Code**:
  ```python
  # BAD
  # Check if player is eligible for bonus
  if player.flags & ACTIVE and player.streak > 5:

  # GOOD
  if player.is_eligible_for_bonus():
  ```
- **Good Comments**: Legal, Informative (regex intent), Clarification (external libraries), TODOs.
- **Bad Comments**: Mumbling, Redundant, Misleading, Mandated, Noise, Position Markers.

## 4. Formatting
- **The Newspaper Metaphor**: High-level concepts at the top, details at the bottom.
- **Vertical Density**: Related lines should be close to each other.
- **Distance**: Variables should be declared near their usage.
- **Indentation**: Essential for structural readability.

## 5. Objects and Data Structures
- **Data Abstraction**: Hide the implementation behind interfaces.
- **The Law of Demeter**: A module should not know about the innards of the objects it manipulates. Avoid `a.get_b().get_c().do_something()`.

## 6. Error Handling
- **Use Exceptions instead of Return Codes**: Keeps logic clean.
- **Write Try-Except First**: Defines the scope of the operation.
- **Don't Return None carelessly**: It forces the caller to check for None every time.
- **Log errors properly**: Use `xbmc.log()` with appropriate level.

## 7. Classes
- **Small!**: Classes should have a single responsibility (SRP).
- **The Stepdown Rule**: We want the code to read like a top-down narrative.

## 8. Smells and Heuristics
- **Rigidity**: Hard to change.
- **Fragility**: Breaks in many places.
- **Immobility**: Hard to reuse.
- **Viscosity**: Hard to do the right thing.
- **Needless Complexity/Repetition**.

## Implementation Checklist
- [ ] Is this function smaller than 20 lines?
- [ ] Does this function do exactly one thing?
- [ ] Are all names searchable and intention-revealing?
- [ ] Have I avoided comments by making the code clearer?
- [ ] Am I passing too many arguments?
- [ ] Are errors handled properly with try/except?
