---
name: anthropic-api-dev
description: >
  Use this skill when building applications with Anthropic Claude models,
  Claude API (Messages API), implementing chat completions, streaming,
  tool use, structured outputs, or needing current model specs.
  Covers REST API usage, Python urllib integration, model selection, and capabilities.
---

# Anthropic Claude API Development Skill

You are an expert at integrating Anthropic Claude API into Python applications.

## Overview

The Claude API provides access to Anthropic's Claude models:
- **Messages API** -- chat completion (NOT OpenAI-compatible format)
- **Streaming** -- real-time SSE token output
- **Tool Use** -- function calling with input_schema
- **Vision** -- image understanding (base64 or URL)
- **Extended Thinking** -- chain-of-thought reasoning
- **Adaptive Thinking** -- model dynamically determines reasoning depth
- **Prompt Caching** -- cache large contexts (1-hour duration, now GA)
- **Context Compaction** -- server-side context summarization (beta)
- **Structured Outputs** -- JSON schema enforcement (GA)
- **Batch API** -- async batch processing

## Current Claude Models

- `claude-opus-4-6-20260205`: most intelligent, adaptive thinking, 1M context (beta), agentic
- `claude-opus-4-5-20250929`: previous flagship, coding, complex reasoning
- `claude-sonnet-5-20260203`: latest Sonnet, balanced performance/cost
- `claude-sonnet-4-5-20250929`: previous Sonnet, high-volume
- `claude-haiku-4-5-20251015`: fastest, most cost-effective

> [!CAUTION]
> **Deprecated models** (do NOT use):
> - `claude-opus-3` -- retired January 2026
> - `claude-haiku-3-5` -- deprecated December 2025
> - `claude-sonnet-3-5` -- retired October 2025
> Migrate to Opus 4.6 or Sonnet 5.

> [!IMPORTANT]
> Anthropic uses its OWN Messages API format -- NOT OpenAI-compatible.
> Key difference: `messages` array uses `content` blocks, not plain strings.
> The `system` prompt is a top-level parameter, not a message role.

## API Base

```
https://api.anthropic.com/v1/
```

Authentication:
- `x-api-key: $ANTHROPIC_API_KEY`
- `anthropic-version: 2023-06-01` (required header)

## Key Endpoints

| Endpoint | Purpose |
|----------|---------|
| `/v1/messages` | Chat completion (main) |
| `/v1/messages` + `stream: true` | Streaming |
| `/v1/messages/batches` | Batch processing |

## Quick Start (Python)

```python
import json
import urllib.request

payload = json.dumps({
    'model': 'claude-sonnet-4-5-20250929',
    'max_tokens': 1024,
    'system': 'You are helpful.',  # system is top-level, NOT in messages
    'messages': [
        {'role': 'user', 'content': user_message}
    ]
}).encode('utf-8')

req = urllib.request.Request(
    'https://api.anthropic.com/v1/messages',
    data=payload,
    headers={
        'x-api-key': api_key,
        'anthropic-version': '2023-06-01',
        'Content-Type': 'application/json'
    }
)

with urllib.request.urlopen(req) as resp:
    data = json.loads(resp.read().decode('utf-8'))
    answer = data['content'][0]['text']
```

## Response Format Differences

```json
// Anthropic response (different from OpenAI!)
{
  "content": [
    { "type": "text", "text": "Hello!" }
  ],
  "model": "claude-sonnet-4-5-20250929",
  "stop_reason": "end_turn",
  "usage": { "input_tokens": 10, "output_tokens": 5 }
}
```

## Tool Use

```python
tools = [{
    'name': 'get_weather',
    'description': 'Get weather for a city',
    'input_schema': {
        'type': 'object',
        'properties': {
            'city': {'type': 'string', 'description': 'City name'}
        },
        'required': ['city']
    }
}]
# Response: content[].type === "tool_use" -> { id, name, input }
# Follow up: role: "user", content: [{ type: "tool_result", tool_use_id, content }]
```

> [!WARNING]
> Tool response format differs from OpenAI:
> - No `tool_calls` array -- uses `content` blocks with `type: "tool_use"`
> - Tool results go as `user` message with `type: "tool_result"`, not `role: "tool"`

## Streaming

```python
payload['stream'] = True
# SSE events: message_start, content_block_start, content_block_delta, message_stop
# Delta: event.delta.text (not choices[0].delta.content)
```

## API Docs

Anthropic does NOT have `llms.txt`. Key docs:

- [Messages API](https://docs.anthropic.com/en/api/messages)
- [Streaming](https://docs.anthropic.com/en/api/streaming)
- [Tool Use](https://docs.anthropic.com/en/docs/build-with-claude/tool-use)
- [Models](https://docs.anthropic.com/en/docs/about-claude/models)
- [Vision](https://docs.anthropic.com/en/docs/build-with-claude/vision)
- [Prompt Caching](https://docs.anthropic.com/en/docs/build-with-claude/prompt-caching)
- [Extended Thinking](https://docs.anthropic.com/en/docs/build-with-claude/extended-thinking)
- [Structured Outputs](https://docs.anthropic.com/en/docs/build-with-claude/structured-outputs)

## Related Skills
- `ai-api` -- general AI integration patterns (Python, streaming, agent loops)
- `system-prompt-master` -- prompt engineering
